//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.ImportFileActivity.getProperties;

import android.annotation.SuppressLint;
import android.content.ActivityNotFoundException;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.AssetManager;
import android.graphics.Rect;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.text.Spanned;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowInsets;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.PopupMenu;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContract;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.constraintlayout.helper.widget.Flow;

import androidx.constraintlayout.widget.ConstraintLayout;
import androidx.core.view.MenuCompat;
import androidx.fragment.app.DialogFragment;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.ViewModel;
import androidx.lifecycle.ViewModelProvider;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import de.rainerhock.eightbitwonders.DownloaderFactory.AdditionalDownloads.DownloadException;
import io.noties.markwon.Markwon;

/**
 * This is the app's default activity users start Emulations from.
 */
public final class MainActivity extends EmulationLauncherActivity {
    private View mLastFocusedView;

    private static final String FRAGMENT_MENU_DIALOG = "FRAGMENT_MENU_DIALOG";
    private static final String TAG = MainActivity.class.getSimpleName();
    private static Serializable mDestroyedData = false;
    private final ActivityResultLauncher<String> mLaunchWebsiteContract = registerForActivityResult(
            new ActivityResultContract<String, Intent>() {
                @Override
                public Intent parseResult(final int result, final @Nullable Intent intent) {
                    return intent;
                }

                @NonNull
                @Override
                public Intent createIntent(final @NonNull Context context, final String s) {
                    return BrowserActivity.createArchiveOrgIntent(MainActivity.this, s,
                            mStreamingURLS);
                }
            }, result -> {
            });
    private final ActivityResultLauncher<Intent> mOpenLocalContentLauncher
            = registerForActivityResult(new FileCreateionContract(),
            result -> {
                boolean success = false;
                if (result != null && result.getData() != null) {
                    try {
                        InputStream is = getContentResolver().openInputStream(
                                result.getData());
                        if (is != null) {
                            Map<String, ByteArrayOutputStream> contents = new HashMap<>();
                            Map<String, ByteArrayOutputStream> savestates = new HashMap<>();
                            Properties props = ImportFileActivity.checkProperties(is);
                            if (props != null && props.containsKey("name")
                                    && props.containsKey("emulation") && props.containsKey("id")) {
                                is.close();
                                is = getContentResolver().openInputStream(
                                        result.getData());
                                props = getProperties(this, is, savestates, contents);
                                storeZip(contents, props, savestates);
                                success = true;
                            } else if (Rp9Configuration.isRp9File(
                                    getContentResolver().openInputStream(result.getData()))) {
                                is.close();
                                is = getContentResolver().openInputStream(
                                        result.getData());
                                if (is != null) {
                                    byte[] data = new byte[is.available()];
                                    //noinspection ResultOfMethodCallIgnored
                                    is.read(data);
                                    new Rp9Configuration(this, data).add();
                                    addLocalConfigurations();
                                    success = true;
                                }
                            } else {
                                success = false;
                            }
                        }
                        updateUsercontentUI();
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    } catch (Rp9Configuration.Rp9ImportException e) {
                        showErrorDialog(getResources().getString(R.string.import_local_content),
                                e.getMessage());
                    }
                    if (!success) {
                        Toast.makeText(this, R.string.import_8bw_only,
                                Toast.LENGTH_LONG).show();
                        openLocalContent(true);
                    }
                }
            });
    private final LinkedList<String> mStreamingURLS = new LinkedList<>();
    private final ActivityResultLauncher<Intent> mGlobalSettingsLauncher
            = registerForActivityResult(new ActivityResultContract<Intent, Intent>() {

                @Override
                public Intent parseResult(final int result, final @Nullable Intent intent) {
                    return intent;
                }

                @NonNull
                @Override
                public Intent createIntent(final @NonNull Context context,
                                           final Intent intent) {
                    return intent;
                }
            },
            result -> {
                if (result != null && result.getBooleanExtra("restored", false)) {
                    mStreamingURLS.clear();
                    mStreamingURLS.addAll(addLocalConfigurations());
                    updateUsercontentUI();
                }
            });

    @Override
    ActivityResultLauncher<Intent> getEmulationContract() {
        return registerForActivityResult(
                new ActivityResultContract<Intent, Intent>() {

                    @Override
                    public Intent parseResult(final int i, final @Nullable Intent intent) {
                        return intent;
                    }

                    @NonNull
                    @Override
                    public Intent createIntent(final @NonNull Context context,
                                               final Intent intent) {
                        return intent;
                    }
                },
                result -> {
                    EmulationLauncherActivity.setEmulationRunning(false);
                    if (result != null) {
                        if (result.getStringExtra("error-message") != null) {
                            showErrorDialog(null, result.getStringExtra("error-message"));
                        }
                    }
                    mStreamingURLS.clear();
                    mStreamingURLS.addAll(addLocalConfigurations());
                    updateUsercontentUI();
                });
    }

    public static final class MainViewModel extends ViewModel {
        private boolean mRefreshRequired = false;
        private final Set<String> mHandledExceptions = new HashSet<>();
        private DownloadException mDownloadException = null;
        private EmulationConfiguration mConfiguration = null;

        boolean isExceptionHandled(final String id) {
            return mHandledExceptions.contains(id);
        }
        void setExceptionHandled(final String id) {
            mHandledExceptions.add(id);
        }
        void requestRefresh() {
            mRefreshRequired = true;
        }
        boolean isRefreshRequired() {
            return mRefreshRequired;
        }
        void setDownloadException(final DownloadException e) {
            mDownloadException = e;
        }
        DownloadException getDownloadException() {
            return mDownloadException;
        }

        void setConfiguration(final EmulationConfiguration conf) {
            mConfiguration = conf;
        }
        EmulationConfiguration getConfiguration() {
            return mConfiguration;
        }

    }

    private BroadcastReceiver mReceiver = null;
    public static class UpdateBroadcastReceiver extends BroadcastReceiver {
        private final MainActivity mActivity;

        @Override
        public final void onReceive(final Context context, final Intent intent) {
            MainViewModel viewModel = new ViewModelProvider(mActivity).get(MainViewModel.class);
            if (mActivity.getLifecycle().getCurrentState().isAtLeast(Lifecycle.State.RESUMED)) {
                mActivity.mStreamingURLS.clear();
                mActivity.mStreamingURLS.addAll(mActivity.addLocalConfigurations());
                mActivity.updateUsercontentUI();
            } else {
                viewModel.requestRefresh();
            }
        }
        UpdateBroadcastReceiver(final @NonNull MainActivity activity) {
            super();
            mActivity = activity;

        }
    }
    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (mReceiver != null) {
            unregisterReceiver(mReceiver);
            mReceiver = null;
        }
    }
    void removeStreamingUrl(final String url) {
        mStreamingURLS.remove(url);
        updateUsercontentUI();
    }

    @Override
    protected void onPause() {
        super.onPause();
        mLastFocusedView = findViewById(android.R.id.content).findFocus();
    }
    @Override
    protected void onResume() {
        super.onResume();
        Log.v("openInfoMenu", "set listener");
        MainViewModel viewModel = new ViewModelProvider(this).get(MainViewModel.class);
        if (viewModel.isRefreshRequired()) {
            mStreamingURLS.clear();
            mStreamingURLS.addAll(addLocalConfigurations());
        }
        updateUsercontentUI();
        final View focusView;
        if (mLastFocusedView != null && mLastFocusedView.getContext() == this) {
            Log.v("FocusTrace", mLastFocusedView + " is not dead (1)");
            focusView = mLastFocusedView;
        } else if (mFirstTile != null && mFirstTile.getContext() == this) {
            focusView = mFirstTile;
            Log.v("FocusTrace", mFirstTile + " is not dead (2)");
        } else {
            focusView = null;
        }
        if (focusView != null) {
            focusView.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                @Override
                //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
                public void onLayoutChange(final View view, final int i, final int i1,
                                           final int i2, final int i3, final int i4,
                                           final int i5, final int i6, final int i7) {
                    Log.v("FocusTrace", "onLayoutChanged");
                    focusView.post(() -> {
                        Log.v("FocusTrace", focusView + "post");
                        focusView.requestFocus();
                        Log.v("FocusTrace", "done");

                    });
                    focusView.removeOnLayoutChangeListener(this);
                    Log.v("FocusTrace", "onLayoutChanged done");
                }
            });
        }


    }
    private static final Map<Integer, Integer> HOMEPAGE_URLS = new HashMap<Integer, Integer>() {{
        put(R.id.about, R.string.app_url);
        put(R.id.manual, R.string.manual_url);
        put(R.id.changelog, R.string.changelog_url);
        put(R.id.feedback, R.string.feedback_url);
        put(R.id.blog, R.string.blog_url);
        put(R.id.license_and_info, R.string.license_url);
        put(R.id.privacy, R.string.privacy_url);
        put(R.id.video_tutorials, R.string.tutorials_url);

    }};
    private void prepareMenu(final Menu m) {
        final String rootUrl = getResources().getString(R.string.app_url);
        for (int i = 0; i < m.size(); i++) {
            Menu submenu = m.getItem(i).getSubMenu();
            if (submenu != null) {
                prepareMenu(submenu);
            }

            if (HOMEPAGE_URLS.containsKey(m.getItem(i).getItemId())) {
                Integer id = HOMEPAGE_URLS.get(m.getItem(i).getItemId());
                String title = Objects.requireNonNull(m.getItem(i).getTitle()).toString();
                if (id != null) {
                    final String url = getResources().getString(id);
                    if (url.startsWith(rootUrl)) {
                        m.getItem(i).setOnMenuItemClickListener(mi -> {
                            startActivity(BrowserActivity.createIntent(this,
                                    title, url, rootUrl, new String[]{".d64"}));
                            return true;
                        });
                    } else {
                        m.getItem(i).setOnMenuItemClickListener(mi -> {
                            try {
                                Intent intent = new Intent(Intent.ACTION_VIEW);
                                intent.setData(Uri.parse(url));
                                startActivity(intent);
                            } catch (ActivityNotFoundException e) {
                                Toast.makeText(this, R.string.cannot_open_app,
                                        Toast.LENGTH_LONG).show();
                            }
                            return true;
                        });
                    }
                }
            }
        }
        m.findItem(R.id.global_settings).setOnMenuItemClickListener(mi -> {
            if (mi.getItemId() == R.id.global_settings) {
                mGlobalSettingsLauncher.launch(new Intent(this, GlobalSettingsActivity.class));
                return true;
            }
            return false;
        });

    }


    private void openLocalContent(final boolean retry) {
        Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.putExtra(Intent.EXTRA_TITLE, R.string.import_local_content);
        intent.setType("application/*");
        if (retry) {
            intent.putExtra("retry", true);
        }
        mOpenLocalContentLauncher.launch(intent);
    }

    private PopupMenu createInfoMenu(final View anchor) {
        PopupMenu pm = new PopupMenu(MainActivity.this, anchor);
        pm.getMenuInflater().inflate(R.menu.menu_info, pm.getMenu());
        prepareMenu(pm.getMenu());
        MenuCompat.setGroupDividerEnabled(pm.getMenu(), true);
        return pm;
    }
    void openInfoDialog(final View view, final boolean calledWithJoystick) {
        if (getSupportFragmentManager()
                .findFragmentByTag(FRAGMENT_MENU_DIALOG) == null) {
            if (calledWithJoystick) {
                BaseActivity.showGamepadMenu(view, createInfoMenu(null).getMenu());
            } else {
                openInfoMenu(view);
            }
        }

    }

    void openInfoMenu(final View view) {
        createInfoMenu(view).show();
    }
    private TileView mFirstTile = null;
    void addEmulationTile(final EmulationConfiguration conf, final boolean first,
                                  final boolean builtin) {
        boolean preinstalled = (conf.isBareMachine()
                || (!conf.isLocallyGenerated() && !conf.isImported())) && !conf.isStream();
        final ViewGroup vg = findViewById(preinstalled ? R.id.preinstalled : R.id.usercontent);
        final GroupHeader gh = findViewById(preinstalled
                ? R.id.gh_preinstalled : R.id.gh_usercontent);
        final Flow flow = findViewById(preinstalled ? R.id.fl_preinstalled : R.id.fl_usercontent);
        final TileView tv;
        if (vg.findViewWithTag(conf.getEmulatorId() + "-" + conf.getId()) == null) {
            Emulationslist.Description desc =
                    Emulationslist.getDescription(conf.getEmulatorId());
            String name = conf.getName();
            if (conf.isBareMachine()) {
                tv = TileView.createMachinePreview(getLayoutInflater(), vg, name,
                        Objects.requireNonNull(desc).getImageResourceId(),
                        getResources().getColor(desc.getColorResourceId()));
            } else {
                tv = TileView.createConfigurationPreview(getLayoutInflater(), vg, name,
                        Objects.requireNonNull(desc).getImageResourceId(), conf.getBitmap());
            }
            tv.setTag(conf.getEmulatorId() + "-" + conf.getId());
            tv.setListener((isJoystick, showExtendedAction) -> {
                if (showExtendedAction) {
                    tv.showExtendedOption(this, conf, isJoystick);

                } else {
                    if (!tv.isPopupOpen() && getSupportFragmentManager()
                            .findFragmentByTag(FRAGMENT_MENU_DIALOG) == null) {
                        startEmulation(conf);
                    }
                }
            });
            tv.setCopy(conf instanceof TileView.CopiedConfiguration);
            tv.setBuiltin(builtin);
            tv.setId(View.generateViewId());
            vg.addView(tv);
            flow.addView(tv);

        } else {
            tv = vg.findViewWithTag(conf.getEmulatorId() + "-" + conf.getId());
            tv.setBitmap(conf.getBitmap());
        }
        if (first && vg.getVisibility() == View.VISIBLE) {
            mFirstTile = tv;
        }
        tv.setOnFocusChangeListener((v, hasFocus) -> {
            if (hasFocus) {
                gh.getChildAt(0).setNextFocusDownId(v.getId());
                findViewById(R.id.bn_info).setNextFocusDownId(v.getId());
            }
        });
        if (builtin && new Useropts(this).getBooleanValue(
                "SHOW_IN_USERCONTENT_" + conf.getEmulatorId() + "_" + conf.getId(),
                false)) {
            addEmulationTile(new TileView.CopiedConfiguration(conf), false, false);
        }
    }
    private void fixAtRuntime(final ConstraintLayout cl) {
        View previous = null;
        List<View> toDelete = new LinkedList<>();
        for (int i = 0; i < cl.getChildCount(); i++) {
            View child = cl.getChildAt(i);
            if (!(child instanceof Flow) && child.getVisibility() != View.VISIBLE) {
                toDelete.add(child);
            }
        }
        for (View v : toDelete) {
            cl.removeView(v);
        }
        for (int i = 0; i < cl.getChildCount(); i++) {
            View child = cl.getChildAt(i);
            if (!(child instanceof Flow)) {
                if (previous != null) {
                    if (child.getId() == View.NO_ID) {
                        child.setId(View.generateViewId());
                    }
                    previous.setNextFocusRightId(child.getId());
                    child.setNextFocusLeftId(previous.getId());
                }
                previous = child;
            }
        }
        cl.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
            @Override
            // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
            public void onLayoutChange(final View v, final int left, final int top,
                                       final int right, final int bottom,
                                       final int oldLeft, final int oldTop,
                                       final int oldRight, final int oldBottom) {

                cl.removeOnLayoutChangeListener(this);
                int parentwidth = right - left;
                int childrenwidth = 0;
                int childwidth = 0;
                Flow flow = null;
                for (int i = 0; i < cl.getChildCount(); i++) {
                    View child = cl.getChildAt(i);
                    if (cl.getChildAt(i) instanceof Flow) {
                        flow = (Flow) child;
                    } else {
                        if (childwidth == 0) {
                            childwidth = child.getMeasuredWidth();
                        }
                        childrenwidth += childwidth;
                    }
                }
                ViewGroup.LayoutParams lp = new ViewGroup.LayoutParams(childwidth, 1);

                while (childwidth > 0 && flow != null && childrenwidth < parentwidth) {
                    View filler = new View(v.getContext());
                    filler.setLayoutParams(lp);
                    filler.setVisibility(View.INVISIBLE);
                    filler.setId(View.generateViewId());
                    childrenwidth += childwidth;
                    cl.addView(filler);
                    flow.addView(filler);
                }
            }
        });
    }
    private void setUseropts(final GroupHeader gh, final String key) {
        Useropts opts = new Useropts(this);
        gh.setViewVisible(opts.getBooleanValue(key, true));
        gh.setStateChangedListener((view, isOpen)
                        -> opts.setValue(Useropts.Scope.GLOBAL, key, isOpen));

    }
    @Override
    void updateUsercontentUI() {
        int visibility = View.GONE;

        ViewGroup vg = findViewById(R.id.usercontent);
        for (int i = 0; i < vg.getChildCount(); i++) {
            if (vg.getChildAt(i) instanceof TileView) {
                visibility = View.VISIBLE;
                break;
            }
        }
        findViewById(R.id.gh_usercontent).setVisibility(visibility);
        findViewById(R.id.usercontent).setVisibility(
                ((GroupHeader) findViewById(R.id.gh_usercontent)).isViewVisible()
                        ? visibility : View.GONE);
        fixAtRuntime(findViewById(R.id.usercontent));

    }

    @SuppressLint("UnspecifiedRegisterReceiverFlag")
    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        // Needed for starts from android studio.
        if (BuildConfig.DEBUG) {
            updateWidgets();
        }
        mReceiver = new UpdateBroadcastReceiver(this);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            registerReceiver(mReceiver,
                    new IntentFilter("de.rainerhock.eightbitwonders.update"),
                    Context.RECEIVER_NOT_EXPORTED);
        } else {
            registerReceiver(mReceiver,
                    new IntentFilter("de.rainerhock.eightbitwonders.update"));

        }
        setContentView(R.layout.activity_main);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT_WATCH) {
            findViewById(R.id.root).setOnApplyWindowInsetsListener(
                    new View.OnApplyWindowInsetsListener() {
                @NonNull
                @Override
                public WindowInsets onApplyWindowInsets(final @NonNull View v,
                                                        final @NonNull WindowInsets insets) {
                    LinearLayout.LayoutParams lp =
                            (LinearLayout.LayoutParams) findViewById(R.id.titlebar)
                                    .getLayoutParams();
                    lp.topMargin = insets.getSystemWindowInsetTop();
                    findViewById(R.id.titlebar).setLayoutParams(lp);
                    return insets;
                }
            });
        }
        findViewById(R.id.bn_info).setOnClickListener(this::openInfoMenu);
        findViewById(R.id.bn_info).setOnKeyListener((view, keycode, keyEvent) -> {
            if (keyEvent.getAction() == KeyEvent.ACTION_UP
                    && EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)) {
                openInfoDialog(view, keycode != KeyEvent.KEYCODE_DPAD_CENTER);
                return true;
            }
            return false;
        });

        setCBMFont(findViewById(R.id.tv_app_title));
        showOneTimeMessage("show_manual_hint", R.string.welcome, R.string.manual_hint, null);

        showChangelogMessage();

        if (EmulationLauncherActivity.isEmulationRunning()
                && Emulationslist.getCurrentEmulation() != null) {
            String pkg = getApplication().getPackageName();
            Intent intent = new Intent(this, EmulationActivity.class);
            intent.putExtra(pkg + ".Relaunch", true);

            if (mDestroyedData != null) {
                intent.putExtra(pkg + ".ViewModel", mDestroyedData);
            }
            launchEmulation(intent);
        }
        boolean first = true;
        for (EmulationConfiguration conf : getMachineConfigurations()) {
            addEmulationTile(conf, first, true);
            first = false;
        }

        addAssets(getResources().getAssets(), "packages", null);
        mStreamingURLS.clear();
        mStreamingURLS.addAll(addLocalConfigurations());
        updateUsercontentUI();

        if (getIntent().getBooleanExtra("recovery", false)) {
            String id = getIntent().getStringExtra("id");
            MainViewModel viewModel = new ViewModelProvider(this).get(MainViewModel.class);
            if (!viewModel.isExceptionHandled(id)) {
                recoverEmulation((EmulationConfiguration) getIntent().getSerializableExtra(
                        "configuration"));
                viewModel.setExceptionHandled(id);
            }

        }
        setUseropts(findViewById(R.id.gh_preinstalled), "SHOW_PREINSTALLED");
        setUseropts(findViewById(R.id.gh_usercontent), "SHOW_USERONTENT");
        setUseropts(findViewById(R.id.gh_imported), "SHOW_IMPORT");
        ConstraintLayout l = findViewById(R.id.imported);
        Flow f = findViewById(R.id.fl_imported);
        updateUsercontentUI();
        ContentSourceView csvImport = new ContentSourceView(this,
                R.drawable.ic_import, 0,
                getResources().getString(R.string.import_local_content),
                getResources().getString(R.string.description_import_local_content));
        csvImport.setId(View.generateViewId());
        csvImport.setAction(() -> executeWithFilesystemAccess(
                null, () -> openLocalContent(false), null,
                () -> openLocalContent(false)));
        l.addView(csvImport);
        f.addView(csvImport);
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            addArchiveOrgSource(new ArchiveOrgAccess.ArchiveOrgC64LibraryAdapter(),
                    R.string.description_software_library_c64, R.color.c64bordercolor);
            addArchiveOrgSource(new ArchiveOrgAccess.ArchiveOrgVic20CommunitySoftware(),
                    R.string.description_software_library_vic20cartridges, R.color.vic20ordercolor);
            addArchiveOrgSource(new ArchiveOrgAccess.ArchiveOrgPetLibraryAdapter(),
                    R.string.description_software_library_pet, R.color.petbordercolor);

        }
        fixAtRuntime(findViewById(R.id.preinstalled));
        fixAtRuntime(findViewById(R.id.usercontent));
        fixAtRuntime(findViewById(R.id.imported));
    }

    private void addArchiveOrgSource(final BrowserActivity.WebContentAdapter adapter,
                                     final int description,
                                     final int background) {
        ConstraintLayout l = findViewById(R.id.imported);
        Flow f = findViewById(R.id.fl_imported);
        ContentSourceView csv = new ContentSourceView(this,
                R.drawable.archiveorg_white,
                background,
                adapter.getTitle(this),
                getResources().getString(description));

        csv.setAction(() -> {
            if (adapter.getLicenseTextId() != View.NO_ID) {
                showOneTimeMessage("show_webcontent_hint_" + adapter.getRootURL(),
                        R.string.note, adapter.getLicenseTextId(),
                        () -> mLaunchWebsiteContract.launch(adapter.getRootURL()));
            } else {
                mLaunchWebsiteContract.launch(adapter.getRootURL());
            }
        });
        csv.setId(View.generateViewId());
        l.addView(csv);
        f.addView(csv);

    }

    public static final class ChangelogDialogFragment extends DialogFragment {
        /**
         * Android requires this constructor.
         */
        public ChangelogDialogFragment() {
            super();
        }

        private static final String CHANGELOG_CONTENT = "CHANGELOG_CONTENT";

        static void show(final MainActivity a) {
            ChangelogDialogFragment f = new ChangelogDialogFragment();
            InputStream is = a.getResources().openRawResource(R.raw.changelog);
            byte[] data;
            try {
                data = new byte[is.available()];
                if (is.read(data) >= 0) {
                    String s = new String(data);
                    Bundle b = new Bundle();
                    b.putString(CHANGELOG_CONTENT, s);
                    f.setArguments(b);
                    f.setStyle(DialogFragment.STYLE_NORMAL, R.style.AlertDialogTheme);
                    f.showNow(a.getSupportFragmentManager(), "CHANGELOG");
                    a.updateChangelogShown();

                }
                is.close();

            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            @SuppressLint("InflateParams")
            View ret =  inflater.inflate(R.layout.dialog_changelog, null);
            final Markwon markwon = Markwon.create(requireContext());
            final Spanned markdown = markwon.toMarkdown(Objects.requireNonNull(
                    requireArguments().getString(CHANGELOG_CONTENT)));
            ((TextView) ret.findViewById(R.id.tv_content)).setText(markdown);
            TextView tv = ret.findViewById(R.id.tv_title);
            tv.setVisibility(View.VISIBLE);
            tv.setText(getResources().getString(R.string.new_in_version, BuildConfig.VERSION_NAME));
            ret.findViewById(R.id.bn_close_changelog).setOnClickListener(v -> dismiss());
            ret.findViewById(R.id.bn_full_changelog).setOnClickListener(v -> {
                String url = getResources().getString(R.string.changelog_url);
                startActivity(BrowserActivity.createIntent(v.getContext(),
                        getResources().getString(R.string.manual),
                        url, getString(R.string.app_url), new String[]{}));
            });
            ret.findViewById(R.id.bn_blog).setOnClickListener(v -> {
                String url = getResources().getString(R.string.blog_url);
                startActivity(BrowserActivity.createIntent(v.getContext(),
                        getResources().getString(R.string.manual),
                        url, getString(R.string.app_url), new String[]{}));
            });

            /*

             */
            return ret;
        }

        @Override
        public void onCreate(final @Nullable Bundle savedInstanceState) {
            setStyle(DialogFragment.STYLE_NO_TITLE, R.style.AlertDialogTheme);
            super.onCreate(savedInstanceState);
        }

    }

    private void showChangelogMessage() {
        if (!BuildConfig.VERSION_NAME.equals(
                new Useropts(this).getStringValue(CHANGELOG_SHOWN, "v0"))) {
            Log.v(TAG, "showChangelogMessage " + BuildConfig.VERSION_NAME + " vs "
                    + new Useropts(this).getStringValue(CHANGELOG_SHOWN, "v0"));

            ChangelogDialogFragment.show(this);

        }
    }


    private void addAssets(final AssetManager am, final String assetFolder,
                           final String filesystemFolder) {
        for (EmulationConfiguration configuration
                : getAssetConfigurations(am, assetFolder, filesystemFolder)) {
            addEmulationTile(configuration, false, true);
            removeInitialSnapshots(configuration);
        }

    }
    static final String CHANGELOG_SHOWN = "CHANGELOG_SHOWN";

    static void setDestroyedData(final Serializable data) {
        mDestroyedData = data;
    }

    private List<String> addLocalConfigurations() {
        List<String> configurations = new LinkedList<>();
        List<String> foundTileTags = new LinkedList<>();
        for (EmulationConfiguration configuration : getLocalConfigurations()) {
            if (configuration instanceof ConfigurationFactory.StreamConfiguration) {
                if (configuration.getWebAdress() != null) {
                    configurations.add(configuration.getWebAdress().toString());
                } else {
                    configuration.getDeinstaller().run();
                }
            }
            if (configuration != null) {
                foundTileTags.add(configuration.getEmulatorId()
                        + "-" + configuration.getId());
                addEmulationTile(configuration, false, false);
            }
        }
        ViewGroup root = findViewById(R.id.usercontent);
        List<View> toRemove = new LinkedList<>();
        for (int i = 0; i < root.getChildCount(); i++) {
            if (root.getChildAt(i) instanceof TileView) {

                TileView tv = (TileView) root.getChildAt(i);
                if (!tv.isCopy() && !foundTileTags.contains(tv.getTag().toString())
                        && !tv.isBuiltin()) {
                    toRemove.add(tv);
                }
            }
        }
        for (View v : toRemove) {
            ((Flow) findViewById(R.id.fl_usercontent)).removeView(v);
            ((ViewGroup) findViewById(R.id.usercontent)).removeView(v);

        }
        makefilenamesUnique();
        return configurations;
    }

    @Override
    protected void onFileOpened(final Uri uri) {
        MainViewModel viewModel = new ViewModelProvider(this).get(MainViewModel.class);
        try {
            byte[] bytes;
            byte[] content = null;
            File zipFile = null;
            boolean hashCorrect = false;
            InputStream inputStream = getContentResolver().openInputStream(uri);
            if (inputStream != null) {
                if (viewModel.getDownloadException().getZipURL() != null) {
                    byte[] buffer = new byte[CHUNK_SIZE];

                    File dir = new File(getCacheDir(), "additional_downloads");
                    if (dir.isDirectory() || dir.mkdirs()) {
                        zipFile = new File(dir, new File(viewModel.getDownloadException()
                                .getZipURL().getPath()).getName());
                        //noinspection ResultOfMethodCallIgnored
                        zipFile.createNewFile();
                        FileOutputStream fos = new FileOutputStream(zipFile);
                        int bytesRead;
                        while ((bytesRead = inputStream.read(buffer)) >= 0) {
                            fos.write(buffer, 0, bytesRead);
                        }
                        inputStream.close();
                        fos.close();
                        //bytes = null;
                        ZipFile zip = new ZipFile(zipFile);
                        ZipEntry ze = zip.getEntry(new File(viewModel.getDownloadException()
                                .getPath()).getName());
                        if (ze != null) {
                            InputStream is = zip.getInputStream(ze);
                            buffer = new byte[CHUNK_SIZE];
                            ByteArrayOutputStream baos = new ByteArrayOutputStream();

                            while ((bytesRead = is.read(buffer)) >= 0) {
                                baos.write(buffer, 0, bytesRead);
                            }

                            content = baos.toByteArray();
                            baos.close();
                        }
                        zip.close();
                    }

                } else {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    byte[] buffer = new byte[CHUNK_SIZE];
                    int bytesRead;

                    while ((bytesRead = inputStream.read(buffer)) >= 0) {
                        baos.write(buffer, 0, bytesRead);
                    }
                    bytes = baos.toByteArray();
                    content = bytes;
                }
                for (String hash : viewModel.getDownloadException().getHashes()) {
                    if (DownloaderFactory.AdditionalDownloads.isHashCorrect(content, hash)) {
                        hashCorrect = true;
                        File dir = new File(viewModel.getDownloadException().getPath())
                                .getParentFile();
                        //noinspection ConstantConditions
                        if (!dir.exists()) {
                            //noinspection ResultOfMethodCallIgnored
                            dir.mkdirs();
                        }
                    }
                }
            }
            if (hashCorrect) {
                DownloaderFactory.AdditionalDownloads.writeLocalData(
                        viewModel.getDownloadException(), content);
                startEmulation(viewModel.getConfiguration());
            } else {
                if (zipFile != null) {
                    //noinspection ResultOfMethodCallIgnored
                    zipFile.delete();
                }
                AlertDialogBuilder builder = new AlertDialogBuilder(this);
                //noinspection ConstantConditions
                final byte[] content2 = content;
                builder.setTitle(R.string.check_file_title)
                        .setMessage(getResources().getString(R.string.check_file_message,
                                FileUtil.getFileName(this, uri),
                                uri.getPath(),
                                viewModel.getDownloadException().getUserfriendlyPath()))
                        .setPositiveButton(R.string.yes, (dialogInterface, i) -> {
                            try {
                                DownloaderFactory.AdditionalDownloads
                                        .writeLocalData(viewModel.getDownloadException(),
                                                content2);
                            } catch (IOException e) {
                                handleDownloadError(viewModel.getConfiguration(),
                                        viewModel.getDownloadException(),
                                        () -> startEmulation(viewModel.getConfiguration()));
                            }
                            dialogInterface.dismiss();
                            startEmulation(viewModel.getConfiguration());
                        })
                        .setNeutralButton(R.string.choose_other, (dialogInterface, i) -> {
                            dialogInterface.dismiss();
                            openLocalFile();
                        })
                        .setOnDismissListener(DialogInterface::dismiss);
                AlertDialogBuilder.showStyledDialog(builder.create());
            }
        } catch (NullPointerException | IOException e) {
            handleDownloadError(viewModel.getConfiguration(),
                    viewModel.getDownloadException(),
                    () -> startEmulation(viewModel.getConfiguration()));
        }
        super.onFileOpened(uri);
    }

    @Override
    protected void onFileOpenCancelled(final boolean wasRetry) {
        MainViewModel viewModel = new ViewModelProvider(this).get(MainViewModel.class);
        handleDownloadError(viewModel.getConfiguration(), viewModel.getDownloadException(),
                () -> startEmulation(viewModel.getConfiguration()));
        super.onFileOpenCancelled(wasRetry);
    }

    private void recoverEmulation(final EmulationConfiguration conf) {
        Intent intent = createStartEmulationIntent(conf);
        intent.putExtra("recovery", true);
        EmulationLauncherActivity.setEmulationRunning(true);
        launchEmulation(intent);

    }

    void setTestConfigurationFolder(final AssetManager am, final String testAssetPath) {
        runOnUiThread(() -> addAssets(am, testAssetPath, "androidTest"));

    }
    public static final class RenameConfigurationFragment extends DialogFragment {
        /**
         * Must be public to allow Android to access it.
         */
        public RenameConfigurationFragment() {
            super();
        }

        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            if (getDialog() != null && getDialog().getWindow() != null) {
                getDialog().getWindow().setSoftInputMode(
                        WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);
            }
        }

        @Override
        public void onCreate(final @Nullable Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
        }

        @NonNull
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_renamepackage, null);
            EmulationConfiguration conf = (EmulationConfiguration) requireArguments()
                    .getSerializable("configuration");
            assert conf != null;
            Button pbOk = root.findViewById(R.id.bn_ok);
            EditText etName = root.findViewById(R.id.tv_name);
            etName.addTextChangedListener(createWatcher(
                    s -> pbOk.setEnabled(!s.isEmpty())));
            etName.setText(conf.getName());
            etName.setSelection(conf.getName().length());
            pbOk.setOnClickListener(view -> {
                String newName = String.valueOf(etName.getText());
                ConfigurationFactory.renameConfiguration(
                        conf, newName);
                View tv = requireActivity()
                        .findViewById(R.id.usercontent)
                        .findViewWithTag(conf.getEmulatorId() + "-" + conf.getId());
                if (tv instanceof TileView) {
                    ((TextView) tv.findViewById(R.id.tv_name)).setText(newName);
                }
                ((BaseActivity) requireActivity()).updateWidgets();
                dismiss();
            });

            return root;
        }
    }

    @Override
    protected void onNewIntent(final Intent intent) {
        super.onNewIntent(intent);
    }
    @Override
    protected void onContentImported(final String name) {
        mStreamingURLS.clear();
        mStreamingURLS.addAll(addLocalConfigurations());
        updateUsercontentUI();
        Rect textRect = new Rect(); //coordinates to scroll to
        ArrayList<View> views = new ArrayList<>();
        findViewById(R.id.usercontent).findViewsWithText(views, name, View.FIND_VIEWS_WITH_TEXT);
        if (!views.isEmpty() && views.get(0).getParent() instanceof View) {
            findViewById(R.id.usercontent).post(() -> {
                View target = (View) views.get(0).getParent();
                target.getHitRect(textRect);
                ((ViewGroup) findViewById(R.id.scrollview)).requestChildRectangleOnScreen(
                        target, textRect, false);
            });
        }
    }
    private void makefilenamesUnique() {
        ViewGroup parent = findViewById(R.id.usercontent);
        Map<String, Integer> counter = new LinkedHashMap<>();
        for (int c = 0; c < parent.getChildCount(); c++) {
            View view = parent.getChildAt(c);
            if (view instanceof TileView) {
                TileView tv = (TileView) view;
                String name = tv.getName();
                Integer others = 0;
                if (counter.containsKey(name)) {
                    others = counter.get(name);
                    //noinspection DataFlowIssue
                    counter.put(name, others + 1);
                } else {
                    counter.put(name, 1);
                }
                tv.setPosition(others);
            }
        }
    }
}
