//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.view.KeyEvent;

import java.text.Normalizer;
import java.util.Objects;

final class KeycodeHelper {

    private KeycodeHelper() {
    }

    static KeyEvent eventFromKey(final KeyEvent src, final int action, final String s) {
        int code;
        int metastate = 0;
        switch (s) {
            case "`":
                code = KeyEvent.KEYCODE_GRAVE;
                metastate = KeyEvent.META_SHIFT_ON;
                break;
            case "ˋ":
                code = KeyEvent.KEYCODE_GRAVE;
                break;
            case "´":
                //noinspection DuplicateBranchesInSwitch
                code = KeyEvent.KEYCODE_GRAVE;
                metastate = KeyEvent.META_SHIFT_ON;
                break;
            case "§":
                code = KeyEvent.KEYCODE_S;
                metastate = KeyEvent.META_SYM_ON;
                break;
            case "ü":
            case "Ü":
                code = KeyEvent.KEYCODE_U;
                metastate = Emulation.META_UMLAUT;
                break;
            case "ö":
            case "Ö":
                code = KeyEvent.KEYCODE_O;
                metastate = Emulation.META_UMLAUT;
                break;
            case "ä":
            case "Ä":
                code = KeyEvent.KEYCODE_A;
                metastate = Emulation.META_UMLAUT;
                break;

            default:
                return null;
        }
        return new KeyEvent(src.getDownTime(), src.getEventTime(),
                action, code,
                src.getRepeatCount(), metastate);
    }
    static boolean translateKeyEvent(final KeyEvent event,
                                      final Emulation.HardwareKeyboardFunctions listener) {
        int action = event.getAction();
        String characters = event.getCharacters();
        if (action == KeyEvent.ACTION_DOWN) {
            return listener.onKeyDown(event);
        }
        if (action == KeyEvent.ACTION_UP) {
            return listener.onKeyUp(event);
        }
        if (action == KeyEvent.ACTION_MULTIPLE) {
            boolean ret = true;

            if (characters != null) {
                StringBuilder targetString = new StringBuilder();
                for (int i = 0; i < characters.length(); i++) {
                    @SuppressWarnings("RegExpDuplicateCharacterInClass")
                    String normalized = Normalizer.normalize(String.valueOf(
                            characters.charAt(i)), Normalizer.Form.NFD)
                            .replaceAll("[\\u0300|\\u0301|\\u0302|\\u0303]",
                                    "");
                    String clean = Normalizer.normalize(normalized, Normalizer.Form.NFC);
                    KeyEvent newEvent = eventFromKey(event, KeyEvent.ACTION_DOWN, clean);
                    if (newEvent != null) {
                        translateKeyEvent(newEvent, listener);
                        translateKeyEvent(
                                Objects.requireNonNull(
                                        eventFromKey(event, KeyEvent.ACTION_UP, clean)), listener);
                    } else {
                        targetString.append(clean);
                    }
                }
                KeyEvent[] events = event.getKeyCharacterMap()
                        .getEvents(targetString.toString().toCharArray());
                if (events != null) {
                    for (KeyEvent e : events) {
                        ret = ret && translateKeyEvent(e, listener);
                    }
                }
            }
            return ret;
        }
        return false;
    }
}
