//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------



package de.rainerhock.eightbitwonders;


import android.content.Context;
import android.graphics.drawable.LayerDrawable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * View for the direction view of a virtual Joystick.
 */
public final class JoystickStickView extends JoystickDirectionView<TouchJoystick>
        implements TouchDisplayRelativeLayout.TouchDisplayElement {

    /**
     * Simple constructor to use when creating a JoystickStickView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public JoystickStickView(final Context context) {
        super(context);
    }
    /**
     * Constructor that is called when inflating a JoystickStickView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickStickView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickStickView(final Context context,
                             final AttributeSet attrs,
                             final int defStyle) {
        super(context, attrs, defStyle);
    }
    private static final List<Integer> DIAGONALS_INDICATORS =
            Arrays.asList(R.id.diagonal_indicator_1, R.id.diagonal_indicator_2,
                    R.id.diagonal_indicator_3, R.id.diagonal_indicator_4);
    private static final List<Integer> ALWAYS_VISIBLE =
            Arrays.asList(R.id.joystick1, R.id.joystickcentered,
                    View.NO_ID);
    private void setFixedDrawables() {
        LayerDrawable ld = (LayerDrawable) getDrawable();
        for (int i = 0; i < ld.getNumberOfLayers(); i++) {
            boolean visibility;
            if (DIAGONALS_INDICATORS.contains(ld.getId(i))) {
                visibility = !getJoystick().isWithLockedDiagonals();
            } else {
                visibility = ALWAYS_VISIBLE.contains(ld.getId((i)));
            }
            ld.getDrawable(i).setAlpha(visibility ? OPAQUE : TRANSPARENT);
        }

    }

    @Override
    void setJoystick(final TouchJoystick device) {
        super.setJoystick(device);
        setFixedDrawables();
    }

    @Override
    protected void initcontrol() {

        super.initcontrol();
        setTag("joystickcentered");

    }
    static {
        STICKIMAGES.put(R.id.joystickcentered, Collections.singletonList(Directionvalue.CENTERED));
        STICKIMAGES.put(R.id.joysticknorthwest,
                Collections.singletonList(Directionvalue.NORTHWEST));
        STICKIMAGES.put(R.id.joysticknortheast,
                Collections.singletonList(Directionvalue.NORTHEAST));
        STICKIMAGES.put(R.id.joysticksouthwest,
                Collections.singletonList(Directionvalue.SOUTHWEST));
        STICKIMAGES.put(R.id.joysticksoutheast,
                Collections.singletonList(Directionvalue.SOUTHEAST));

        STICKIMAGES.put(R.id.joysticknorth, Arrays.asList(
                Directionvalue.NORTH, Directionvalue.NORTHEAST, Directionvalue.NORTHWEST));
        STICKIMAGES.put(R.id.joystickeast, Arrays.asList(
                Directionvalue.EAST, Directionvalue.NORTHEAST, Directionvalue.SOUTHEAST));
        STICKIMAGES.put(R.id.joysticksouth, Arrays.asList(
                Directionvalue.SOUTH, Directionvalue.SOUTHWEST, Directionvalue.SOUTHEAST));
        STICKIMAGES.put(R.id.joystickwest, Arrays.asList(
                Directionvalue.WEST, Directionvalue.NORTHWEST, Directionvalue.SOUTHWEST));

    }
    protected void showNewDirection(final Directionvalue dv) {
        LayerDrawable ld = (LayerDrawable) getDrawable();
        //Log.v(getClass().getSimpleName(), "dv=" + dv.toString());
        StringBuilder tag = new StringBuilder();
        for (int i = 0; i < ld.getNumberOfLayers(); i++) {
            if (STICKIMAGES.containsKey(ld.getId(i))) {
                //noinspection ConstantConditions
                boolean draw = JoystickStickView.STICKIMAGES.get(ld.getId(i)).contains(dv);
                ld.getDrawable(i).setAlpha(
                        draw ? OPAQUE : TRANSPARENT);
                if (draw) {
                    if (tag.length() > 0) {
                        tag.append("|");
                    }
                    tag.append(getResources().getResourceEntryName(ld.getId(i)));
                }

            }
        }
        setTag(tag.toString());
        setLastDirection(dv);
    }


    @Override
    public void setLayoutParams(final ViewGroup.LayoutParams params) {
        super.setLayoutParams(params);
    }

    boolean isPointingNorth() {
        return JoystickStickView.DIRECTION_IS_UP.contains(getLastDirection());
    }

    boolean isPointingEast() {
        return JoystickStickView.DIRECTION_IS_RIGHT.contains(getLastDirection());
    }

    boolean isPointingSouth() {
        return JoystickStickView.DIRECTION_IS_DOWN.contains(getLastDirection());
    }

    boolean isPointingWest() {
        return JoystickStickView.DIRECTION_IS_LEFT.contains(getLastDirection());
    }
    private static final List<RangeChecker> CHECKERS_WO_DIAGONALS
            = new LinkedList<RangeChecker>() {{
        // CHECKSTYLE DISABLE MagicNumber FOR 5 LINES
        add(new RangeChecker(0, 45, Directionvalue.NORTH));
        add(new RangeChecker(45, 135, Directionvalue.EAST));
        add(new RangeChecker(135, 225, Directionvalue.SOUTH));
        add(new RangeChecker(225, 315, Directionvalue.WEST));
        add(new RangeChecker(315, 360, Directionvalue.NORTH));
        }};
        private static final List<RangeChecker> CHECKERS_WITH_DIAGONALS
            = new LinkedList<RangeChecker>() {{
        // CHECKSTYLE DISABLE MagicNumber FOR 9 LINES
        add(new RangeChecker(0, 25, Directionvalue.NORTH));
        add(new RangeChecker(25, 65, Directionvalue.NORTHEAST));
        add(new RangeChecker(65, 115, Directionvalue.EAST));
        add(new RangeChecker(115, 155, Directionvalue.SOUTHEAST));
        add(new RangeChecker(155, 205, Directionvalue.SOUTH));
        add(new RangeChecker(205, 245, Directionvalue.SOUTHWEST));
        add(new RangeChecker(245, 295, Directionvalue.WEST));
        add(new RangeChecker(295, 335, Directionvalue.NORTHWEST));
        add(new RangeChecker(335, 360, Directionvalue.NORTH));
    }};
    protected List<RangeChecker> getRanges() {
        if (getJoystick().isWithLockedDiagonals()) {
            return CHECKERS_WO_DIAGONALS;
        } else {
            return CHECKERS_WITH_DIAGONALS;
        }

    }
    @Override
    public void moveTo(final float x, final float y) {
        press(x, y);
    }
}
