//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;

import androidx.annotation.NonNull;

/**
 * View for the fire button of a virtual Joystick.
 */
public final class JoystickFireView extends JoystickElementView<VirtualJoystick>
implements TouchDisplayRelativeLayout.TouchDisplayElement {

    private VirtualJoystick mHandler = null;

    void setJoystick(@NonNull  final VirtualJoystick handler) {
        mHandler = handler;
    }

    @SuppressLint("ClickableViewAccessibility")
    protected void initcontrol() {
        setTag("unpressed");
        super.initcontrol();
        setClickable(true);
    }

    /**
     * Simple constructor to use when creating a JoystickFireView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public JoystickFireView(final Context context) {
        super(context);
    }
    /**
     * Constructor that is called when inflating a JoystickFireView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickFireView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickFireView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }

    boolean isButtonPressed() {
         return mPressed;
    }

    private boolean mPressed = false;
    static final int[] STATES_PRESSED = new int[]{android.R.attr.state_pressed};
    static final int[] STATES_UNPRESSED = new int[]{};
    private void setFirePressed(final boolean pressed) {
        if (mPressed != pressed) {
            mPressed = pressed;
            mHandler.onChanged();
            getDrawable().setState(pressed ? STATES_PRESSED : STATES_UNPRESSED);
            setTag(pressed ? "pressed" : "unpressed");
        }

    }
    @Override
    public void press(final float x, final float y) {
        setFirePressed(true);
    }

    @Override
    public void release() {
        setFirePressed(false);
    }

    @Override
    public void moveTo(final float x, final float y) {
        press(x, y);
    }
}
