//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.util.AttributeSet;
import android.util.Log;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

abstract class JoystickDirectionView<T extends Joystick>
        extends JoystickElementView<T> implements TouchDisplayRelativeLayout.TouchDisplayElement {
    protected enum Directionvalue { CENTERED, NORTH, NORTHWEST, WEST,
        SOUTHWEST, SOUTH, SOUTHEAST, EAST, NORTHEAST
    }
    protected static final Map<Integer, List<Directionvalue>> STICKIMAGES = new LinkedHashMap<>();
    protected static final List<Directionvalue> DIRECTION_IS_LEFT = Arrays.asList(
            Directionvalue.NORTHWEST,
            Directionvalue.WEST,
            Directionvalue.SOUTHWEST);
    protected static final List<Directionvalue> DIRECTION_IS_RIGHT = Arrays.asList(
            Directionvalue.NORTHEAST,
            Directionvalue.EAST,
            Directionvalue.SOUTHEAST);
    protected static final List<Directionvalue> DIRECTION_IS_UP = Arrays.asList(
            Directionvalue.NORTHWEST,
            Directionvalue.NORTH,
            Directionvalue.NORTHEAST);
    protected static final List<Directionvalue> DIRECTION_IS_DOWN = Arrays.asList(
            Directionvalue.SOUTHWEST,
            Directionvalue.SOUTH,
            Directionvalue.SOUTHEAST);
    private Directionvalue mLastDirection = Directionvalue.CENTERED;
    protected Directionvalue getLastDirection() {
        return mLastDirection;
    }
    protected void setLastDirection(final Directionvalue value)  {
        mLastDirection = value;
    }
    JoystickDirectionView(final Context context) {
        super(context);
    }

    JoystickDirectionView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }

    JoystickDirectionView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }

    private static final float DEAD_CENTER_SIZE = 0.125f;




    protected abstract void showNewDirection(Directionvalue dv);
        protected static class RangeChecker {
        private final double mFrom;
        private final double mTo;
        private final Directionvalue mDirection;

        RangeChecker(final double thetaFrom, final double thetaTo, final Directionvalue direction) {
            mFrom = thetaFrom;
            mTo = thetaTo;
            mDirection = direction;

        }
        boolean match(final double theta) {
            return theta >= mFrom && theta <= mTo;
        }
        Directionvalue getDirection() {
            return mDirection;
        }
    }
    protected abstract List<RangeChecker> getRanges();
    protected double getThetaOutOfDeadRange(final double theta) {
        return theta;
    }
    private  Directionvalue calculateDirection(final float x, final float y) {
        double d = Math.sqrt((x * x) + (y * y));
        if (d < DEAD_CENTER_SIZE) {
            Log.v("calculateDirection", "in center");
            return Directionvalue.CENTERED;
        }
        // Theta, 0° = north, 90° = east, 180° = south, 270° = west
        double theta = Math.atan2(-y, -x);
        Log.v("calculateDirection", "theta from x = " + x + ", y = " + y
                + " is " + theta);
        // CHECKSTYLE DISABLE MagicNumber FOR 4 LINES
        double thetaDeg = Math.toDegrees(theta) - 90;
        if (thetaDeg < 0) {
            thetaDeg += 360;
        }
        Log.v("calculateDirection", "theta from x = " + x + ", y = " + y
                + " is " + thetaDeg);
        for (JoystickStickView.RangeChecker r : getRanges()) {
            if (r.match(getThetaOutOfDeadRange(thetaDeg))) {
                Log.v("calculateDirection", "getDirection from theta " + thetaDeg
                        + " is " + r.getDirection());
                return r.getDirection();
            }
        }
        return null;
    }

    @Override
    public void press(final float x, final float y) {
        Directionvalue newDirection = calculateDirection(x, y);
        if (newDirection != null && getLastDirection() != newDirection) {
            showNewDirection(newDirection);
            if (getJoystick() != null) {
                getJoystick().onChanged();
            }
        }

    }
    @Override
    public void release() {
        Log.v("onTouchEvent", getClass().getSimpleName()  + " released.");
        if (getLastDirection() != Directionvalue.CENTERED) {
            showNewDirection(Directionvalue.CENTERED);
            if (getJoystick() != null) {
                getJoystick().onChanged();
            }
        }
    }
}
