//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.Nullable;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

public final class GlobalSettingsActivity extends BaseActivity {
    private final ActivityResultLauncher<Intent> mBackupUserdataLauncher
            = registerForActivityResult(new FileCreateionContract(), result -> {
        if (result != null && result.getData() != null) {
            backupAllUserdata(result.getData());
        }
    });
    private final ActivityResultLauncher<Intent> mRestoreUserdataLauncher
            = registerForActivityResult(new FileCreateionContract(),
    result -> {
        if (result != null && result.getData() != null) {
            restoreAllUserdata(result.getData());
        }
    });

    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_global_settings);
        findViewById(R.id.bn_backup).setOnClickListener(view -> {
            Runnable r = () -> {
                Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
                intent.addCategory(Intent.CATEGORY_OPENABLE);
                intent.putExtra(Intent.EXTRA_TITLE, "EightBitWondersData.zip");
                intent.setType("application/zip");
                mBackupUserdataLauncher.launch(intent);
            };
            executeWithFilesystemAccess(null, r,  null, r);
        });
        findViewById(R.id.bn_restore).setOnClickListener(view -> {
            AlertDialogBuilder builder = new AlertDialogBuilder(view.getContext());
            builder.setIcon(android.R.drawable.ic_dialog_info);
            builder.setTitle(R.string.warning);
            builder.setMessage(R.string.will_delete_data);
            builder.setPositiveButton(android.R.string.ok, (dialogInterface, i1) -> {
                Runnable r = () -> {
                    Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
                    intent.addCategory(Intent.CATEGORY_OPENABLE);
                    intent.putExtra(Intent.EXTRA_TITLE, "EightBitWondersData.zip");
                    intent.setType("application/zip");
                    mRestoreUserdataLauncher.launch(intent);
                };
                executeWithFilesystemAccess(null, r, null, r);
            });
            AlertDialogBuilder.showStyledDialog(builder.create());
        });
    }
    private void backupAllUserdata(final Uri uri) {
        try {
            OutputStream os = getContentResolver().openOutputStream(uri);
            ZipOutputStream zos = new ZipOutputStream(os);
            List<File> output = new LinkedList<>();
            boolean globalOptsWritten = false;
            List<String> machineConfigsWritten = new LinkedList<>();
            File folder = new File(getFilesDir(), "packages");
            listFiles(folder, output);
            for (File f : output) {
                byte[] data = addFile(f, zos, "_packages_");
                if (f.getName().equals("properties")
                        && !Objects.requireNonNull(f.getParentFile()).getName()
                        .equals(".originals")) {
                    Properties p = new Properties();
                    p.load(new ByteArrayInputStream(data));
                    String emulation = p.getProperty("emulation");
                    Useropts opts = new Useropts();
                    opts.setCurrentEmulation(this, emulation, p.getProperty("id"));
                    StringWriter yaml = new StringWriter();
                    opts.yamlify(Useropts.Scope.CONFIGURATION, yaml);
                    ZipEntry zeYaml = new ZipEntry(String.format("_config_/%s_%s.yml",
                            emulation, p.getProperty("id")));
                    zos.putNextEntry(zeYaml);
                    zos.write(yaml.toString().getBytes(StandardCharsets.UTF_8));
                    zos.closeEntry();
                    if (!machineConfigsWritten.contains(emulation)) {
                        machineConfigsWritten.add(emulation);
                        yaml = new StringWriter();
                        opts.yamlify(Useropts.Scope.EMULATOR, yaml);
                        zeYaml = new ZipEntry(String.format("_config_/%s.yml", emulation));
                        zos.putNextEntry(zeYaml);
                        zos.write(yaml.toString().getBytes(StandardCharsets.UTF_8));
                        zos.closeEntry();
                        opts = new Useropts();
                        opts.setCurrentEmulation(this, emulation, "default");
                        yaml = new StringWriter();
                        opts.yamlify(Useropts.Scope.CONFIGURATION, yaml);
                        zeYaml = new ZipEntry(String.format("_config_/%s_default.yml", emulation));
                        zos.putNextEntry(zeYaml);
                        zos.write(yaml.toString().getBytes(StandardCharsets.UTF_8));
                        zos.closeEntry();


                    }
                    if (!globalOptsWritten) {
                        globalOptsWritten = true;
                        yaml = new StringWriter();
                        opts.yamlify(Useropts.Scope.GLOBAL, yaml);
                        zeYaml = new ZipEntry("_config_/_global_.yml");
                        zos.putNextEntry(zeYaml);
                        zos.write(yaml.toString().getBytes(StandardCharsets.UTF_8));
                        zos.closeEntry();

                    }
                }

            }
            folder = new File(getFilesDir(), "snapshots");
            output.clear();
            listFiles(folder, output);
            for (File f : output) {
                addFile(f, zos, "_snapshots_");
            }
            ZipEntry identifier = new ZipEntry("8bw");
            zos.putNextEntry(identifier);
            zos.write("8bw".getBytes(StandardCharsets.UTF_8));
            zos.closeEntry();
            zos.close();
            setResult(RESULT_OK);

            Toast.makeText(this, getResources().getString(R.string.userdata_backup_saved,
                    FileUtil.getFileName(this, uri)), Toast.LENGTH_SHORT).show();
            finish();

        } catch (IOException e) {
            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            } else {
                Log.v(getClass().getSimpleName(), "Exception occured", e);
                showInfoWrongZip(e);
            }
        }
    }
    private void listFiles(final File folder, final List<File> output) {
        File[] files = folder.listFiles();
        if (files != null) {
            for (File f : files) {
                if (f.isFile() && f.canRead()) {
                    output.add(f);
                }
                if (f.isDirectory() && !f.getName().equals("androidTest")) {
                    listFiles(f, output);
                }
            }
        }
    }


    private byte[] addFile(final File f, final ZipOutputStream zos,
                           final String rootfolder) throws IOException {
        ZipEntry ze = new ZipEntry(f.getAbsolutePath().replace(
                getFilesDir().getAbsolutePath(), rootfolder));
        zos.putNextEntry(ze);
        byte[] data;
        //noinspection IOStreamConstructor
        try (InputStream is = new FileInputStream(f)) {
            data = new byte[is.available()];
            if (is.read(data) >= 0) {
                zos.write(data);
            } else {
                data = null;
            }
            zos.closeEntry();
        }
        return data;
    }
    private void showInfoWrongZip(final @Nullable Throwable reason) {
        if (reason != null) {
            showErrorDialog(getResources().getString(R.string.failed_to_restore),
                    reason.getLocalizedMessage());
        } else {
            showErrorDialog(getResources().getString(R.string.failed_to_restore),
                    getResources().getString(R.string.wrong_zip, getString(R.string.app_name)));
        }
    }
    private void restoreAllUserdata(final Uri uri) {
        try {
            InputStream inputStream = getContentResolver().openInputStream(uri);
            ZipInputStream zis = new ZipInputStream(inputStream);
            Map<String, byte[]> mZipContents = new HashMap<>();
            ZipEntry ze;
            boolean isCorrectZip = false;
            ze = zis.getNextEntry();
            //boolean isCorrectZip = FileUtil.getFileName(this,uri);
            boolean hasSecurityIssue = false;
            String localName = FileUtil.getFileName(this, uri);
            if (!localName.toLowerCase(Locale.getDefault()).endsWith(".zip")) {
                Log.v(getClass().getSimpleName(), localName.toLowerCase(Locale.getDefault())
                + " does not end with .zip");
                showInfoWrongZip(null);
                return;
            }
            while (ze != null) {
                if (ze.getName().equals("8bw")) {
                    byte[] data =  extractZipEntry(zis);
                    if (data != null
                            && Arrays.equals(data, "8bw".getBytes(StandardCharsets.UTF_8))) {
                        isCorrectZip = true;
                    }
                }
                File f = new File("/a/b/c/d", ze.getName());
                String canonicalPath = f.getCanonicalPath();
                if (!canonicalPath.startsWith("/a/b/c/d")) {
                    hasSecurityIssue = true;
                }
                if (ze.getName().startsWith("_files_/")
                        || ze.getName().startsWith("_packages_/")
                        || ze.getName().startsWith("_snapshots_/")
                        || ze.getName().startsWith("_config_/")) {
                    byte[] data = extractZipEntry(zis);
                    if (data != null) {
                        mZipContents.put(ze.getName(), data);
                    }
                }

                zis.closeEntry();
                ze = zis.getNextEntry();
            }
            zis.close();
            if (isCorrectZip && !hasSecurityIssue) {
                for (String s : Arrays.asList("packages", "snapshot")) {
                    File folder = new File(getFilesDir(), s);
                    if (folder.exists()) {
                        deleteFile(folder);
                    }
                }
                for (String key : mZipContents.keySet()) {
                    String[] parts = key.split("/", 2);
                    File target;
                    if (parts[0].equals("_packages_") || parts[0].equals("_snapshots_")) {
                        target = new File(getFilesDir(), parts[1]);
                    } else if (parts[0].equals("_config_")) {
                        new Useropts().readFromYaml(this, new StringReader(
                                new String(mZipContents.get(key), StandardCharsets.UTF_8)));
                        target = null;
                    } else {
                        target = null;
                    }
                    if (target != null && target.getParentFile() != null) {
                        if (target.getParentFile().exists() || target.getParentFile().mkdirs()) {
                            byte[] data = mZipContents.get(key);
                            if (data != null) {
                                FileOutputStream fos = new FileOutputStream(target);
                                fos.write(mZipContents.get(key));
                                fos.close();
                            }
                        }
                    }
                }
                Intent res = new Intent();
                res.putExtra("restored", true);
                setResult(RESULT_OK, res);
                Toast.makeText(this, R.string.userdata_restored, Toast.LENGTH_SHORT).show();
                finish();
            } else {
                if (hasSecurityIssue) {
                    Log.v(getClass().getSimpleName(),
                        localName.toLowerCase(Locale.getDefault()) + "has security issues");
                }
                if (!isCorrectZip) {
                    Log.v(getClass().getSimpleName(),
                            localName.toLowerCase(Locale.getDefault()) + "has no 8bw file");
                }
                showInfoWrongZip(null);
            }
        } catch (IOException e) {
            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            } else {
                Log.v(getClass().getSimpleName(), "Exception occured", e);
                showInfoWrongZip(e);
            }

        }

    }
    private static final int CHUNKSIZE = 2048;
    private byte[] extractZipEntry(final ZipInputStream zis) throws IOException {

        byte[] buffer = new byte[CHUNKSIZE];
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        int len;
        while ((len = zis.read(buffer)) > 0) {
            baos.write(buffer, 0, len);
        }
        return baos.size() > 0 ? baos.toByteArray() : null;
    }
    static void cleanUp(final Context ctx) {
        try {
            Class<?> clz = Objects.requireNonNull(ctx.getClassLoader())
                    .loadClass("org.junit.Test");
            if (clz != null) {
                deleteFile(ctx.getFilesDir());
                deleteFile(ctx.getCacheDir());
                deleteFile(new File(ctx.getCacheDir().getParentFile(), "shared_prefs"));

            } else {
                throw new IllegalStateException(GlobalSettingsActivity.class.getSimpleName()
                        + ".cleanUp was called outside unit test.");
            }
        } catch (ClassNotFoundException e) {
            if (BuildConfig.DEBUG) {
                throw new IllegalStateException("Was called outside unit test");
            }
        }
    }
    private static boolean deleteFile(final File element) {
        boolean ret = true;
        File[] files = element.listFiles();
        if (files != null) {
            for (File sub : files) {
                ret = ret && deleteFile(sub);
            }
        }
        if (ret) {
            ret = element.delete();
        }
        return ret;
    }
}
