//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;
import android.content.ContentUris;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Environment;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.provider.OpenableColumns;
import android.util.Log;

import androidx.annotation.Keep;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

final class FileUtil {

    private static final String TAG = FileUtil.class.getSimpleName();


    private FileUtil() { }

    /**
     * Get the correct filename for an uri.
     * @param context any context
     * @param uri uri the filename is asked
     * @return filename for the given uri
     */
    static String getFileName(final Context context, final Uri uri) {
        String result = null;
        if (uri.getScheme() != null) {
            if (uri.getScheme().equals("content")) {
                Cursor cursor = context.getContentResolver().query(uri, null, null, null, null);
                if (cursor != null) {
                    try {
                        if (cursor.moveToFirst()) {
                            int col = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
                            if (col >= 0) {
                                result = cursor.getString(col);
                            }

                        }
                    } finally {
                        cursor.close();
                    }
                }
            }
        }
        if (result == null) {
            result = uri.getPath();
        }
        if (result != null) {
            int cut = result.lastIndexOf('/');
            if (cut >= 0) {
                result = result.substring(cut + 1);
            }
        }
        return result;
    }
    /**
     * Try to get a accessible file from local filesystems.
     * @param context any context
     * @param uri an uri
     * @return a {@link File} if the uri points to a local file and this can be accessed.
     */
    static File tryToGetAsFile(final Context context, final Uri uri) {
        File f;
        try {
            String path = getPath(context, uri);
            if (path != null) {
                f = new File(path);
                if (f.canRead()) {
                    return f;
                }
            }
        } catch (SecurityException e) {
            return null;
        }
        return null;
    }
    /** Get a file path from a Uri. This will get the the path for Storage Access.
     * Framework Documents, as well as the _data field for the MediaStore and
     * other file-based ContentProviders.
     *
     * @param context The context.
     * @param uri The Uri to query.
     * @author paulburke
     * @return file path
     */
    static String getPath(final Context context, final Uri uri) throws SecurityException {
        // DocumentProvider
        if (DocumentsContract.isDocumentUri(context, uri)) {
            // ExternalStorageProvider
            if (isExternalStorageDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];
                String storageDefinition;

                if ("primary".equalsIgnoreCase(type)) {
                    return Environment.getExternalStorageDirectory() + "/" + split[1];
                } else {

                    if (Environment.isExternalStorageRemovable()) {
                        storageDefinition = "EXTERNAL_STORAGE";
                        if (System.getenv(storageDefinition) == null) {
                            storageDefinition = "SECONDARY_STORAGE";
                        }
                    } else {
                        storageDefinition = "SECONDARY_STORAGE";
                        if (System.getenv(storageDefinition) == null) {
                            storageDefinition = "EXTERNAL_STORAGE";
                        }
                    }

                    return System.getenv(storageDefinition) + "/" + split[1];
                }

                // TODO handle non-primary volumes
            } else if (isDownloadsDocument(uri)) { // DownloadsProvider
                final String id = DocumentsContract.getDocumentId(uri);
                final Uri contentUri = ContentUris.withAppendedId(
                        Uri.parse("content://downloads/public_downloads"), Long.parseLong(id));

                return getDataColumn(context, contentUri, null, null);
            } else if (isMediaDocument(uri)) { // MediaProvider
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }

                final String selection = "_id=?";
                final String[] selectionArgs = new String[] {
                        split[1]
                };

                return getDataColumn(context, contentUri, selection, selectionArgs);
            }
        } else if ("content".equalsIgnoreCase(uri.getScheme())) { // MediaStore (and general)
            return getDataColumn(context, uri, null, null);
        } else if ("file".equalsIgnoreCase(uri.getScheme())) { // File
            return uri.getPath();
        }

        return null;
    }

    /**
     * Get the value of the data column for this Uri. This is useful for
     * MediaStore Uris, and other file-based ContentProviders.
     *
     * @param context The context.
     * @param uri The Uri to query.
     * @param selection (Optional) Filter used in the query.
     * @param selectionArgs (Optional) Selection arguments used in the query.
     * @return The value of the _data column, which is typically a file path.
     */
    static String getDataColumn(final Context context, final Uri uri, final String selection,
                                       final String[] selectionArgs) {

        Cursor cursor = null;
        final String column = "_data";
        final String[] projection = {
                column
        };

        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs,
                    null);
            if (cursor != null && cursor.moveToFirst()) {
                final int columnIndex = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(columnIndex);
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return null;
    }


    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is ExternalStorageProvider.
     */
    static boolean isExternalStorageDocument(final Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is DownloadsProvider.
     */
    static boolean isDownloadsDocument(final Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is MediaProvider.
     */
    static boolean isMediaDocument(final Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }
    private static void logContentAction(final String action, final String uri, final byte[] data) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(data);
            byte[] digest = md.digest();
            StringBuilder sb = new StringBuilder(digest.length * 2);
            for (byte b: digest) {
                sb.append(String.format("%02x", b));
            }
            Log.v(TAG, String.format("%s (%s), %d bytes, hash is %s",
                    action, uri, data.length, sb));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }

    }
    /**
     * Open a content with given uri and return binary contents.
     * @param context Any context
     * @param address uri, fulfilling the security requirements for opening it.
     * @return contents binary data of the uri's content or null if any error occured.
     */
    static byte[] getContentData(final Context context, final String address) {
        final InputStream inputStream;
        try {
            inputStream = context.getContentResolver().openInputStream(Uri.parse(address));
            if (inputStream != null) {
                byte[] bytes = new byte[inputStream.available()];
                int read = inputStream.read(bytes);
                inputStream.close();
                if (read >= 0) {
                    logContentAction("getContentData", address, bytes);
                    return bytes;
                }
            }
        } catch (IOException e) {
            Log.v(FileUtil.class.getSimpleName(), String.format("Cannot open %s", address), e);
        }
        return null;
    }
    private static final int READ_WRITE_ACCESS = 6;
    /**
     * Return Information about write and read access.
     * @param context Any context
     * @param adress The uri to be checked
     * @return A combination of IOUTIL_ACCESS_R_OK and IOUTIL_ACCESS_W_OK
     */
    @Keep
    static int getContentAccess(final Context context, final String adress) {
        int ret = 0;
        Uri uri = Uri.parse(adress);
        Cursor c = context.getContentResolver().query(uri, null, null, null, null);
        if (c != null) {
            if (c.moveToFirst()) {
                int col = c.getColumnIndex("flags");
                if (col >= 0) {

                    if ((c.getInt(col) & DocumentsContract.Document.FLAG_SUPPORTS_WRITE)
                            == DocumentsContract.Document.FLAG_SUPPORTS_WRITE) {
                        ret = READ_WRITE_ACCESS;
                    }
                }
            }
            c.close();
        } else {
            // qualified guess.
            ret = READ_WRITE_ACCESS;
        }
        if (ret == 0) {
            try {
                InputStream is = context.getContentResolver().openInputStream(uri);
                if (is != null) {
                    ret = 4;
                    is.close();
                }
            } catch (IOException e) {
                // make compiler smile
            }
        }
        if (BuildConfig.DEBUG) {
            Log.v(FileUtil.class.getSimpleName(),
                String.format("getContentData(%s) returned %d.", adress, ret));
        }
        return ret;
    }

    /**
     * Write binary content to a content with given uri.
     * @param context Any context
     * @param data data to be written.
     * @param address uri, fulfilling the security requirements for opening it.
     */
    static void putContentData(final Context context,
                                      final String address,
                                      final byte[] data) {
        final OutputStream outputStream;
        try {

            outputStream = context.getContentResolver().openOutputStream(Uri.parse(address));
            BufferedOutputStream bos = new BufferedOutputStream(outputStream);
            bos.write(data);
            bos.close();
            logContentAction("putContentData", address, data);

        } catch (IOException e) {
            Log.v(FileUtil.class.getSimpleName(), String.format("Cannot open %s", address), e);
        }
    }

}
