//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;


import de.rainerhock.eightbitwonders.vice.ViceFactory;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * List of all available Emulations.
 */
final class Emulationslist {
    private Emulationslist() {
        throw new UnsupportedOperationException();
    }
    /**
     * Description of an emulation to be presented in a view on the {@link MainActivity}.
     */
    interface Description {
        /**
         * id of the emulations name.
         * @return string-id
         */
        int getNameResourceId();

        /**
         * id of the bitmap-resource with the emulation's real device.
         * @return bitmap-id
         */
        int getImageResourceId();

        /**
         * id of the color-resource for the view's background.
         * @return color id
         */
        int getColorResourceId();
    }

    /**
     * get IDs for all available emulations.
     * @return List with the emulations unique keys.
     */
    static List<String> getEmulationIds() {
        return Arrays.asList("C64", "VIC20", "PET", "C128", "PLUS4");
    }
    /**
     * get the emulations's description for a unique emulation-key.
     * @param emulationId one of the keys returned from {@link #getEmulationIds()}
     * @return the {@link Description} for the id
     */
    static Description getDescription(final String emulationId) {
        if ("C64".equals(emulationId)) {
            return new Description() {
                @Override
                public int getNameResourceId() {
                    return R.string.name_c64;
                }

                @Override
                public int getImageResourceId() {
                    return R.drawable.image_c64;
                }

                @Override
                public int getColorResourceId() {
                    return R.color.c64bordercolor;
                }

            };

        }
        if ("C128".equals(emulationId)) {
            return new Description() {
                @Override
                public int getNameResourceId() {
                    return R.string.name_c128;
                }

                @Override
                public int getImageResourceId() {
                    return R.drawable.image_c128;
                }

                @Override
                public int getColorResourceId() {
                    return R.color.c128bordercolor;
                }

            };

        }
        if ("PET".equals(emulationId)) {
            return new Description() {
                @Override
                public int getNameResourceId() {
                    return R.string.name_pet;
                }

                @Override
                public int getImageResourceId() {
                    return R.drawable.image_pet;
                }

                @Override
                public int getColorResourceId() {
                    return R.color.petbordercolor;
                }
            };
        }
        if ("VIC20".equals(emulationId)) {
            return new Description() {
                @Override
                public int getNameResourceId() {
                    return R.string.name_vic20;
                }

                @Override
                public int getImageResourceId() {
                    return R.drawable.image_vic20;
                }

                @Override
                public int getColorResourceId() {
                    return R.color.vic20ordercolor;
                }

            };

        }
        if ("PLUS4".equals(emulationId)) {
            return new Description() {
                @Override
                public int getNameResourceId() {
                    return R.string.name_plus4;
                }

                @Override
                public int getImageResourceId() {
                    return R.drawable.image_plus4;
                }

                @Override
                public int getColorResourceId() {
                    return R.color.plus4bordercolor;
                }

            };

        }

        return null;
    }

    /**
     * create an Emulation.
     * @param ui object to handle user-interaction
     * @param config The emulation configuration
     * @return a new {@link Emulation} or null if no emulation was found for the given emulationId
     */
    static Emulation createEmulation(final EmulationUi ui,
                                            final EmulationConfiguration config) {
        Emulation ret = null;
        List<EmulationFactory> factories = Collections.singletonList(new ViceFactory());
        if (getCurrentEmulation() != null) {
            getCurrentEmulation().terminate(null);
            disposeCurrentEmulation();
        }

        for (EmulationFactory factory: factories) {
            if (ret == null && factory.getEmulatorIds().contains(config.getEmulatorId())) {
                ret = factory.newEmulation(ui, config);
            }
        }
        currentEmulation = ret;
        return ret;
    }

    private static Emulation currentEmulation = null;

    /**
     * get the current {@link Emulation}.
     * @return current emulation or null if no emulation is active.
     */
    static Emulation getCurrentEmulation() {
        return currentEmulation;
    }
    static final List<EmulationFactory> FACTORIES = Collections.singletonList(
            new ViceFactory());

    static void disposeCurrentEmulation() {
        currentEmulation = null;
    }
    static DownloaderFactory.Downloader getEmulationPreparer(
            final Context context,
            final EmulationConfiguration config)
            throws Exception {
        DownloaderFactory.Downloader ret = null;
        String emulationId = config.getEmulatorId();
        for (EmulationFactory factory : FACTORIES) {
            if (ret == null && factory.getEmulatorIds().contains(emulationId)) {
                ret = factory.getPrepareEmulationRunnable(context, emulationId);
            }
        }
        return ret;
    }
}
