package de.rainerhock.eightbitwonders;

import static android.graphics.Bitmap.CompressFormat.PNG;

import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.net.Uri;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ScrollView;
import android.widget.Toast;

import androidx.annotation.IdRes;
import androidx.annotation.Nullable;
import androidx.core.os.ConfigurationCompat;
import androidx.fragment.app.Fragment;

import com.jakewharton.processphoenix.ProcessPhoenix;


import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

class EmulationUiImpl implements EmulationUi {
    private final EmulationActivity mActivity;
    private final List<Joystick> mAvailableJoysticks;
    private final AudioService mService;
    private final EmulationConfiguration mConfiguration;
    private final Map<Integer, Point> mCanvasSize = new HashMap<>();
    private final Map<Integer, Float> mPixelAspectRatio = new HashMap<>();
    private final Map<Integer, Bitmap> mBitmap = new HashMap<>();
    private final Map<Integer, Float> mScreenAspectRatio = new HashMap<>();
    private boolean mScreenRefresh = true;

    EmulationUiImpl(final EmulationActivity activity) {
        mActivity = activity;
        mService = null;
        mAvailableJoysticks = activity.getAvailableJoysticks();
        mConfiguration = null;
    }
    EmulationUiImpl() {
        mActivity = null;
        mService = null;
        mAvailableJoysticks = new LinkedList<>();
        mConfiguration = null;
    }
    private EmulationConfiguration getConfiguration() {
        if (mConfiguration != null) {
            return mConfiguration;
        } else if (mActivity != null && mActivity.getViewModel() != null) {
            return mActivity.getViewModel().getConfiguration();
        }
        return null;
    }
    EmulationUiImpl(final AudioService service, final EmulationConfiguration conf) {
        mService = service;
        mActivity = null;
        mAvailableJoysticks = new LinkedList<>();
        mConfiguration = conf;
    }
    EmulationActivity getActivity() {
        return mActivity;
    }
    private EmulationViewModel getViewModel() {
        if (mActivity != null) {
            return mActivity.getViewModel();
        } else {
            return null;
        }

    }
    private boolean isTv() {
        return mActivity.isTv();
    }
    int getKeyboardVisibility() {
        if (mActivity != null) {
            return mActivity.findViewById(R.id.keyboardview).getVisibility();
        }
        return View.GONE;
    }
    @Nullable
    @Override
    public Context getContext() {
        if (mActivity != null) {
            return mActivity;
        }
        return mService;
    }
    @Override
    public void onEmulatorException(final Emulation emu, final NativeSignalException exception) {
        if (getConfiguration() != null && mActivity != null) {
            Intent i = new Intent(mActivity, MainActivity.class);
            i.putExtra("configuration", getConfiguration());
            i.putExtra("recovery", true);
            i.putExtra("id", UUID.randomUUID().toString());
            i.putExtra("recovery-written", exception.isRecoveryWritten());
            ProcessPhoenix.triggerRebirth(mActivity, i);
        }
    }
    @Override
    public void onEmulatorException(final Emulation emu, final Exception exception) {
        if (mActivity != null) {
            Intent i = new Intent();
            i.putExtra("recovery", false);
            i.putExtra("configuration", mActivity.getViewModel().getConfiguration());
            i.putExtra("error-message", exception.getLocalizedMessage());
            mActivity.finishWithResult(i);
        }
    }
    @Override
    public void onEmulatorInitialized(final Emulation emu) {
        if (mActivity != null) {
            mActivity.onEmulatorInitialized(emu);
        }
    }
    @Override
    public boolean isInRecovery() {
        if (mActivity != null) {
            return mActivity.getIntent().getBooleanExtra("recovery", false);
        } else {
            return false;
        }
    }
    @Override
    public Intent newIntentForUi() {
        if (mActivity != null) {
            return new Intent(mActivity, EmulationActivity.class);
        } else {
            return null;
        }
    }

    @Override
    public void runOnUiThread(final Runnable r) {
        if (mActivity != null) {
            mActivity.runOnUiThread(r);
        }
    }
    private static final List<String> NTSC_COUNTRIES = Arrays.asList("CA", "MX", "US", "AG",
            "AW", "BS", "BB", "BZ", "BM", "VG", "KY", "CR", "CU", "DM", "DO", "SV", "GT", "GD",
            "HT", "HN", "JM", "MS", "AN", "NI", "PA", "PR", "KN", "LC", "TT", "VI", "BO", "CL",
            "CO", "EC", "GY", "PY", "PE", "SR", "VE", "JP", "PH", "KR", "TW", "BU", "KH", "VN",
            "TH", "AS", "GU", "MP", "MH", "PW", "FJ", "DG");

    @Override
    public boolean isInNtscRegion() {
        Locale l;
        try {
            //noinspection unchecked
            Class<EmulationActivity.TestInterface> clz =
                    (Class<EmulationActivity.TestInterface>)
                            Objects.requireNonNull(EmulationActivity.class.getClassLoader())
                    .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("getTestLocale");
            l = (Locale) method.invoke(clz.newInstance());
            if (l == null) {
                l = Locale.getDefault();
            }
        } catch (ClassNotFoundException e) {
            l = Locale.getDefault();
        } catch (NoSuchMethodException | InvocationTargetException
                 | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }
        return NTSC_COUNTRIES.contains(l.getCountry());
    }

    @Override
    public boolean isInPalRegion() {
        return !isInNtscRegion();
    }

    @Override
    public Useropts getCurrentUseropts() {
        if (mActivity != null) {
            return mActivity.getCurrentUseropts();
        } else if (mService != null) {
            Useropts ret = new Useropts();
            if (Emulationslist.getCurrentEmulation() != null) {
                ret.setCurrentEmulation(mService,
                        Emulationslist.getCurrentEmulation().getEmulatorId(),
                        Emulationslist.getCurrentEmulation().getConfigurationId());
            }
            return ret;
        } else {
            return null;
        }
    }
    private <T extends View> T findViewById(final @IdRes int id) {
        if (mActivity != null) {
            return mActivity.findViewById(id);
        } else {
            return null;
        }
    }
    private DriveStatusListener getDriveStatusListener() {
        if (mActivity != null) {
            return findViewById(R.id.st_statusbar);
        } else {
            return new DriveStatusListener() {
                @Override
                public void updateDiskdriveList(final EmulationViewModel viewModel) {

                }

                @Override
                public void setDiskdriveState(final Object drive, final boolean active) {

                }

                @Override
                public void updateTapedriveList(final List<?> tapedrives) {

                }

                @Override
                public void setTapedriveState(final Object tape, final TapedriveState state) {

                }

                @Override
                public void setTapedriveMotor(final Object tape, final boolean active) {

                }

                @Override
                public void setTapedriveCounter(final Object tape, final int tapeCount) {

                }
            };
        }
    }

    @Override
    public void updateDiskdriveList(final EmulationViewModel viewModel) {
        DriveStatusListener l = getDriveStatusListener();
        if (l != null) {
            l.updateDiskdriveList(viewModel);
            if (l instanceof StatusbarView) {
                ((StatusbarView) getDriveStatusListener()).setVisibleIfNotEmpty();
            }

        }
    }
    private boolean mToastAboutSoftkeysSent = false;
    @Override
    public void createSoftkey(final String id, final String text) {
        if (getViewModel() != null) {
            if (getViewModel().hasJsSoftkey(id)) {
                destroySoftkey(id);
            }
            getViewModel().addJsSoftkey(id, text);
            getViewModel().setSoftkeyEnabled(id, true);
            if (!isTv()) {
                SoftkeysLinearLayout root = findViewById(R.id.softkeys);
                if (root != null) {
                    root.createSoftkey(id, text, true, getKeyboardVisibility(),
                            getViewModel().getEmulation().getSoftkeyFunctions(), null, null);
                }
            } else {
                if (!mToastAboutSoftkeysSent && getContext() != null) {
                    View screen = findViewById(R.id.screen);
                    if (screen != null) {
                        screen.post(() ->
                                Toast.makeText(getContext().getApplicationContext(),
                                        R.string.softkeys_created, Toast.LENGTH_SHORT).show());
                    }
                    mToastAboutSoftkeysSent = true;
                }
            }
        }
    }
    @Override
    public void destroySoftkey(final String id) {
        if (getViewModel() != null) {
            SoftkeysLinearLayout root = findViewById(R.id.softkeys);
            if (root != null) {
                getViewModel().removeJsSoftkey(id);
                root.destroySoftkey(id, getViewModel().getEmulation().getSoftkeyFunctions());
            }
        }
    }

    @Override
    public void enableSoftkey(final String key, final boolean enable) {
        if (getViewModel() != null) {
            getViewModel().setSoftkeyEnabled(key, enable);
            SoftkeysLinearLayout root = findViewById(R.id.softkeys);
            if (root != null) {
                root.enableSoftkey(key, enable);
            }
        }
    }

    @Override
    public void setDiskdriveState(final Object drive, final boolean active) {
        if (getViewModel() != null) {
            getViewModel().setDiskdriveState(drive, active);
            getDriveStatusListener().setDiskdriveState(drive, active);
        }
    }

    @Override
    public void updateTapedriveList(final List<?> tapedrives) {
        if (getViewModel() != null) {
            getViewModel().updateTapedriveList(tapedrives);
            getDriveStatusListener().updateTapedriveList(tapedrives);
        }
    }

    @Override
    public void setTapedriveState(final Object tape, final TapedriveState state) {
        getDriveStatusListener().setTapedriveState(tape, state);
    }

    @Override
    public void setTapedriveMotor(final Object tape, final boolean active) {
        if (getViewModel() != null) {
            getViewModel().setTapedriveMotor(tape, active);
        }
        getDriveStatusListener().setTapedriveMotor(tape, active);
    }

    @Override
    public void setTapedriveCounter(final Object tape, final int tapeCount) {
        getDriveStatusListener().setTapedriveCounter(tape, tapeCount);
    }

    @Override
    public void onCanvasSizeChanged(final int monitor, final int canvasWidth,
                                    final int canvasHeight,
                                    final float pixelAspectRatio,
                                    final float screenAspectRatio) {
        if (getViewModel() != null) {
            int display = getViewModel().getCurrentDisplayId();
            if (display == monitor || display == EmulationViewModel.NO_DISPLAY_SELECTED) {
                getViewModel().setCanvasSize(monitor, new Point(canvasWidth, canvasHeight));
                getViewModel().setPixelAspectRatio(monitor, pixelAspectRatio);
                getViewModel().setScreenAspectRatio(monitor, screenAspectRatio);
                MonitorGlSurfaceView gl = mActivity.findViewById(R.id.gv_monitor);
                mActivity.runOnUiThread(() -> gl.setCanvasGeometry(
                        canvasWidth, canvasHeight, pixelAspectRatio, screenAspectRatio));
            }
        } else {
            mCanvasSize.put(monitor, new Point(canvasWidth, canvasHeight));
            mPixelAspectRatio.put(monitor, pixelAspectRatio);
            mScreenAspectRatio.put(monitor, screenAspectRatio);
        }
    }

    @Override
    public void showFliplistDialog() {
        if (mActivity != null) {
            mActivity.showFliplistDialog();
        }
    }
    @Override
    public void setBitmap(final int monitor, final Bitmap bitmap) {
        if (mActivity != null && mScreenRefresh) {
            mActivity.setBitmap(monitor, bitmap);
        }
        if (mService != null && monitor == 0) {
            mService.setBitmap(bitmap);
        }
        mBitmap.put(monitor, bitmap);

    }
    void setScreenRefresh(final boolean value) {
        mScreenRefresh = value;
    }

    @Override
    public void updateDiskdriveList(final List<?> diskdrives) {
        if (getViewModel() != null) {
            getViewModel().updateDiskdriveList(diskdrives);
            DriveStatusListener l = getDriveStatusListener();
            if (l != null) {
                getDriveStatusListener().updateDiskdriveList(getViewModel());
            }
        }
    }

    @Override
    public Bitmap getCurrentBitmap() {
        if (mActivity != null) {
            return mActivity.getCurrentBitmap();
        } else {
            return mBitmap.get(mVisibleMonitor);
        }
    }

    @Override
    public boolean isKeycodeAvailable(final int keycode) {
        for (Joystick joy: mAvailableJoysticks) {
            if (joy.deviceHasButtonWithKeycode(keycode)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void showVirtualKeyboard(final int visibleScreenpart) {
        if (getViewModel() != null) {
            if (!getViewModel().isHardwareKeyboardAvailable()) {
                getViewModel().setKeyboardRequesteByEmulation(true);
                final MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
                final ScrollView sv = findViewById(R.id.scroll);
                if (sv != null && gl != null) {
                    Runnable r;
                    switch (visibleScreenpart) {
                        case (EmulationUi.SCREENPART_TOP):
                            r = () -> sv.fullScroll(ScrollView.FOCUS_UP);
                            break;
                        case (EmulationUi.SCREENPART_BOTTOM):
                            r = () -> sv.fullScroll(ScrollView.FOCUS_DOWN);
                            break;
                        case (EmulationUi.SCREENPART_CENTER):
                            r = () -> sv.scrollTo(0, (gl.getHeight() - sv.getHeight()) / 2);
                            break;
                        default:
                            return;
                    }

                    runOnUiThread(() -> {
                        mActivity.setKeyboardVisibility(View.VISIBLE);
                        sv.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                            @Override
                            //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
                            public void onLayoutChange(final View view, final int i, final int i1,
                                                       final int i2, final int i3, final int i4,
                                                       final int i5, final int i6, final int i7) {
                                r.run();
                                sv.removeOnLayoutChangeListener(this);
                            }
                        });
                    });
                }
                getViewModel().setKeyboardForced(true);
            }
        }
    }
    @Override
    public void restoreVirtualKeyboardVisibility() {
        if (getViewModel() != null && getViewModel().isKeyboardRequesteByEmulation()) {
            getViewModel().setKeyboardRequesteByEmulation(false);
            getViewModel().setKeyboardForced(false);
            runOnUiThread(() -> mActivity.setKeyboardVisibility(View.GONE));
        }
    }

    @Override
    public boolean getDeviceFeature(final int feature) {
        switch (feature) {
            case DEVICEFEATURE_HARDWARE_KEYBOARD:
                if (getViewModel() != null) {
                    return getViewModel().isHardwareKeyboardAvailable();
                } else {
                    return false;
                }
            case DEVICEFEATURE_TOUCH:
                return mActivity.getPackageManager()
                        .hasSystemFeature("android.hardware.touchscreen");
            case DEVICEFEATURE_TV:
                return isTv();
            default:
                return false;
        }
    }
    private List<Field> getStaticInts(@SuppressWarnings("SameParameterValue") final Class<?> cls) {
        List<Field> ret = new LinkedList<>();
        Field[] declaredFields = cls.getDeclaredFields();
        for (Field field : declaredFields) {
            if (java.lang.reflect.Modifier.isStatic(field.getModifiers())) {
                if (field.getType() == int.class) {
                    ret.add(field);
                }
            }
        }
        return ret;
    }
    @Override
    public void getConstants(final EmulationUi.SetConstantsCallback callback) {
        //noinspection rawtypes
        for (Class clz: Arrays.asList(EmulationUi.class, Joystick.class)) {
            for (Field field : getStaticInts(clz)) {
                try {
                    callback.setFeature(field.getName(), field.getInt(null));
                } catch (IllegalAccessException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        for (int keycode : Joystick.MAPPED_BUTTONS.values()) {
            callback.setFeature(KeyEvent.keyCodeToString(keycode), keycode);
        }
    }

    @Override
    public String getSystemLanguage() {
        Locale l;
        try {
            //noinspection unchecked
            Class<EmulationActivity.TestInterface> clz =
                    (Class<EmulationActivity.TestInterface>) Objects.requireNonNull(
                                    EmulationActivity.class.getClassLoader())
                            .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("getTestLocale");
            l = (Locale) method.invoke(clz.newInstance());
        } catch (ClassNotFoundException e) {
            l = ConfigurationCompat.getLocales(Resources.getSystem().getConfiguration()).get(0);
        } catch (NoSuchMethodException | IllegalAccessException | InstantiationException
                 | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
        if (l == null) {
            l = Locale.getDefault();
        }
        return l.getLanguage();
    }

    @Override
    public byte[] getContentData(final String address) {

        if (getConfiguration() != null) {
            byte[] cached = BaseActivity.getStreamData(getConfiguration().getId(), address);
            if (cached != null) {
                File f = new File(getConfiguration().getFilepath(
                        BaseActivity.hexSha256Hash(cached) + ".ovl"));
                if (f.exists()) {
                    try {
                        FileInputStream fis = new FileInputStream(f);
                        byte[] overlay = new byte[fis.available()];
                        if (fis.read(overlay) >= 0) {
                            for (int i = 0; i < Math.min(cached.length, overlay.length); i++) {
                                cached[i] += overlay[i];
                            }
                        }
                        fis.close();
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                return cached;
            }
        }
        if (getContext() != null) {
            return FileUtil.getContentData(getContext(), address);
        }
        return null;

    }
    @Override
    public void putContentData(final String address,
                               final byte[] data) {
        if (getConfiguration() != null) {
            byte[] original = BaseActivity.getStreamData(getConfiguration().getId(), address);
            if (original != null) {
                if (!Arrays.equals(original, data)) {
                    byte[] overlay = Arrays.copyOf(original, original.length);
                    for (int i = 0; i < Math.min(original.length, data.length); i++) {
                        overlay[i] -= original[i];
                    }
                    File f = new File(getConfiguration().getFilepath(
                            BaseActivity.hexSha256Hash(original) + ".ovl"));
                    if (f.getParentFile() != null && f.getParentFile().mkdirs()) {
                        try {
                            FileOutputStream fos = new FileOutputStream(f);
                            fos.write(overlay);
                            fos.close();
                            return;
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
            }
        }
        if (getContext() != null) {
            FileUtil.putContentData(getContext(), address, data);
        }
    }
    @Override
    public int getContentAccess(final String adress) {
        if (getContext() != null) {
            return FileUtil.getContentAccess(getContext(), adress);
        }
        return 0;
    }
    @Override
    public String getFileName(final Uri uri) {
        if (getContext() != null) {
            return FileUtil.getFileName(getContext(), uri);
        }
        return null;
    }
    @Override
    public File tryToGetAsFile(final Uri uri) {
        if (getContext() != null) {
            return FileUtil.tryToGetAsFile(getContext(), uri);
        }
        return null;
    }

    @Override
    public File getInitialSnapshotPath(final EmulationConfiguration conf, final int level) {
        if (mActivity != null) {
            return mActivity.getInitialSnapshotPathImpl(conf, level);
        }
        return null;
    }
    private static final int FULLQUALITY = 100;
    @Override
    public void storeJunitScreenshot(final Bitmap bmp, final String filename) {
        if (getContext() != null) {
            try {
                Class<?> clz = Objects.requireNonNull(getClass().getClassLoader())
                        .loadClass("org.junit.Test");
                if (clz != null) {
                    File target = new File(getContext().getCacheDir(),
                            filename.endsWith(".png") ? filename : filename + ".png");
                    try {
                        bmp.compress(PNG, FULLQUALITY, new FileOutputStream(target));
                    } catch (FileNotFoundException e) {
                        // should not happen
                    }
                }
            } catch (ClassNotFoundException e) {
                // nothing;
            }
        }
    }

    @Override
    public EmulationUi.JoystickToKeysHelper
    createJoystickToKeysHelper(final int port, final Emulation.SoftkeyFunctions mapper) {
        return new JoystickToKeysHelperImpl(mActivity, port, mapper);
    }

    @Nullable
    @Override
    public File getDirtyClosedSaveState(final EmulationConfiguration conf) {
        if (mActivity != null) {
            return mActivity.getInstanceStatePath(conf);
        }
        return null;
    }
    private int mVisibleMonitor = 0;
    private final Map<View, Fragment> mTemporaryFragments = new HashMap<>();
    @Override
    public void setVisibleMonitor(final int active) {
        mVisibleMonitor = active;
        if (mActivity != null) {
            mActivity.setVisibleMonitor(active);
            mActivity.restoreStatusbar(mActivity.findViewById(R.id.st_statusbar));
        }
    }
    @Override
    public void removeInternalKeyboardFragmentView(final View v) {
        if (mActivity != null) {
            if (mTemporaryFragments.containsKey(v)) {
                Fragment f = mTemporaryFragments.get(v);
                if (f != null && mActivity.getSupportFragmentManager().getFragments().contains(f)) {
                    mActivity.getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                    mTemporaryFragments.remove(v);
                }
            }
        }
    }
    @Override
    public View createInternalKeyboardFragmentView(final KeyboardFragment f) {
        if (mActivity != null) {
            String tag = new Object().toString();
            mActivity.getSupportFragmentManager().beginTransaction().add(f, tag).commitNow();
            LayoutInflater inflater = (LayoutInflater) mActivity
                    .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            ViewGroup vg = new ViewGroup(mActivity) {
                @Override
                protected void onLayout(final boolean changed, final int l, final int t,
                                        final int r, final int b) {

                }
            };
            mTemporaryFragments.put(vg, f);
            return f.onCreateView(inflater, vg, f.getArguments());
        }
        return null;
    }

}
