//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.view.KeyEvent;
import android.view.View;

import androidx.annotation.Keep;
import androidx.annotation.Nullable;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * Functions for interactions from a running emulation to the user interface.
 */
@Keep
public interface EmulationUi extends DriveStatusListener {
    /**
     * These joystick buttons can start an action as long as there is no other consumer
     * of the button events.
     * This is the case outside of the emulation activity
     * or in dialogs within the emulation acitivity.
     * see {@link #JOYSTICK_MINIMAL_ACTION_BUTTONS}
     */
    List<Integer> JOYSTICK_MAXIMAL_ACTIONS_BUTTONS = Arrays.asList(
            KeyEvent.KEYCODE_DPAD_CENTER, KeyEvent.KEYCODE_BUTTON_2, KeyEvent.KEYCODE_BUTTON_A,
            KeyEvent.KEYCODE_BUTTON_9, KeyEvent.KEYCODE_BUTTON_10, KeyEvent.KEYCODE_BUTTON_START,
            KeyEvent.KEYCODE_BUTTON_C, KeyEvent.KEYCODE_BUTTON_SELECT);
    /**
     * These buttons are part of the DPAD.
     */
    List<Integer> JOYSTICK_DPAD_DIRECTION_BUTTONS = Arrays.asList(
            KeyEvent.KEYCODE_DPAD_UP, KeyEvent.KEYCODE_DPAD_DOWN,
            KeyEvent.KEYCODE_DPAD_LEFT, KeyEvent.KEYCODE_DPAD_RIGHT);
    /**
     * These buttons are part of a remote control and trigger special functions.
     */
    List<Integer> JOYSTICK_RC_FUNCTIONS = Arrays.asList(
            KeyEvent.KEYCODE_MEDIA_PAUSE, KeyEvent.KEYCODE_SETTINGS,
            KeyEvent.KEYCODE_MEDIA_FAST_FORWARD, KeyEvent.KEYCODE_MENU,
            KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE, KeyEvent.KEYCODE_MEDIA_REWIND,
            KeyEvent.KEYCODE_SETTINGS);
    /**
     * These are buttons that the app should not consume.
     */
    List<Integer> SYSTEM_KEYS = Arrays.asList(KeyEvent.KEYCODE_VOLUME_DOWN,
            KeyEvent.KEYCODE_VOLUME_UP, KeyEvent.KEYCODE_VOLUME_MUTE, KeyEvent.KEYCODE_HOME,
            KeyEvent.KEYCODE_APP_SWITCH);
    /**
     * These joystick buttons can trigger a select action.
     */

    List<Integer> JOYSTICK_SELECT_BUTTONS = Arrays.asList(KeyEvent.KEYCODE_BUTTON_9,
            KeyEvent.KEYCODE_BUTTON_SELECT);
    /**
     * These joystick buttons can trigger a start action.
     */
    List<Integer> JOYSTICK_START_BUTTONS = Arrays.asList(KeyEvent.KEYCODE_BUTTON_10,
            KeyEvent.KEYCODE_BUTTON_START, KeyEvent.KEYCODE_BUTTON_C);

    /**
     * These joystick buttons can close a menu.
     */
    List<Integer> JOYSTICK_BACK_BUTTONS = Arrays.asList(KeyEvent.KEYCODE_BUTTON_3,
            KeyEvent.KEYCODE_BUTTON_B, KeyEvent.KEYCODE_BACK);
    /**
     * These joystick buttons can start an action and should not collide with emulation.
     * For situations without concurrency see {@link #JOYSTICK_MAXIMAL_ACTIONS_BUTTONS}.
     */
    List<Integer> JOYSTICK_MINIMAL_ACTION_BUTTONS = Arrays.asList(KeyEvent.KEYCODE_BUTTON_SELECT,
            KeyEvent.KEYCODE_BUTTON_9);

    /**
     * Get the android-context.
     * @return the {@link Context} to which the emulation is connected.
     */
    @Nullable Context getContext();

    /**
     * To be called when the emulation has been shut down due to a native exception.
     * @param emu the emulation
     * @param exception the exception that occurred
     */
    void onEmulatorException(Emulation emu, NativeSignalException exception);

    /**
     * To be called when the emulator is initialized and starts running.
     * @param emu the emulation
     */
    void onEmulatorInitialized(Emulation emu);
    /**
     * To be called when the emulation has been shut down due any but native exception.
     * @param emu the emulation
     * @param exception the exception that occurred
     */
    void onEmulatorException(Emulation emu, Exception exception);
    /**
     * Check if the emulation has been started by the user or by an automatic relaunch of the
     * application after crash.
     * @return true if relaunch after crash, false if started by user
     */
    boolean isInRecovery();

    /**
     * Create an intent.
     * @return new {@link Intent} with the implementation of this interface.
     */
    Intent newIntentForUi();

    /**
     * Runs the specified action on the UI thread.
     * @param r action to be run.
     */
    void runOnUiThread(Runnable r);

    /**
     * Check if current device is in a country where PAL was/is tv-standard, using the
     * language settings.
     * @return true if in PAL-region
     * @noinspection unused
     */
    boolean isInPalRegion();
    /**
     * Check if current device is in a country where NTSC was/is tv-standard, using the
     * language settings.
     * @return true if in NTSC-region
     */

    boolean isInNtscRegion();

    /**
     * Get the currently valid Useropts.
     * @return Useropts instance with current values.
     */
    Useropts getCurrentUseropts();

    /**
     * Notify about new canvas size.
     * @param monitor to identify the canvas on emulations with
     *                  multiple monitors or monitor setups.
     * @param canvasWidth canvas width in pixels
     * @param canvasHeight canvas height in pixels
     * @param pixelAspectRatio aspect ratio of each pixel
     * @param screenAspectRatio aspect ratio of screen
     */
    void onCanvasSizeChanged(int monitor, int canvasWidth,
                        int canvasHeight,
                        float pixelAspectRatio,
                        float screenAspectRatio);


    /**
     * Show a dialog where a file to be attached can be selected.
     * {@link de.rainerhock.eightbitwonders.Emulation.FliplistFunctions} will be used to
     * interact with the emulation.
     */
    void showFliplistDialog();

    /**
     * Update the contents of the view to draw on.
     * @param monitor to identify the canvas on emulations with
     *                  multiple monitors or monitor setups.
     * @param bitmap the new contents
     */
    void setBitmap(int monitor, Bitmap bitmap);
    /**
     * Show/update the currently attached disk-drives.
     * @param diskdrives each member of the list represents a device, it visualized by its
     *                   toString() method
     */
    void updateDiskdriveList(List<?> diskdrives);

    /**
     * Create a softkey, that is shown on a touch screen when no keyboard is enabled.
     * @param id id of a softkey, will be used as first parameter in
     * {@link Emulation.SoftkeyFunctions#pressKey(String)} and
     * {@link Emulation.SoftkeyFunctions#releaseKey(String)}.
     * @param text Text to be shown
     */
    void createSoftkey(String id, String text);

    /**
     * Remove a softkey.
     * @param id id of the softkey, see {@link #createSoftkey(String, String)}
     */
    void destroySoftkey(String id);

    /**
     * Enable or disable or softkey.
     * @param id id of the softkey, see {@link #createSoftkey(String, String)}
     * @param enable true to enable, false to disable the softkey
     */
    void enableSoftkey(String id, boolean enable);

    /**
     * Get the current screen data as bitmap.
     * @return current screen data
     */
    Bitmap getCurrentBitmap();
    /**
     * Return true if any attached hardware input device has the given key.
     * Key can be a key on a keyboard or a gamepad button.
     * @param keycode keycode according to {@link KeyEvent}
     * @return if any device has the given key.
     */
    boolean isKeycodeAvailable(int keycode);
    /** To be used as parameter visibleScreenpart in #showVirtualKeyboard to show the top
     * part of the screen.
     */
    int SCREENPART_TOP = 1;
    /** To be used as parameter visibleScreenpart in #showVirtualKeyboard to show the center
     * part of the screen.
     */
    int SCREENPART_CENTER = 2;
    /** To be used as parameter visibleScreenpart in #showVirtualKeyboard to show the bottom
     * part of the screen.
     */
    int SCREENPART_BOTTOM = 3;

    /**
     * Show virtual keyboard if required.
     * Show a virtual keyboard if keyboard input is required and there is no hardware keyboard.
     * If there is not enough space on the screen to show the keyboard beneath
     * the emulated display only the part given in visibleScreenpart will be shown initially.
     * @param visibleScreenpart describe which part of the emulated screen will be shown if there
     *                          is not enough space, muste be one of {@link #SCREENPART_TOP},
     *                          {@link #SCREENPART_CENTER}, {@link #SCREENPART_BOTTOM}
     */
    void showVirtualKeyboard(int visibleScreenpart);

    /**
     * Restore a virtual keyboard visibility if keyboard was set to visible
     * by calling {@link #showVirtualKeyboard(int)}.
     * If the keyboard was not set to visible by #showVirtualKeyboard since either there is
     * a hardware keyboard connected  or the virtual keyboard was visible anyway this
     * method will do nothing.
     */
    void restoreVirtualKeyboardVisibility();

    /**
     * Use as parameter for #getDeviceFeature to check if a device has a touch display.
     */
    int DEVICEFEATURE_TOUCH = 1;
    /**
     * Use as parameter for #getDeviceFeature to check if a device has a hardware keyboard.
     */
    int DEVICEFEATURE_HARDWARE_KEYBOARD = 2;
    /**
     * Use as parameter for #getDeviceFeature to check if a device is a tv.
     */
    int DEVICEFEATURE_TV = 3;
    /**
     * Use this as parameter displayId to draw a bitmap to the current display id.
     */
    int CURRENT_DISPLAY = -1;
    /**
     * Request device features.
     * @param feature feature to be requested, see static fields {@link EmulationUi}.DEVICEFEATURE*.
     * @return true if the device has the given feature.
     */
    boolean getDeviceFeature(int feature);

    /**
    boolean isFeatureAvailable(int feature);
    /**
    Get all input device features to be stored somewhere.
     */
    interface SetConstantsCallback {
        /**
         * Store a device feature by name and value.
         * @param label name of the device feature
         * @param value value of the device feature
         */
        void setFeature(String label, int value);
    }

    /**
     * Enrypt and decrypt data that should not be stored as plain text.
     */
    interface Encryptor {
        /**
         * Encrypt data to be decrypted with {@link #decrypt(byte[])}.
         * @param plaindata dat to be encrypted
         * @return encrypted data.
         */
        byte[] encrypt(byte[] plaindata);
        /**
         *  Decrypt data encrypted with {@link #encrypt(byte[])}.
         * @param encrypted data encrypted with {@link #encrypt(byte[])}
         * @return unencrypted data.
         */
        byte[] decrypt(byte[] encrypted);
    }
    /**
     * Get all feature values.
     * @param callback @see {@link SetConstantsCallback}
     */
    void getConstants(SetConstantsCallback callback);

    /**
     * Get system language according to ISO 639.
     * @return system language
     */
    String getSystemLanguage();
    /**
     * Open a content with given uri and return binary contents.
     * @param address uri, fulfilling the security requirements for opening it.
     * @return contents binary data of the uri's content or null if any error occured.
     */
    byte[] getContentData(String address);
    /**
     * Write binary content to a content with given uri.
     * @param data data to be written.
     * @param address uri, fulfilling the security requirements for writing to it.
     */
    void putContentData(String address, byte[] data);
    /**
     * Return Information about write and read access.
     * @param adress The uri to be checked
     * @return A combination of IOUTIL_ACCESS_R_OK and IOUTIL_ACCESS_W_OK
     */
    int getContentAccess(String adress);
    /**
     * Get the correct local filename for an uri.
     * @param uri uri the filename is asked
     * @return filename for the given uri
     */
    String getFileName(Uri uri);
    /**
     * Try to get a accessible file from local filesystems.
     * @param uri an uri
     * @return a {@link File} if the uri points to a local file and this can be accessed.
     */
    File tryToGetAsFile(Uri uri);

    /** Get a file where initial snapshots can be stored.
     * Initial snapshots are files created to speed up an emulations start
     * e.q. by creating them when initial loading of a file has been finished.
     * They can be created by javascript and will be used automatically.
     * @param conf #EmulationConfiguration the snapshot is taken for.
     * @param level level of the snapshot. the snapshot with the highest level will be taken.
     * @return File to store the data in or null in case of obscure errors (disk full)
     */
    File getInitialSnapshotPath(EmulationConfiguration conf, int level);

    /**
     * Store screenhots of the emulator's screen when running a unit test
     * (mostly an instrumented test).
     * @param bmp Bitmap to be stored.
     * @param filename basename of the file to be stored.
     */
    void storeJunitScreenshot(Bitmap bmp, String filename);

    /**
     * Interface for an element in a {@link MappedSpinner}.
     */
    interface SettingSpinnerElement extends Emulation.DataEntry,
            MappedSpinner.MappedSpinnerElement {
        /**
         * This value will be stored in the useropts representing the selected element.
         * @return user's choice (internal id)
         */
        String getId();

        /**
         * This is a user friendly text for an element.
         * @return user friendly text
         */
        String getText();
    }
    interface JoystickToKeysHelper {
        /**
         * Utitility class that takes joystick actions, translates them to keys and calls
         * the emulation's {@link de.rainerhock.eightbitwonders.Emulation.SoftkeyFunctions}
         * with theese keys.
         * @param north new value for pointing north/up
         * @param south new value for pointing south/down
         * @param east new value for pointing right/east
         * @param west new value for pointing left/west
         * @param buttons new value for pressed buttons
         */
        void onJoystickChanged(boolean north, boolean south, boolean east,
                               boolean west, Set<Emulation.InputButton> buttons);
    }

    /**
     * Create a {@link JoystickToKeysHelper} to translate joystick actions, translate them to keys.
     * and calls a JoystickToKeyboardMapper
     * @param port joystick port
     * @param mapper JoystickToKeyboardMapper that takes the translated keys.
     * @return an instance of {@link JoystickToKeysHelper}
     */
    JoystickToKeysHelper createJoystickToKeysHelper(int port, Emulation.SoftkeyFunctions
                                                            mapper);
    /**
     * Get File where instance state from last run not finished cleanly is stored.
     * @param conf emulation configuration
     * @return File with stored data or null if the last run was closed clean.
     */
    @Nullable File getDirtyClosedSaveState(EmulationConfiguration conf);

    /**
     * Set a visible monitor.
     * @param active the monitor to show. Must be the first parameter in one of the calls of
     * {@link #setBitmap(int, Bitmap)} and {@link #onCanvasSizeChanged(int, int, int, float, float)}
     */
    void setVisibleMonitor(int active);

    /**
     * Inflate the view of the fragment and pass it as return value.
     * The view is for internal use (like enumerating the available keys) and not for display.
     * @param f the fragment the view has to be returned.
     * @return the fragment's root view.
     */
    View createInternalKeyboardFragmentView(KeyboardFragment f);
    /**
     * Remove a view inflated with {@link #createInternalKeyboardFragmentView(KeyboardFragment)}.
     * @param v a return value from {@link #createInternalKeyboardFragmentView(KeyboardFragment)}
     */
    void removeInternalKeyboardFragmentView(View v);
}
