//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.graphics.Bitmap;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.File;
import java.io.Serializable;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Set;

public interface EmulationConfiguration extends Serializable, EmulationUi.Encryptor {
    /**
     * Configure Joysticks, especially set Ports fitting the configurations's requirements.
     *
     * @param activity current activity
     * @param joystickfunctions the current emulation's
     * @param availableJoysticks list of available joysticks
     * @param preferredJoysticktypes the preferred virtual joystick types (joystick/wheel).
     * Modifications in the available joysticks will be effective no matter what the return value
     * is.
     */
    void adaptJoysticks(BaseActivity activity,
                        Emulation.JoystickFunctions joystickfunctions,
                        List<Joystick> availableJoysticks,
                        Map<Integer, PreferredJoystickType> preferredJoysticktypes);

    /**
     * Check if the configuration locks touch joystick diagonals.
     * Good for maze-runners like pac-man
     * @return true if diagonals shall be locked, touch joystick then only has for directions.
     */
    boolean getTouchJoystickDiagonalsLocked();

    /**
     * Value for a type of joystick.
     */
    enum PreferredJoystickType { DIRECTIONAL, WHEEL }

    /**
     * Get the preferred virtual joystick types (joystick/wheel).
     * @return Map with types, key ist the joystickport, value the joysticktype joystick/wheel
     */
    Map<Integer, PreferredJoystickType> getVirtualJoystickTypes();

    /**
     * Get the required input device type (joystick/paddle/mouse).
     * @return Map with types, key ist the joystickport, value the input device type
     */
    Map<Integer, Emulation.InputDeviceType> getInputDeviceTypes();
    /**
     * Value for device orientation.
     */
    enum Orientation { LANDSCAPE, PORTRAIT, DEFAULT }
    /** Get the unique ID for the emulation.
     * @return id, must be none-null;
     */
    @NonNull String getId();

    /**
     * Get the Bitmap to be displayed on start screen.
     * @return a Bitmap or null if there is none.
     */
    Bitmap getBitmap();
    /**
     * Get the ID for the emulator that runs this configuration.
     * @return id, must be element of the return value from
     * {@link de.rainerhock.eightbitwonders.vice.ViceFactory#getEmulatorIds()}
     */
    @NonNull String getEmulatorId();
    /**
     * The best mapping device orientation. in PORTRAIT mode a virtual will be visible
     * if the emulator supports it.
     * @return orientation or null if the current device orientation has to be used.
     */
    Orientation getBestOrientation();

    /**
     * Apply emulation-specific values to the running emulation.
     * @param emu target emulation
     * @param ui for interactions with the user interface
     */
    void apply(EmulationActivity ui, Emulation emu);

    /**
     * Returns default values to be handled by the emulation class.
     * @return Map with emulation-specific-values
     */
    Map<String, String> getPresettings();

    /**
     * A Runnable-style Downloader to be called when additional files have to be downloaded.
     * @param context context to be used for accessing resources.
     * @return a Downloader to be run or null if nothing has to be downloaded
     */
    DownloaderFactory.Downloader getAdditionalDownloader(BaseActivity context);

    /**
     * get user friendly name.
     * @return value to be displayed
     */
    String getName();

    /**
     * If the configuration is a bare machine or some kind of pre-configuration.
     * @return true if the configuration is a bare machine, otherwise false
     */
    boolean isBareMachine();
    /**
     * If the configuration was generated locally by using "add to start screen".
     * @return true if the configuration was created by using the
     * "add to start screen" feature
     */
    boolean isLocallyGenerated();
    /**
    If the configuration was imported from ab website or filesystem.
     @return true if the configuration was downloaded or imported
     */
    boolean isImported();

    /**
     * If the configuration was downloaded from a website to be streamed.
     * @return true if the configuration was downloaded
     * from a website to be streamed.
     */
    boolean isStream();
    /**
     * Get Web Adress for the configuration.
     * @return URL to be opened in a WebView
     */
    URL getWebAdress();

    /**
     * A Runnable that undoes all changes made by the user or null if this is not needed.
     * @return a Runnable to be run to undo all changes by the user.
     */
    Runnable getFactoryResetter();

    /**
     * A Runnable that deinstalls the configuration if this is possible.
     * @return a Runnable to be run to deinstall configuration
     */
    Runnable getDeinstaller();

    /**
     * Return all Joystickports used by the configuration.
     * The configuration will use theese ports, but also run without them.
     * @param allJoystickports List of Joystickports the emulation can use
     * @return List of Joystickports
     */
    List<Integer> getUsedJoystickports(List<Integer> allJoystickports);

    /**
     * Return all Joystickports required by the configuration.
     * The configuration will not run without theese ports connected.
     * @param allJoystickports all available Joystickports
     * @return List of Joystickports
     */
    @NonNull List<Integer> getRequiredJoystickports(List<Integer> allJoystickports);

    /**
     * Keyboard usage of this configuration.
     * @return true if the configuration uses a keyboard.
     */
    boolean isKeyboardUsed();

    /**
     * The configuration contains a fixed list of files that can be attached to devices
     * without having to access the file system.
     * @return files that can be attached to emulated devices.
     */
    Set<Uri> getFliplist();

    /**
     * To be called when an emulation changes the fliplist automatically or by user interaction.
     * @param emulatorlist the fliplist from the emulation.
     */
    void onEmulatorFliplistChanged(Set<Uri> emulatorlist);

    /**
     * Return the full path of file with the given basename.
     * @param basename basename of the file
     * @return full path of a file with the given name in a configuration-specific path.
     */
    String getFilepath(String basename);
    /**
     * Encrypt data to be stored permanently if encryption is required.
     * Encrypt data using attributes of the EmulationConfiguration
     * @param plaindata data to be encrypted
     * @return the encrypted data or plaindata if no encryption is required.
     */
    byte[] encrypt(byte[] plaindata);
    /**
     * Decrypt data to that was encrypted by {@link #encrypt(byte[])}.
     * Decrypt data using attributes of the EmulationConfiguration
     * @param encrypted data to be encrypted.
     * @return the decrypted data or encrypted if {@link #encrypt} did only a copy.
     */
    byte[] decrypt(byte[] encrypted);

    /**
     * Get folder where the configuration is stored.
     * @return storage folder or null for volatile or built in configurations.
     */
    @Nullable File getLocalPath();

    /**
     * Get a property handled by eight bit wonders (not the emulators).
     * Emulators should not call this.
     * @param key one value stored in the "properties" file of each emulation.
     * @param defaultvalue default value to be returned if the properties do not contain a value
     *                     for the given key.
     * @return stored value if found or defaultvalue.
     */
    String getProperty(String key, String defaultvalue);
}
