//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------

package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.DialogsView.UiElement.BACKGROUND_MODE;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.FLIPLIST;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.JOYSTICKSELECTION;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.PAUSE;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.PAUSE_AFTER_TIMEMACHINE;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.SAVESTATES;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.SOFTKEYS;
import static de.rainerhock.eightbitwonders.Joystick.PORT_NOT_CONNECTED;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.SWITCH_MONITOR;
import static de.rainerhock.eightbitwonders.DialogsView.UiElement.TAPE_CONTROL;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ActivityManager;
import android.app.Dialog;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.os.PowerManager;
import android.util.Log;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContract;

import android.view.ViewTreeObserver;
import android.view.WindowInsets;
import android.view.WindowManager;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.ViewModelProvider;

import java.io.File;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * This is the activity the emulations run in. Emulations communicate with it via the
 * interface {@link EmulationUi}.
 */
public final class EmulationActivity extends DeviceSupportActivity {
    static final String PAUSE_FROM_NOTIFICATION = "PAUSE_FROM_NOTIFICATION";
    private static final String TAG = EmulationActivity.class.getSimpleName();
    static final String FRAGMENT_DIALOG = "FRAGMENT_DIALOG";
    private static final String REMOTE_KEYBOARD = "FRAGMENT_REMOTE_KEYBOARD";
    private EmulationViewModel mViewModel = null;
    private final List<Joystick> mAvailableJoysticks = new LinkedList<>();
    static final String KEYBOARDFRAGMENT = "KEYBOARDFRAGMENT";
    private boolean mMonitorSet = false;
    private boolean mGlobalLaoyutListenerAdded = true;

    private SoftkeyConfigView getSoftkeyConfigView() {
        return findViewById(R.id.softkeyconfig);
    }

    private void applySettings(final Intent result) {
        evaluateSettings(result);
        getViewModel().updateKeyboardDefaultVisibility(this,
                getViewModel().getConfiguration());
        applyKeyboardSettings();
        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
    }
    private final ActivityResultLauncher<SettingsActivity.SettingsConfiguration> mSettingsLauncher
            = registerForActivityResult(new SettingsActivity.ShowSettingsContract(),
            result -> {
                applySettings(result);
                handleCommonLaunchresult(result);
            });
    private final ActivityResultLauncher<Object> mNewFilePropertiesLauncher
            = registerForActivityResult(new ActivityResultContract<Object, Intent>() {
        @Override
        public Intent parseResult(final int result, @Nullable final Intent intent) {
            handleCommonLaunchresult(intent);
            return intent;

        }

        @NonNull
        @Override
        public Intent createIntent(@NonNull final Context context, final Object dummy) {
            if (getViewModel().getEmulation().getFileCreationFunction() != null) {
                return getViewModel().getEmulation().getFileCreationFunction()
                        .getNewFilePropertiesIntent(EmulationActivity.this);
            }
            return new Intent();
        }
    }, result -> {
        if (result != null) {
            SubActivityBlockFragment.add(this);
            mCreateFileData = result.getSerializableExtra("createdata");
            createFile(result.getStringExtra("filename"));
        }
    });
    private final ActivityResultLauncher<Intent> mShareLauncher
            = registerForActivityResult(new ShareEmulationActivity.ShareContract(), result -> {
            handleCommonLaunchresult(null);
            });
    private final ActivityResultLauncher<Intent> mShareLauncherFromDialog
            = registerForActivityResult(new ShareEmulationActivity.ShareContract(), result -> {
        if (result) {
            handleCommonLaunchresult(null);
        } else {
            showSnapshotsDialog();
        }
    });
    private final ActivityResultLauncher<SettingsActivity.SettingsConfiguration>
            mInitialSettingsLauncher = registerForActivityResult(
            new SettingsActivity.ShowSettingsContract(), result -> {
                applySettings(result);
                getViewModel().initializeKeyboardDefaultVisibility(this,
                        getViewModel().getConfiguration());
                updateOrientation(getViewModel().getConfiguration());
                runEmulation();
                handleCommonLaunchresult(result);

            });
    private boolean mSettingsEvaluated = false;

    EmulationViewModel getViewModel() {
        if (mViewModel == null) {
            mViewModel = new ViewModelProvider(this).get(EmulationViewModel.class);
        }
        return mViewModel;
    }

    void onEmulatorInitialized(final Emulation emu) {
        synchronized ((mUpdateUiMonitor)) {
            if (!mUpdateUiDone) {
                try {
                    mUpdateUiMonitor.wait();
                } catch (InterruptedException e) {
                    Log.e(TAG, e.getMessage(), e);
                }
            }
        }
        if (mPauseEmulationImmediately) {
            emu.setPaused(true);
            mPauseEmulationImmediately = false;
        }
        ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor)).onGeometryChanged();
    }
    void showSoftkeyDialog() {
        getDialogsController().showUiElement(SOFTKEYS);
    }

    protected void onFileCreated(@NonNull final Uri uri) {
        if (mCreateFileData != null && getViewModel().getEmulation().getFileCreationFunction()
                .createFile(getContentResolver(), uri, mCreateFileData)) {
            onFileOpened(uri);
        } else {
            Toast.makeText(this, getResources().getString(R.string.could_not_open_file,
                            uri.getPath() != null ? new File(uri.getPath()).getName() : "???"),
                            Toast.LENGTH_LONG).show();
            unmuteEmulation();
        }
        super.onFileCreated(uri);
    }
    static final  int PAUSE_TICK_DURATION = 100;
    static final  int PAUSE_TICK_COUNT = 10;
    private void pauseAfterTimeMachine(final @NonNull  Runnable runWhenEnding) {
        for (Joystick joy:mAvailableJoysticks) {
            joy.startStateBuffering();
        }
        Object pauseReason = addPauseReason();
        new Thread(() -> runOnUiThread(()
                -> getDialogsController().showUiElement(PAUSE_AFTER_TIMEMACHINE, () -> {
                    for (Joystick joy:mAvailableJoysticks) {
                        if (joy.isBuffered()) {
                            joy.readFromStateBuffer();
                        }
                    }
                    removePauseReason(pauseReason);
                    runOnUiThread(runWhenEnding);
                }))).start();
    }

    private Runnable mRestoreFromMovieMode = null;
    void playInMovieMode() {
        Log.v("moviemode-test", "playInMovieMode called from "
                + new Exception().getStackTrace()[1].toString());
        final int orientation = getRequestedOrientation();
        final List<Integer> restoreViews = new LinkedList<>();
        List<Integer> allViews = new LinkedList<>(((TouchDisplayRelativeLayout)
                findViewById(R.id.crowded_area)).getUiElements());
        allViews.add(R.id.statusbar);
        allViews.add(R.id.keyboardview);
        allViews.add(R.id.ib_hamburger_menu);
        allViews.remove(Integer.valueOf(R.id.scroll));
        setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_USER_LANDSCAPE);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        for (int id : allViews) {
            View v = findViewById(id);
            if (v != null && v.getVisibility() == View.VISIBLE) {
                restoreViews.add(id);
                v.setVisibility(View.GONE);
            }
        }
        Toast.makeText(this, R.string.tap_to_leave_move_mode, Toast.LENGTH_SHORT).show();
        mRestoreFromMovieMode = () -> {
            for (int id : restoreViews) {
                findViewById(id).setVisibility(View.VISIBLE);
            }
            setRequestedOrientation(orientation);
            restoreCanvasSize();
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        };
        DialogsView v = getDialogsController();
        v.post(() -> v.setVisibility(View.VISIBLE));
    }

    boolean leaveMovieMode() {
        Log.v("moviemode-test", "leaveMovieMode called from "
                + new Exception().getStackTrace()[1].toString());

        if (mRestoreFromMovieMode != null) {
            mRestoreFromMovieMode.run();
            mRestoreFromMovieMode = null;
            return true;
        }
        return false;
    }
    public static class SubActivityBlockFragment extends Fragment {
        /**
         * Needed by android, do not use.
         */
        public SubActivityBlockFragment() {
            super();
        }

        static final String TAG = "SUBACTIVITY_BLOCK";

        static void add(final EmulationActivity a) {
            a.getSupportFragmentManager().beginTransaction()
                    .add(new SubActivityBlockFragment(), TAG).commit();
        }

        @SuppressWarnings("BooleanMethodIsAlwaysInverted")
        static boolean exists(final EmulationActivity a) {
            for (Fragment f : a.getSupportFragmentManager().getFragments()) {
                if (f instanceof SubActivityBlockFragment) {
                    return true;
                }
            }
            return false;
        }

        static void remove(final EmulationActivity a) {
            Fragment toDelete = null;
            for (Fragment f : a.getSupportFragmentManager().getFragments()) {
                if (f instanceof SubActivityBlockFragment) {
                    toDelete = f;
                    break;
                }
            }
            if (toDelete != null) {
                a.getSupportFragmentManager().beginTransaction().remove(toDelete).commit();
            }
        }
    }
    void shareContent(final Intent i, final boolean fromDialog) {
        SubActivityBlockFragment.add(this);
        muteEmulation();
        if (fromDialog) {
            mShareLauncherFromDialog.launch(i);
        } else {
            mShareLauncher.launch(i);
        }
    }

    private Serializable mCreateFileData = null;

    @Override
    protected boolean openFile(final boolean isRetry) {
        SubActivityBlockFragment.add(this);
        getViewModel().addSubActivityPauseReason();
        return super.openFile(isRetry);
    }
    void muteEmulation() {
        if (getViewModel().getEmulation().getSoundFunctions() != null) {
            getViewModel().getEmulation().getSoundFunctions().mute();
        }
    }
    void unmuteEmulation() {
        if (getViewModel().getEmulation().getSoundFunctions() != null) {
            getViewModel().getEmulation().getSoundFunctions().unmute();
        }
    }
    @Override
    protected void onFileOpened(final Uri uri) {

        final Runnable r = () -> {
            boolean followUpAction = false;
            Intent emulationIntent = getViewModel().getEmulation().getFileFunctions()
                    .getIntent(uri, FileUtil.getFileName(this, uri));
            if (emulationIntent != null && emulationIntent.getComponent() != null) {
                applyScreenOrientation(emulationIntent);
                if (!emulationIntent.getComponent().getClassName().equals(getClass().getName())) {
                    SubActivityBlockFragment.add(this);
                    followUpAction = true;
                    if (!getViewModel().getEmulation().isPaused()) {
                        getViewModel().getEmulation().setPaused(true,
                                () -> runOnUiThread(() ->
                                        mHandleImageLauncher.launch(emulationIntent)));
                    } else {
                        mHandleImageLauncher.launch(emulationIntent);
                    }
                }
                String errormessage = emulationIntent.getStringExtra("errormessage");
                if (errormessage != null) {
                    showErrorMessage(errormessage);
                    if (emulationIntent.getBooleanExtra("retry", false)) {
                        getViewModel().removeSubActivityPauseReason();
                        followUpAction = true;
                        if (!getViewModel().getEmulation().isPaused()) {
                            getViewModel().getEmulation().setPaused(true, () ->
                                    runOnUiThread(() -> openFile(true)));
                        } else {
                            openFile(true);
                        }
                    } else {
                        unmuteEmulation();
                    }
                }
            }
            if (!followUpAction) {
                unmuteEmulation();
                invalidateOptionsMenu();
                getViewModel().removeSubActivityPauseReason();
            }
        };
        if (getViewModel().getEmulation().isPaused()) {
            r.run();
        } else {
            getViewModel().getEmulation().setPaused(true, () -> runOnUiThread(r));
        }
        super.onFileOpened(uri);
    }

    @Override
    protected void onFileOpenCancelled(final boolean wasRetry) {
        unmuteEmulation();
        getViewModel().removeSubActivityPauseReason();
    }
    protected void handleCommonLaunchresult(final Intent resultData) {

        disposeInstanceState(getViewModel().getConfiguration());
        getViewModel().setStopped(false);
        if (resultData != null) {
            String message = resultData.getStringExtra(getPackageName() + ".SuccessMessage");
            if (message != null) {
                Toast.makeText(getApplicationContext(), message, Toast.LENGTH_LONG).show();
            }
            message = resultData.getStringExtra(getPackageName() + ".ErrorMessage");
            if (message != null) {
                showErrorMessage(message);
            }
        }
        invalidateOptionsMenu();
        unmuteEmulation();
        getViewModel().removeSubActivityPauseReason();
        restoreStatusbar(findViewById(R.id.st_statusbar));
        TouchDisplayRelativeLayout tdrl = findViewById(R.id.crowded_area);
        tdrl.post(() -> tdrl.setBottomMargin(getKeyboardVisibility() == View.GONE));

    }

    void applyScreenOrientation(final @NonNull Intent i) {
        int orientation = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
        switch (getEmulationScreenOrientation(getViewModel().getConfiguration())) {
            case PORTRAIT:
                orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                break;
            case LANDSCAPE:
                orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                break;
            default:
                break;
        }
        i.putExtra("screen_orientation", orientation);
    }

    private EmulationConfiguration getInitialConfiguration() {
        EmulationConfiguration conf =
                getViewModel().getConfiguration();
        if (conf == null && getIntent().getAction() != null
                && getIntent().getAction().startsWith("WIDGET_CALL:")) {
            String[] parts = getIntent().getAction().split(":", 2);
            if (parts.length == 2) {
                String id = parts[1];
                conf = getWidgetConfiguration(id);
            }
        }
        if (conf == null) {
            conf = (EmulationConfiguration) getIntent()
                    .getSerializableExtra(SettingsActivity.CONFIGURATION);
        }
        return conf;
    }
    @Nullable
    private EmulationConfiguration getWidgetConfiguration(final String widgetId) {
        EmulationConfiguration conf = null;
        if (Emulationslist.getEmulationIds().contains(widgetId)) {
            conf = new ConfigurationFactory.MachineConfiguration(this, widgetId);
        } else {
            for (EmulationConfiguration c : getAssetConfigurations(
                    getResources().getAssets(), "packages", null)) {
                if (c.getLocalPath() != null
                        && c.getLocalPath().getAbsolutePath().equals(widgetId)) {
                    conf = c;
                    break;
                }
            }
            if (conf == null) {
                for (EmulationConfiguration c : getLocalConfigurations()) {
                    if (c.getLocalPath() != null
                            && c.getLocalPath().getAbsolutePath().equals(widgetId)) {
                        conf = c;
                        break;
                    }
                }
            }
        }
        return conf;
    }

    private boolean mPauseEmulationImmediately = false;
    private void evaluateSettings(final Intent result) {
        if (!mSettingsEvaluated) {
            Runnable runAfter = () -> {
                if (!isTv()) {
                    runOnUiThread(this::createTouchKeyboard);
                }
            };
            Emulation.MachineSettingsFunctions msf =
                    Emulationslist.getCurrentEmulation() != null
                            ? Emulationslist.getCurrentEmulation().getMachineSettingsFunction()
                            : null;
            if (msf != null) {
                msf.applyUseropts(getCurrentUseropts(), runAfter);
            } else {
                runAfter.run();
            }
            if (result != null) {
                if (result.getSerializableExtra(SettingsActivity.OPENSECTIONS) != null) {
                    //noinspection unchecked
                    getViewModel().setOpenSettingsSections((List<Integer>)
                            result.getSerializableExtra(SettingsActivity.OPENSECTIONS));
                }
            } else {
                runAfter.run();
            }
            mMonitorSet = false;
            updateUi();
            EmulationConfiguration conf = getViewModel().getConfiguration();
            updateOrientation(conf);
            mSettingsEvaluated = true;
        }
    }
    DialogsView getDialogsController() {
        return findViewById(R.id.dialogsview);
    }
    void addOnBackPressedCallback(final LifecycleOwner owner) {
        getOnBackPressedDispatcher().addCallback(owner, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                if (isEnabled() && !getDialogsController().dismissMenu()
                        && !getDialogsController().closeNewestDialog()) {
                    if (!isTv() && !getViewModel().isKeyboardVisibibleFromStart(isInLandscape())
                            && getKeyboardVisibility() == View.VISIBLE) {
                        setKeyboardVisibility(View.GONE);
                        Fragment f = getSupportFragmentManager()
                                .findFragmentByTag(KEYBOARDFRAGMENT);
                        if (getViewModel().getEmulation().getSoftwareKeyboardFunctions() != null
                                && f instanceof KeyboardFragment) {
                            getViewModel().getEmulation().getSoftwareKeyboardFunctions()
                                    .onKeyboardFragmentDismiss((KeyboardFragment) f);
                        }
                        getViewModel().setKeyboardVisible(isInLandscape(), View.GONE);
                    } else {
                        quitEmulation();
                    }
                }
            }
        });
    }
    private void setImmersiveMode() {
        View decorView = getWindow().getDecorView();
        int uiOptions = (View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                | View.SYSTEM_UI_FLAG_FULLSCREEN);
        if (!BuildConfig.DEBUG || android.os.Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            uiOptions |= View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION;
        }
        decorView.setSystemUiVisibility(uiOptions);
    }

    @Override
    public void onWindowFocusChanged(final boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);
        if (hasFocus) {
            setImmersiveMode();
        }
    }
    private int mBottomInset = 0;
    int getBottomInset() {
       return mBottomInset;
    }
    @SuppressLint({"SourceLockedOrientationActivity", "MissingInflatedId"})
    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        System.loadLibrary("jni");
        super.onCreate(savedInstanceState);
        addOnBackPressedCallback(this);
        setImmersiveMode();

        if (savedInstanceState != null) {
            mPauseEmulationImmediately = savedInstanceState.getBoolean("saved", false);
        } else {
            mPauseEmulationImmediately = false;
        }
        if (android.os.Debug.isDebuggerConnected()) {
            android.os.Debug.waitForDebugger();
        }
        setContentView(R.layout.activity_emulation);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT_WATCH) {
            findViewById(R.id.screen).setOnApplyWindowInsetsListener(
                    new View.OnApplyWindowInsetsListener() {
                        @NonNull
                        @Override
                        public WindowInsets onApplyWindowInsets(final @NonNull View v,
                                                                final @NonNull WindowInsets insets) {
                            mBottomInset = Math.max(mBottomInset, insets.getSystemWindowInsetBottom());
                            v.onApplyWindowInsets(insets);
                            return insets;
                        }
                    });
        }
        mGlobalLaoyutListenerAdded = false;
        findViewById(R.id.screen).setOnTouchListener((v, event)
                -> findViewById(R.id.crowded_area).dispatchTouchEvent(event)
                || findViewById(R.id.statusbar).dispatchTouchEvent(event)
                || findViewById(R.id.screen).performClick());
        mSurfaceView = findViewById(R.id.gv_monitor);
        SoftkeysLinearLayout softkeys = findViewById(R.id.softkeys);
        softkeys.setOnVisibilityChangedListener(() ->
                ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor)).onGeometryChanged());
        String pkg = getApplication().getPackageName();
        String errormessage = getIntent().getStringExtra("errormessage");
        if (getIntent().getSerializableExtra(pkg + ".ViewModel") != null) {
            mViewModel = (EmulationViewModel)
                    getIntent().getSerializableExtra(pkg + ".ViewModel");
            if (mViewModel != null) {
                mViewModel.setEmulation(Emulationslist.getCurrentEmulation());
            }
        }


        EmulationConfiguration conf = getInitialConfiguration();
        getViewModel().setConfiguration(conf);
        getViewModel().initializeKeyboardDefaultVisibility(this, conf);
        updateOrientation(conf);


        prepareEmulationStart(conf);
        if (!isTv()) {
            createTouchKeyboard();
        }
        if (errormessage != null) {
            Toast.makeText(this, errormessage, Toast.LENGTH_LONG).show();
        }

    }
    private boolean launchInitialSetting(final EmulationConfiguration conf) {
        Useropts opts = new Useropts();
        opts.setCurrentEmulation(this, conf.getEmulatorId(), conf.getId());
        if (conf instanceof ConfigurationFactory.StreamConfiguration
                && opts.getBooleanValue("__show_settings__", true)) {
            opts.setValue(Useropts.Scope.CONFIGURATION, "__show_settings__", false);
            SubActivityBlockFragment.add(this);
            ArrayList<Integer> opensections = new ArrayList<>(
                    getViewModel().getOpenSettingsSections());
            for (int section : Arrays.asList(R.id.joystickgroup, R.id.mobile_options)) {
                if (!opensections.contains(section)) {
                    opensections.add(section);
                }
            }
            getViewModel().setOpenSettingsSections(opensections);
            mSettingsEvaluated = false;
            mInitialSettingsLauncher.launch(createSettingsConfiguration(false));
            return true;
        }
        return false;
    }
    private void prepareEmulationStart(final @NonNull  EmulationConfiguration conf) {
        String pkg = getApplication().getPackageName();
        getViewModel().setConfiguration(conf);
        if (getViewModel().getEmulation() == null) {
            if (getIntent().getBooleanExtra(pkg + ".Relaunch",
                    false)) {
                getViewModel().setEmulation(Emulationslist.getCurrentEmulation());
            } else {
            getViewModel().setEmulation(Emulationslist.createEmulation(
                    getEmulationUi(), conf));
            }
            getViewModel().getEmulation().setEmulationUI(getEmulationUi());
            if (getViewModel().getEmulation() != null) {
                getViewModel().setSnapshotPath(getSnapshotPath(conf));
                if (getViewModel().getPreferredJoystickTypes() == null) {
                    getViewModel().setPreferredJoystickTypes(conf.getVirtualJoystickTypes());
                }
            }
        }
        if (getViewModel().getEmulation() != null) {
            conf.apply(this, getViewModel().getEmulation());
        }
        if (!launchInitialSetting(conf)) {
            findViewById(R.id.screen).addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                //CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                @Override
                public void onLayoutChange(final View view, final int i, final int i1, final int i2,
                                           final int i3, final int i4, final int i5, final int i6,
                                           final int i7) {
                    Runnable r = () -> {
                        Emulation emu = getViewModel().getEmulation();
                        if (emu != null) {
                            if (!emu.isRunning()) {
                                new Thread(() -> runOnUiThread(() -> {
                                    emu.startThread();
                                    if (!getViewModel().hasNoPauseReasons()) {
                                        emu.setPaused(true);
                                    }
                                    removePreparingDialogs();
                                })).start();
                            }
                        }
                    };
                    if (getIntent().getAction() != null && getIntent().getAction()
                            .startsWith("WIDGET_CALL:")) {

                        executeEmulationStart(conf, () -> downloadFiles(conf, r,
                                de -> showErrorMessage(de.getLocalizedMessage())));

                    } else {
                        executeEmulationStart(conf, r);
                    }
                    view.removeOnLayoutChangeListener(this);
                }
            });

        }
    }

    private  SettingsActivity.SettingsConfiguration createSettingsConfiguration(
            final boolean isModelFix) {
        return new SettingsActivity.SettingsConfiguration(getViewModel(), isModelFix,
                new SettingsActivity.CurrentEmulationCallbacks(getViewModel().getEmulation(),
                        getViewModel().getConfiguration(), getViewModel()));
    }

    @SuppressLint("SourceLockedOrientationActivity")
    private void updateOrientation(final EmulationConfiguration conf) {
        switch (getEmulationScreenOrientation(conf)) {
            case PORTRAIT:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_USER_PORTRAIT);
                break;
            case LANDSCAPE:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_USER_LANDSCAPE);
                break;
            default:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_FULL_USER);
                break;
        }
    }

    @SuppressLint("MissingSuperCall")
    @Override
    protected void onNewIntent(final Intent intent) {
        if (intent.getAction() != null) {
            String[] parts = intent.getAction().split(":", 2);
            if (parts.length == 2) {
                String id = parts[1];
                EmulationConfiguration newConf = getWidgetConfiguration(id);
                EmulationConfiguration oldConf = getViewModel().getConfiguration();
                if (newConf != null && oldConf != null
                        && newConf.getId().equals(oldConf.getId())
                        && newConf.getEmulatorId().equals(oldConf.getEmulatorId())) {
                    return;
                }
            }
            if (Emulationslist.getCurrentEmulation() != null
                    && getViewModel().getConfiguration() != null
                    && intent.getAction() != null
                    && intent.getAction().startsWith("WIDGET_CALL:")) {
                getDialogsController().hidePauseDialog();
                getViewModel().setLifecyclePauseReason(null);
                getViewModel().getEmulation().setPaused(true);
                if (parts.length == 2) {
                    String id = parts[1];
                    EmulationConfiguration newConf = getWidgetConfiguration(id);
                    if (newConf != null) {
                        SubActivityBlockFragment.add(this);
                        cancelRunningEmulation(getViewModel().getConfiguration().getName(),
                                newConf.getName(), () -> {
                                    getViewModel().setEmulation(Emulationslist
                                            .createEmulation(getEmulationUi(), newConf));
                                    getViewModel().setConfiguration(newConf);
                                    Runnable r = () -> {
                                        if (getViewModel().getEmulation() != null) {
                                            if (!getViewModel().getEmulation().isRunning()) {
                                                getViewModel().getEmulation().startThread();
                                            }
                                        }
                                    };
                                    if (!launchInitialSetting(newConf)) {
                                        Runnable rStart = () -> downloadFiles(newConf, r,
                                                de -> showErrorMessage(de.getLocalizedMessage()));
                                        executeEmulationStart(newConf, rStart);
                                        evaluateSettings(intent);
                                        getViewModel().updateKeyboardDefaultVisibility(
                                                this, getViewModel().getConfiguration());
                                        applyKeyboardSettings();
                                        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
                                        updateUi();
                                    }

                                }, () -> {
                                    if (!SubActivityBlockFragment.exists(this)) {
                                        showPauseDialog(false);
                                    }
                                });
                    }
                }
            } else {
                if (!SubActivityBlockFragment.exists(this)
                        && !getDialogsController().isDialogVisible()) {
                    showPauseDialog(false);
                }
            }
        } else {
            EmulationConfiguration conf = AudioService.extractConfiguration(intent);
            if (conf != null) {
                getViewModel().setConfiguration(conf);
            }
            if (intent.hasExtra(PAUSE_FROM_NOTIFICATION)) {
                getIntent().putExtra(PAUSE_FROM_NOTIFICATION,
                        intent.getBooleanExtra(PAUSE_FROM_NOTIFICATION, false));
            } else {
                getIntent().removeExtra(PAUSE_FROM_NOTIFICATION);
            }
            super.onNewIntent(intent);
        }
    }

    private final Set<Integer> mPressedButtons = new LinkedHashSet<>();
    private Bitmap mBitmap = null;

    @Override
    protected void onResume() {
        super.onResume();
        AudioService.hideResumeNotification(this);
        applyKeyboardSettings();
        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
        getViewModel().getEmulation().setEmulationUI(getEmulationUi());
        MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
        gl.increaseLayoutBlockers();
        gl.setScrollView(findViewById(R.id.scroll));
        mPressedButtons.clear();
        getDialogsController().restoreMenu();
        boolean blockebydialog = getDialogsController().isDialogVisible();
        if (getViewModel().isResumed()) {
            if (!SubActivityBlockFragment.exists(this)) {
                if (!blockebydialog) {
                    showPauseDialog(false);
                } else {
                    getViewModel().getEmulation().setPaused(true);
                }
            } else {
                if (mViewModel.isAskingForPermissions()) {
                    mViewModel.setAskingForPermissions(false);
                } else {
                    SubActivityBlockFragment.remove(this);
                }
            }
        }
        if (blockebydialog) {
            findViewById(R.id.dialogsview).setVisibility(View.VISIBLE);
        }
        getViewModel().setResumed(true);
        updateOrientation(getViewModel().getConfiguration());
        updateUi();
        if (getViewModel().getLifecyclePauseReason() != null) {
            if (blockebydialog) {
                mViewModel.removePauseReason(getViewModel().getLifecyclePauseReason());
            } else {
                removePauseReason(getViewModel().getLifecyclePauseReason());
            }
        }
        for (Fragment f : getSupportFragmentManager().getFragments()) {
            if (f instanceof DialogsView.BaseFragment) {
                DialogsView.BaseFragment bf = (DialogsView.BaseFragment) f;
                if (bf.getFragmentType() == BACKGROUND_MODE) {
                    removePauseReason(bf.getPauseReason());
                    getEmulationUi().setScreenRefresh(false);
                }
            }
        }
        if (getIntent().hasExtra(PAUSE_FROM_NOTIFICATION)) {
            if (getIntent().getBooleanExtra(PAUSE_FROM_NOTIFICATION, true)) {
                showPauseDialog(false);
                Fragment toRemove = null;
                for (Fragment f : getSupportFragmentManager().getFragments()) {
                    if (f instanceof DialogsView.BaseFragment) {
                        DialogsView.BaseFragment bf = (DialogsView.BaseFragment) f;
                        if (bf.getFragmentType() == BACKGROUND_MODE) {
                            toRemove = f;
                        }
                    }
                }
                if (toRemove != null) {
                    getSupportFragmentManager().beginTransaction().remove(toRemove).commitNow();
                }
            } else {
                getViewModel().getEmulation().setPaused(false);
            }
        }

        if (!EmulationWatchdogService.isRunning(this)) {
            final Intent i = new Intent(this, EmulationWatchdogService.class);
            File f = mEmulationUi.getDirtyClosedSaveState(getViewModel().getConfiguration());
            if (f != null) {
                i.putExtra("path", f.getAbsolutePath());
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM) {
                try {
                    startService(i);
                } catch (Exception e) {
                    findViewById(android.R.id.content).postDelayed(() -> {
                        try {
                            startService(i);
                        } catch (Exception e2) {
                            // so what...
                        }
                    }, 250);

                }
            } else {
                startService(i);
            }
        }
        if (getViewModel() != null && getViewModel().getEmulation() != null
                && getViewModel().getEmulation().isPaused()) {
            gl.setBitmap(getViewModel().getEmulation()
                    .getCurrentBitmap(getViewModel().getCurrentDisplayId()));

        }
        gl.getViewTreeObserver().addOnGlobalLayoutListener(
                new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                gl.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                gl.decreaseLayoutBlockers();
            }
        });
    }

    private void applyKeyboardSettings() {
        if (getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE
                || getViewModel().isKeyboardForced()) {
                setKeyboardVisibility(autoShowKeyboard() ? View.VISIBLE : View.GONE);
        } else {
            setKeyboardVisibility(View.GONE);
        }
    }
    private Configuration mCurrentConfig = new Configuration();
    @Override
    public void onConfigurationChanged(@NonNull final Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        mSurfaceView.setScrollView(findViewById(R.id.scroll));
        mGlobalLaoyutListenerAdded = false;
        int orientation = mCurrentConfig.orientation;
        View rootView = findViewById(android.R.id.content);
        if (orientation == 0) {
            if (rootView.getWidth() < rootView.getHeight()) {
                orientation = Configuration.ORIENTATION_PORTRAIT;
            } else {
                orientation = Configuration.ORIENTATION_LANDSCAPE;
            }
        }
        mSurfaceView.increaseLayoutBlockers();
        if (newConfig.orientation != orientation) {
            mSurfaceView.calcTransformation();
            getDialogsController().populateMenu();
            findViewById(R.id.screen).addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                @Override
                // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
                public void onLayoutChange(final View v,
                                           final int left, final int top,
                                           final int right, final int bottom,
                                           final int oldLeft, final int oldTop,
                                           final int oldRight, final int oldBottom) {
                    findViewById(R.id.screen).removeOnLayoutChangeListener(this);
                }
            });

        }
        SoftkeysLinearLayout softkeys = findViewById(R.id.softkeys);
        softkeys.setOnVisibilityChangedListener(() ->
            mSurfaceView.onGeometryChanged());
        if (!isTv()) {
            createTouchKeyboard();
        }
        for (Joystick joy : mAvailableJoysticks) {
            joy.onConfigurationChanged(newConfig);
        }
        updateUi();
        getViewModel().setResumed(true);

        mSurfaceView.setVisibility(View.VISIBLE);
        DialogFragment f = (DialogFragment) getSupportFragmentManager()
                .findFragmentByTag(FRAGMENT_DIALOG);
        if (f != null) {
            getSupportFragmentManager()
                    .beginTransaction()
                    .detach(f)
                    .commitNowAllowingStateLoss();
            getSupportFragmentManager()
                    .beginTransaction()
                    .attach(f)
                    .commitAllowingStateLoss();
        }
        mCurrentConfig = new Configuration(newConfig);
        rootView.getViewTreeObserver().addOnGlobalLayoutListener(
                new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                rootView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                if (mGlobalLaoyutListenerAdded) {
                    rootView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                }
                rootView.post(() -> ((TouchDisplayRelativeLayout) findViewById(R.id.crowded_area))
                        .setBottomMargin(getKeyboardVisibility() == View.GONE));
                mSurfaceView.decreaseLayoutBlockers();
            }
        });
        setKeyboardMargin();
        mGlobalLaoyutListenerAdded = true;
    }

    @Override
    public void onPause() {
        super.onPause();
        if (mRestoreFromMovieMode != null) {
            mRestoreFromMovieMode.run();
            mRestoreFromMovieMode = null;
        }
        if (getViewModel().getEmulation() != null) {
            if (!getViewModel().isPausedByUser()) {
                PowerManager pm = (PowerManager) getSystemService(Context.POWER_SERVICE);
                getViewModel().setStopped(getViewModel().isStopped() || !pm.isScreenOn());
            }
        }
        disconnectJoysticks();
        String pkg = getPackageName();
        Intent intent = new Intent(this, EmulationActivity.class);
        intent.putExtra(pkg + ".Relaunch", true);
        intent.putExtra(pkg + ".ViewModel", getViewModel());
        boolean playInBackground = false;
        for (Fragment f : getSupportFragmentManager().getFragments()) {
            if (f instanceof DialogsView.BaseFragment) {
                DialogsView.BaseFragment bf = (DialogsView.BaseFragment) f;
                if (bf.getFragmentType() == BACKGROUND_MODE) {
                    playInBackground = true;
                    break;
                }
            }
        }
        if (!isFinishing()) {
            if (!playInBackground) {
                getViewModel().setLifecyclePauseReason(addPauseReason());
            }
            if (!SubActivityBlockFragment.exists(this)) {
                if (playInBackground) {
                    AudioService.showResumeNotification(this, intent, getViewModel()
                            .getConfiguration().getName(), getViewModel().getEmulation()
                            .isPaused());
                }
            }
        }
    }

    @Override
    public boolean dispatchKeyEvent(final KeyEvent event) {
        return onKey(event.getKeyCode(), event) || super.dispatchKeyEvent(event);
    }
    @Override
    protected void onAvailableKeyboardsChanged(final boolean hardware, final boolean software) {
        getViewModel().setHardwareKeyboardAvailable(hardware);
        getViewModel().setSoftwareKeyboardAvailable(software);
        super.onAvailableKeyboardsChanged(hardware, software);
        if (isJoystickListChanged(mAvailableJoysticks)) {
            checkRequiredJoysticks(mAvailableJoysticks);
        }
        updateTouchKeyboard();
    }

    List<Joystick> getAvailableJoysticks() {
        return mAvailableJoysticks;
    }

    private boolean isJoystickListChanged(final List<Joystick> newList) {
        final List<Joystick> oldList = mAvailableJoysticks;
        if (oldList.size() != newList.size()) {
            return true;
        }
        for (Joystick o : oldList) {
            boolean found = false;
            for (Joystick n : newList) {
                if (o.getId().equals(n.getId())) {
                    found = true;
                }
            }
            if (!found) {
                return true;
            }
        }
        return false;
    }
    private void updateJoystickSettings() {
        int requiredcount;
        int requiredport = PORT_NOT_CONNECTED;
        if (Emulationslist.getCurrentEmulation() != null) {
            Emulation.JoystickFunctions jf = Emulationslist.getCurrentEmulation()
                    .getJoystickFunctions();
            if (jf != null) {
                Emulation.InputDeviceType type = null;

                List<Integer> allPorts = new ArrayList<>(jf.getJoystickports().keySet());
                List<Integer> requiredPorts = new ArrayList<>(getViewModel().getConfiguration()
                        .getRequiredJoystickports(allPorts));
                requiredcount = requiredPorts.size();
                if (requiredcount == 1) {
                    requiredport = requiredPorts.get(0);
                    type = getViewModel().getConfiguration().getInputDeviceTypes()
                            .get(requiredcount);
                }
                Map<Integer, Emulation.InputDeviceType> types = new HashMap<>();
                for (int port : allPorts) {
                    types.put(port, null);
                }
                for (Joystick joy : mAvailableJoysticks) {
                    if (requiredcount == 1) {
                        if (joy.getSupportedDeviceTypes().contains(type)) {
                            joy.setPortnumber(requiredport);
                            types.put(requiredport, type);
                        } else {
                            joy.setPortnumber(PORT_NOT_CONNECTED);
                        }
                    } else {
                        joy.readUseropts(this);
                        types.put(joy.getPortnumber(), joy.getCurrentDeviceType());
                    }
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        if (joy instanceof TouchJoystick) {
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.DIRECTIONAL);
                        }
                        if (joy instanceof WheelJoystick) {
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.WHEEL);
                        }
                        if (getViewModel().isJoystickAutofireToggled(joy.getPortnumber())) {
                            joy.setAutofiring(true);
                        }
                    }
                }
                for (int port : types.keySet()) {
                    jf.setActiveInputDeviceType(port, types.get(port));
                }
                checkRequiredJoysticks(mAvailableJoysticks);
            }
        }
        getSoftkeyConfigView().setJoysticks(mAvailableJoysticks, findViewById(R.id.softkeys));
        restoreSoftkeys();
    }
    private final ActivityResultLauncher<Intent> mCreateDocumentLauncher
            = registerForActivityResult(new FileCreateionContract(), result -> {
        if (result != null && result.getData() != null) {
            onFileCreated(result.getData());
        } else {
            onFileOpenCancelled(false);
        }
    });
    private static final String RESULT = "____RESULT__";
    private final ActivityResultLauncher<Intent> mHandleImageLauncher
            = registerForActivityResult(new ActivityResultContract<Intent, Intent>() {
                @Override
                public Intent parseResult(final int result, final @Nullable Intent intent) {
                    if (intent != null) {
                        intent.putExtra(RESULT, result);
                    }
                    return intent;
                }

                @NonNull
                @Override
                public Intent createIntent(final @NonNull Context context, final Intent intent) {
                    return intent;
                }
            }, result -> {
                if (result != null && result.getBooleanExtra(RESULT, false) && !openFile(false)) {
                    showErrorDialog(getString(R.string.could_not_access_storage),
                            getString(R.string.check_content_provider));
                }
                handleCommonLaunchresult(result);
            });
    void createFile(final String defaultFilename) {
        SubActivityBlockFragment.add(this);
        getViewModel().addSubActivityPauseReason();
        Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.putExtra(Intent.EXTRA_TITLE, defaultFilename);
        intent.setType("application/*");
        muteEmulation();
        mCreateDocumentLauncher.launch(BaseActivity.createChooserIntent(intent, null));
    }
    @Override
    protected void onAvailableJoysticksChanged(final @NonNull  List<Joystick> joysticks) {
        if (!joysticks.isEmpty()) {
            joysticks.add(new MultiplexerJoystick(this, joysticks));
        }
        super.onAvailableJoysticksChanged(joysticks);
        boolean changed = isJoystickListChanged(joysticks);
        Log.v(getClass().getSimpleName(), "onAvailableJoysticksChanged: " + changed);
        if (changed) {
            mAvailableJoysticks.clear();
            mAvailableJoysticks.addAll(joysticks);
            updateJoystickSettings();
            setDynamicUiElements();
            Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
            if (jf != null) {
                List<Integer> allPorts = new ArrayList<>(jf.getJoystickports().keySet());
                List<Integer> requiredPorts = new ArrayList<>(getViewModel().getConfiguration()
                        .getRequiredJoystickports(allPorts));
                int requiredcount = requiredPorts.size();
                for (Joystick joy : joysticks) {
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        requiredcount--;
                    }
                }
                if (requiredcount == 1) {
                    for (Joystick joy : joysticks) {
                        joy.setPortnumber(requiredPorts.get(0));
                    }
                }
                Runnable removeOldFragment = () -> {
                    for (Fragment f : getSupportFragmentManager().getFragments()) {
                        if (f instanceof RequiredJoysticksFragment) {
                            getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                        }
                    }
                };
                if (requiredcount > 0) {
                    Object additionalPauseReason = addPauseReason();
                    removeOldFragment.run();
                    removePauseReason(additionalPauseReason);
                    EmulationConfiguration conf = getViewModel().getConfiguration();
                    getDialogsController().showUiElement(JOYSTICKSELECTION,
                                    RequiredJoysticksFragment.getBundle(
                                            conf.getVirtualJoystickTypes(),
                                            conf.getInputDeviceTypes(),
                                            requiredPorts
                                            ));
                } else {
                    removeOldFragment.run();
                    List<String> joysticknames = new ArrayList<>();
                    List<Integer> configuredJoysticks = getViewModel().getConfiguration()
                            .getUsedJoystickports(allPorts);
                    if (configuredJoysticks != null) {
                        for (int i : configuredJoysticks) {
                            boolean found = false;
                            for (Joystick joy : mAvailableJoysticks) {
                                if (joy.getPortnumber() == i) {
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                joysticknames.add(jf.getJoystickports().get(i));
                            }
                        }
                    }
                    if (!joysticknames.isEmpty()) {
                        int length = isInUnitTest() ? Toast.LENGTH_LONG : Toast.LENGTH_SHORT;
                        Toast.makeText(this,
                                getString(R.string.not_connected,
                                        String.join(", ", joysticknames)),
                                length).show();
                    }
                }
            }
        }
    }

    @Override
    void updateHardware() {
        super.updateHardware();
        if (getViewModel().getEmulation().getJoystickFunctions() != null) {
            autosetJoystickports(getViewModel().getEmulation()
                            .getJoystickFunctions(), getAvailableJoysticks());
        }
    }

    private void autosetJoystickports(final Emulation.JoystickFunctions jf,
                                      final List<Joystick> joysticks) {
        Map<Integer, EmulationConfiguration.PreferredJoystickType> preferred =
                getViewModel().getPreferredJoystickTypes();
        if (preferred.isEmpty()) {
            preferred = getViewModel().getConfiguration().getVirtualJoystickTypes();
        }
        getViewModel().getConfiguration().adaptJoysticks(this, jf, joysticks,
                preferred);
    }

    private void checkRequiredJoysticks(final List<Joystick> joysticks) {
        if (getViewModel().getEmulation() != null) {
            Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
            if (jf != null && !joysticks.isEmpty()) {
                autosetJoystickports(jf, joysticks);
            }
        }
    }

    void onKeyboardVisibilityChanged(final int visibility, final InputDevice dev) {
        getViewModel().setKeyboardVisible(isInLandscape(), visibility);
        setKeyboardVisibility(visibility, dev);
        ((TouchDisplayRelativeLayout) findViewById(R.id.crowded_area))
                .setBottomMargin(visibility == View.GONE);

    }
    void setKeyboardVisibility(final int visibility) {
        setKeyboardVisibility(visibility, null);
    }
    @SuppressLint("SourceLockedOrientationActivity")
    void setKeyboardVisibility(final int visibility, final InputDevice dev) {
        View kv = findViewById(R.id.keyboardview);
        if (visibility != findViewById(R.id.keyboardview).getVisibility()) {
            if (isTv()) {
                Emulation.SoftwareKeyboardFunctions skf = mViewModel.getEmulation()
                        .getSoftwareKeyboardFunctions();
                if (skf != null) {
                    if (visibility == View.VISIBLE) {
                        Fragment f = getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD);
                        if (f != null) {
                            getSupportFragmentManager().beginTransaction().remove(f).commit();
                        }
                        ViewGroup screen = findViewById(R.id.screen);
                        screen.setDescendantFocusability(ViewGroup.FOCUS_AFTER_DESCENDANTS);
                        TvKeyboardFragment dlg = new TvKeyboardFragment(
                                mViewModel.getEmulation().getSoftwareKeyboardFunctions());
                        dlg.setIsShownListener(new TvKeyboardFragment.TvKeyboardShownListener(
                                EmulationActivity.this, () -> getSupportFragmentManager()
                                .beginTransaction().remove(dlg).commitNow()));
                        dlg.setInputDevice(dev);
                        dlg.show(getSupportFragmentManager(), REMOTE_KEYBOARD);
                    } else {
                        Fragment f = getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD);
                        if (f != null) {
                            getSupportFragmentManager().beginTransaction().remove(f).commit();
                        }
                    }
                }
            } else {
                kv.setVisibility(visibility);
                ((SoftkeysLinearLayout) findViewById(R.id.softkeys))
                        .onKeyboardVisibleChanged(kv.getVisibility() == View.VISIBLE);
                ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                        .onGeometryChanged();
                setKeyboardMargin();
            }
        }
    }
    private void setKeyboardMargin() {
        if (!isTv()) {
            View kv = findViewById(R.id.keyboardview);
            RelativeLayout.LayoutParams lp = (RelativeLayout.LayoutParams) kv.getLayoutParams();
            lp.bottomMargin = getResources().getBoolean(R.bool.is_landscape) ? 0 : getResources().getDimensionPixelSize(R.dimen.toplevel_margin);
            kv.setLayoutParams(lp);
        }
    }

    @Override
    public boolean onGenericMotionEvent(final MotionEvent event) {
        if (mRestoreFromMovieMode == null) {
            for (Joystick joy : mAvailableJoysticks) {
                if (joy instanceof View.OnGenericMotionListener) {
                    if (((View.OnGenericMotionListener) joy).onGenericMotion(null, event)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private boolean autoShowKeyboard() {
        if (getViewModel().isKeyboardRequesteByEmulation()) {
            return true;
        }
        if (getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE) {
            return true;
        }
        return getViewModel().isKeyboardVisibibleFromStart(isInLandscape()) && !isTv();
    }

    void showFliplistDialog() {
        getDialogsController().showUiElement(FLIPLIST);
    }

    private MonitorGlSurfaceView mSurfaceView = null;

    void setBitmap(final int displayId, final Bitmap bitmap) {
        if (mSurfaceView != null && displayId == mViewModel.getCurrentDisplayId()) {
            mSurfaceView.setBitmap(bitmap);
        }
        if (bitmap != null && displayId == mVisibleMonitor) {
            mBitmap = bitmap;
        }
    }
    void switchMonitor() {
        Emulation.DualMonitorFunctions mf = getViewModel().getEmulation()
                .getDualMonitorFunctions();
        if (mf != null && mf.getDisplays().size() == 2) {
            for (int key : mf.getDisplays().keySet()) {
                if (key != getViewModel().getCurrentDisplayId()) {
                    getViewModel().setCurrentDisplayId(key);
                    setVisibleMonitor(key);
                    getCurrentUseropts().setValue(
                            Useropts.Scope.CONFIGURATION, "active_monitor", key);
                    restoreStatusbar(findViewById(R.id.st_statusbar));
                    Toast.makeText(this,
                            getResources().getString(R.string.toast_switched_monitor,
                                    mf.getDisplays().get(key)), Toast.LENGTH_SHORT).show();
                    break;
                }
            }
        }
        if (mf != null && mf.getDisplays().size() > 2) {
            getDialogsController().showUiElement(SWITCH_MONITOR, false);
        }
    }
    Bitmap getCurrentBitmap() {
        return mBitmap;
    }

    void removePauseReasonAfterDelay(final Object pauseReason) {
        pauseAfterTimeMachine(() -> removePauseReason(pauseReason));
    }

    int getSoftkeyboardType() {
        String key = getResources().getString(R.string.useropt_keyboard_variants);
        String valCompact = getResources().getString(R.string.value_keyboard_variant_compact);
        String valExact = getResources().getString(R.string.value_keyboard_variant_exact);

        if (getCurrentUseropts().getStringValue(key, valCompact).equals(valExact)) {
            return Emulation.SOFTKEYBOARD_EXACT_IF_POSSIBLE;
        } else {
            return Emulation.SOFTKEYBOARD_ALWAYS_COMPACT;
        }
    }
    private void createTouchKeyboard() {
        Emulation.SoftwareKeyboardFunctions skf =
                getViewModel().getEmulation().getSoftwareKeyboardFunctions();
        if (skf != null) {
            int type = getSoftkeyboardType();
            Fragment f = skf.createKeyboardFragment(type);
            try {
                FragmentManager fm = getSupportFragmentManager();
                if (f != null) {
                    Fragment toDelete = fm.findFragmentByTag(KEYBOARDFRAGMENT);
                    FragmentTransaction fta = fm.beginTransaction();
                    if (toDelete != null) {
                        fta = fta.remove(toDelete);
                    }
                    fta.add(R.id.keyboardview, f, KEYBOARDFRAGMENT).commitNow();
                }
            } catch (IllegalStateException e) {
                // dirty
            }
        }
    }

    int getKeyboardVisibility() {
        return findViewById(R.id.keyboardview).getVisibility();
    }

    private void updateTouchKeyboard() {
        Integer keyboardVisibility;
        if (getViewModel().isSoftwareKeyboardAvailable()) {
            keyboardVisibility = getViewModel().getKeyboardVisible(isInLandscape());
        } else {
            if (getViewModel().isKeyboardVisibibleFromStart(isInLandscape())) {
                keyboardVisibility = View.VISIBLE;
            } else {
                keyboardVisibility = View.GONE;
            }
        }
        if (keyboardVisibility != null) {
            if (getKeyboardVisibility() != View.INVISIBLE
                    && getKeyboardVisibility() != keyboardVisibility) {
                setKeyboardVisibility(keyboardVisibility);
            }
        }
    }

    void showPauseDialog(final boolean extended) {
        getDialogsController().showUiElement(
                extended ? DialogsView.UiElement.PAUSE_EXTENED
                        : DialogsView.UiElement.PAUSE);
    }

    private void runEmulation() {
        Runnable starter = () -> {
            hidePreparingDialogs();
            if (getViewModel().getEmulation() != null) {
                if (!getViewModel().getEmulation().isRunning()) {
                    getViewModel().getEmulation().startThread();
                }
            }
        };
        if (getIntent().getAction() != null
                && getIntent().getAction().startsWith("WIDGET_CALL:")) {
            downloadFiles(getViewModel().getConfiguration(), starter, de
                    -> AlertDialogBuilder.showStyledDialog(new AlertDialogBuilder(this)
                    .setIcon(android.R.drawable.ic_dialog_alert)
                    .setTitle(R.string.download_error)
                    .setMessage(getResources().getString(R.string.restart_tile_from_main,
                            de.getMessage(),
                            getViewModel().getConfiguration().getName(),
                            getResources().getString(R.string.app_name)))
                    .setPositiveButton(getResources().getString(R.string.start_app,
                                    getResources().getString(R.string.app_name)),
                            (dialogInterface, i) -> {
                                Intent intent = new Intent(this, MainActivity.class);
                                startActivity(intent);
                                dialogInterface.dismiss();
                            })
                    .setOnDismissListener(dialogInterface -> finish())
                    .create()));
        } else {
            prepareEmulationStart(getViewModel().getConfiguration());
        }
    }

    void restoreStatusbar(final StatusbarView sb) {
        sb.restore(getViewModel().getEmulation(), getViewModel(), mAvailableJoysticks, isTv(),
                () -> getDialogsController().showUiElement(TAPE_CONTROL, false));
        setVisibleMonitor(getViewModel().getCurrentDisplayId());
    }
    private EmulationUiImpl mEmulationUi = null;
    EmulationUiImpl getEmulationUi() {
        if (mEmulationUi == null || mEmulationUi.getActivity() != this) {
            mEmulationUi = new EmulationUiImpl(this);
        }
        return mEmulationUi;
    }
    void updateUi() {
        ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor)).setUseropts(getCurrentUseropts());

        if (getViewModel().getEmulation() != null) {
            if (!isTv()) {
                updateTouchKeyboard();
            }
        }
        updateHardware();
        updateJoystickSettings();
        setDynamicUiElements();
        Emulation.DualMonitorFunctions dmf = getViewModel().getEmulation()
                .getDualMonitorFunctions();
        if (!mMonitorSet) {
            if (dmf != null) {
                Resources res = getResources();
                String machineSettingsVal = res.getString(R.string.value_initial_monitor_machine);
                String monitorKey = res.getString(R.string.useropt_initial_monitor);
                int monitor = dmf.getDefaultMonitor();
                if (!machineSettingsVal.equals(
                        getCurrentUseropts().getStringValue(monitorKey, null))) {
                    monitor = getCurrentUseropts().getIntegerValue("active_monitor", monitor);
                    dmf.setActiveMonitor(monitor);
                }
                getViewModel().setCurrentDisplayId(monitor);
            } else {
                getViewModel().setCurrentDisplayId(0);
            }
            mMonitorSet = true;
        }
        restoreStatusbar(findViewById(R.id.st_statusbar));
        restoreSoftkeys();

        synchronized ((mUpdateUiMonitor)) {
            mUpdateUiDone = true;
            mUpdateUiMonitor.notifyAll();
        }
    }
    private int mVisibleMonitor = 0;
    void setVisibleMonitor(final int active) {
        getViewModel().setCurrentDisplayId(active);
        if (getViewModel().getEmulation().getDualMonitorFunctions() != null) {
            getViewModel().getEmulation().getDualMonitorFunctions().setActiveMonitor(active);
        }
        mVisibleMonitor = active;
    }

    private void restoreSoftkeys() {

        SoftkeysLinearLayout root = findViewById(R.id.softkeys);
        root.removeAllViews();
        addSoftkeyViews(root, null, null);
        root.setVisibility(isTv() ? View.GONE : root.isSoftkeyVisible(
                getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE));

    }

    void addSoftkeyViews(final SoftkeysLinearLayout vg, @Nullable  final Runnable runAfterPress,
                         @Nullable  final Runnable runAfterRelease) {
        if (getViewModel().getEmulation() != null
                && getViewModel().getEmulation().getSoftkeyFunctions() != null) {
            Emulation.SoftkeyFunctions functions = getViewModel().getEmulation()
                    .getSoftkeyFunctions();
            for (SoftkeyConfigView.Softkey key : getViewModel().getUiSoftkeys()) {
                vg.createSoftkey(key, functions, runAfterPress, runAfterRelease);
            }
        }
        for (String key : getViewModel().getSoftkeyIds()) {
            vg.createSoftkey(key, getViewModel().getJsSoftkey(key),
                    getViewModel().isSoftkeyEnabled(key),
                    getKeyboardVisibility(),
                    getViewModel().getEmulation().getSoftkeyFunctions(),
                    runAfterPress, runAfterRelease);
        }
    }

    private final Object mUpdateUiMonitor = new Object();
    private boolean mUpdateUiDone = false;

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean isTvKeyboardVisible() {
        return getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD) != null;
    }

    private void recenterTriggeringJoystick(final KeyEvent event) {
        for (Joystick joy : mAvailableJoysticks) {
            if (joy.getHardwareId() == event.getDeviceId()) {
                joy.reset();
                joy.notifyListener();
            }
        }
    }
    boolean onKey(final int keyCode, final KeyEvent event) {
        if (getDialogsController().onKey(event)) {
            return true;
        }
        if (event.getAction() == KeyEvent.ACTION_DOWN) {
            mPressedButtons.add(keyCode);
            if (keyCode == KeyEvent.KEYCODE_MEDIA_PAUSE
                    //|| keyCode == KeyEvent.KEYCODE_BACK
                    || keyCode == KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE
                    || keyCode == KeyEvent.KEYCODE_SETTINGS
                    || keyCode == KeyEvent.KEYCODE_MEDIA_FAST_FORWARD
                    || keyCode == KeyEvent.KEYCODE_MENU
                    || keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                    || EmulationUi.JOYSTICK_MINIMAL_ACTION_BUTTONS.contains(keyCode)
                    || EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)
            ) {
                return true;
            }

        }
        boolean wasHitHere = false;
        if (event.getAction() == KeyEvent.ACTION_UP) {
            wasHitHere = mPressedButtons.contains(keyCode);
            mPressedButtons.remove(keyCode);
            if (wasHitHere) {
                if (keyCode == KeyEvent.KEYCODE_MEDIA_PAUSE
                        || keyCode == KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE) {
                    showPauseDialog(false);
                    return true;
                }
                if (EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)) {
                    recenterTriggeringJoystick(event);
                    showPauseDialog(!isTvKeyboardVisible());
                    return true;
                }
                if (keyCode == KeyEvent.KEYCODE_SETTINGS) {
                    SubActivityBlockFragment.add(this);
                    mSettingsLauncher.launch(createSettingsConfiguration(
                            !getViewModel().getConfiguration().isBareMachine()));
                    return true;
                }
                if (keyCode == KeyEvent.KEYCODE_MEDIA_FAST_FORWARD) {
                    if (getViewModel().getEmulation().getSpeedUpFunctions() != null) {
                        Emulation e = getViewModel().getEmulation();
                        e.getSpeedUpFunctions().setMaximumSpeedMode(
                                !e.getSpeedUpFunctions().getMaximumSpeedMode());
                    }
                }
                if (keyCode == KeyEvent.KEYCODE_MENU
                        || keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                        || EmulationUi.JOYSTICK_MINIMAL_ACTION_BUTTONS.contains(keyCode)) {
                    recenterTriggeringJoystick(event);
                    if (!isTvKeyboardVisible()) {
                        showPopupMenu();
                    }
                    return true;
                }
            }
        }
        if (mRestoreFromMovieMode == null && delegateToEmulatedInputDevices(keyCode, event)) {
            return true;
        }
        if (event.getAction() == KeyEvent.ACTION_DOWN) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                return true;
            }
        }
        if (event.getAction() == KeyEvent.ACTION_UP) {
            if (keyCode == KeyEvent.KEYCODE_BACK
                    && event.getRepeatCount() == 0) {
                if (!getViewModel().containsActiveMouse(event.getDeviceId())) {
                    onBackPressed();
                }
                return true;
            }
        }
        if (wasHitHere && keyCode == KeyEvent.KEYCODE_F12) {
            showPopupMenu();
            return true;
        }
        return false;
    }
    private boolean delegateToGamepadFunctions(final SoftkeyConfigView scv,
                                               final KeyEvent event) {
        if (event.getAction() == KeyEvent.ACTION_DOWN && scv.onButtonPressed(event.getKeyCode(),
                 getViewModel().getEmulation().getSoftkeyFunctions())) {
            return true;
        }
        //noinspection RedundantIfStatement
        if (event.getAction() == KeyEvent.ACTION_UP && scv.onButtonReleased(event.getKeyCode(),
                        getViewModel().getEmulation().getSoftkeyFunctions())) {
            return true;
        }
        return false;
    }

    private boolean delegateToGamepadFunctions(final Emulation.GamepadFunctions gf,
                                               final KeyEvent event) {
        if (gf != null) {
            if (event.getAction() == KeyEvent.ACTION_DOWN
                    && gf.onButtonPressed(event.getKeyCode())) {
                return true;
            }
            //noinspection RedundantIfStatement
            if (event.getAction() == KeyEvent.ACTION_UP
                    && gf.onButtonReleased(event.getKeyCode())) {
                return true;
            }
        }
        return false;
    }

    boolean delegateToEmulatedInputDevices(final int keyCode, final KeyEvent event) {
        if (delegateToGamepadFunctions(getViewModel()
                .getEmulation().getGamepadFunctions(), event)) {
            return true;
        }
        if (delegateToGamepadFunctions(getSoftkeyConfigView(), event)) {
            return true;
        }
        for (Joystick joy : mAvailableJoysticks) {
            if (joy instanceof View.OnKeyListener) {
                View.OnKeyListener kbj = (View.OnKeyListener) joy;
                boolean handled = kbj.onKey(null, keyCode, event);
                if (handled) {
                    return true;
                }
            }
        }
        if (getViewModel().getEmulation().getHardwareKeyboardFunctions() != null
                && mViewModel.isHardwareKeyboardAvailable()) {
            if (keyCode != KeyEvent.KEYCODE_BACK) {
                boolean handled = KeycodeHelper.translateKeyEvent(event,
                        getViewModel().getEmulation().getHardwareKeyboardFunctions());
                if (handled) {
                    getViewModel().setKeyboardUsed();
                }
                return handled;
            }
        }
        return false;
    }

    void setDynamicUiElements() {
        Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
        EmulationConfiguration.PreferredJoystickType joystickType = null;
        if (jf != null) {
            boolean multiplexer = false;
            for (Joystick j : mAvailableJoysticks) {
                if (j instanceof MultiplexerJoystick && j.getPortnumber() != PORT_NOT_CONNECTED) {
                    joystickType = ((MultiplexerJoystick) j).getPreferredVirtualType(this);
                    multiplexer = true;
                }
            }
            if (!multiplexer) {
                for (Joystick joy : mAvailableJoysticks) {
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        if (joy instanceof TouchJoystick) {
                            joystickType = EmulationConfiguration.PreferredJoystickType.DIRECTIONAL;
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.DIRECTIONAL);
                        }
                        if (joy instanceof WheelJoystick) {
                            joystickType = EmulationConfiguration.PreferredJoystickType.WHEEL;
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.WHEEL);
                        }
                    }
                }
            }
            MultiplexerJoystick muxer = new MultiplexerJoystick(this, mAvailableJoysticks);
            List<Integer> ports = getViewModel().getConfiguration().getUsedJoystickports(
                    new LinkedList<>(jf.getJoystickports().keySet()));
            muxer.readUseropts(this);

            if (muxer.getPortnumber() == Joystick.PORT_NOT_CONNECTED
                    && ports != null && ports.size() == 1) {
                muxer.setPortnumber(ports.get(0));
            }

            for (Joystick joystick : mAvailableJoysticks) {
                if (!(joystick instanceof VirtualJoystick)) {
                    joystickType = null;
                }

                joystick.connect(this, (joy, devicetype, portnumber,
                                        isAbsolutePosition, xvalue, yvalue, fire) -> {
                    getViewModel().getEmulation().getJoystickFunctions()
                            .onJoystickChanged(joy, devicetype, portnumber,
                                    isAbsolutePosition, xvalue, yvalue, fire);
                    StatusbarView sb = findViewById(R.id.st_statusbar);
                    sb.onJoystickChanged(joy, devicetype, portnumber,
                            isAbsolutePosition, xvalue, yvalue, fire);
                }, getViewModel().getEmulation().getGamepadFunctions(), muxer);
            }
        }
        boolean withBottomMargins = SubActivityBlockFragment.exists(this) ? getKeyboardVisibility() == View.GONE : !autoShowKeyboard();
        ((TouchDisplayRelativeLayout) findViewById(R.id.crowded_area))
                .updateAllViews(this, mAvailableJoysticks, this::showPopupMenu,
                        withBottomMargins, joystickType);
    }
    boolean isInMovieMode() {
        return mRestoreFromMovieMode != null;
    }
    void launchOpenFileActivity() {
        if (!openFile(false)) {
            Object pauseReason = addPauseReason();
            showErrorDialog(getString(R.string.could_not_access_storage),
                    getString(R.string.check_content_provider),
                    () -> removePauseReason(pauseReason));
        }
    }

    void openSettingsDialog() {
        SubActivityBlockFragment.add(this);
        mSettingsEvaluated = false;
        mSettingsLauncher.launch(createSettingsConfiguration(
                !getViewModel().getConfiguration().isBareMachine()));

    }
    void openCreateFilesDialog() {
        SubActivityBlockFragment.add(this);
        mNewFilePropertiesLauncher.launch(null);
    }

    void showPopupMenu() {
        getViewModel().setActiveDialogState(null);
        ((DialogsView) findViewById(R.id.dialogsview)).showMainMenu();
    }

    private void disconnectJoysticks() {
        for (Joystick j : mAvailableJoysticks) {
            j.disconnect();
        }
    }

    void showErrorMessage(final String text) {
        if (text != null && !text.isEmpty()) {
            Toast.makeText(this, text, Toast.LENGTH_LONG).show();
        }
    }
    void finishWithResult(final Intent i) {
        setResult(Activity.RESULT_OK, i);
        removeStreamData(mViewModel.getConfiguration());
        finish();
    }
    void onEmulatorFinished() {
        Emulationslist.disposeCurrentEmulation();
        setResult(RESULT_OK, new Intent());
        try {
            //noinspection unchecked
            Class<TestInterface> clz = (Class<TestInterface>) Objects.requireNonNull(
                    EmulationActivity.class.getClassLoader())
                    .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("isIsolatedEmulationActivityTest");
            if (Boolean.FALSE.equals(method.invoke(clz.newInstance()))) {
                finish();
            }

        } catch (ClassNotFoundException e) {
            removeStreamData(mViewModel.getConfiguration());
            setResult(RESULT_OK);
            finish();
        } catch (NoSuchMethodException | InvocationTargetException
                 | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }
    }

    void quitEmulation() {
        getDialogsController().showUiElement(DialogsView.UiElement.QUIT);
    }

    interface IsShownListener {
        void isShown(Dialog d);
    }

    void showSnapshotsDialog() {
        getDialogsController().showUiElement(SAVESTATES, SaveStateFragment.getBundle(
                this, getCurrentBitmap(), SaveStateFragment.NO_TIMEMACHINE));
    }

    void restoreCanvasSize() {
        MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
        gl.setCanvasGeometry(
        Objects.requireNonNull(getViewModel()
                        .getCanvasSize(getViewModel().getCurrentDisplayId())).x,
                Objects.requireNonNull(getViewModel()
                                .getCanvasSize(getViewModel().getCurrentDisplayId())).y,
                getViewModel().getPixelAspectRatio(getViewModel().getCurrentDisplayId()),
                getViewModel().getScreenAspectRatio(getViewModel().getCurrentDisplayId()));
    }

    @Override
    protected void onStop() {
        super.onStop();
        if (!isFinishing()) {
            getViewModel().setStopped(true);
            MainActivity.setDestroyedData(getViewModel());
        }
    }

    @Override
    protected void onRestoreInstanceState(@NonNull final Bundle savedInstanceState) {
        if (savedInstanceState.getBoolean("saved", false)) {
            if (!isFinishing()) {
                boolean show = true;
                for (Fragment f : getSupportFragmentManager().getFragments()) {
                    if (f instanceof DialogsView.BaseFragment) {
                        DialogsView.BaseFragment bf = (DialogsView.BaseFragment) f;
                        if (bf.getFragmentType() == PAUSE) {
                            show = false;
                            break;
                        }
                    }
                }
                if (show) {
                    showPauseDialog(false);
                }
            }
        }
        super.onRestoreInstanceState(savedInstanceState);
    }
    File getInstanceStatePath(final EmulationConfiguration conf) {
        if (conf != null) {
            return new File(getCacheDir(),
                    File.separator + "instancestate_" + conf.getEmulatorId()
                            + "_" + conf.getId());
        }
        return null;

    }
    private void disposeInstanceState(final EmulationConfiguration conf) {
        File f = getInstanceStatePath(conf);
        if (f != null) {
            if (!f.delete()) {
                f.deleteOnExit();
            }
        }
    }

    @Override
    protected void onSaveInstanceState(@NonNull final Bundle outState) {
        if (!isFinishing() && getViewModel().getEmulation().createStateForNextRestart(
                getInstanceStatePath(getViewModel().getConfiguration()))) {
            outState.putBoolean("saved", true);
        }
        super.onSaveInstanceState(outState);
    }

    private static long counter = 0;
    Serializable addPauseReason() {
        Serializable lock = counter;
        counter = counter + 1;
        mViewModel.addPauseReason(lock);
        if (BuildConfig.DEBUG) {
            Log.v(TAG, String.format("pausereason %d added from %s, now %d", lock,
                    new Exception().getStackTrace()[1].toString(),
                    mViewModel.getPauseReasonsCount()));
        }
        if (!mViewModel.getEmulation().isPaused()) {
            mViewModel.getEmulation().setPaused(true);
        }
        return lock;
    }
    void clearPauseReasons() {
        mViewModel.clearPauseReasons();
        Emulation emu = mViewModel.getEmulation();
        emu.setPaused(false);
        emu.setEmulationUI(getEmulationUi());
        getEmulationUi().setScreenRefresh(true);
        findViewById(R.id.gv_monitor).invalidate();
    }
    void removePauseReason(final Object lock) {
        if (BuildConfig.DEBUG) {
            Log.v(TAG, String.format("pausereason %s to be removed from %s, now %d",
                    lock.toString(), new Exception().getStackTrace()[1].toString(),
                    mViewModel.getPauseReasonsCount()));
        }
        if (mViewModel.removePauseReason(lock)) {
            if (mViewModel.hasNoPauseReasons()) {
                Log.v(TAG, "list is now empty, set pause = false");
                mViewModel.getEmulation().setEmulationUI(getEmulationUi());
                mViewModel.getEmulation().setPaused(false);
            }
        } else if (mViewModel.hasNoPauseReasons()) {
            Log.v(TAG, "list was empty, set pause = false");
            mViewModel.getEmulation().setPaused(false);
            Log.w(getClass().getSimpleName(), String.format(
                    "%s to be removed was never added, may result in a hanging emulation.", lock));
        }
    }

    @Override
    public void finish() {
        disposeInstanceState(getViewModel().getConfiguration());
        EmulationConfiguration conf = getViewModel().getConfiguration();
        if (conf.getId().startsWith("_")) {
            Useropts.delete(this, conf.getEmulatorId(), conf.getId());
            new Deleter(getSnapshotPath(conf.getEmulatorId(), conf.getId()))
                    .run();
        }
        super.finish();
    }

    public static class EmulationWatchdogService extends Service {
        static boolean isRunning(final Context context) {
            final ActivityManager activityManager = (ActivityManager) context.getSystemService(
                    Context.ACTIVITY_SERVICE);
            final List<ActivityManager.RunningServiceInfo> services = activityManager
                    .getRunningServices(Integer.MAX_VALUE);
            String myService = EmulationWatchdogService.class.getName();
            for (ActivityManager.RunningServiceInfo runningServiceInfo : services) {
                Log.d(TAG, String.format("Service:%s", runningServiceInfo.service.getClassName()));
                if (runningServiceInfo.service.getClassName().equals(myService)) {
                    return true;
                }
            }
            return false;
        }
        private File mFile = null;

        @Nullable
        @Override
        public final IBinder onBind(final Intent intent) {
            return null;
        }

        @Override
        public final int onStartCommand(final Intent intent, final int flags, final int startId) {
            if (intent != null) {
                String path = intent.getStringExtra("path");
                if (path != null) {
                    mFile = new File(path);
                }
            }
            return START_STICKY;
        }

        @Override
        public final void onTaskRemoved(final Intent rootIntent) {
            if (mFile != null && mFile.exists()) {
                if (!mFile.delete()) {
                    mFile.deleteOnExit();
                }
            }
            super.onTaskRemoved(rootIntent);
        }
    }
}
