//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.BaseActivity.HTTP_STATUS_SSL_ERROR;

import android.content.Context;
import android.os.Build;
import android.util.Log;

import androidx.annotation.NonNull;

import org.yaml.snakeyaml.Yaml;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.URL;
import java.nio.file.NoSuchFileException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import javax.net.ssl.SSLProtocolException;


/**
 * Download additional assets via https, store them in cache and open them just like local assets.
 */
public class DownloaderFactory {

    private static final int BLOCKSIZE = 8192;
    private static final String TAG = Downloader.class.getSimpleName();
    private static final int HEX_MASK = 0x0F;
    interface HttpResult {
        int getResultcode();
        byte[] getBody();
    }
    interface HttpRequest {
        @NonNull  HttpResult execute(URL url) throws IOException;
    }
    /**
     * Variants of {@link Runnable} to throw Exceptions.
     */
    public abstract static class Downloader {
        /**
         * Start Download.
         * @param httpRequests List of @see {{@link HttpRequest}} instances
         *                     to be called one by one until success
         *                     downloading.
         * @throws AdditionalDownloads.DownloadException if download failed
         * @throws IOException local error, the localized message will be shown as error.
         */
        abstract void run(List<HttpRequest> httpRequests) throws NullPointerException, IOException,
                AdditionalDownloads.DownloadException;
        private URL[] mServices;

        final boolean checkForServices() {
            Log.v(TAG, "checkForServices started");
            if (unittestFakeNetworkError()) {
                return false;
            }
            for (URL url : mServices) {
                final List<Boolean> result = new LinkedList<>();
                new Thread(() -> {
                    try {
                        int port = url.getPort();
                        if (port < 0) {
                            port = url.getDefaultPort();
                        }
                        Socket sock = new Socket();
                        SocketAddress sockaddr = new InetSocketAddress(url.getHost(), port);

                        sock.connect(sockaddr, TIMEOUT_MS);
                        sock.close();
                        Log.v(TAG, "checkForServices: success for " + url.getHost());
                        result.add(true);

                    } catch (IOException e) {
                        result.add(false);
                        Log.v(TAG, "checkForServices: fail for " + url.getHost());
                    }
                    synchronized (result) {
                        result.notifyAll();
                    }
                }).start();
                try {
                    while (result.isEmpty()) {
                        synchronized (result) {
                            result.wait();
                            if (result.get(0)) {
                                Log.v(TAG, "checkForServices return true");
                                return true;
                            }
                        }
                    }
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
            Log.v(TAG, "checkForServices return false");
            return false;
        }

        final void setInvolvedServices(final URL[] involvedServers) {
            mServices = involvedServers.clone();
        }
        abstract boolean isContentFullyDownloaded();
        private static boolean unittestFakeNetworkError() {
            if (BaseActivity.isInUnitTest()) {
                return !BaseActivity.getUnittestNetworkEnabled();
            }
            return false;

        }
    }
    static final String DOWNLOADS_FOLDER = "/additional_downloads/";
    private final Context mContext;
    /**
     * Constructor.
     * @param context Android {@link Context} for asset handling
     */
    public DownloaderFactory(final Context context) {
        mContext = context;
    }
    static String basenameForDownload(final String line) throws MalformedURLException {
        if (!line.contains(">")) {
            return new File(new URL(line).getPath()).getName();
        }
        return line.split(">", 2)[1];
    }

    static List<String> downloadFilesFromYaml(final InputStream is) {
        AdditionalDownloads downloads = AdditionalDownloads.load(is);
        return downloads.getFiles();

    }
    private boolean zipExists(final DownloaderFactory.AdditionalDownloads downloads) {
        if (downloads.getZipURL() != null) {
            File dir = new File(mContext.getCacheDir(), "additional_downloads");
            if (dir.isDirectory()) {
                return new File(dir, new File(downloads.getZipURL().getPath()).getName())
                        .exists();
            }
        }

        return false;
    }
    /**
     * Download file to a local folder from line-by-line-URLs in a file "additional_downloads"
     * in the given folder.
     * @param path local folder with file addition_downloads where the downloads will bes stored
     * @return a Runnable-style {@link Downloader}, if files have to be updated, null if not.
     */

    public final Downloader createPackageDownloader(final String path) {
        boolean allDownloaded;
        final AdditionalDownloads downloads;
        AdditionalDownloads dsLoaded;
        try {
            dsLoaded = AdditionalDownloads.load(
                    new FileInputStream(path + "/additional_downloads.yml"));
            allDownloaded = allDownloadedFromYAML(path, dsLoaded);
        } catch (IOException e) {
            if (e instanceof  FileNotFoundException) {
                dsLoaded = null;
                allDownloaded = true;
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
                    && e instanceof NoSuchFileException) {
                dsLoaded = null;
                allDownloaded = true;
            } else {
                throw new RuntimeException(e);
            }
        }
        downloads = dsLoaded;
        if (!allDownloaded) {
            Downloader ret = new Downloader() {
                @Override
                void run(final List<HttpRequest> httpClients) throws NullPointerException,
                        AdditionalDownloads.DownloadException {
                    boolean zipExists = zipExists(downloads);

                    if (BaseActivity.isInUnitTest()) {
                        if (!zipExists) {
                            BaseActivity.fakeNetworkError(
                                    mContext,
                                    path, downloads);
                        }
                    }

                    long now = System.currentTimeMillis();
                    // isOnline
                    downloads.downloadFiles(mContext, httpClients,
                            new File(path), checkForServices());
                    try {
                        if (!zipExists && BaseActivity.isInUnitTest()) {
                            long t = getTestMinimalDelayMs()
                                    - (System.currentTimeMillis() - now);
                            if (t > 0) {
                                try {
                                    Thread.sleep(t);
                                } catch (InterruptedException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }
                    } catch (ClassCastException e) {
                        throw new RuntimeException(e);
                    }
                }

                @Override
                boolean isContentFullyDownloaded() {


                    try {
                        //noinspection IOStreamConstructor
                        AdditionalDownloads dl = AdditionalDownloads.load(
                                new FileInputStream(path + "/additional_downloads.yml"));
                        for (String file : dl.getRequiredFiles()) {
                            String fullpath = path
                                    + "/" + file;
                            if (!new File(fullpath).exists()) {
                                return false;
                            }
                        }
                        return true;
                    } catch (IOException e) {
                        if (BuildConfig.DEBUG) {
                            throw new RuntimeException(e);
                        }
                        return false;
                    }
                }
            };
            ret.setInvolvedServices(downloads.getInvolvedServices());
            return ret;
        }
        return null;

    }
    private static final int TIMEOUT_MS = 1500;
    public static final class AdditionalDownloads {

        private URL mZipUrl;

        static void writeLocalData(final DownloadException downloadException, final byte[] bytes)
                throws  IOException {
            writeLocalData(downloadException.getPath(), bytes);
        }
        List<String> getAllHashes(final String localfile) {
            LinkedList<String> ret = new LinkedList<>();
            for (DownloadFile df : mFiles) {
                if (df.mName.equals(localfile)) {
                    for (DownloadSource ds : df.mDownloads) {
                        ret.add(ds.mHash);
                    }
                }
            }
            return ret;
        }
        static void writeLocalData(final String path, final byte[] bytes) throws IOException {

            if (bytes != null) {
                FileOutputStream os = new FileOutputStream(
                        path);
                os.write(bytes);
                os.close();
                FileWriter fw = new FileWriter(path + "__md5__");
                try {
                    fw.write(getMd5Result(bytes));
                    fw.close();
                } catch (NoSuchAlgorithmException e) {
                    throw new RuntimeException(e);
                }
            }

        }
        URL[] getInvolvedServices() {
            ArrayList<URL> list = new ArrayList<>();
            if (mZipUrl != null) {
                try {
                    list.add(new URL(mZipUrl.getProtocol(), mZipUrl.getHost(), mZipUrl.getPort(),
                            ""));
                } catch (MalformedURLException e) {
                    throw new RuntimeException(e);
                }
            }
            for (DownloadFile f : mFiles) {
                for (DownloadSource s: f.mDownloads) {
                    for (URL u: s.mParts) {
                        try {
                            list.add(new URL(u.getProtocol(), u.getHost(), u.getPort(), ""));
                        } catch (MalformedURLException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
            }
            URL[] array = new URL[list.size()];
            array = list.toArray(array);

            return array;
        }
        URL getZipURL() {
            return mZipUrl;
        }

        public static final class DownloadException extends Exception {
            private final List<String> mHashes = new LinkedList<>();
            private final String mUserfriendlyPath;
            private final URL mZipURL;
            private final boolean mSslError;

            URL getZipURL() {
                return mZipURL;
            }
            boolean isSslError() {
                return mSslError;
            }
            List<String> getHashes() {
                return mHashes;
            }
            String getUserfriendlyPath() {
                return mUserfriendlyPath;
            }
            String getPath() {
                return mPath;
            }
            private final String mPath;
            DownloadException(final String path, final String userfriendlyPath,
                              final List<String> hashes, final URL zipURL,
                              final boolean sslError) {
                mPath = path;
                mSslError = sslError;
                mUserfriendlyPath = userfriendlyPath;
                if (hashes != null) {
                    mHashes.addAll(hashes);
                }
                mZipURL = zipURL;

            }
        }
        @SuppressWarnings({"unchecked", "rawtypes"})
        private static AdditionalDownloads parse(final Map root) {
            AdditionalDownloads ret = new AdditionalDownloads();
            ret.mFiles = new LinkedList<>();
            if (root != null) {
                if (root.get("fallback-url") != null) {
                    try {
                        ret.mZipUrl = new URL((String) root.get("fallback-url"));
                    } catch (MalformedURLException e) {
                        throw new RuntimeException(e);
                    }
                } else {
                    ret.mZipUrl = null;
                }
                List<Map> files = (List<Map>) root.get("files");
                if (files != null) {
                    for (Object oFile : files) {
                        Map mFile = (Map) oFile;
                        if (mFile != null) {
                            DownloadFile df = new DownloadFile();
                            df.mName = (String) mFile.get("name");
                            if (mFile.get("required") != null) {
                                //noinspection ConstantConditions
                                df.mRequired = (Boolean) mFile.get("required");
                            } else {
                                df.mRequired = false;
                            }

                            List<DownloadSource> sources = new LinkedList<>();
                            Object oDownloads = mFile.get("downloads");
                            List lDownloads = (List<Map>) oDownloads;
                            if (lDownloads != null) {
                                for (Object oDownload : lDownloads) {
                                    Map mDownload = (Map) oDownload;
                                    DownloadSource ds = new DownloadSource();
                                    ds.mHash = (String) mDownload.get("hash");

                                    ds.mParts = new LinkedList<>();
                                    //noinspection ConstantConditions
                                    for (Object oPart : (List) mDownload.get("parts")) {
                                        String sPart = (String) oPart;
                                        try {
                                            ds.mParts.add(new URL(sPart));
                                        } catch (MalformedURLException e) {
                                            throw new RuntimeException(e);
                                        }
                                    }
                                    sources.add(ds);
                                }
                                df.mDownloads = sources;
                            }
                            ret.mFiles.add(df);
                        }

                    }
                }
            }
            return ret;

        }
        static AdditionalDownloads load(final Reader r) {
            return parse(new Yaml().load(r));
        }
        static AdditionalDownloads load(final InputStream is) {
            return parse(new Yaml().load(is));
        }
        private List<DownloadFile> mFiles;
        public static class DownloadSource {
            private String mHash;
            private List<URL> mParts;
        }
        public static class DownloadFile {
            private String mName;
            private boolean mRequired;
            private List<DownloadSource> mDownloads;
        }
        boolean hashMatches(final String name, final String hash) {
            for (DownloadFile f : mFiles) {
                if (f.mName.equals(name)) {
                    for (DownloadSource s:f.mDownloads) {
                        if (s.mHash.equals(hash)) {
                            return true;
                        }
                    }
                }
            }
            return false;
        }
        List<String> getRequiredFiles() {
            LinkedList<String> ret = new LinkedList<>();
            for (DownloadFile file : mFiles) {
                if (file.mRequired) {
                    ret.add(file.mName);
                }
            }
            return ret;

        }

        List<String> getFiles() {
            LinkedList<String> ret = new LinkedList<>();
            for (DownloadFile file : mFiles) {
                ret.add(file.mName);
            }
            return ret;
        }
        static boolean isHashCorrect(final byte[] data, final String hash) {
            try {
                String md5result = getMd5Result(data);
                return md5result.toLowerCase(Locale.ROOT)
                        .equals(hash.toLowerCase(Locale.ROOT));
            } catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(e);
            }
        }

        @NonNull
        private static String getMd5Result(final byte[] data) throws NoSuchAlgorithmException {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(data);
            byte[] digest = md.digest();
            char[] hexChars = new char[digest.length * 2];
            for (int j = 0; j < digest.length; j++) {
                int v = digest[j] & 0xFF;
                hexChars[j * 2] = HEX_ARRAY[v >>> 4];
                hexChars[j * 2 + 1] = HEX_ARRAY[v & HEX_MASK];
            }
            return new String(hexChars);
        }
        private static final char[] HEX_ARRAY = "0123456789abcdef".toCharArray();
        private void downloadFiles(final Context context, final List<HttpRequest> httpRequests,
                                   final File folder, final boolean serviceReachable)
                throws DownloadException {
            for (DownloadFile dl : mFiles) {
                File local = new File(folder, dl.mName);
                File hashFile = new File(folder, dl.mName + "__md5__");
                boolean download = true;
                if (local.canRead() && hashFile.canRead()) {
                    try {
                        String hash = new BufferedReader(new FileReader(hashFile)).readLine();
                        if (hash != null) {
                            for (DownloadSource ds:dl.mDownloads) {
                                if (hash.equals(ds.mHash)) {
                                    download = false;
                                    break;
                                }
                            }
                        }
                    } catch (IOException e) {
                        // that's ok.
                    }
                }
                if (download && !dl.mDownloads.isEmpty()) {
                    boolean sslError = false;
                    boolean success = false;
                    for (DownloadSource ds : dl.mDownloads) {
                        if (!success) {
                            ByteArrayOutputStream baos = new ByteArrayOutputStream();
                            String md5result = null;
                            if (serviceReachable) {
                                for (URL url : ds.mParts) {
                                        byte[] buffer = null;
                                        for (HttpRequest req : httpRequests) {

                                            try {
                                                HttpResult res = req.execute(url);
                                                if (res.getResultcode()
                                                        < HttpURLConnection.HTTP_MULT_CHOICE
                                                        && res.getResultcode()
                                                        >= HttpURLConnection.HTTP_OK
                                                        && res.getBody() != null) {
                                                    buffer = res.getBody();
                                                    break;
                                                }
                                                sslError = res.getResultcode()
                                                        == HTTP_STATUS_SSL_ERROR;
                                            } catch (SSLProtocolException e) {
                                                sslError = true;
                                            } catch (IOException e) {
                                                // ok
                                            }
                                        }
                                        if (buffer != null) {
                                            Log.v(TAG, "Writing from " + url + " to bytearray");
                                            baos.write(buffer, 0, buffer.length);
                                            success = true;
                                        } else {
                                            success = false;
                                            break;
                                        }
                                }
                                if (success) {
                                    success = isHashCorrect(baos.toByteArray(), ds.mHash);
                                    Log.v(TAG, "hash match for byte array: " + success);
                                    if (success) {
                                        md5result = ds.mHash;
                                    }
                                }
                            }
                            if (success) {
                                try {
                                    Log.v(TAG, "Writing from byte array to " + local);
                                    if (dl.mName.endsWith(".zip")) {
                                        unzip(new ByteArrayInputStream(
                                                baos.toByteArray()), folder);

                                    }
                                    FileOutputStream os = new FileOutputStream(local);
                                    os.write(baos.toByteArray());
                                    os.close();
                                    FileWriter fw = new FileWriter(hashFile);
                                    fw.write(md5result);
                                    fw.close();

                                } catch (IOException e) {
                                    Log.v(TAG, "Error writing from byte array to " + local, e);
                                    success = false;
                                }
                            }
                        }
                    }
                    if (!success && mZipUrl != null) {
                        Log.v(TAG, "Checking for zip");
                        File dir = new File(context.getCacheDir(), "additional_downloads");
                        if (dir.isDirectory() || dir.mkdirs()) {
                            File zipFile = new File(dir, new File(mZipUrl.getPath()).getName());
                            Log.v(TAG, zipFile.getAbsolutePath() + " exists: "
                                    + (zipFile.exists() ? "true" : "false"));
                            try {
                                ZipFile zip = new ZipFile(zipFile);
                                ZipEntry ze = zip.getEntry(dl.mName);
                                if (ze != null) {
                                    Log.v(TAG, dl.mName + "found");
                                    File f = new File(folder, ze.getName());
                                    String cp = f.getCanonicalPath();
                                    if (cp.startsWith(folder.getCanonicalPath())) {
                                        byte[] data = new byte[(int) ze.getSize()];
                                        int bytesRead;
                                        ByteArrayOutputStream baos =
                                                new ByteArrayOutputStream();
                                        InputStream is = zip.getInputStream(ze);
                                        while ((bytesRead = is.read(data)) >= 0) {
                                            baos.write(data, 0, bytesRead);
                                        }
                                        Log.v(TAG, "Writing data from zipfile to"
                                                + local.getAbsolutePath());
                                        writeLocalData(local.getAbsolutePath(),
                                                baos.toByteArray());
                                        success = true;
                                    }
                                } else {
                                    Log.v(TAG, dl.mName + "not found");
                                }
                                zip.close();
                            } catch (IOException e) {
                                boolean ok = e instanceof FileNotFoundException
                                        || e instanceof ZipException;
                                if (!ok && Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                                    if (e instanceof NoSuchFileException) {
                                        ok = true;
                                    }
                                }
                                if (!ok) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }

                    }
                    if (!success) {
                        if (dl.mRequired) {
                            List<String> hashes = new LinkedList<>();
                            for (DownloadSource ds:dl.mDownloads) {
                                if (!hashes.contains(ds.mHash)) {
                                    hashes.add(ds.mHash);
                                }
                            }
                            throw new DownloadException(local.getAbsolutePath(),
                                    folder.getName() + "/" + dl.mName, hashes,  mZipUrl,
                                    sslError);
                        } else {
                            if (sslError) {
                                Log.v("Downloader", "sslError for " + dl.mName);
                            }
                        }
                    }
                }
            }
        }
    }
    /**
     * Runnable-style downloader to download files from a list of folders.
     * @param folders folders to be updated, each folder must contain a file named
     *                additional_downloads with a list of Download-URLs
     * @return a Runnable-style {@link Downloader}, if files have to be updated, null if not.
     */

    public final Downloader createAssetDownloader(final List<String> folders)
            throws IOException {
        boolean allDownloaded = true;
        ArrayList<URL> involvedServers = new ArrayList<>();
        for (String s:folders) {
            AdditionalDownloads downloads = AdditionalDownloads.load(
                    new BufferedReader(new InputStreamReader(
                    mContext.getResources().getAssets()
                            .open(s + "/additional_downloads.yml"))));

            if (!allDownloadedFromYAML(s, downloads)) {
                allDownloaded = false;
                involvedServers.addAll(Arrays.asList(downloads.getInvolvedServices()));
                break;
            }
        }
        if (!allDownloaded) {

            Downloader ret = new Downloader() {
                @Override
                void run(final List<HttpRequest> httpClients) throws NullPointerException,
                        IOException, AdditionalDownloads.DownloadException {
                    long now = System.currentTimeMillis();
                    for (String s:folders) {
                        AdditionalDownloads downloads = AdditionalDownloads.load(new BufferedReader(
                                new InputStreamReader(mContext.getResources().getAssets()
                                        .open(s + "/additional_downloads.yml"))));
                        if (BaseActivity.isInUnitTest()) {
                            BaseActivity.fakeNetworkError(mContext, s, downloads);
                        }
                        String targetFolder = mContext.getCacheDir() + DOWNLOADS_FOLDER + s;

                        if (!requiredsDownloadedFromYAML(s, downloads)) {
                            File folder = new File(targetFolder);
                            if (!(folder.exists() || folder.mkdirs())) {
                                throw new IOException(
                                        mContext.getString(R.string.exception_occured));
                            }
                            downloads.downloadFiles(mContext, httpClients, folder,
                                    checkForServices());
                            try {
                                if (BaseActivity.isInUnitTest()) {
                                    long t = getTestMinimalDelayMs()
                                            - (System.currentTimeMillis() - now);
                                    if (t > 0) {
                                        try {
                                            Thread.sleep(t);
                                        } catch (InterruptedException e) {
                                            throw new RuntimeException(e);
                                        }
                                    }
                                }
                            } catch (ClassCastException e) {
                                throw new RuntimeException(e);
                            }

                        }
                    }
                }

                @Override
                boolean isContentFullyDownloaded() {
                    for (String s:folders) {
                        try {
                            AdditionalDownloads downloads =
                                    AdditionalDownloads.load(new BufferedReader(
                                    new InputStreamReader(mContext.getResources().getAssets()
                                            .open(s + "/additional_downloads.yml"))));
                            for (String file : downloads.getRequiredFiles()) {
                                String fullpath = mContext.getCacheDir() + DOWNLOADS_FOLDER + s
                                        + "/" + file;
                                if (!new File(fullpath).exists()) {
                                    return false;
                                }
                            }

                        } catch (IOException e) {
                            if (BuildConfig.DEBUG) {
                                throw new RuntimeException(e);
                            }
                            return false;
                        }
                    }
                    return true;
                }
            };
            URL[] array = new URL[involvedServers.size()];
            array = involvedServers.toArray(array);
            ret.setInvolvedServices(array);
            return ret;
        }
        return null;
    }
    private boolean filesDownloadedFromYAML(final String path, final AdditionalDownloads downloads,
                                            final List<String> files) throws IOException {
        boolean allDownloaded = true;
        for (String file : files) {
            String fullpath;
            if (new File(path).isAbsolute()) {
                fullpath = path + "/" + file;
            } else {
                fullpath = mContext.getCacheDir() + DOWNLOADS_FOLDER + path + "/" + file;
            }
            if (!new File(fullpath).canRead()) {
                Log.v(TAG, String.format("%s missing", fullpath));
                allDownloaded = false;
            } else {
                if (!new File(fullpath + "__md5__").canRead()) {
                    Log.v(TAG, String.format("%s hash missing", fullpath));
                    allDownloaded = false;
                } else {
                    String hash = new BufferedReader(new FileReader(fullpath + "__md5__"))
                            .readLine();
                    if (!downloads.hashMatches(file, hash)) {
                        Log.v(TAG, String.format("%s hash error [%s is wrong]",
                                fullpath, hash !=  null ? hash : "<null>"));
                        allDownloaded = false;
                    }
                }
            }
        }
        return allDownloaded;
    }
    private boolean allDownloadedFromYAML(final String path, final AdditionalDownloads downloads)
            throws IOException {
        return filesDownloadedFromYAML(path, downloads, downloads.getFiles());
    }
    private boolean requiredsDownloadedFromYAML(final String path,
                                                final AdditionalDownloads downloads)
            throws IOException {
        return filesDownloadedFromYAML(path, downloads, downloads.getRequiredFiles());
    }

    private static void checkZip(final ZipInputStream zis) throws IOException {
        ZipEntry ze;
        boolean ok = true;
        try {
            while (((ze = zis.getNextEntry()) != null)) {
                File f = new File("/a/b/c/d", ze.getName());
                String canonicalPath = f.getCanonicalPath();
                if (!canonicalPath.startsWith("/a/b/c/d")) {
                    ok = false;
                }
            }
            zis.reset();
        } catch (IOException e) {
            ok = false;
        }
        if (!ok) {
            throw new IOException("zip damaged.");
        }
    }

    private static void unzip(final InputStream is, final File targetDirectory) throws IOException {
        try (ZipInputStream zis = new ZipInputStream(
                new BufferedInputStream(is))) {
            ZipEntry ze;
            int count;
            checkZip(zis);
            byte[] buffer = new byte[BLOCKSIZE];
              while ((ze = zis.getNextEntry()) != null) {
                File file = new File(targetDirectory, ze.getName());
                String canonicalPath = file.getCanonicalPath();
                if (canonicalPath.startsWith(targetDirectory.getCanonicalPath())) {
                    File dir = ze.isDirectory() ? file : file.getParentFile();
                    if (dir != null && !dir.isDirectory() && !dir.mkdirs()) {
                        throw new FileNotFoundException("Failed to ensure directory: "
                                + dir.getAbsolutePath());
                    }
                    if (ze.isDirectory()) {
                        continue;
                    }
                    try (FileOutputStream fout = new FileOutputStream(file)) {
                        while ((count = zis.read(buffer)) >= 0) {
                            fout.write(buffer, 0, count);
                        }

                    } finally {
                        long time = ze.getTime();
                        if (time > 0) {
                            //noinspection ResultOfMethodCallIgnored
                            file.setLastModified(time);
                        }
                    }
                }
            }
        }
    }

    /**
     * Open an asset either from the cache or the local asset folder.
     * @param context Android {@link Context} for asset and cache handling
     * @param fileName filename as in {@link android.content.res.AssetManager#open(String)}
     * @return InputStream either from a downloaded file or a local asset
     */
    public static InputStream open(final Context context, final String fileName)
            throws IOException {
        InputStream ret;
        File overlay = new File(context.getCacheDir() + DOWNLOADS_FOLDER + fileName);
        if (overlay.exists()) {
            //noinspection IOStreamConstructor
            ret = new FileInputStream(overlay);
        } else {
            ret = context.getResources().getAssets().open(fileName);
        }
        return ret;
    }

    /**
     * Check if a file exists either as download or as local asset.
     * @param context Android {@link Context} for asset and cache handling
     * @param fileName filename as in {@link android.content.res.AssetManager#open(String)}
     * @return existence of either a download or a local asset
     */
    public static boolean exists(final Context context, final String fileName) {
        InputStream inputStream = null;
        File overlay = new File(
                context.getCacheDir() + DOWNLOADS_FOLDER + fileName);
        if (overlay.exists()) {
            return true;
        }
        try {
            inputStream = context.getResources().getAssets().open(fileName);
            return true;
        } catch (IOException e) {
            // make checkstyle smile
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (IOException e) {
                // ok
            }
        }
        return false;
    }
    private static final int TEST_MINIMAL_DELAY = 2000;
    private long getTestMinimalDelayMs() {
        return TEST_MINIMAL_DELAY;
    }
}
