package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.BaseActivity.isTv;
import static de.rainerhock.eightbitwonders.BaseActivity.remapKeyEvent;

import static de.rainerhock.eightbitwonders.EmulationUi.JOYSTICK_BACK_BUTTONS;
import static de.rainerhock.eightbitwonders.EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.util.AttributeSet;
import android.util.Log;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.SeekBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageButton;
import androidx.core.content.res.ResourcesCompat;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.lifecycle.Lifecycle;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;

public final class DialogsView extends RelativeLayout {


    private static final String MAIN_MENU_VISIBLE = "MAIN_MENU_VISIBLE";


    static void initTimemachineSlider(final @NonNull  ViewGroup root,
                                      final @NonNull Bitmap currentBitmap,
                                      final @NonNull Emulation.TimeMachineFunctions tmf) {
        SeekBar sb = root.findViewById(R.id.sb_seekbar);
        int newestSnapshot = tmf.getNewestMoment();
        sb.setMax(tmf.getOldestMoment() - newestSnapshot);
        Bitmap bmp = tmf.getScreenPreview(newestSnapshot);
        ImageView iv = root.findViewById(R.id.iv_bitmap);
        iv.setImageBitmap(bmp);
        TextView tv = root.findViewById(R.id.tv_time_offset);
        tv.setText(StatefulFragment.elapsedTimeAsString(root.getContext(), newestSnapshot));
        sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int i, final boolean b) {
                if (b) {
                    if (i > 0) {
                        iv.setImageBitmap(tmf.getScreenPreview(i + newestSnapshot));
                        tv.setText(StatefulFragment.elapsedTimeAsString(
                                root.getContext(), i + newestSnapshot));
                    } else {
                        iv.setImageBitmap(currentBitmap);
                        tv.setText(R.string.now);
                    }

                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        sb.setProgress(0);

    }
    void showMainMenu() {
        if (getActivity() != null) {
            getActivity().getViewModel().setMenuState(new MainMenuView.MenuState());
        }
        findViewById(R.id.mainmenu).setVisibility(View.VISIBLE);
        setVisibility(View.VISIBLE);
    }

    void populateMenu() {
        MainMenuView mmv = findViewById(R.id.mainmenu);
        if (mmv.getVisibility() == View.VISIBLE) {
            mmv.populate();
        }
    }

    boolean dismissMenu() {
        MainMenuView mmv = findViewById(R.id.mainmenu);
        if (mmv.getVisibility() == View.VISIBLE) {
            mmv.dismiss();
            return true;
        }
        return false;
    }

    void restoreMenu() {
        EmulationActivity activity = getActivity();
        if (activity != null && activity.getViewModel().getMenuState() != null) {
            MainMenuView mmv = findViewById(R.id.mainmenu);
            mmv.setVisibility(View.VISIBLE);
        }
    }

    abstract static class BaseFragment extends DialogFragment {
        private static final String PERMANENT = "PERMANENT";
        private boolean mDelayAfterDialog = false;
        private Runnable mOnDismissRunnable = null;

        @Override
        public void onDetach() {
            super.onDetach();
            if (showAsDialog() && getActivity() instanceof EmulationActivity) {
                ((EmulationActivity) getActivity())
                        .getDialogsController().hideAfterDialogsClosed();
            }
        }

        boolean showAsDialog() {
            return false;
        }
        protected void setPermanent(final boolean value) {
            if (getArguments() != null) {
                getArguments().putBoolean(PERMANENT, value);
            }
        }
        Serializable getPauseReason() {
            return requireArguments().getSerializable(PAUSEREASON);
        }
        @Override
        public void onCreate(final @Nullable Bundle savedInstanceState) {
            setStyle(DialogFragment.STYLE_NO_TITLE, getTheme());
            super.onCreate(savedInstanceState);
        }

        protected void setDelayAfterDialog(final boolean value) {
            mDelayAfterDialog = value;
        }
        protected DialogsView getContainerView() {
            return ((EmulationActivity) Objects.requireNonNull(
                    requireActivity())).getDialogsController();
        }
        @Override
        public void onStart() {
            super.onStart();
            if (getDialog() == null) {
                DialogsView v = getContainerView();
                v.onFragmentStart(this);
                if (getDialog() == null) {
                    v.setVisibility(View.VISIBLE);
                }
            }

        }
        int getContainerViewId() {
            return R.id.menucontainer_touch_friendly;

        }
        UiElement getFragmentType() {
            Bundle args = getArguments();
            if (args != null) {
                return (UiElement) args.getSerializable(FRAGMENT_TYPE);
            }
            return null;
        }
        @Override
        public void onStop() {
            super.onStop();
            boolean otherFragment = false;
            for (Fragment f : requireActivity().getSupportFragmentManager().getFragments()) {
                Bundle b = f.getArguments();
                if (f != this && b != null && b.getBoolean(IS_EMULATION_POPUP)) {
                    otherFragment = true;
                }
            }
            if (getArguments() != null) {
                Serializable pauseReason = getArguments().getSerializable(PAUSEREASON);
                if (pauseReason != null && getActivity() instanceof EmulationActivity) {
                    EmulationActivity activity = ((EmulationActivity) getActivity());
                    if (mDelayAfterDialog) {
                        activity.removePauseReasonAfterDelay(pauseReason);
                    } else {
                        activity.removePauseReason(pauseReason);
                    }
                }
            }
            DialogsView v = getContainerView();
            if (getActivity() != null && getActivity().getLifecycle().getCurrentState()
                    .isAtLeast(Lifecycle.State.STARTED)) {
                if (getDialog() == null) {
                    v.onFragmentStop(this);
                }
            }
            if (!otherFragment && getDialog() == null) {
                v.setVisibility(View.GONE);
            }
        }
        @Override
        public final void dismiss() {
            if (getDialog() == null) {
                    if (getActivity() != null) {
                        getActivity().getSupportFragmentManager().beginTransaction()
                                .remove(this).commit();
                }
            } else {
                super.dismiss();

            }
            FragmentActivity a = getActivity();
            if (a != null) {
                Fragment f = a.getSupportFragmentManager().findFragmentByTag(MAIN_MENU_VISIBLE);
                if (f != null) {
                    a.getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                }
            }
            if (mOnDismissRunnable != null) {
                mOnDismissRunnable.run();
            }
        }

        void setOnDismissRunnable(final Runnable r) {
            mOnDismissRunnable = r;
        }

        void addDialogParameters(final Context ctx, final Bundle b, final int bottomMargin) {
            if (ctx != null && !isTv(ctx)) {
                b.putInt(GRAVITY, Gravity.BOTTOM | Gravity.END);
                b.putParcelable(GRAVITY_OFFSET,
                        new Point(ctx.getResources().getDimensionPixelSize(
                                R.dimen.main_menu_endmargin), bottomMargin));
            }
        }
    }

    private static final float MILLIS_PER_S = 1_000f;
    abstract static class StatefulFragment<T extends Serializable> extends BaseFragment {


        abstract T createDefaultState();
        @NonNull
        abstract View createUi(@NonNull LayoutInflater inflater,
                               @NonNull EmulationActivity activity,
                               @NonNull Emulation emu,
                               @NonNull EmulationConfiguration conf,
                               T state);
        @Nullable
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            Serializable s = requireArguments().getSerializable(DIALOGSTATE);
            T state;
            if (s != null) {
                try {
                    //noinspection unchecked
                    state = (T) s;
                } catch (ClassCastException e) {
                    state = createDefaultState();
                }
            } else {
                state = createDefaultState();
            }
            EmulationActivity ea = (EmulationActivity) requireActivity();
            View ret =  createUi(inflater, ea, ea.getViewModel().getEmulation(),
                    ea.getViewModel().getConfiguration(), state);
            ret.post(ret::requestFocus);
            if (requireArguments().keySet().contains(GRAVITY)) {
                int gravity = requireArguments().getInt(GRAVITY);
                Window window = requireDialog().getWindow();
                if (window != null) {
                    WindowManager.LayoutParams wlp = window.getAttributes();
                    wlp.gravity = gravity;
                    wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
                    window.setAttributes(wlp);
                    Point p = requireArguments().getParcelable(GRAVITY_OFFSET);
                    if (p != null) {
                        wlp.x = p.x;
                        wlp.y = p.y;
                    }
                }
            }
            return ret;
        }
        protected static String elapsedTimeAsString(final Context context,
                                                  final int elapsedMilliSecs) {
            //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
            String s = String.format(context.getResources().getConfiguration().locale,
                    "%.2f", elapsedMilliSecs / MILLIS_PER_S);
            return context.getResources().getString(R.string.elapsed, s);

        }

    }
    private void onFragmentStart(final BaseFragment fragment) {
        Bundle args = fragment.getArguments();
        if (args != null && getActivity() != null) {
            getActivity().getViewModel().setCurrentFragment(
                    (UiElement) args.getSerializable(FRAGMENT_TYPE),
                    args.getBoolean(FRAGMENT_KEEP_STATE));
        }
    }

    boolean recreateLastFragment() {
        if (getActivity() != null) {
            EmulationViewModel vm = getActivity().getViewModel();
            if (vm.getFragmentElementType() != null) {
                showUiElement(vm.getFragmentElementType(), vm.getFragmentElementKeepState());
                return true;
            }
        }
        return false;
    }
    void onFragmentStop(final BaseFragment fragment) {
        Bundle args = fragment.getArguments();

        if (args != null && getActivity() != null) {
            EmulationViewModel vm = getActivity().getViewModel();

            if (args.getSerializable(FRAGMENT_TYPE) == vm.getFragmentElementType()) {
                vm.setCurrentFragment(null, false);
            }
        }
    }

    enum UiElement  { QUIT, PAUSE_EXTENED, PAUSE, TAPE_CONTROL, SWITCH_MONITOR,
    TIMEMACHINE, SAVESTATES, JOYSTICKSELECTION, FLIPLIST, SOFTKEYS, ADD_TO_START,
        PAUSE_AFTER_TIMEMACHINE, SPEED_SELECTION, BACKGROUND_MODE }
    private BaseFragment getNewestFragment() {
        BaseFragment newest = null;
        long timestamp = 0;
        EmulationActivity ea = getActivity();
        if (ea != null) {
            for (Fragment f : ea.getSupportFragmentManager().getFragments()) {
                if (f instanceof BaseFragment) {
                    if (f.getArguments() != null
                            && f.getArguments().getLong(TIMESTAMP) > timestamp) {
                        timestamp = f.getArguments().getLong(TIMESTAMP);
                        newest = (BaseFragment) f;
                    }
                }
            }
        }
        return newest;
    }
    void hideAfterDialogsClosed() {
        EmulationActivity ea = getActivity();
        if (ea != null) {
            for (Fragment f : ea.getSupportFragmentManager().getFragments()) {
                if (f instanceof BaseFragment) {
                    return;
                }
            }
            setVisibility(View.GONE);
        }

    }
    boolean closeNewestDialog() {
        boolean ret = false;
        EmulationActivity ea = getActivity();
        long timestamp = 0;
        int fragments = 0;
        if (ea != null) {
            if (!ea.leaveMovieMode()) {
                Fragment newest = null;
                for (Fragment f : ea.getSupportFragmentManager().getFragments()) {
                    if (f instanceof BaseFragment) {
                        fragments++;
                        if (f.getArguments() != null
                                && f.getArguments().getLong(TIMESTAMP) > timestamp) {
                            timestamp = f.getArguments().getLong(TIMESTAMP);
                            newest = f;
                        }
                    }
                }
                if (newest != null) {
                    ea.getSupportFragmentManager().beginTransaction().remove(newest).commitNow();
                    ret = true;
                }
            } else {
                ret = true;
            }
            if (fragments <= 1) {
                ea.getEmulationUi().setScreenRefresh(true);
                setVisibility(View.GONE);
            }
        }
        MainMenuView mmv = findViewById(R.id.mainmenu);
        if (mmv.getVisibility() == View.VISIBLE) {
            mmv.dismiss();
        }
        return ret;
    }

    abstract static class DialogDescription implements Serializable {
        abstract String getTitle(Context context);
        abstract List<Emulation.MenuFeature> getFeatures(EmulationActivity activity,
                                                Emulation emu,
                                                EmulationConfiguration conf, Bundle args);

        int getContainerViewId() {
            return R.id.menucontainer_featurelist_anchor;
        }
    }
    void showUiElement(final UiElement element, final Runnable r) {
        showUiElement(element, false, null, r);
    }

    void showUiElement(final UiElement element) {
        showUiElement(element, false, null, null);
    }
    void showUiElement(final UiElement element, final Bundle extradata) {
        showUiElement(element, false, extradata, null);
    }
    void showUiElement(final UiElement element, final boolean keepState) {
        showUiElement(element, keepState, null, null);
    }
    boolean isDialogVisible() {
        if (getActivity() != null) {
            MainMenuView mmv = findViewById(R.id.mainmenu);
            if (mmv.getVisibility() == View.VISIBLE) {
                return true;
            }
            for (Fragment f : getActivity().getSupportFragmentManager().getFragments()) {
                if (f instanceof BaseFragment) {
                    return true;
                }
            }
        }
        return recreateLastFragment();
    }
    boolean onKey(final KeyEvent event) {
        View menu = findViewById(R.id.mainmenu);
        if (menu.getVisibility() == View.VISIBLE && menu.dispatchKeyEvent(event)) {
            return true;
        }
        if (getActivity() != null) {
            Fragment f = getNewestFragment();
            if (f != null && f.getView() != null) {
                Log.v("onKey", "fragment: " + f + " focus: " + f.getView().findFocus());
                return f.getView().dispatchKeyEvent(event);
            }
        }
        return false;

    }
    void hidePauseDialog() {
        if (getActivity() != null) {
            List<Fragment> toRemove = new LinkedList<>();
            for (Fragment f : getActivity().getSupportFragmentManager().getFragments()) {
                if (f instanceof BaseFragment && f.getArguments() != null
                        && f.getArguments().getSerializable(DIALOGDESCRIPTION)
                        instanceof PauseFunctions) {
                    toRemove.add(f);
                }
            }
            if (!toRemove.isEmpty()) {
                FragmentTransaction ft = getActivity().getSupportFragmentManager()
                        .beginTransaction();
                for (Fragment f : toRemove) {
                    ft = ft.remove(f);
                }
                ft.commitNow();
            }
        }
    }
    private static final String GRAVITY = "GRAVITY";
    private static final String GRAVITY_OFFSET = "GRAVITY_OFFSET";
    private void showUiElement(final UiElement element, final boolean keepState,
                               final Bundle extradata, final Runnable r) {
        BaseFragment f;
        DialogDescription desc;
        Bundle b = new Bundle();
        int dialogBottomMargin = Objects.requireNonNull(getActivity()).getResources()
                .getDimensionPixelOffset(R.dimen.dialog_margin_bottom)
                - getActivity().getBottomInset();
        switch (element) {
            case QUIT:
                f = new MenuFeaturesFragment();
                desc = new QuitFunctions();
                break;

            case SPEED_SELECTION:
                f = new MenuFeaturesFragment();
                desc = new SpeedSelectionFunctions();
                break;
            case BACKGROUND_MODE:
                f = new MenuFeaturesFragment();
                desc = new BackgroundModeFunctions();
                break;
            case PAUSE_EXTENED:
                f = new MenuFeaturesFragment();
                desc = new PauseFunctions(true);
                break;
            case PAUSE:
                f = new MenuFeaturesFragment();
                desc = new PauseFunctions(false);
                break;
            case TAPE_CONTROL:
                f = new MenuFeaturesFragment();
                desc = new TapeFunctions();
                break;
            case SWITCH_MONITOR:
                f = new MenuFeaturesFragment();
                desc = new SwitchMonitorFunctions();
                break;
            case TIMEMACHINE:
                f = new TimeMachineFragment();
                f.addDialogParameters(getContext(), b, dialogBottomMargin);
                desc = null;
                break;
            case SAVESTATES:
                f = new SaveStateFragment();
                f.addDialogParameters(getContext(), b, dialogBottomMargin);
                desc = null;
                break;
            case FLIPLIST:
                f = new FliplistDialogFragment();
                desc = null;
                break;
            case SOFTKEYS:
                f = new SoftkeysDialogFragment();
                desc = null;
                b.putInt(GRAVITY, Gravity.BOTTOM | Gravity.CENTER_HORIZONTAL);
                break;
            case ADD_TO_START:
                f = new SaveStateFragment.AddtoStartScreenFragment();
                desc = null;
                break;
            case PAUSE_AFTER_TIMEMACHINE:
                f = new BePreparedDialogFragment();
                desc = null;
                break;
            case JOYSTICKSELECTION:
                f = new RequiredJoysticksFragment();
                desc = null;
                break;
            default:
                throw new IllegalStateException("Must not reach this piece of code. "
                        + element + " was not handled");
        }
        EmulationActivity a = getActivity();
        if (a != null) {
            b.putSerializable(FRAGMENT_TYPE, element);
            b.putBoolean(FRAGMENT_KEEP_STATE, keepState);
            b.putBoolean(IS_EMULATION_POPUP, true);
            b.putSerializable(DIALOGDESCRIPTION, desc);
            b.putLong(TIMESTAMP, System.currentTimeMillis());
            if (extradata != null) {
                b.putAll(extradata);
            }
            if (getActivity() != null) {
                if (keepState) {
                    b.putSerializable(DIALOGSTATE, getActivity()
                            .getViewModel().getActiveDialogState());
                }
                b.putSerializable(PAUSEREASON, getActivity().addPauseReason());
            }
            f.setArguments(b);
            if (r != null) {
                f.setOnDismissRunnable(r);
            }
            if (isTv(getContext()) || f.showAsDialog()) {
                f.showNow(a.getSupportFragmentManager(), null);
            } else {
                View v = a.findViewById(R.id.menucontainer_featurelist_anchor);
                RelativeLayout.LayoutParams lp = (LayoutParams) v.getLayoutParams();
                lp.setMargins(lp.leftMargin, lp.topMargin, lp.rightMargin,
                        getResources().getDimensionPixelOffset(
                                a.getKeyboardVisibility() == View.VISIBLE
                                        ? R.dimen.minimal : R.dimen.menufeatures_margin_bottom));
                v.setLayoutParams(lp);
                a.getSupportFragmentManager().beginTransaction().add(f.getContainerViewId(), f)
                        .commitNow();
            }
        }
    }
    private static final String DIALOGDESCRIPTION = "DIALOGDESCRIPTION";
    private static final String TIMESTAMP = "TIMESTAMP";
    private static final String IS_EMULATION_POPUP = "IS_EMULATION_POPUP";
    private static final String FRAGMENT_TYPE = "FRAGMENT_TYPE";
    private static final String FRAGMENT_KEEP_STATE = "FRAGMENT_KEEP_STATE";
    private static final String DIALOGSTATE = "DIALOGSTATE";
    private static final String PAUSEREASON = "PAUSEREASON";

    static class PauseFunctions extends CenteredDialogDescription {
        private final boolean mExtended;

        PauseFunctions(final boolean extended) {
            mExtended = extended;
        }

        @Override
        public String getTitle(final Context context) {
            if (mExtended) {
                return null;
            } else {
                return context.getResources().getString(R.string.pause);
            }
        }

        @Override
        public List<Emulation.MenuFeature> getFeatures(final EmulationActivity ea,
                                                       final Emulation emu,
                                                       final EmulationConfiguration conf,
                                                       final Bundle args) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            if (args != null && args.getBoolean("extended", true)) {
                if (mExtended) {
                    ret.add(new BaseActivity.PopupCallback(ea, R.string.menu, 0,
                            () -> {
                                ea.getViewModel().setNotPausedByUser();
                                ea.showPopupMenu();
                            }, false, false));
                    EmulationViewModel vm = ea.getViewModel();
                    if (ea.isTv() && !(vm.getSoftkeyIds().isEmpty()
                            && vm.getUiSoftkeys().isEmpty())) {
                        ret.add(new BaseActivity.PopupCallback(ea, R.string.softkeys, 0,
                                () -> {
                                    ea.getViewModel().setNotPausedByUser();
                                    ea.showSoftkeyDialog();
                                }, false, false));

                    }
                }
            }
            ret.add(new BaseActivity.PopupCallback(ea, R.string.cont, 0,
                    () -> {
                        if (ea.getViewModel() != null) {
                            ea.getViewModel().setNotPausedByUser();
                            new Thread(() -> ea.runOnUiThread(ea::clearPauseReasons)).start();
                        }
                    }, false, false));
            return ret;
        }
    }
    abstract static class CenteredDialogDescription extends DialogDescription {
        @Override
        int getContainerViewId() {
            return R.id.menucontainer_center_anchor;
        }
    }
    static class BackgroundModeFunctions extends CenteredDialogDescription {

        @Override
        String getTitle(final Context context) {
            return context.getResources().getString(R.string.running_in_background);
        }

        @Override
        List<Emulation.MenuFeature> getFeatures(final EmulationActivity activity,
                                                final Emulation emu,
                                                final EmulationConfiguration conf,
                                                final Bundle args) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            ret.add(new Emulation.MenuFeature() {
                @Override
                public String getName() {
                    return activity.getResources().getString(R.string.bring_back);
                }

                @Override
                public int getIconResource() {
                    return 0;
                }

                @Override
                public Runnable getRunnable() {
                    return () -> {
                        activity.getViewModel().setRunningInBackground(false);
                        activity.getEmulationUi().setScreenRefresh(true);
                    };
                }
            });
            return ret;
        }
    }
    static class QuitFunctions extends CenteredDialogDescription {

        @Override
        public String getTitle(final Context context) {
            return null;            }

        @Override
        public List<Emulation.MenuFeature> getFeatures(final EmulationActivity activity,
                                                       final Emulation emu,
                                                       final EmulationConfiguration conf,
                                                       final Bundle args) {
            List<Emulation.MenuFeature> functions = new LinkedList<>();
            functions.add(new BaseActivity.PopupCallback(activity, R.string.quit, 0, () -> {
                Log.v("terminate", "terminate");
                emu.terminate(() -> activity.runOnUiThread(activity::onEmulatorFinished));
            }, false, false));
            return functions;
        }
    }
    static class SwitchMonitorFunctions extends DialogDescription {

        @Override
        public String getTitle(final Context context) {
            return null;
        }

        @Override
        public List<Emulation.MenuFeature> getFeatures(final EmulationActivity activity,
                                                       final Emulation emu,
                                                       final EmulationConfiguration conf,
                                                       final Bundle args) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            if (emu.getDualMonitorFunctions() != null) {
                for (int i : emu.getDualMonitorFunctions().getDisplays().keySet()) {
                    if (activity != null && activity.getViewModel() != null
                            && activity.getViewModel().getCurrentDisplayId() != i) {
                        ret.add(new BaseActivity.PopupCallback(
                                emu.getDualMonitorFunctions().getDisplays().get(i), 0,
                                () -> activity.getViewModel().setCurrentDisplayId(i),
                                false, false));
                    }
                }
            }
            return ret;
        }
    }
    static class TapeFunctions extends DialogDescription {

        @Override
        public String getTitle(final Context context) {
            return context.getResources().getString(R.string.IDS_MP_DATASETTE_CONTROL);
        }

        @Override
        public List<Emulation.MenuFeature> getFeatures(final EmulationActivity ctx,
                                                       final Emulation emu,
                                                       final EmulationConfiguration conf,
                                                       final Bundle args) {
            Emulation.TapeDeviceFunctions tdf = emu.getTapeDeviceFunctions();
            List<Emulation.MenuFeature> functions = new LinkedList<>();
            if (tdf != null) {
                functions.add(new BaseActivity.PopupCallback(ctx,
                        R.string.press_play,
                        R.drawable.ic_media_play, tdf.getPlayCommand(),
                        false, false));
                functions.add(new BaseActivity.PopupCallback(ctx,
                        R.string.IDS_MI_DATASETTE_STOP,
                        R.drawable.ic_media_stop, tdf.getStopCommand(),
                        false, false));
                functions.add(new BaseActivity.PopupCallback(ctx,
                        R.string.IDS_MI_DATASETTE_FORWARD,
                        R.drawable.ic_media_ff, tdf.getForwardCommand(),
                        false, false));
                functions.add(new BaseActivity.PopupCallback(ctx,
                        R.string.IDS_MI_DATASETTE_REWIND,
                        R.drawable.ic_media_rew, tdf.getRewindCommand(),
                        false, false));
                functions.add(new BaseActivity.PopupCallback(ctx,
                        R.string.press_record_and_play,
                        R.drawable.ic_media_record, tdf.getRecordCommand(),
                        false, false));
            }
            return functions;
        }
        @Override
        int getContainerViewId() {
            return R.id.menucontainer_touch_friendly;
        }

    }
    /**
     * Constructor that is called when inflating a MenuContainerView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */

    public DialogsView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyleAttr An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public DialogsView(final Context context, final AttributeSet attrs,
                       final int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }
    private void init(final EmulationActivity activity) {
        LayoutInflater inflater = (LayoutInflater) activity
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        View root = inflater.inflate(R.layout.view_menucontainer, this).findViewById(R.id.root);
        root.findViewById(R.id.tap_to_close).setOnClickListener(v -> closeNewestDialog());
        FragmentManager fm = activity.getSupportFragmentManager();
        int visibility = fm.findFragmentByTag(MAIN_MENU_VISIBLE) == null ? View.GONE : View.VISIBLE;
        for (Fragment f : fm.getFragments()) {
            if (f instanceof BaseFragment) {
                visibility = View.VISIBLE;
                break;
            }
        }
        setVisibility(visibility);
    }

    private EmulationActivity getActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof EmulationActivity) {
                return (EmulationActivity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        return null;
    }
    private void init() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof EmulationActivity) {
                init((EmulationActivity) context);
                return;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        throw new IllegalStateException(getClass().getSimpleName()
                + " can only be instanciated in an EmulationActivity");
    }
    public static final class SpeedSelectionFunctions extends DialogDescription {

        @Override
        String getTitle(final Context context) {
            return context.getResources().getString(R.string.emulation_speed);
        }

        @Override
        List<Emulation.MenuFeature> getFeatures(final EmulationActivity activity,
                                                final Emulation emu,
                                                final EmulationConfiguration conf,
                                                final Bundle args) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            for (int speed: emu.getSpeedUpFunctions().getAvailableNormalSpeeds()) {
                ret.add(new Emulation.MenuFeature() {
                    @Override
                    public String getName() {
                        if (speed == Emulation.SpeedUpFunctions.NORMAL_SPEED) {
                            return activity.getResources().getString(R.string.normal_speed);
                        } else {
                            return String.format(Locale.getDefault(), "%d %%", speed);
                        }
                    }
                    @Override
                    public int getIconResource() {
                        return 0;
                    }

                    @Override
                    public Runnable getRunnable() {
                        return () -> emu.getSpeedUpFunctions().setNormalSpeed(speed);
                    }
                });
            }
            return ret;
        }

        @Override
        int getContainerViewId() {
            return R.id.menucontainer_touch_friendly;
        }
    }
    public static final class MenuFeaturesFragment extends BaseFragment {
        @Nullable
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            @SuppressLint("InflateParams")
            ViewGroup ret = (ViewGroup) inflater.inflate(R.layout.viewgroup_emulation_fragment,
                    null, false);

            if (getArguments() != null) {
                ViewGroup parent = ret.findViewById(R.id.ll_root);
                Serializable s = getArguments().getSerializable(DIALOGDESCRIPTION);
                EmulationActivity ea = (EmulationActivity) getActivity();
                if (s instanceof DialogDescription && ea != null) {
                    EmulationConfiguration conf = ea.getViewModel().getConfiguration();
                    Emulation emu = ea.getViewModel().getEmulation();
                    DialogDescription d = (DialogDescription) s;
                    String title = d.getTitle(ea);
                    if (title != null) {
                        TextView tv = ret.findViewById(R.id.tv_title);
                        tv.setVisibility(View.VISIBLE);
                        tv.setText(title);
                    }
                    boolean withIcons = false;
                    for (Emulation.MenuFeature f : d.getFeatures(ea, emu, conf, getArguments())) {
                        if (f.getIconResource() >= 0) {
                            withIcons = true;
                        }
                    }
                    for (Emulation.MenuFeature f : d.getFeatures(ea, emu, conf, getArguments())) {
                        MenuButton b = ret.findViewById(R.id.bn_first);

                        if (b == null) {
                            b = inflater.inflate(R.layout.view_emulation_popup_menubutton,
                                    parent, true).findViewById(R.id.bn_text);
                        }
                        b.setTextAlignment(withIcons
                                ? View.TEXT_ALIGNMENT_VIEW_START : View.TEXT_ALIGNMENT_CENTER);
                        b.setId(View.NO_ID);
                        b.setText(f.getName());
                        if (f.getIconResource() > 0) {
                            b.setIcon(ResourcesCompat.getDrawable(b.getResources(),
                                    f.getIconResource(), null));
                        } else {
                            b.setIcon(ResourcesCompat.getDrawable(b.getResources(),
                                    R.drawable.ic_blank, null));
                        }
                        b.setStateIcon(MenuButton.State.EMPTY);
                        final MenuButton b2 = b;
                        if (f.getRunnable() != null) {
                            BaseActivity.setViewListeners(b, () -> {
                                Log.v("viewlistener called", b2.toString());
                                f.getRunnable().run();
                                dismiss();
                            }, this::dismiss);
                        } else {
                            b.setEnabled(false);
                        }
                        Log.v("viewlistener registered", b2.toString());
                    }
                }
            }
            return ret;

        }

        @Override
        int getContainerViewId() {
            if (getArguments() != null) {
                Serializable s = getArguments().getSerializable(DIALOGDESCRIPTION);
                if (s instanceof DialogDescription) {
                    return ((DialogDescription) s).getContainerViewId();

                }
            }
            return super.getContainerViewId();
        }
    }
    static boolean handleKeyUpOnView(final View v, final KeyEvent event, final int keyCode,
                                     final OnKeyListener mOnBackListener) {
        if (JOYSTICK_BACK_BUTTONS.contains(keyCode) && mOnBackListener != null
                && mOnBackListener.onKey(v, keyCode, event)) {
            return true;
        }
        if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                || keyCode == KeyEvent.KEYCODE_SPACE
                || keyCode == KeyEvent.KEYCODE_ENTER
                || JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keyCode)) {
            return v.callOnClick();
        }
        return false; // keyCode != KeyEvent.KEYCODE_BACK && keyCode != KeyEvent.KEYCODE_ESCAPE;
    }
    static boolean handleKeyDownOnView(final View v, final KeyEvent event, final int keyCode,
                                       final OnKeyListener mOnBackListener) {
        if (JOYSTICK_BACK_BUTTONS.contains(keyCode) && mOnBackListener != null
                && mOnBackListener.onKey(v, keyCode, event)) {
            return true;
        }
        if (keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
            View next = v.focusSearch(FOCUS_DOWN);
            if (next != null) {
                next.requestFocus();
                return true;
            }
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_UP) {
            View prev = v.focusSearch(FOCUS_UP);
            if (prev != null) {
                prev.requestFocus();
                return true;
            }
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_RIGHT) {
            View prev = v.focusSearch(FOCUS_RIGHT);
            if (prev != null) {
                prev.requestFocus();
                return true;
            }
        } else if (keyCode == KeyEvent.KEYCODE_DPAD_LEFT) {
            View prev = v.focusSearch(FOCUS_LEFT);
            if (prev != null) {
                prev.requestFocus();
                return true;
            }
        }
        return keyCode != KeyEvent.KEYCODE_BACK && keyCode != KeyEvent.KEYCODE_ESCAPE;
    }
    public static final class MenuButton extends androidx.appcompat.widget.AppCompatButton {


        private void init(final Context context, final AttributeSet attrs) {
            int[] attrIds = new int[]{R.attr.leftDrawable};
            //noinspection resource
            TypedArray a = context.getResources().obtainAttributes(attrs, attrIds);
            try {
                Drawable left = a.getDrawable(R.styleable.menubutton_leftDrawable);
                setIcon(left);
            } finally {
                a.recycle();
            }
        }
        void setIcon(final Drawable icon) {
            Drawable[] d = getCompoundDrawables();
            setCompoundDrawablesWithIntrinsicBounds(icon != null ? icon : d[0],
                    d[1], d[2], d[3]);

        }
        enum State { NONE, EMPTY }
        void setStateIcon(final State state) {
            Drawable[] d = getCompoundDrawables();
            Drawable right = d[2];
            if (Objects.requireNonNull(state) == State.EMPTY) {
                right = ResourcesCompat.getDrawable(
                        getResources(), R.drawable.ic_blank, null);
            }
            setCompoundDrawablesWithIntrinsicBounds(d[0],
                    d[1], right, d[3]);

        }
        // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
        public MenuButton(final Context context, final AttributeSet attrs) {
            super(context, attrs);
            init(context, attrs);
        }

        // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
        public MenuButton(final Context context, final AttributeSet attrs,
                          final int defStyleAttr) {
            super(context, attrs, defStyleAttr);
            init(context, attrs);
        }

        private OnKeyListener mOnBackListener = null;

        void setOnBackPressedListener(final OnKeyListener listener) {
            mOnBackListener = listener;
        }

        @Override
        public boolean onKeyUp(final int keyCode, final KeyEvent event) {
            return DialogsView.handleKeyUpOnView(this, event,
                    remapKeyEvent(event).getKeyCode(), mOnBackListener);
        }

        @Override
        public boolean onKeyDown(final int keyCode, final KeyEvent event) {
            return DialogsView.handleKeyDownOnView(this, event,
                    remapKeyEvent(event).getKeyCode(), mOnBackListener);
        }
    }
    public static final class MoreFunctionsButton extends AppCompatImageButton {
        // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
        public MoreFunctionsButton(final @NonNull Context context) {
            super(context);
        }
        // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
        public MoreFunctionsButton(final @NonNull Context context,
                                   final @Nullable AttributeSet attrs) {
            super(context, attrs);
        }
        // CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
        public MoreFunctionsButton(final @NonNull Context context,
                                   final @Nullable AttributeSet attrs,
                                   final int defStyleAttr) {
            super(context, attrs, defStyleAttr);
        }
        @Override
        public boolean onKeyUp(final int keyCode, final KeyEvent event) {
            return DialogsView.handleKeyUpOnView(this, event,
                    remapKeyEvent(event).getKeyCode(), null);
        }

        @Override
        public boolean onKeyDown(final int keyCode, final KeyEvent event) {
            return DialogsView.handleKeyDownOnView(this, event,
                    remapKeyEvent(event).getKeyCode(), null);
        }

    }
}
