//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.hardware.input.InputManager;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.NonNull;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;


abstract class DeviceSupportActivity extends BaseActivity
        implements InputManager.InputDeviceListener {

    protected final boolean isInLandscape() {
        return getResources().getBoolean(R.bool.is_landscape);
    }

    protected void onAvailableJoysticksChanged(final @NonNull List<Joystick> joysticks) {
    }

    protected void onAvailableKeyboardsChanged(final boolean hardware, final boolean software) {
    }

    @SuppressWarnings({"unused", "EmptyMethod"})
    protected void onAvailableKeyboardListenersChanged(final List<View.OnKeyListener> listeners) {

    }


    private interface DeviceChecker {
        boolean check(InputDevice dev, int sources);
    }

    private boolean checkDevice(final DeviceChecker checker) {
        boolean ret = false;
        for (int deviceId : InputDevice.getDeviceIds()) {
            InputDevice dev = InputDevice.getDevice(deviceId);
            if (dev != null) {
                if (checker.check(dev, dev.getSources())) {
                    ret = true;
                }
            }
        }
        return ret;

    }

    private boolean isSourceSupported(final int allSources, final int requestedSource) {
        return ((allSources & requestedSource) == requestedSource);
    }

    private final List<Joystick> mAttachedJoysticks = new LinkedList<>();
    private boolean mSoftwareKeyboardAvailable = false;
    private boolean mHardwareKeyboardAvailable = false;
    private InputManager mInputmanager = null;

    //protected abstract EmulationViewModel getViewModel();

    @Override
    protected void onResume() {
        super.onResume();
        mInputmanager = (InputManager) getSystemService(Context.INPUT_SERVICE);
        mInputmanager.registerInputDeviceListener(this, null);

    }

    @Override
    protected void onPause() {
        super.onPause();
        mInputmanager.unregisterInputDeviceListener(this);

    }

    interface KeyboardUnitTest {
        boolean isKeyboardEnabled();
        Set<Class<Joystick>> getDisabledJoystickClasses();
    }
    //CHECKSTYLE DISABLE MethodLength FOR 1 LINES
    void updateHardware() {
        final int oldJoystickHash = mAttachedJoysticks.hashCode();
        updateDevices();
        if (oldJoystickHash != mAttachedJoysticks.hashCode()) {
            for (Joystick joy:mAttachedJoysticks) {
                Log.v("AVAILABLE_JOYSTICK", joy.toString()
                        + " Buttons: " + joy.getHardwareButtons());
            }
            onAvailableJoysticksChanged(mAttachedJoysticks);
        }
    }
    private DeviceChecker getHardwareKeyboardChecker(final List<Integer> keyboardDeviceIds) {
        return (dev, sources) -> {
            if (isSourceSupported(sources, InputDevice.SOURCE_KEYBOARD)
                    && !isSourceSupported(sources, InputDevice.SOURCE_GAMEPAD)
                    && !dev.isVirtual()
                    && dev.getKeyboardType() == InputDevice.KEYBOARD_TYPE_ALPHABETIC) {
                keyboardDeviceIds.add(dev.getId());
                return true;
            }
            return false;
        };
    }
    private DeviceChecker getGamepadChecker(final DeviceChecker isHardwareKeyboardChecker) {
        return (dev, sources) -> {
            GameControllerJoystick dpad = null;
            GameControllerJoystick left = null;
            GameControllerJoystick right = null;
            if (isSourceSupported(sources, InputDevice.SOURCE_JOYSTICK)) {
                List<Integer> axes = new LinkedList<>();
                for (InputDevice.MotionRange range : dev.getMotionRanges()) {
                    axes.add(range.getAxis());
                }
                // DPAD
                if (axes.contains(MotionEvent.AXIS_HAT_X)
                        && axes.contains(MotionEvent.AXIS_HAT_Y)
                        && (dev.hasKeys(KeyEvent.KEYCODE_BUTTON_L1)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_R1)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_A)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_B)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_X)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_Y)[0])) {
                    List<Integer> buttonlist = getCorrespondingButtons(axes);
                    Integer[] buttons = buttonlist.toArray(new Integer[]{});
                    dpad = GameControllerJoystick.getInstance(
                            this, dev, MotionEvent.AXIS_HAT_X, MotionEvent.AXIS_HAT_Y,
                            new Integer[]{}, buttons, Joystick.ValueType.POSITION);
                }
                // Left Joystick
                if (axes.contains(MotionEvent.AXIS_X) && axes.contains(MotionEvent.AXIS_Y)
                        && (dev.hasKeys(KeyEvent.KEYCODE_BUTTON_THUMBL)[0])) {
                    left = GameControllerJoystick.getInstance(this, dev,
                            MotionEvent.AXIS_X, MotionEvent.AXIS_Y,
                            new Integer[]
                                    {MotionEvent.AXIS_LTRIGGER, MotionEvent.AXIS_BRAKE},
                            new Integer[]
                                    {KeyEvent.KEYCODE_BUTTON_THUMBL, KeyEvent.KEYCODE_BUTTON_L2,
                                    KeyEvent.KEYCODE_BUTTON_X, KeyEvent.KEYCODE_BUTTON_Y},
                            Joystick.ValueType.MOVEMENT);
                }
                // Right Joystick
                if (axes.contains(MotionEvent.AXIS_Z) && axes.contains(MotionEvent.AXIS_RZ)
                        && (dev.hasKeys(KeyEvent.KEYCODE_BUTTON_R1)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_R2)[0]
                        || dev.hasKeys(KeyEvent.KEYCODE_BUTTON_THUMBR)[0])) {
                    right = GameControllerJoystick.getInstance(this, dev,
                            MotionEvent.AXIS_Z, MotionEvent.AXIS_RZ,
                            new Integer[]{MotionEvent.AXIS_RTRIGGER, MotionEvent.AXIS_THROTTLE,
                                    MotionEvent.AXIS_GAS},
                            new Integer[]{KeyEvent.KEYCODE_BUTTON_R1, KeyEvent.KEYCODE_BUTTON_R2,
                                    KeyEvent.KEYCODE_BUTTON_THUMBR},
                            Joystick.ValueType.MOVEMENT);
                }
                if (axes.contains(MotionEvent.AXIS_X) && axes.contains(MotionEvent.AXIS_Y)
                        && (dev.hasKeys(KeyEvent.KEYCODE_BUTTON_1)[0])
                        && left == null && dpad == null && right == null) {

                    dpad = GameControllerJoystick.getInstance(
                            this, dev, MotionEvent.AXIS_X, MotionEvent.AXIS_Y,
                            new Integer[]{}, new Integer[]{KeyEvent.KEYCODE_BUTTON_1,
                                    KeyEvent.KEYCODE_BUTTON_2, KeyEvent.KEYCODE_BUTTON_3,
                                    KeyEvent.KEYCODE_BUTTON_4, KeyEvent.KEYCODE_BUTTON_5,
                                    KeyEvent.KEYCODE_BUTTON_6, KeyEvent.KEYCODE_BUTTON_7,
                                    KeyEvent.KEYCODE_BUTTON_8, KeyEvent.KEYCODE_BUTTON_11,
                                    KeyEvent.KEYCODE_BUTTON_12, KeyEvent.KEYCODE_BUTTON_13,
                                    KeyEvent.KEYCODE_BUTTON_14, KeyEvent.KEYCODE_BUTTON_15},
                            Joystick.ValueType.POSITION);

                }
                if (dpad != null) {
                    mAttachedJoysticks.add(dpad);
                    dpad.setOtherControllerjoysticks(new GameControllerJoystick[]{left, right});
                }
                if (left != null) {
                    mAttachedJoysticks.add(left);
                    left.setOtherControllerjoysticks(new GameControllerJoystick[]{dpad, right});
                }
                if (right != null) {
                    mAttachedJoysticks.add(right);
                    right.setOtherControllerjoysticks(new GameControllerJoystick[]{dpad, left});
                }
            }
            if ((dpad == null && isSourceSupported(sources, InputDevice.SOURCE_GAMEPAD))
                    && !dev.isVirtual() && !isHardwareKeyboardChecker.check(dev, dev.getId())) {
                mAttachedJoysticks.add(new DpadJoystick(dev));
                return true;
            }
            return dpad != null || left != null || right != null;
        };
    }
    private void updateDevices() {
        List<Integer> keyboardDeviceIds = new ArrayList<>();
        DeviceChecker isHardwareKeyboardChecker = getHardwareKeyboardChecker(keyboardDeviceIds);
        boolean hardwareKeyboard;
        try {
            // unittest
            @SuppressWarnings("unchecked")
            Class<KeyboardUnitTest> clz = (Class<KeyboardUnitTest>)
                    Objects.requireNonNull(EmulationActivity.class.getClassLoader())
                    .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("isKeyboardEnabled");

            hardwareKeyboard = Boolean.TRUE.equals(method.invoke(clz.newInstance()));
            keyboardDeviceIds.add(new KeyEvent(
                    KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_ENTER).getDeviceId());
        } catch (ClassNotFoundException e) {
            // not in unittest.
            hardwareKeyboard = checkDevice(isHardwareKeyboardChecker);

        } catch (NullPointerException | NoSuchMethodException | InvocationTargetException
                 | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }


        for (Joystick joy : mAttachedJoysticks) {
            joy.setPortnumber(Joystick.PORT_NOT_CONNECTED);
            joy.readUseropts(this);
        }
        mAttachedJoysticks.clear();
        checkDevice(getGamepadChecker(isHardwareKeyboardChecker));
        checkDevice((dev, sources) -> {
            if (isSourceSupported(sources, InputDevice.SOURCE_MOUSE)
                    && dev.getMotionRange(MotionEvent.AXIS_VSCROLL) != null
                    && Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                mAttachedJoysticks.add(new MouseController(dev));
            }
            return false;
        });
        if (getPackageManager().hasSystemFeature("android.hardware.touchscreen")) {
            mAttachedJoysticks.add(
                    new TouchJoystick(getResources().getString(R.string.virtual_touch_joystick)));
            mAttachedJoysticks.add(
                    new WheelJoystick(getResources().getString(R.string.virtual_wheel_joystick)));

        }
        if (hardwareKeyboard) {
            mAttachedJoysticks.add(new KeyboardJoystick(KeyboardJoystick.Keyset.KEYSET_A,
                    getResources().getString(R.string.IDS_KEYSET_A), keyboardDeviceIds));
            mAttachedJoysticks.add(new KeyboardJoystick(KeyboardJoystick.Keyset.KEYSET_B,
                    getResources().getString(R.string.IDS_KEYSET_B), keyboardDeviceIds));
        }

        boolean softwareKeyboard =
                checkDevice((dev, sources) ->
                        isSourceSupported(sources, InputDevice.SOURCE_KEYBOARD)
                        && !isSourceSupported(sources, InputDevice.SOURCE_GAMEPAD)
                        && !isSourceSupported(sources, InputDevice.SOURCE_JOYSTICK)
                        && dev.isVirtual()
                        && dev.getKeyboardType() == InputDevice.KEYBOARD_TYPE_ALPHABETIC);

        if (softwareKeyboard != mSoftwareKeyboardAvailable
                || hardwareKeyboard != mHardwareKeyboardAvailable) {
            mSoftwareKeyboardAvailable = softwareKeyboard;
            mHardwareKeyboardAvailable = hardwareKeyboard;

            onAvailableKeyboardsChanged(mHardwareKeyboardAvailable,
                    softwareKeyboard);
        }
        try {
            @SuppressWarnings("unchecked")
            Class<Joystick.UnitTestClass> clz = (Class<Joystick.UnitTestClass>) getClassLoader()
                    .loadClass("de.rainerhock.eightbitwonders.TestBase");
            Joystick.UnitTestClass instance = clz.newInstance();
            final Method methodGetJoysticks = clz.getMethod("getTestJoysticks");
            final Method methodUseJoystick = clz.getMethod("useThisJoystick", Joystick.class);
            @SuppressWarnings("unchecked")
            List<Joystick> result = (List<Joystick>) methodGetJoysticks.invoke(instance);
            if (result != null) {
                for (Joystick joy: result) {
                    joy.readUseropts(this);
                }
                mAttachedJoysticks.addAll(result);
            }
            List<Joystick> toRemove = new LinkedList<>();
            for (Joystick joy:mAttachedJoysticks) {
                Object ret = methodUseJoystick.invoke(instance, joy);
                if (Boolean.FALSE.equals(ret)) {
                    toRemove.add(joy);

                }
            }
            for (Joystick joy:toRemove) {
                mAttachedJoysticks.remove(joy);
            }
        } catch (InstantiationException | NoSuchMethodException
                 | IllegalAccessException | InvocationTargetException e) {
            throw new RuntimeException(e);
        } catch (ClassNotFoundException e) {
            // OK, when not in unit test.
        }
    }

    @NonNull
    private static List<Integer> getCorrespondingButtons(final List<Integer> axes) {
        List<Integer> buttonlist = new ArrayList<>();
        buttonlist.add(KeyEvent.KEYCODE_BUTTON_L1);
        int others = 2;
        if (!axes.contains(MotionEvent.AXIS_X) || !axes.contains(MotionEvent.AXIS_Y)) {
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_THUMBL);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_L2);
            others--;
        }
        if (!axes.contains(MotionEvent.AXIS_Z) || !axes.contains(MotionEvent.AXIS_RZ)) {
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_R1);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_R2);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_THUMBR);
            others--;
        }
        if (others == 0) {
            buttonlist.add(KeyEvent.KEYCODE_UNKNOWN);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_A);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_B);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_X);
            buttonlist.add(KeyEvent.KEYCODE_BUTTON_Y);
        }
        return buttonlist;
    }

    @Override
    public void onInputDeviceAdded(final int deviceId) {
        updateHardware();
    }

    @Override
    public void onInputDeviceRemoved(final int deviceId) {
        updateHardware();
    }

    @Override
    public void onInputDeviceChanged(final int deviceId) {
        updateHardware();

    }


    @Override
    protected void onSaveInstanceState(@NonNull final Bundle outState) {
        updateDevices();
        super.onSaveInstanceState(outState);
    }

}
