package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.ColorRes;
import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public final class ContentSourceView extends LinearLayout {

    private Runnable mAction = null;

    /**
     * Simple constructor to use when creating a ContentSourceView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     * @param bitmap icon to be displayed
     * @param title (short) title to bre displayed
     * @param description descriptive text to be displayed
     */
    public ContentSourceView(final @NonNull Context context, final @NonNull Bitmap bitmap,
                             final @NonNull String title, final @NonNull String description) {
        super(context);
        init(context);
        ((ImageView) findViewById(R.id.iv_bitmap)).setImageBitmap(bitmap);
        ((TextView) findViewById(R.id.tv_title)).setText(title);
        ((TextView) findViewById(R.id.tv_description)).setText(description);
    }

    /**
     * Create ContentSourceView programatically.
     * @param context application context
     * @param colorResId background color for the image
     * @param drawableResId resource id for the image
     * @param title title (should be 30 chars max)
     * @param description description (should be 30 chars max)
     */
    public ContentSourceView(final Context context, final @DrawableRes int drawableResId,
                             final @ColorRes int colorResId,
                             final @NonNull String title, final @NonNull  String description) {
        super(context);
        init(context);
        ((ImageView) findViewById(R.id.iv_bitmap)).setImageResource(drawableResId);
        if (colorResId != Resources.ID_NULL) {
            findViewById(R.id.iv_bitmap).setBackgroundColor(getResources().getColor(colorResId));
        }
        ((TextView) findViewById(R.id.tv_title)).setText(title);
        ((TextView) findViewById(R.id.tv_description)).setText(description);
    }

    /**
     * Action to be performed (mouse/touch/remote control/gamepad) when the view is selected.
     * @param action Action to be performed.
     */
    void setAction(final Runnable action) {
        mAction = action;
    }

    private void init(final Context context, final AttributeSet attrs) {
        init(context);
        //noinspection resource
        TypedArray a = context.getResources().obtainAttributes(attrs, R.styleable.contentsource);
        try {
            int drawableId = a.getResourceId(R.styleable.contentsource_icon, View.NO_ID);
            String title = a.getString(R.styleable.contentsource_header);
            String description = a.getString(R.styleable.contentsource_description);
            if (title != null) {
                ((TextView) findViewById(R.id.tv_title)).setText(title);
            }
            if (description != null) {
                ((TextView) findViewById(R.id.tv_description)).setText(description);
            }

            if (drawableId != View.NO_ID) {
                ((ImageView) findViewById(R.id.iv_bitmap)).setImageResource(drawableId);
            }


        } finally {
            a.recycle();
        }
    }
    private void init(final Context context) {
        LayoutInflater inflater = (LayoutInflater) context.getSystemService(
                Context.LAYOUT_INFLATER_SERVICE);
        inflater.inflate(R.layout.view_contentsource, this);

        setOnClickListener(v -> {
            if (mAction != null) {
                mAction.run();
            }
        });

        setOnKeyListener((v, keyCode, event) -> {
            if (event.getAction() == KeyEvent.ACTION_UP
                    && EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keyCode)
                    && mAction != null) {
                mAction.run();
                return true;
            }
            return false;
        });
    }
    /**
     * Constructor that is called when inflating a ContentSourceView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public ContentSourceView(final Context context, final @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }


    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public ContentSourceView(final Context context, final @Nullable AttributeSet attrs,
                             final int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);
    }
}
