//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------

package de.rainerhock.eightbitwonders;

import static android.graphics.Bitmap.CompressFormat.PNG;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.util.Log;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

@SuppressWarnings("IOStreamConstructor")
final class ConfigurationFactory {


    public static final int FULLQUALITY = 100;

    static Intent prepareCreatePackageIntent(final EmulationActivity ui,
                                             final Emulation emu,
                                             final EmulationConfiguration conf,
                                             final Runnable runOnPrepare,
                                             final Bitmap screenshot) {
        Intent i = new Intent(ui, ShareEmulationActivity.class);
        ui.applyScreenOrientation(i);
        if (ui.findViewById(R.id.jv_wheel) != null) {
            i.putExtra(ShareEmulationActivity.VIRTUAL_WHEELJOYSTICK, true);
        }
        i.putExtra("bitmap", screenshot);
        i.putExtra("configuration", conf);
        Emulation.PackCurrentStateFunctions pcsf = emu.getPackCurrentStateFunctions();
        runOnPrepare.run();
        if (pcsf.getProperties() != null) {
            i.putExtra("emu_props", new HashMap<>(pcsf.getProperties()));
        }
        i.putExtra("exportable_snapshotdata", ui.getSnapshotPath(
                conf).getAbsolutePath());
        if (conf.getTouchJoystickDiagonalsLocked()
                || !ui.getCurrentUseropts().getStringValue(ui.getResources()
                        .getString(R.string.key_joystick_lock_diagonals), "0")
                .equals("0")) {
            i.putExtra("touch_diagonals_locked", true);
        }
        List<Uri> attachedFiles = pcsf.getAttachedFiles();
        if (attachedFiles != null) {
            i.putExtra("attached_files", new LinkedList<>(attachedFiles));
            List<String> visibleFiles = new LinkedList<>();
            for (Uri uri : attachedFiles) {
                if (pcsf.showFiletoUser(uri)) {
                    visibleFiles.add(uri.toString());
                }
            }
            i.putExtra("visible_files", new LinkedList<>(visibleFiles));

        }
        i.putExtra("keyboardvisible", ui.getKeyboardVisibility() == View.VISIBLE);
        i.putExtra("hardwarekeyboard", ui.getViewModel().isKeyboardUsed());

        Emulation.JoystickFunctions jf = emu.getJoystickFunctions();
        if (jf != null) {
            i.putExtra("joysticknames", new HashMap<>(jf.getJoystickports()));
            LinkedHashMap<Integer, Emulation.InputDeviceType> usedJoysticks
                    = new LinkedHashMap<>();
            for (Joystick joy : ui.getAvailableJoysticks()) {
                if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                    usedJoysticks.put(joy.getPortnumber(), joy.getCurrentDeviceType());
                }
            }
            i.putExtra(ShareEmulationActivity.CONNECTED_JOYSTICKS, usedJoysticks);
        }
        if (emu.getPackCurrentStateFunctions() != null) {
            ArrayList<Uri> a = new ArrayList<>(emu.getPackCurrentStateFunctions()
                    .getAttachedFiles());
            i.putExtra("attachedfiles", a);
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        screenshot.compress(PNG, FULLQUALITY, baos);
        i.putExtra("bitmap", baos.toByteArray());
        return i;
    }

    @SuppressWarnings("CheckStyle")
    static class PackagedConfiguration implements EmulationConfiguration {
        private final Map<String, String> mEmuProperties;
        private final Map<String, String> mProperties;
        private final String mEmulation;
        private final String mId;
        private final String mName;
        private final byte[] mBitmapdata;
        private final String mTargetfolder;
        private final Runnable mDeinstaller;
        private final Runnable mResetter;
        private final Set<String> mFliplist = new HashSet<>();
        private final boolean mLocallyCreated;
        private final boolean mTouchJoystickDiagonalsLocked;
        private final Map<Integer, Emulation.InputDeviceType> mInputDevicesTypes
                = new LinkedHashMap<>();
        private final boolean mImported;

        //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
        PackagedConfiguration(final String emulation,
                              final String id,
                              final String name,
                              final Bitmap bmp,
                              final Map<String, String> properties,
                              final Map<String, String> emuproperties,
                              final String targetfolder,
                              final Runnable deinstaller,
                              final boolean locallyCreated,
                              final boolean imported,
                              final boolean diagonalsLocked) {
            mImported = imported;
            mEmuProperties = emuproperties;
            mProperties = properties;
            mEmulation = emulation;
            mResetter = null;
            mLocallyCreated = locallyCreated;
            mTargetfolder = targetfolder;
            mDeinstaller = deinstaller;
            mTouchJoystickDiagonalsLocked = diagonalsLocked;
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            if (bmp != null) {
                bmp.compress(Bitmap.CompressFormat.PNG, FULLQUALITY, stream);
                mBitmapdata = stream.toByteArray();
            } else {
                mBitmapdata = null;
            }
            mId = id;
            mName = name;
            boolean isValue = false;
            if (emuproperties.containsKey("__openfiles__")
                    && emuproperties.get("__openfiles__") != null) {
                for (String s : Objects.requireNonNull(emuproperties.get("__openfiles__"))
                        .split(" ")) {
                    if (isValue) {
                        mFliplist.add(Uri.decode(s));
                    }
                    isValue = !isValue;
                }
            }
            int index = 0;

            while (properties.containsKey(getIndexedParam("additional_file%d", index))) {
                mFliplist.add(Uri.fromFile(new File(
                        targetfolder + File.separator + Uri.decode(properties.get(
                                getIndexedParam("additional_file%d", index))))).toString());
                index++;
            }


        }
        private final Map<Integer, PreferredJoystickType> mJoysticktypes = new LinkedHashMap<>();
        @Override
        public void adaptJoysticks(final BaseActivity context,
                                   final Emulation.JoystickFunctions joystickFunctions,
                                   final List<Joystick> availableJoysticks,
                                   final Map<Integer, PreferredJoystickType> preferredTypes) {
            int firstport = Joystick.PORT_NOT_CONNECTED;
            for (int i:joystickFunctions.getJoystickports().keySet()) {
                String key = getIndexedParam("joystick%d", i);
                if (mProperties.containsKey(key)) {
                    String value = mProperties.get(key);
                    if (value != null) {
                        mJoysticktypes.put(i, PreferredJoystickType.valueOf(
                                value.toUpperCase(Locale.ROOT)));
                    }
                    if (firstport == Joystick.PORT_NOT_CONNECTED) {
                        firstport = i;
                    }
                    Emulation.InputDeviceType idt = Emulation.InputDeviceType.DSTICK;
                    String keyInputDeviceType = getIndexedParam("joystick%d_devicetype", i);
                    if (mProperties.containsKey(keyInputDeviceType)) {
                        String valueIdt = mProperties.get(keyInputDeviceType);
                        if (valueIdt != null && !"null".equals(valueIdt)) {
                            try {
                                idt = Emulation.InputDeviceType.valueOf(valueIdt);
                            } catch (IllegalArgumentException e) {
                                if (BuildConfig.DEBUG) {
                                    throw (e);
                                }
                            }
                        }
                    }
                    mInputDevicesTypes.put(i, idt);
                }
            }
            //switch (mJoysticktypes.size()) {
            if (mJoysticktypes.isEmpty() && !isLocallyGenerated()) {
                for (Joystick joy : availableJoysticks) {
                    joy.setPortnumber(Joystick.PORT_NOT_CONNECTED);
                }
            } else if (mJoysticktypes.size() == 1 && !isLocallyGenerated()) {
                Emulation.InputDeviceType requiredIdt = mInputDevicesTypes.get(firstport);
                boolean hardwareJoystickAvailable = false;
                for (Joystick joy : availableJoysticks) {
                    if (!(joy instanceof VirtualJoystick)
                            && !(joy instanceof KeyboardJoystick)
                            && !(joy instanceof MultiplexerJoystick)) {
                        if (joy.getSupportedDeviceTypes().contains(requiredIdt)) {
                            hardwareJoystickAvailable = true;
                            joy.setPortnumber(firstport);
                            joy.setCurrentdevicetype(requiredIdt);
                        }
                    }
                }
                for (Joystick joy : availableJoysticks) {
                    if (joy instanceof VirtualJoystick
                            && joy.getSupportedDeviceTypes().contains(requiredIdt)) {
                        if (hardwareJoystickAvailable) {
                            joy.setPortnumber(Joystick.PORT_NOT_CONNECTED);
                        } else {
                            if (((VirtualJoystick) joy).isPreferredType(
                                    preferredTypes.get(firstport))) {
                                joy.setPortnumber(firstport);
                                joy.setCurrentdevicetype(requiredIdt);
                            } else {
                                joy.setPortnumber(Joystick.PORT_NOT_CONNECTED);
                            }
                        }
                    }
                }
            } else {
                for (Joystick joy : availableJoysticks) {
                    joy.readUseropts(context);
                }
            }
        }

        @Override
        public boolean getTouchJoystickDiagonalsLocked() {
            return mTouchJoystickDiagonalsLocked;
        }

        @Override
        public Map<Integer, PreferredJoystickType> getVirtualJoystickTypes() {
            return mJoysticktypes;
        }
        public Map<Integer, Emulation.InputDeviceType> getInputDeviceTypes() {
            return mInputDevicesTypes;
        }

        @NonNull
        @Override
        public String getId() {
            return mId;
        }
        @NonNull
        @Override
        public Bitmap getBitmap() {
            if (mBitmapdata != null) {
                return BitmapFactory.decodeByteArray(mBitmapdata,
                        0, mBitmapdata.length);
            }
            return Bitmap.createBitmap(1, 1, Bitmap.Config.RGB_565);
        }

        @NonNull
        @Override
        public String getEmulatorId() {
            return mEmulation;
        }


        @Override
        public Orientation getBestOrientation() {
            try {
                if (mProperties.containsKey("orientation")
                        && mProperties.get("orientation") != null) {

                    //noinspection ConstantConditions
                    return Enum.valueOf(Orientation.class,
                            mProperties.get("orientation").toUpperCase(Locale.ROOT));
                } else {
                    return Orientation.DEFAULT;
                }
            } catch (IllegalArgumentException e) {
                return Orientation.DEFAULT;
            }
        }

        @Override
        public void apply(final EmulationActivity ui, final Emulation emu) {
        }

        @Override
        public Map<String, String> getPresettings() {
            return mEmuProperties;
        }

        @Override
        public DownloaderFactory.Downloader getAdditionalDownloader(final BaseActivity activity) {
            return new DownloaderFactory(activity).createPackageDownloader(mTargetfolder);
        }

        @Override
        public String getName() {
            String ret = ConfigurationFactory.readConfigurationName(this);
            if (ret == null) {
                ret = mName;
            }
            return ret;
        }



        @Override
        public boolean isBareMachine() {
            return false;
        }

        @Override
        public boolean isLocallyGenerated() {
            return mLocallyCreated;
        }

        @Override
        public boolean isImported() {
            return mImported;
        }

        @Override
        public boolean isStream() {
            return false;
        }

        @Override
        public URL getWebAdress() {
            try {
                return new URL(mProperties.get("url"));
            } catch (MalformedURLException e) {
                return null;
            }

        }

        @Override
        public Runnable getFactoryResetter() {
            return mResetter;
        }

        @Override
        public Runnable getDeinstaller() {
            return mDeinstaller;
        }
        private List<Integer> getJoystickports(final String keyTemplate,
                                               final List<Integer> allJoystickports) {
            List<Integer> ret = new LinkedList<>();
            for (Integer i:allJoystickports) {
                if (mProperties.containsKey(getIndexedParam(keyTemplate, i))) {
                    ret.add(i);
                }
            }
            return ret;

        }
        @Override
        public List<Integer> getUsedJoystickports(final List<Integer> allJoystickports) {
            if (isLocallyGenerated()) {
                return null;
            }
            return getJoystickports("joystick%d", allJoystickports);
        }

        @NonNull
        @Override
        public List<Integer> getRequiredJoystickports(final List<Integer> allJoystickports) {
            return getJoystickports("joystick%d_required", allJoystickports);
        }

        @Override
        public boolean isKeyboardUsed() {
            if (mProperties.containsKey("keyboard")) {
                return !"0".equals(mProperties.get("keyboard"));
            }
            return true;
        }

        @Override
        public Set<Uri> getFliplist() {
            Set<Uri> ret = new HashSet<>();
            for (String s:mFliplist) {
                ret.add(Uri.parse(s));
            }
            return ret;
        }

        @Override
        public void onEmulatorFliplistChanged(final Set<Uri> emulatorlist) {
            mFliplist.clear();
            for (Uri uri:emulatorlist) {
                mFliplist.add(uri.toString());
            }
        }

        @Override
        public String getFilepath(final String basename) {
            String name = Uri.decode(basename);
            if (new File(name).isAbsolute()) {
                return name;
            }
            return mTargetfolder + File.separator + name;
        }

        @Override
        public byte[] encrypt(final byte[] plaindata) {
            return plaindata;
        }

        @Override
        public byte[] decrypt(final byte[] encrypted) {
            return encrypted;
        }

        @Nullable
        @Override
        public File getLocalPath() {
            return new File(mTargetfolder);
        }

        @Override
        public String getProperty(final String key, final String defaultvalue) {
            return mProperties.containsKey(key) ? mProperties.get(key) : defaultvalue;
        }
    }
    private static final int BLOCKSIZE = 8192;
    private void copyFile(final InputStream reader,
                            final String filename,
                            final String targetfolder) throws IOException {
        byte[] buffer = new byte[BLOCKSIZE];
        String target = targetfolder + "/" + filename;
        OutputStream os = new FileOutputStream(target);
        int length;
        while ((length = reader.read(buffer)) > 0) {
            os.write(buffer, 0, length);
        }
        os.close();
    }

    interface AvailableResourcesInvestigator     {
        List<String> getAvailableResources();
    }
    private List<String> remotefiles(final InputStream data) {
        List<String> ret = new LinkedList<>();
        String line;
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(data));
            line = br.readLine();
            while (line != null) {
                ret.add(DownloaderFactory.basenameForDownload(line));
                line = br.readLine();
            }
        } catch (IOException e) {
            if (e.getMessage() != null) {
                Log.v(getClass().getSimpleName(), e.getMessage());
            } else {
                Log.v(getClass().getSimpleName(), e.toString());
            }
        }
        return ret;
    }
    // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
    static StreamConfiguration createConfiguration(@NonNull final Context activity,
                                      @NonNull final String name,
                                      @NonNull final URL siteUrl,
                                      final URL imageUrl,
                                      @NonNull final String emulatorId,
                                      @NonNull final Map<String, String> config,
                                      @NonNull final Map<String, String> emuConfig,
                                      final Map<String, URL> assets,
                                      final boolean keep) {
        String id = BaseActivity.hexSha256Hash(
                siteUrl.toString().getBytes(StandardCharsets.UTF_8));
        if (!keep) {
            id = "_" + id;
        }
        StreamConfiguration ret = new StreamConfiguration(
                emulatorId,
                id,
                name, siteUrl.toString(), null, config, emuConfig, assets, null, null, false);
        final File parent;
        if (keep) {
            parent = new File(new File(activity.getFilesDir(), "packages"), "streams");
        } else {
            parent = new File(activity.getCacheDir(), "streams");
            parent.deleteOnExit();
        }
        File folder = new File(parent, ret.mId);

        if (!folder.exists()) {
            //noinspection ResultOfMethodCallIgnored
            folder.mkdirs();
        }
        ret.mSiteURL = siteUrl;
        ret.mImageURL = imageUrl;
        ret.mKeep = keep;
        ret.mTargetfolder = folder.getAbsolutePath();
        return ret;
    }

    EmulationConfiguration createConfiguration(final BaseActivity activity,
                                              final InputStreamCreator streamcreator,
                                             final AvailableResourcesInvestigator investigator,
                                              final Runnable deinstaller,
                                              final boolean bareMachine,
                                              final boolean imported,
                                              final File existingLocation) {
        Properties properties = new Properties();
        List<String> resources = investigator.getAvailableResources();
        InputStream isProps = streamcreator.getInputStream("properties");
        if (isProps != null) {
            try {
                properties.load(isProps);
                String emulation = properties.getProperty("emulation");
                String id = properties.getProperty("id");
                String name = properties.getProperty("name");
                String url = properties.getProperty("url");
                String imagefile = properties.getProperty("image");
                if (imagefile != null & !resources.contains(imagefile)) {
                    imagefile = null;
                }
                if (emulation != null & id != null && name != null && imagefile != null) {
                    Properties emuProperties = new Properties();
                    try {
                        InputStream isEmuProps = streamcreator.getInputStream("emu_properties");
                        if (isEmuProps != null) {
                            emuProperties.load(isEmuProps);
                        }
                        String target;
                        if (existingLocation != null) {
                            if (existingLocation.isAbsolute()) {
                                target = existingLocation.getAbsolutePath();
                            } else {
                                target = activity.getFilesDir().toString() + "/packages/"
                                        + existingLocation.getPath() + "/"
                                        + getClass().getSimpleName() + "_"
                                        + properties.getProperty("emulation")
                                        + properties.getProperty("id");

                            }
                        } else {
                            target = activity.getFilesDir().toString() + "/packages/"
                                    + getClass().getSimpleName() + "_"
                                    + properties.getProperty("emulation")
                                    + properties.getProperty("id");
                        }
                        Map<String, String> emuPropertiesMap = new LinkedHashMap<>();
                        List<String> files = new LinkedList<>();
                        boolean doCopy = !new File(target).exists();
                        if (doCopy) {
                            try {
                                //noinspection ResultOfMethodCallIgnored
                                new File(target).mkdirs();
                            } catch (SecurityException e) {
                                throw new IOException(e);
                            }
                        }
                        doCopy |= (existingLocation != null && !existingLocation.isAbsolute());
                        for (String file : investigator.getAvailableResources()) {
                            if (!(streamcreator.getInputStream(file) instanceof FileInputStream)) {
                                if (doCopy || imagefile.equals(file)
                                        || file.equals("vice-actions.js")
                                        || file.equals("properties")) {
                                    copyFile(streamcreator.getInputStream(file),
                                            file, target);
                                }
                            }
                            if (file.equals("additional_downloads")) {
                                files.addAll(remotefiles(streamcreator.getInputStream(file)));
                            } else if (file.equals("additional_downloads.yml")) {
                                files.addAll(DownloaderFactory.downloadFilesFromYaml(
                                        streamcreator.getInputStream(file)));
                            } else {
                                files.add(file);
                            }
                        }
                        Useropts opts = new Useropts();
                        opts.setCurrentEmulation(activity, emulation, id);
                        if (opts.isEmpty() && resources.contains("default-useropts.yml")) {
                            InputStream is = streamcreator.getInputStream("default-useropts.yml");
                            BufferedReader br = new BufferedReader(new InputStreamReader(is));
                            List<String> data = new LinkedList<>();
                            String line;
                            while ((line = br.readLine()) != null) {
                                data.add(line);
                            }
                            String result = String.join("\n", data);
                            opts.readFromYaml(activity, new StringReader(result));

                        }
                        // DA!
                        boolean initUseropts = !opts.getBooleanValue("__initialized__", false);
                        for (String key : emuProperties.stringPropertyNames()) {
                            String value = emuProperties.getProperty(key);
                            if (!(key.startsWith("__") && key.endsWith("__"))) {
                                if (initUseropts) {
                                    opts.setValue(Useropts.Scope.CONFIGURATION, key, value);
                                } else {
                                    value = opts.getStringValue(key, "");
                                }
                            }
                            StringBuilder valueWithPaths = new StringBuilder();
                            for (String word : value.split(" ")) {
                                if (files.contains(word) || files.contains(Uri.decode(word))) {
                                    valueWithPaths.append(target).append("/").append(word)
                                            .append(" ");
                                } else {
                                    valueWithPaths.append(word).append(" ");
                                }
                                emuPropertiesMap.put(key, valueWithPaths.toString().trim());
                            }
                        }

                        opts.setValue(Useropts.Scope.CONFIGURATION, "__initialized__", true);
                        Bitmap bmp = BitmapFactory.decodeStream(
                                streamcreator.getInputStream(imagefile));
                        Map<String, String> mapProperties = new LinkedHashMap<>();
                        for (String key : properties.stringPropertyNames()) {
                            mapProperties.put(key, properties.getProperty(key));
                        }
                        boolean touchJoystickDiagonalsLocked = "1".equals(
                                properties.getProperty("touchjoystick_diagonals_locked", null));
                        if (streamcreator.getInputStream("assets") != null) {
                            Properties propDownloads = new Properties();
                            propDownloads.load(streamcreator.getInputStream("assets"));
                            Map<String, URL> downloads = new LinkedHashMap<>();
                            for (String s : propDownloads.stringPropertyNames()) {
                                String value = propDownloads.getProperty(s);
                                downloads.put(s, new URL(value));
                            }
                            return new StreamConfiguration(
                                    emulation, id, name, url, bmp, mapProperties,
                                    emuPropertiesMap, downloads, target, deinstaller,
                                    touchJoystickDiagonalsLocked);
                        }
                        return new PackagedConfiguration(
                                emulation, id, name, bmp, mapProperties, emuPropertiesMap,
                                target, deinstaller, bareMachine, imported,
                                touchJoystickDiagonalsLocked);

                    } catch (IOException e) {
                        if (e.getMessage() != null) {
                            Log.v(getClass().getSimpleName(), e.getMessage());
                        } else {
                            Log.v(getClass().getSimpleName(), e.toString());
                        }
                    }
                }
            } catch (Exception e) {
                if (BuildConfig.DEBUG) {
                    throw new RuntimeException(e);
                }
            }
        }
        return null;
    }
    static Map<String, String> mapFromProperties(final Properties p) {
        HashMap<String, String> retMap = new HashMap<>();
        for (Map.Entry<Object, Object> entry : p.entrySet()) {
            retMap.put(String.valueOf(entry.getKey()), String.valueOf(entry.getValue()));
        }
        return retMap;
    }
    static Properties createAbsolutePathProperties(final Properties source,
                                                   final String targetpath,
                                                   final Set<String> files) {
        Properties ret = new Properties();
        for (String key : source.stringPropertyNames()) {
            String value = source.getProperty(key);
            if (key.startsWith("__") && key.endsWith("__")) {
                StringBuilder valueWithPaths = new StringBuilder();
                for (String word : value.split(" ")) {
                    if (files.contains(word) || files.contains(Uri.decode(word))) {
                        valueWithPaths.append(targetpath).append("/").append(word)
                                .append(" ");
                    } else {
                        valueWithPaths.append(word).append(" ");
                    }
                    value = valueWithPaths.toString().trim();
                }
            }
            ret.setProperty(key, value);
        }
        return ret;

    }
    static EmulationConfiguration  createOneTimeZipConfiguration(
            final BaseActivity activity, final Map<String, ByteArrayOutputStream> zipcontents,
            final Map<String, ByteArrayOutputStream> savestates) throws IOException {
        String targetpath = new File(activity.getCacheDir(), "streams").getAbsolutePath();
        if (zipcontents.containsKey("properties")) {
            for (String filename : zipcontents.keySet()) {
                if (!new File(targetpath).exists()) {
                    new File(targetpath).mkdirs();
                }
                if (!new File(targetpath, ".originals").exists()) {
                    new File(targetpath,  ".originals").mkdirs();
                }
                for (String f : new String[]{targetpath + "/" + filename,
                        targetpath + "/.originals/" + filename}) {
                    FileOutputStream fos = new FileOutputStream(f);
                    fos.write(Objects.requireNonNull(zipcontents
                            .get(filename)).toByteArray());
                    fos.close();
                }
            }

            Properties p = new Properties();
            try {
                ByteArrayOutputStream propsData = zipcontents.get("properties");
                if (propsData != null) {
                    p.load(new ByteArrayInputStream(propsData.toByteArray()));
                    String emulation = p.getProperty("emulation", null);
                    String name = p.getProperty("name", null);
                    String id = p.getProperty("id", null);
                    if (id != null) {
                        id = "_" + id;
                    } else {
                        id = "_" + UUID.randomUUID().toString();
                    }
                    if (savestates != null && !savestates.isEmpty()) {
                        File root = activity.getSnapshotPath(emulation, id);
                        for (String snapshot: savestates.keySet()) {
                            ByteArrayOutputStream baos = savestates.get(snapshot);
                            File parent = new File(root, snapshot).getParentFile();
                            if (baos != null && parent != null) {
                                if (parent.isDirectory() || parent.mkdirs()) {
                                    FileOutputStream fos = new FileOutputStream(
                                            new File(root, snapshot));
                                    fos.write(baos.toByteArray());
                                    fos.close();
                                }
                            }
                        }
                    }
                    String image = p.getProperty("image", null);
                    if (name != null && emulation != null) {
                        ByteArrayOutputStream emuPropsdata = zipcontents.get("emu_properties");
                        Properties emuProps = new Properties();
                        if (emuPropsdata != null) {
                            emuProps.load(new ByteArrayInputStream(emuPropsdata.toByteArray()));
                        }
                        ByteArrayOutputStream imagedata = zipcontents.get(image);
                        Bitmap bmp;
                        if (imagedata != null) {
                            bmp = BitmapFactory.decodeStream(
                                    new ByteArrayInputStream(imagedata.toByteArray()));
                        } else {
                            bmp = null;
                        }
                        return new PackagedConfiguration(emulation, id, name, bmp,
                            mapFromProperties(p),
                            mapFromProperties(createAbsolutePathProperties(
                                    emuProps, targetpath, zipcontents.keySet())),
                            targetpath,
                            null, false, true, false);
                    }
                }
            } catch (IOException e) {
                if (BuildConfig.DEBUG) {
                    throw new RuntimeException(e);
                }
            }

        }
        return null;

    }
    MachineConfiguration createConfiguration(final Context context, final String emulationId) {
        return new MachineConfiguration(context, emulationId);
    }
    static void updateBitmap(final String path, final Bitmap bmp) {
        try {
            FileOutputStream out = new FileOutputStream(path);
            bmp.compress(Bitmap.CompressFormat.PNG, COMPRESS_MAX_QUALITY, out);
            out.flush();
            out.close();
        } catch (IOException e) {
            // ok
        }

    }
    static void updateBitmap(final EmulationConfiguration conf, final Bitmap bmp) {
        updateBitmap(new File(conf.getLocalPath(), "screenshot.png").getAbsolutePath(), bmp);
    }
    private static final int COMPRESS_MAX_QUALITY = 100;
    static void storeConfiguration(final DeviceSupportActivity activity,
                                   final String id,
                                   final Map<String, String> properties,
                                   final Map<String, String> emuProperties,
                                   final List<Uri> attachedFiles,
                                   final List<Joystick> connectedJoysticks,
                                   final Bitmap bmp) {
        String folder = activity.getApplicationContext().getFilesDir() + "/packages/user/" + id;
        try {
            if (!new File(folder).mkdirs()) {
                // make checkstyle smile
                Log.v(ConfigurationFactory.class.getSimpleName(),
                        String.format("Cannot create %s", folder));
            }
            activity.getCurrentUseropts().copyToConfiguration(activity, id);
            Properties props = new Properties();
            props.putAll(properties);
            props.put("image", "screenshot.png");
            for (Joystick joy : connectedJoysticks) {
                String key = getIndexedParam("joystick%d", joy.getPortnumber());
                if (joy instanceof WheelJoystick) {
                    props.put(key,
                            EmulationConfiguration.PreferredJoystickType.WHEEL.toString());
                } else {
                    props.put(key,
                            EmulationConfiguration.PreferredJoystickType.DIRECTIONAL.toString());
                }
            }

            props.store(new FileOutputStream(new File(folder, "properties")), "");

            props = new Properties();
            props.putAll(emuProperties);

            props.store(new FileOutputStream(new File(folder, "emu_properties")), "");
            for (Uri uri : attachedFiles) {

                InputStream is = null;
                if (uri.getScheme() != null) {
                    if ("content".equals(uri.getScheme())) {
                        activity.getContentResolver().takePersistableUriPermission(uri,
                                Intent.FLAG_GRANT_READ_URI_PERMISSION
                                | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);

                    } else if (uri.toString().startsWith(
                            Uri.fromFile(activity.getCacheDir()).toString())) {
                        is = new FileInputStream(uri.getPath());
                    }
                } else {
                    is = new FileInputStream(uri.getPath());
                }
                if (is != null) {
                    byte[] data = new byte[is.available()];
                    //noinspection ResultOfMethodCallIgnored
                    is.read(data);
                    OutputStream os = new FileOutputStream(
                            new File(folder, FileUtil.getFileName(activity, uri)));
                    os.write(data);
                    is.close();
                }
            }
        } catch (IOException e) {
            // ok
        }
        updateBitmap(new File(new File(folder), "screenshot.png").getAbsolutePath(), bmp);
    }

    static void saveConfiguration(final EmulationActivity activity, final String name,
                                  final Bitmap bmp, final File dump) {
        Emulation currentEmulation = activity.getViewModel().getEmulation();
        Emulation.PackCurrentStateFunctions pcsf = currentEmulation.getPackCurrentStateFunctions();
        EmulationUi.Encryptor enc = activity.getViewModel().getConfiguration();
        saveConfiguration(activity, name, pcsf.getImportContentsRunnable(dump, enc), bmp,
                UUID.randomUUID().toString());
    }
    static void saveConfiguration(final EmulationActivity activity,
                                  final String name,
                                  final int timeOffset) {
        Emulation currentEmulation = activity.getViewModel().getEmulation();
        EmulationUi.Encryptor enc = activity.getViewModel().getConfiguration();
        Emulation.TimeMachineFunctions tmf = currentEmulation.getTimeMachineFunctions();
        Emulation.PackCurrentStateFunctions pcsf = currentEmulation.getPackCurrentStateFunctions();
        Bitmap bmp;
        if (timeOffset == 0 || tmf == null) {
            bmp = activity.getCurrentBitmap();
        } else {
            bmp = tmf.getScreenPreview(timeOffset);
        }
        saveConfiguration(activity, name, pcsf.getDumpContentsRunnable(timeOffset, enc),
                bmp, UUID.randomUUID().toString());
    }
    private static void saveConfiguration(final @NonNull EmulationActivity activity,
                                  final @NonNull String name,
                                  final @NonNull Runnable runOnStart,
                                  final @NonNull Bitmap bmp,
                                  final @NonNull String id) {
        Emulation currentEmulation = activity.getViewModel().getEmulation();
        Emulation.PackCurrentStateFunctions pcsf = currentEmulation.getPackCurrentStateFunctions();
        Map<String, String> properties = new LinkedHashMap<>();
        if (!name.isEmpty()) {
            runOnStart.run();
            properties.put("name", name);
            properties.put("emulation",
                    currentEmulation.getEmulatorId());
            properties.put("id", id);
            Map<String, String> emuProperties = pcsf.getProperties();
            List<Joystick> connectedJoysticks = new LinkedList<>();
            for (Joystick joy:activity.getAvailableJoysticks()) {
                joy.readUseropts(activity);
                if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                    connectedJoysticks.add(joy);
                }
            }
            ConfigurationFactory.storeConfiguration(activity,
                    id, properties, emuProperties,
                    pcsf.getAttachedFiles(),
                    connectedJoysticks,
                    bmp);
        }
    }



    static class MachineConfiguration implements EmulationConfiguration {
        private final String mEmulationId;
        private final String mName;
        private final Map<String, String> mProperties;

        MachineConfiguration(final Context context, final String emulationId) {
            mEmulationId = emulationId;
            mName = context.getResources().getString(Objects.requireNonNull(
                    Emulationslist.getDescription(emulationId)).getNameResourceId());
            mProperties = new LinkedHashMap<>();
            mProperties.put("name", mName);
            mProperties.put("emulation", mEmulationId);

        }
        @Override
        public void adaptJoysticks(final BaseActivity context,
                                   final Emulation.JoystickFunctions joystickFunctions,
                                   final List<Joystick> availableJoysticks,
                                   final Map<Integer, PreferredJoystickType> preferred) {
        }

        @Override
        public boolean getTouchJoystickDiagonalsLocked() {
            return false;
        }

        @Override
        public Map<Integer, PreferredJoystickType> getVirtualJoystickTypes() {
            return null;
        }
        @Override
        public Map<Integer, Emulation.InputDeviceType> getInputDeviceTypes() {
            return null;
        }
        @NonNull
        @Override
        public String getId() {
            return "default";
        }
        @NonNull
        @Override
        public Bitmap getBitmap() {
            return Bitmap.createBitmap(1, 1, Bitmap.Config.ARGB_8888);
        }
        @NonNull
        @Override
        public String getEmulatorId() {
            return mEmulationId;
        }
        @Override
        public Orientation getBestOrientation() {
            return Orientation.DEFAULT;
        }
        @Override
        public void apply(final EmulationActivity ui, final Emulation emu) {

        }
        @Override
        public Map<String, String> getPresettings() {
            return new LinkedHashMap<>();
        }
        @Override
        public DownloaderFactory.Downloader getAdditionalDownloader(final BaseActivity activity) {
            return null;
        }
        @Override
        public String getName() {
            return mName;
        }

        @Override
        public boolean isBareMachine() {
            return true;
        }

        @Override
        public boolean isLocallyGenerated() {
            return false;
        }

        @Override
        public boolean isImported() {
            return false;
        }

        @Override
        public boolean isStream() {
            return false;
        }

        @Override
        public URL getWebAdress() {
            return null;
        }

        @Override
        public Runnable getFactoryResetter() {
            return null;
        }

        @Override
        public Runnable getDeinstaller() {
            return null;
        }

        @Override
        public List<Integer> getUsedJoystickports(final List<Integer> allJoystickports) {
            return null;
        }

        @NonNull
        @Override
        public List<Integer> getRequiredJoystickports(final List<Integer> allJoystickports) {
            return new ArrayList<>();
        }

        @Override
        public boolean isKeyboardUsed() {
            return true;
        }
        @Override
        public Set<Uri> getFliplist() {
            return new HashSet<>();
        }

        @Override
        public void onEmulatorFliplistChanged(final Set<Uri> emulatorlist) {

        }

        @Override
        public String getFilepath(final String basename) {
            return Uri.decode(basename);
        }

        @Override
        public byte[] encrypt(final byte[] plaindata) {
            return plaindata;
        }

        @Override
        public byte[] decrypt(final byte[] encrypted) {
            return encrypted;
        }

        @Nullable
        @Override
        public File getLocalPath() {
            return null;
        }

        @Override
        public String getProperty(final String key, final String defaultvalue) {
            return mProperties.containsKey(key) ? mProperties.get(key) : defaultvalue;
        }
    }
    interface InputStreamCreator {
        InputStream getInputStream(String resourcename);
    }
    static String getIndexedParam(final String template, final int index) {
        return String.format(Locale.getDefault(), template, index);
    }
    static class StreamConfiguration implements EmulationConfiguration {
        private static final String TAG = EmulationConfiguration.class.getSimpleName();
        private final byte[] mBitmapdata;
        private boolean mKeep = true;
        private final Runnable mDeinstaller;
        private final boolean mDiagonalsLocked;
        //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
        StreamConfiguration(final String emulation,
                            final String id,
                            final String name,
                            final String url,
                            final Bitmap bmp,
                            final Map<String, String> properties,
                            final Map<String, String> emuproperties,
                            final Map<String, URL> downloads,
                            final String targetFolder,
                            final Runnable deinstaller,
                            final boolean diagonalsLocked) {
            this(emulation, id, name, url, bmp, properties, emuproperties, downloads, null,
                    targetFolder, deinstaller, diagonalsLocked);
        }

        //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
        private StreamConfiguration(final String emulation,
                                   final String id,
                                   final String name,
                                   final String url,
                                   final Bitmap bmp,
                                   final Map<String, String> properties,
                                   final Map<String, String> emuproperties,
                                   final Map<String, URL> downloads,
                                    final Map<String, ByteArrayOutputStream> localdata,
                                   final String targetFolder,
                                   final Runnable deinstaller,
                                   final boolean diagonalsLocked) {
            mEmulatorSettings = emuproperties;
            mPresettings = properties;
            mId = id;
            mEmulatorId = emulation;
            mName = name;
            mDiagonalsLocked = diagonalsLocked;
            mDeinstaller = deinstaller;
            mAssets = downloads;
            mTargetfolder = targetFolder;
            try {
                mSiteURL = new URL(url);
            } catch (MalformedURLException e) {
                mSiteURL = null;
            }
            if (bmp != null) {
                ByteArrayOutputStream stream = new ByteArrayOutputStream();
                bmp.compress(Bitmap.CompressFormat.PNG, FULLQUALITY, stream);
                mBitmapdata = stream.toByteArray();
            } else {
                mBitmapdata = null;
            }
            if (localdata != null) {
                for (String filename : localdata.keySet()) {
                    File f = new File(targetFolder, filename);
                    try {
                        ByteArrayOutputStream baos = localdata.get(filename);
                        if (baos != null) {
                            FileOutputStream fos = new FileOutputStream(f);
                            fos.write(baos.toByteArray());
                            fos.close();
                        }
                    } catch (IOException e) {
                        if (BuildConfig.DEBUG) {
                            throw new RuntimeException(e);
                        }
                    }
                }
            }
        }
        private final Map<String, String> mEmulatorSettings;
        private final Map<String, URL> mAssets;
        private String mTargetfolder;
        void downloadFile(final List<DownloaderFactory.HttpRequest> httpclients,
                          final URL src, final OutputStream os)
                throws DownloaderFactory.AdditionalDownloads.DownloadException {
            DownloaderFactory.HttpResult result = null;
            for (DownloaderFactory.HttpRequest client : httpclients) {
                try {
                    result = client.execute(src);
                    if (result.getResultcode() >= HttpURLConnection.HTTP_OK
                            && result.getResultcode() < HttpURLConnection.HTTP_MULT_CHOICE
                            && result.getBody() != null) {
                        os.write(result.getBody());
                        break;
                    }
                } catch (IOException e) {
                    throw new DownloaderFactory.AdditionalDownloads.DownloadException(
                            src.toString(), src.toString(), null, null,
                            result != null && result.getResultcode()
                                    == BaseActivity.HTTP_STATUS_SSL_ERROR);
                }
            }
        }
        private URL mImageURL = null;
        @Override
        public void adaptJoysticks(final BaseActivity activity,
                                   final Emulation.JoystickFunctions joystickfunctions,
                                   final List<Joystick> availableJoysticks,
                                   final Map<Integer, PreferredJoystickType>
                                                   preferredJoysticktypes) {

        }

        @Override
        public boolean getTouchJoystickDiagonalsLocked() {
            return mDiagonalsLocked;
        }

        @Override
        public Map<Integer, PreferredJoystickType> getVirtualJoystickTypes() {
            return null;
        }
        @Override
        public Map<Integer, Emulation.InputDeviceType> getInputDeviceTypes() {
            return null;
        }
        private final String mId;
        @NonNull
        @Override
        public String getId() {
            return mId;
        }

        @Override
        public Bitmap getBitmap() {
            if (mBitmapdata != null) {
                return BitmapFactory.decodeByteArray(mBitmapdata,
                        0, mBitmapdata.length);
            }
            //return Bitmap.createBitmap(1, 1, Bitmap.Config.RGB_565);
            return null;
        }
        private final String mEmulatorId;
        @NonNull
        @Override
        public String getEmulatorId() {

            return mEmulatorId;
        }

        @Override
        public Orientation getBestOrientation() {
            return Orientation.DEFAULT;
        }

        @Override
        public void apply(final EmulationActivity ui, final Emulation emu) {

        }
        private String mEncryptionKey = "";
        private final Map<String, String> mPresettings;
        @Override
        public Map<String, String> getPresettings() {
            return mEmulatorSettings;
        }
        @Override
        public DownloaderFactory.Downloader getAdditionalDownloader(final BaseActivity activity) {
            DownloaderFactory.Downloader ret = new DownloaderFactory.Downloader() {
                @Override
                void run(final List<DownloaderFactory.HttpRequest> httpRequests)
                        throws NullPointerException,
                        DownloaderFactory.AdditionalDownloads.DownloadException {
                    StringBuilder sb = new StringBuilder(mSiteURL.toString());
                    if (mAssets != null) {
                        for (URL url : mAssets.values()) {
                            byte[] data = null;
                            if (url.toString().contains(".zip#")) {
                                try {
                                    URL zipUrl = new URL(url.toString().split("#", 2)[0]);
                                    byte[] zipdata = (activity.getStreamData(
                                            StreamConfiguration.this.getId(), zipUrl.toString()));
                                    if (zipdata == null) {
                                        ByteArrayOutputStream baos = new ByteArrayOutputStream();
                                        downloadFile(activity.getHttpRequests(), zipUrl, baos);
                                        zipdata = baos.toByteArray();
                                        activity.cacheStreamingData(
                                                StreamConfiguration.this, url, zipdata);
                                    }
                                    ZipInputStream zis = new ZipInputStream(
                                            new ByteArrayInputStream(zipdata));
                                    String entryname = url.toString().split("#", 2)[1];
                                    //noinspection TryFinallyCanBeTryWithResources
                                    try {
                                        ZipEntry ze = zis.getNextEntry();
                                        while (ze != null) {
                                            byte[] buffer = new byte[BaseActivity.CHUNK_SIZE];
                                            if (ze.getName().equals(entryname)) {
                                                ByteArrayOutputStream output =
                                                        new ByteArrayOutputStream();
                                                int len;
                                                while ((len = zis.read(buffer)) > 0) {
                                                    output.write(buffer, 0, len);
                                                }
                                                data = output.toByteArray();
                                                break;
                                            }
                                            ze = zis.getNextEntry();
                                        }
                                    } catch (IOException e) {
                                        // ok
                                    } finally {
                                        try {
                                            zis.close();
                                        } catch (IOException e) {
                                            // ok
                                        }
                                    }
                                } catch (IOException e) {
                                    // ok
                                }
                            } else {
                                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                                downloadFile(activity.getHttpRequests(), url, baos);
                                data = baos.toByteArray();
                            }
                            if (data != null) {
                                activity.cacheStreamingData(
                                        StreamConfiguration.this, url, data);
                                sb.append(BaseActivity.hexSha256Hash(data));
                            }
                        }
                        mEncryptionKey = sb.toString();
                    }
                }

                @Override
                boolean isContentFullyDownloaded() {
                    return false;
                }
            };
            final URL[] services = new URL[mAssets.size()];
            ret.setInvolvedServices(mAssets.values().toArray(services));
            return ret;
        }
        private final String mName;
        @Override
        public String getName() {
            String ret = ConfigurationFactory.readConfigurationName(this);
            if (ret == null) {
                ret = mName;
            }
            return ret;
        }
        @Override
        public boolean isBareMachine() {
            return false;
        }

        @Override
        public boolean isLocallyGenerated() {
            return false;
        }

        @Override
        public boolean isImported() {
            return true;
        }

        @Override
        public boolean isStream() {
            return true;
        }

        private URL mSiteURL;
        @Override
        public URL getWebAdress() {
            return mSiteURL;
        }

        @Override
        public Runnable getFactoryResetter() {
            return null;
        }

        @Override
        public Runnable getDeinstaller() {
            return mDeinstaller;
        }

        @Override
        public List<Integer> getUsedJoystickports(final List<Integer> allJoystickports) {
            return null;
        }

        @NonNull
        @Override
        public List<Integer> getRequiredJoystickports(final List<Integer> allJoystickports) {
            return new LinkedList<>();
        }

        @Override
        public boolean isKeyboardUsed() {
            return true;
        }

        @Override
        public Set<Uri> getFliplist() {
            if (mAssets != null && mAssets.size() > 1) {
                Set<Uri> ret = new HashSet<>();
                for (URL u: mAssets.values()) {
                    ret.add(Uri.parse(u.toString()));
                }
                return ret;
            }
            return null;
        }

        @Override
        public void onEmulatorFliplistChanged(final Set<Uri> emulatorlist) {

        }

        @Override
        public String getFilepath(final String basename) {
            return new File(mTargetfolder, basename).getAbsolutePath();
        }
        private byte[] crypt(final byte[] src, final int mode) {

            SecretKey key = new SecretKeySpec(BaseActivity.sha256Hash(
                    mEncryptionKey.getBytes()), "AES");

            try {
                Cipher c = Cipher.getInstance("AES/CBC/PKCS5Padding");
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                while (baos.size() < c.getBlockSize()) {
                    baos.write(mSiteURL.toString().getBytes());
                }
                byte[] data = baos.toByteArray();
                byte[] iv = new byte[c.getBlockSize()];
                System.arraycopy(data, 0, iv, 0, iv.length);
                c.init(mode, key, new IvParameterSpec(iv));
                return c.doFinal(src);
            } catch (NoSuchAlgorithmException | NoSuchPaddingException | InvalidKeyException
                     | IllegalBlockSizeException | BadPaddingException | IOException
                     | InvalidAlgorithmParameterException e) {
                throw new RuntimeException(e);
            }


        }
        @Override
        public byte[] encrypt(final byte[] plaindata) {
            return crypt(plaindata, Cipher.ENCRYPT_MODE);
        }

        @Override
        public byte[] decrypt(final byte[] encrypted) {
            return crypt(encrypted, Cipher.DECRYPT_MODE);
        }

        @Nullable
        @Override
        public File getLocalPath() {
            return mKeep ? new File(mTargetfolder) : null;
        }

        @Override
        public String getProperty(final String key, final String defaultvalue) {
            return mPresettings.containsKey(key) ? mPresettings.get(key) : defaultvalue;
        }

        void store(final BaseActivity activity, final File rootFolder) {
            File myFolder = new File(rootFolder, mId);
            //noinspection ResultOfMethodCallIgnored
            myFolder.mkdirs();
            Properties emuProps = new Properties();
            Properties props = new Properties();
            Properties assetProps = new Properties();

            for (String key : mPresettings.keySet()) {
                props.setProperty(key, mPresettings.get(key));
            }
            props.setProperty("name", mName);
            props.setProperty("emulation", mEmulatorId);
            props.setProperty("id", mId);
            props.setProperty("url", mSiteURL.toString());
            try {
                FileOutputStream fosImage = new FileOutputStream(
                        new File(myFolder, "screenshot.png"));
                if (mImageURL != null) {
                    downloadFile(activity.getHttpRequests(), mImageURL, fosImage);
                }
                fosImage.close();
                props.setProperty("image", "screenshot.png");


            } catch (IOException | DownloaderFactory.AdditionalDownloads.DownloadException e) {
                props.setProperty("imagePath", mImageURL.toString());
            }

            for (String key : mEmulatorSettings.keySet()) {
                emuProps.setProperty(key, mEmulatorSettings.get(key));
            }
            if (mAssets != null && !mAssets.isEmpty()) {
                for (String key: mAssets.keySet()) {
                    if (mAssets.get(key) != null) {
                        assetProps.put(key, Objects.requireNonNull(mAssets.get(key))
                                .toString());
                    }
                }
            }

            try {
                props.store(
                        new FileOutputStream(new File(myFolder, "properties")), "");
                emuProps.store(
                        new FileOutputStream(new File(myFolder, "emu_properties")), "");
                if (!assetProps.keySet().isEmpty()) {
                    assetProps.store(
                            new FileOutputStream(new File(myFolder, "assets")), "");
                }

            } catch (IOException e) {
                Log.e(TAG, "Exception occured", e);
                throw new RuntimeException(e);
            }
        }

    }
    static String readConfigurationName(final EmulationConfiguration conf) {
        Properties props = new Properties();
        try {
            String abspath = conf.getFilepath("properties");
            props.load(new FileInputStream(abspath));
            return props.getProperty("name");
        } catch (IOException e) {
            Log.e(ConfigurationFactory.class.getSimpleName(),
                    "could not open properties, exception is " + e.getMessage());
            return null;
        }
    }
    static void renameConfiguration(final EmulationConfiguration conf,
                                    final String newName) {
        Properties props = new Properties();
        try {
            String abspath = conf.getFilepath("properties");
            props.load(new FileInputStream(abspath));
            props.setProperty("name", newName);
            props.store(new FileOutputStream(abspath), "");
        } catch (IOException e) {
            Log.e(ConfigurationFactory.class.getSimpleName(),
                    "could not open properties, exception is " + e.getMessage());
        }

    }
}
