package de.rainerhock.eightbitwonders;

import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.ViewActions.click;
import static androidx.test.espresso.assertion.ViewAssertions.doesNotExist;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.ViewMatchers.isChecked;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.isRoot;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static androidx.test.espresso.matcher.ViewMatchers.withText;

import static org.hamcrest.Matchers.allOf;

import android.content.Context;
import android.os.SystemClock;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.PointerIcon;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.test.espresso.UiController;
import androidx.test.espresso.ViewAction;

import androidx.test.filters.SdkSuppress;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.junit.Test;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

public class MouseControllerTest extends C64EmulationTestBase{
    private final static String UNITTEST_MOUSE_JOYSTICK = "_UNITTEST_MOUSE_JOYSTICK_";
    static class TestMouseController extends MouseController {
        TestMouseController(final Context ctx) {
            super(null);
        }

        @Override
        String getHardwareDescriptor() {
            return UNITTEST_MOUSE_JOYSTICK;
        }
        @Override
        int getHardwareId() {
            return DEVICE_ID;
        }

        @Override
        String getId() {
            return UNITTEST_MOUSE_JOYSTICK;
        }

        @NonNull
        @Override
        public String toString() {
            return UNITTEST_MOUSE_JOYSTICK;
        }
        @Override
        protected boolean isDeviceMatch(InputDevice dev) {
            return true;
        }
    }
    static Matcher<View> isMousePointerVisible(boolean visible) {
        return new BaseMatcher<View>() {
            @Override
            public boolean matches(Object item) {
                if (withId(R.id.screen).matches(item)) {
                    View v = (View) item;
                    int iconType;
                    if (visible) {
                        iconType = PointerIcon.TYPE_DEFAULT;
                    } else {
                        iconType = PointerIcon.TYPE_NULL;
                    }
                    return PointerIcon.getSystemIcon(v.getContext(), iconType).equals(v.getPointerIcon());
                }
                return false;
            }

            @Override
            public void describeMismatch(Object item, Description mismatchDescription) {
                mismatchDescription.appendValue(item);
                if (visible) {
                    mismatchDescription.appendValue("does not show mousepointer, expected one");
                } else {
                    mismatchDescription.appendValue("does show mousepointer, expected none");
                }
            }

            @Override
            public void describeTo(Description description) {
                description.appendValue("Check for mouse pointer visibility");

            }
        };
    }
    static ViewAction connectMouse() {
        return new ViewAction() {
            @Override
            public Matcher<View> getConstraints() {
                return withId(R.id.screen);
            }

            @Override
            public String getDescription() {
                return "Sets mouse cursor";
            }

            @Override
            public void perform(UiController uiController, View view) {
                view.setPointerIcon(PointerIcon.getSystemIcon(view.getContext(), PointerIcon.TYPE_DEFAULT));
            }
        };
    }
    private ViewAction setButton(int button, boolean pressed) {
        return new ViewAction() {
            @Override
            public Matcher<View> getConstraints() {
                return isDisplayed();
            }

            @Override
            public String getDescription() {
                String ret;
                if (pressed) {
                    ret = "Press %s button ";
                } else {
                    ret = "Release %s button ";
                }
                if (button == 1) {
                    ret = String.format(ret, "primary");
                } else {
                    ret = String.format(ret, "secondary");
                }
                return ret;
            }

            @Override
            public void perform(UiController uiController, View view) {
                int action;
                if (pressed) {
                    action = MotionEvent.ACTION_BUTTON_PRESS;
                } else {
                    action = MotionEvent.ACTION_BUTTON_RELEASE;
                }
                MotionEvent e = MotionEvent.obtain(0,0, action, button , 0, 0);
                Objects.requireNonNull(getActivity()).dispatchGenericMotionEvent(e);

            }
        };
    }
    private ViewAction pressButton(int button) {
        return setButton(button, true);
    }
    private ViewAction releaseButton(int button) {
        return setButton(button, false);
    }
    private ViewAction scrollWheel (float value) {
        return new ViewAction() {
            @Override
            public Matcher<View> getConstraints() {
                return isDisplayed();
            }

            @Override
            public String getDescription() {
                if (value < 0) {
                    return "Scroll up with scroll wheel";
                } else {
                    return "Scroll down with scroll wheel";
                }
            }

            @Override
            public void perform(UiController uiController, View view) {
                MotionEvent e = MotionEvent.obtain(0,0, MotionEvent.ACTION_SCROLL, 0, value, 0);
                Objects.requireNonNull(getActivity()).dispatchGenericMotionEvent(e);
            }
        };

    }
    private ViewAction scrollWheelDown() {
        return scrollWheel(-1);
    }
    private ViewAction scrollWheelUp() {
        return scrollWheel( 1);
    }

    private ViewAction moveMouse(float dx, float dy, int repeat, int waitPerStep) {
        return new ViewAction() {
            @Override
            public Matcher<View> getConstraints() {
                return isDisplayed();
            }

            @Override
            public String getDescription() {
                return String.format("move mouse pointer (dx=%f, dy=%f) for %d times with %d wait after each time",
                        dx, dy, repeat, waitPerStep);
            }
            @Override
            public void perform(UiController uiController, View view) {
                for (int i = 0; i < repeat; i++) {
                    MotionEvent e = MotionEvent.obtain(0, SystemClock.uptimeMillis(), MotionEvent.ACTION_MOVE, dx, dy, 0);
                    Objects.requireNonNull(getActivity()).dispatchGenericMotionEvent(e);
                    try {
                        waitForIdle(waitPerStep, TimeUnit.MILLISECONDS);
                        Thread.sleep(waitPerStep);
                    } catch (InterruptedException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            }
        };
    }
    @Test
    @SdkSuppress(minSdkVersion=26)
    public void test_paddle_buttons() {
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect mouse device as paddles to port 1
        Expected result:
        * Mouse pointer still visible
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK, R.string.paddle, true);
        onView(withId(R.id.mouse_usage_hint)).check(matches(isDisplayed()));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.screen)).perform(connectMouse());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(true)));
        /*
         * Action: press secondary button
         * Expected result: Dialog "really quit" is visible
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 10, TimeUnit.SECONDS)));
        waitForIdle(250, TimeUnit.MILLISECONDS);
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        waitForIdle(250, TimeUnit.MILLISECONDS);
        onView(withText(R.string.quit)).check(matches(isDisplayed()));
        /*
         * Action: press back button
         * Expected result: Dialog "really quit" is cancelled
         */

        pressBack();
        /*
        Actions:
        * scroll down with scroll wheel
        Expected result:
        * No mouse pointer visible
        * show potentiometer values 25/0 and 255 (defaults for just plugged in device)
         */

        onView(isRoot()).perform(scrollWheelDown());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(false)));
        onView(withId(R.id.screen)).perform(moveMouse(0,5,250,20));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: press secondary mouse button
        Expected results:
        * Dialog "really quit" is not visible
        * Digital value is 247 (secondary button pressed)
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        onView(withText(R.string.quit)).check(doesNotExist());
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-y.pixelbuffer", 25, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_PRIMARY));
        /*
        Action: press primary mouse button
        Expected result: digital value is 243 (both buttons pressed)
         */
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-xy.pixelbuffer", 10, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_PRIMARY));
        /*
        Action: release both buttons
        Expected result: digital value is 255 (no button pressed)
         */
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: open settings and check "Swap buttons", and apply
        Expected result: Emulation is shown.
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK, R.string.paddle, true);
        onView(allOf(isDisplayed(), withId(R.id.cb_swapbuttons))).perform(click()).check(matches(isChecked()));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        /*
        Action: press secondary mouse button
        Expected results:
        * Dialog "really quit" is not visible
        * Digital value is 251 (primary button pressed)
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        onView(withText(R.string.quit)).check(doesNotExist());
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-x.pixelbuffer", 25, TimeUnit.SECONDS)));
        /*
        Action: release secondary mouse button
        Expected result: Digital value is 255 (no button pressed)
        */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BACK, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_PRIMARY));
        /*
        Action: press primary mouse button
        Expected result: Digital value is 251 (primary button pressed)
        */
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-y.pixelbuffer", 10, TimeUnit.SECONDS)));
        onView(isRoot()).perform(scrollWheelUp());
        /*
        Action: scroll up with scroll wheel
        Expected results:
        * Digital value is 255 (no button pressed)
        * Mouse Pointer is visible
         */
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(true)));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
    }
    @Test
    @FlakyMouseTest
    @SdkSuppress(minSdkVersion=26)
    public void test_paddle_movements() {
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect mouse device as paddles to port 1
        Expected result:
        * Mouse pointer still visible
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK, R.string.paddle, true);
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.screen)).perform(connectMouse());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(true)));
        /*
        Actions:
        * scroll down with scroll wheel
        Expected result:
        * No mouse pointer visible
        * show potentiometer values 25/0 and 255 (defaults for just plugged in device)
         */

        onView(isRoot()).perform(scrollWheelDown());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(false)));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: move mouse to the top and right
        Expected result: Analog values are 0/0
         */
        onView(withId(R.id.screen)).perform(moveMouse(5,-5, 250,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-right.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: move mouse to the left
        Expected result: Analog values are 25/0
         */
        onView(withId(R.id.screen)).perform(moveMouse(-5,0, 250,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: move mouse down
        Expected result: Analog values are 25/25
         */
        onView(withId(R.id.screen)).perform(moveMouse(0,5, 250,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-left.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: move mouse right
        Expected result: Analog values are 0/25
         */
        onView(withId(R.id.screen)).perform(moveMouse(5,0, 250,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-right.pixelbuffer", 10, TimeUnit.SECONDS)));

    }
    @Test
    @FlakyMouseTest
    @SdkSuppress(minSdkVersion=26)
    public void test_mouse_movements() {
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect mouse device as paddles to port 1
        Expected result:
        * Mouse pointer still visible
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK,
                R.string.mouse, true);
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.screen)).perform(connectMouse());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(true)));
        /*
        Actions:
        * scroll down with scroll wheel
        Expected result:
        * No mouse pointer visible
        * show potentiometer values 6/6 and 255 (defaults for just plugged in device)
         */

        onView(isRoot()).perform(scrollWheelDown());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(false)));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 120 units to the right
        Expected result: show potentiometer values 18/6 and 255
         */
        onView(withId(R.id.screen)).perform(moveMouse(10,0, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120x.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 60 units to the right
        Expected result: show potentiometer values 11/6 and 255
         */
        onView(withId(R.id.screen)).perform(moveMouse(5,0, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+180x.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 180 units to the left
        Expected result: show potentiometer values 6/6 and 255
         */
        onView(withId(R.id.screen)).perform(moveMouse(-15,0, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 120 units down
        Expected result: show potentiometer values 6/7 and 255
         */
        onView(withId(R.id.screen)).perform(moveMouse(0,10, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120y.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 60 units down
        Expected result: show potentiometer values 6/7 and 255
         */

        onView(withId(R.id.screen)).perform(moveMouse(0,5, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+180y.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 180 units up
        Expected result: show potentiometer values 6/6 and 255
         */
        onView(withId(R.id.screen)).perform(moveMouse(0,-15, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move mouse 60 units up and 60 units left
        Expected result: show potentiometer values 6/6 and 255
         */

        onView(withId(R.id.screen)).perform(moveMouse(-5,-5, 12,40));
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse--60x-60y.pixeldata", 10, TimeUnit.SECONDS)));
    }
    @Test
    @SdkSuppress(minSdkVersion=26)
    public void test_mouse_buttons() {
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect mouse device as paddles to port 1
        Expected result:
        * Mouse pointer still visible
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK,
                R.string.mouse, true);
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.screen)).perform(connectMouse());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(true)));
        /*
        Action: press primary button
        Expected result: screen stays unchanged
         */
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_PRIMARY));
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_PRIMARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * scroll down with scroll wheel
        Expected result:
        * No mouse pointer visible
        * show potentiometer values 6/6 and 255 (defaults for just plugged in device)
         */
        onView(isRoot()).perform(scrollWheelDown());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(false)));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));

        /*
        Action: press primary button
        Expected result: digital value 239 is displayed
         */
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_PRIMARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-primary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: press secondary button
        Expected result: digital value 238 is displayed
         */
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_SECONDARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-both-buttons.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release primary button
        Expected result: digital value 254 is displayed
         */
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_PRIMARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-secondary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release secondary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_SECONDARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: open settings and click on checkbox "swap buttons"
        Expected result: checkbox is checked
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK, R.string.mouse, true);
        onView(allOf(isDisplayed(), withId(R.id.cb_swapbuttons))).perform(click()).check(matches(isChecked()));
        /*
        Action: press button appy
        Expected result: Emulation is running, digital value 255 is displayed
         */
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: press primary button
        Expected result: digital value 254 is displayed
         */
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_PRIMARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-secondary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release primary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_PRIMARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: press secondary button
        Expected result: digital value 239 is displayed
         */
        onView(withId(R.id.screen)).perform(pressButton(MotionEvent.BUTTON_SECONDARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-primary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release secondary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(releaseButton(MotionEvent.BUTTON_SECONDARY));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
    }
    @Test
    @SdkSuppress(minSdkVersion=26)
    public void test_mouse_speed() throws InterruptedException {
        final int length = 775 * 8;
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect device as paddles to port 1, set speed to minimum
        Expected result:
        * show potentiometer values 6/6 and 255 (defaults for just plugged in device)
         */
        setControllerPort1(UNITTEST_MOUSE_JOYSTICK,
                R.string.mouse, true);
        /*
        onView(allOf(isDisplayed(), withId(R.id.sb_sensitivy)))
                .perform(scrollTo())
                .perform(new GeneralClickAction(Tap.SINGLE, GeneralLocation.CENTER_LEFT, Press.FINGER));

         */
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * scroll down with scroll wheel
        Expected result:
        * No mouse pointer visible
        * show potentiometer values 6/6 and 255 (defaults for just plugged in device)
         */
        onView(isRoot()).perform(scrollWheelDown());
        onView(withId(R.id.screen)).check(matches(isMousePointerVisible(false)));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));

        /*
        Action: Move stick to the right for three times the original time
        Expected result: show potentiometer values 18/6 and 255
        */
        onView(withId(R.id.screen)).perform(moveMouse(10,0, 12,40));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120x.pixeldata", 10, TimeUnit.SECONDS)));

    }
}

