package de.rainerhock.eightbitwonders;

import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.ViewActions.click;
import static androidx.test.espresso.action.ViewActions.pressMenuKey;
import static androidx.test.espresso.action.ViewActions.scrollTo;
import static androidx.test.espresso.assertion.ViewAssertions.doesNotExist;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.RootMatchers.withDecorView;
import static androidx.test.espresso.matcher.ViewMatchers.isChecked;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.isRoot;
import static androidx.test.espresso.matcher.ViewMatchers.withChild;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static androidx.test.espresso.matcher.ViewMatchers.withParent;
import static androidx.test.espresso.matcher.ViewMatchers.withText;
import static androidx.test.platform.app.InstrumentationRegistry.getInstrumentation;


import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.startsWith;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.util.Log;
import android.view.View;
import android.webkit.WebView;

import androidx.annotation.StringRes;
import androidx.appcompat.widget.SwitchCompat;
import androidx.test.espresso.NoMatchingViewException;
import androidx.test.espresso.UiController;
import androidx.test.espresso.ViewAction;
import androidx.test.espresso.intent.Intents;
import androidx.test.espresso.matcher.ViewMatchers;
import androidx.test.espresso.util.TreeIterables;
import androidx.test.ext.junit.rules.ActivityScenarioRule;
import androidx.test.platform.app.InstrumentationRegistry;

import junit.framework.AssertionFailedError;

import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.Rule;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

public class MainActivityTestBase extends TestBase {
    static ViewAction navigateTo(final String url) {
        return new ViewAction() {
            @Override
            public String getDescription() {
                return "load url [" + url + "] in a WebView";
            }

            @Override
            public Matcher<View> getConstraints() {
                return instanceOf(WebView.class);
            }

            @Override
            public void perform(UiController uiController, View view) {
                WebView wv = (WebView) view;
                wv.loadUrl(url);
            }
        };
    }
    static Matcher<View> isSaveStateIndicator(final int index) {
        return ViewMatchers.withTagValue(equalTo(String.format(Locale.getDefault(), "VP_%d", index)));
    }
    void setNetworkEnabled(boolean val) {
        BaseActivity.setUnittestNetworkEnabled(val);
    }

    protected void waitForEmulation() {
        waitForEmulation("snapshot-machine-ready.pixelbuffer");
    }

    /** @noinspection SameParameterValue*/
    protected void waitForEmulation(final String expectedBitmap, final long duration, final TimeUnit unit) {
        long ms = unit.toMillis(duration);
        while (ms > 0) {
            try {
                waitForActivity(EmulationActivity.class, 100, TimeUnit.MILLISECONDS);
                onView(withId(R.id.gv_monitor)).check(matches(isDisplayed()));
                break;
            } catch (RuntimeException e) {
                waitForIdle(100, TimeUnit.MILLISECONDS);
                ms -= 100;
            }

        }
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(expectedBitmap,duration,TimeUnit.SECONDS)));
        onView(withId(R.id.gv_monitor))
                .check(matches(isScreenUpdating()));
    }
    protected void waitForEmulation(final String expectedBitmap) {
        waitForEmulation(expectedBitmap, 300, TimeUnit.SECONDS);
    }
    protected ViewAction createdTileAction( @StringRes int labelRes, ViewAction realAction) {
        return createdTileAction(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(labelRes), realAction);
    }
    protected ViewAction createdTileAction(String label, ViewAction realAction) {
        return new ViewAction() {
            @Override
            public String getDescription() {
                return "Perform action "+realAction.getDescription()+ " on tile with label "+label;
            }

            @Override
            public Matcher<View> getConstraints() {
                return isRoot();
            }

            @Override
            public void perform(UiController uiController, View view) {
                Matcher<View> m = withChild(withText(label));
                for (View child : TreeIterables.breadthFirstViewTraversal(view)) {
                    if (m.matches(child)) {
                        scrollTo().perform(uiController, child);
                        uiController.loopMainThreadForAtLeast(250);
                        realAction.perform(uiController, child);
                        return;
                    }
                }
                for (View c : TreeIterables.breadthFirstViewTraversal(view)) {
                    Log.e(getClass().getSimpleName(),c.toString());
                }
                throw new AssertionError("No matching view found.");
            }
        };
    }

    protected void removePackageDownloads(@SuppressWarnings("SameParameterValue") String system, @SuppressWarnings("SameParameterValue") String packageid) {
        File zipfolder = new File(InstrumentationRegistry.getInstrumentation().getTargetContext().getCacheDir(), "additional_downloads");
        if (zipfolder.isDirectory() && zipfolder.listFiles() != null) {
            for (File zip : Objects.requireNonNull(zipfolder.listFiles())) {
                if (zip.getName().endsWith(".zip")) {
                    //noinspection ResultOfMethodCallIgnored
                    zip.delete();
                }
            }
        }
        File d = new File(new File(InstrumentationRegistry.getInstrumentation().getTargetContext().getFilesDir(), "packages"),
        "ConfigurationFactory_"+system+packageid);
        if (d.isDirectory() && d.listFiles() != null) {
            File yaml = new File(d,"additional_downloads.yml");
            if (yaml.exists()) {
                try {
                    InputStream is = new FileInputStream(yaml);
                    for (String f : DownloaderFactory.downloadFilesFromYaml(is)) {
                        File toDelete = new File(d, f);
                        if (toDelete.exists()) {
                            //noinspection ResultOfMethodCallIgnored
                            toDelete.delete();
                        }
                    }
                }
                catch (FileNotFoundException e) {
                    // that's ok
                }
            }
        }
    }
    protected void removeSystemDownloads(String system) {
        File d = new File(new File(InstrumentationRegistry.getInstrumentation().getTargetContext().getCacheDir(), "additional_downloads/VICE_RESOURCES"), system);
        if (d.isDirectory() && d.listFiles() != null) {
            for (File f: Objects.requireNonNull(d.listFiles())) {
                if (!f.delete()) {
                    throw new RuntimeException("Could not delete "+f.getAbsolutePath());
                }
            }
        }



    }
    protected ActivityScenarioRule<MainActivity> getActivityRule()
    {
        Context context = getInstrumentation().getTargetContext();
        prepareUseropts();
        return new ActivityScenarioRule<>(new Intent(
                context,
                MainActivity.class));
    }
    protected void closeAndRestartActivity() {
        mActivityRule = getActivityRule();
    }
    @Rule
    public ActivityScenarioRule<MainActivity> mActivityRule = getActivityRule();

    @Override
    protected Activity getCurrentScenarioActivity() {
        final BaseActivity[] activity = new BaseActivity[1];
        mActivityRule.getScenario().onActivity(a -> activity[0] = a);
        return activity[0];
    }

    @Before
    public void init() {
        super.init();
    }
    protected void prepareMachine(int machineResId) {
        waitForIdle();
        onView(isRoot())
                .check(matches(isDisplayed()))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false));
        Matcher<View> matcher = withText(getInstrumentation().getTargetContext().getString(machineResId));
        try {
            onView(matcher).check(matches(isDisplayed()));
        } catch (AssertionFailedError e) {
            onView(withId(R.id.gh_preinstalled)).perform(click());
            waitForIdle();
            onView(matcher).check(matches(isDisplayed()));
        }
        onView(matcher).perform(scrollTo()).perform(click());
        waitForIdle(5, TimeUnit.SECONDS);
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);

    }

    protected void testExactKeyboardVisibility(final @StringRes int name, final Runnable setModel,
                                               final String screenshotPrefix) {
        prepareMachine(name);
        if (setModel != null) {
            setModel.run();

        }
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForActivity(SettingsActivity.class, 2, TimeUnit.SECONDS);
        onView(withId(R.id.gh_keyboard)).perform(scrollTo()).perform(click());
        onView(withId(R.id.rb_keyboard_always_compact)).perform(click());
        waitForIdle();
        onView(withId(R.id.gh_mobile_device_settings)).perform(scrollTo()).perform(click());
        waitForIdle();
        onView(withId(R.id.show_keyboard)).perform(scrollTo());
        onView(allOf(withText(R.string.always), withParent(withId(R.id.show_keyboard))))
                .perform(scrollTo()).perform(click())
                .check(matches(isChecked()));
        try {
            onView(withText(R.string.show_statusbar)).check(matches(isChecked()));
            onView(withText(R.string.show_statusbar)).perform(scrollTo()).perform(click());
        } catch (AssertionError e) {
            // that's ok
        }
        onView(withText(R.string.show_statusbar)).check(matches(not(isChecked())));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);
        waitForIdle(1, TimeUnit.SECONDS);
        onView(isRoot()).perform(orientationPortrait());
        waitForIdle();
        onView(isC64Key("Q")).check(matches(isDisplayed()));
        onView(withId(R.id.kb_fullsized)).check(doesNotExist());
        if (screenshotPrefix != null) {
            captureDeviceScreen( screenshotPrefix+"-portrait-compactkeyboard");
            try {
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
                captureDeviceScreen(screenshotPrefix + "-portrait-compactkeyboard-num");
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
            }
            catch (NoMatchingViewException e) {
                // that's ok
            }

        }
        onView(isRoot()).perform(orientationLandscape());
        onView(isC64Key("Q")).check(matches(isDisplayed()));
        onView(withId(R.id.kb_fullsized)).check(doesNotExist());
        if (screenshotPrefix != null) {
            captureDeviceScreen(screenshotPrefix+"-landscape-compactkeyboard");
            try {
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
                captureDeviceScreen(screenshotPrefix + "-portrait-compactkeyboard-num");
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
            }
            catch (NoMatchingViewException e) {
                // that's ok
            }

        }
        waitForIdle(1, TimeUnit.SECONDS);
        onView(isRoot()).perform(orientationPortrait());
        onView(isRoot()).perform(pressMenuKey());
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        onView(withId(R.id.gh_keyboard)).perform(scrollTo());
        onView(withId(R.id.rb_keyboard_exact_if_possible))
                .perform(click())
                .check(matches(isChecked()));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);
        waitForIdle(1, TimeUnit.SECONDS);
        onView(isRoot()).perform(orientationPortrait());
        onView(isC64Key("Q")).check(matches(isDisplayed()));
        onView(withId(R.id.kb_fullsized)).check(matches(isDisplayed()));
        if (screenshotPrefix != null) {
            captureDeviceScreen(screenshotPrefix + "-portrait-exactkeyboard");
            try {
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
                captureDeviceScreen(screenshotPrefix + "-portrait-exactkeyboard-num");
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
            }
            catch (NoMatchingViewException e) {
                // that's ok
            }
        }
        onView(isRoot()).perform(orientationLandscape());
        onView(isC64Key("Q")).check(matches(isDisplayed()));
        onView(allOf(withParent(withId(R.id.keyboardview)), withId(R.id.kb_fullsized))).check(matches(isDisplayed()));
        if (screenshotPrefix != null) {
            captureDeviceScreen(screenshotPrefix+"-landscape-exactkeyboard");
            try {
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
                captureDeviceScreen(screenshotPrefix+"-landscape-exactkeyboard-num");
                onView(allOf(isDisplayed(), instanceOf(SwitchCompat.class))).perform(tap());
                waitForIdle();
            } catch (NoMatchingViewException e) {
                // that's ok
            }
        }

        waitForIdle(5, TimeUnit.SECONDS);
        pressBack();
        try {
            onView(withText(R.string.quit)).perform(click());
        } catch (NoMatchingViewException e) {
            waitForIdle(5, TimeUnit.SECONDS);
            pressBack();
            onView(withText(R.string.quit)).perform(click());
        }
        waitForIdle(2, TimeUnit.SECONDS);
        waitForActivity(MainActivity.class, 2, TimeUnit.SECONDS);
        Intents.release();
        Intents.init();
    }
    protected void waitForDialog(final Matcher<View> match) {
        final BaseActivity[] activity = new BaseActivity[1];
        mActivityRule.getScenario().onActivity(a -> activity[0] = a);

        NoMatchingViewException exception = null;
        for (int i = 0; i < 10; i++) {
            try {
                onView(match).inRoot(withDecorView(not(is(activity[0].getWindow().getDecorView()))))
                        .check(matches(isDisplayed()));
                return;
            }
            catch (NoMatchingViewException e) {
                exception = e;
                waitForIdle(100, TimeUnit.MILLISECONDS);
            }
        }
        throw exception;
    }

    protected void waitForDownloadDialog() {
        waitForDialog(withText(R.string.downloading));
    }

    protected void waitForError() {
        waitForDialog(anyOf(withText(R.string.download_error),withText(R.string.check_file_title)));
    }

    protected void waitForError(String filename) {
        waitForDialog(withText(startsWith(filename)));
    }

    protected Matcher<View> isContentSourceViewWithTitle(String text) {
        return allOf(
                instanceOf(ContentSourceView.class),
                withChild(withChild(allOf(withId(R.id.tv_title), withText(text)))));

    }
    protected Matcher<View> isContentSourceViewWithTitle(@StringRes int text) {
        return allOf(
                instanceOf(ContentSourceView.class),
                withChild(withChild(allOf(withId(R.id.tv_title), withText(text)))));

    }
}
