#!/usr/bin/python3
import argparse
import os
import re
import sys
import xml.dom.minidom
from xml.dom.minidom import parse
import xml.etree.ElementTree

written = []
class Parser:
    def matches(self, node):
        return False
    def check_contents(self, node):
        return True

class Test(Parser):
    def check_contents(self, node):
        return node
    def matches(self, node):
        return False


class SimpleWidget(Parser):
    NO_WIDGETS=["include", "requestFocus", "ViewStub", "View", "Space", "Flow"]
    def matches(self, node):
        if not node.childNodes:
            return not node.localName.endswith("Layout") and not node.localName.split(".")[-1] in SimpleWidget.NO_WIDGETS
        if node.childNodes.length == 1:
            if node.firstChild.localName == "requestFocus":
                return True
    def check_contents(self, node):
        return True


class MissingStyleAttribute(SimpleWidget):
    def check_contents(self, node):
        for attribute, value in node.attributes.items():
            if attribute=="style":
                return True

class NameConvention(SimpleWidget):
    undefined_prefixes=[]
    prefixes = {"tv" : re.compile("^TextView$"),
                "bn" : re.compile(".*Button$"),
                "rb" : re.compile("^RadioButton$"),
                "rg": re.compile("^RadioGroup$"),
                "cb": re.compile(".*CheckBox$"),
                "sp": re.compile(".*Spinner$"),
                "sw": re.compile("^SwitchCompat$"),
                "et": re.compile("^EditText$"),
                "jv": re.compile("^Joystick.*View$"),
                "se": re.compile("^SeekBar$"),
                "st": re.compile("^StatusbarView$"),
                "wv": re.compile("^WebView$"),
                "gv": re.compile("^MonitorGlSurfaceView$"),
                "ib": re.compile("^ImageButton$"),
                "iv": re.compile("^ImageView$"),
                "gh": re.compile("^GroupHeader$"),
                "lv": re.compile("^ListView$"),
                "sb": re.compile("^SeekBar$"),
                "pb": re.compile("^ProgressBar$")

                }
    REGEX = re.compile("[a-z][a-z]_[a-z|0-9|_]*$")
    def check_contents(self, node):
        attr = node.attributes.get("android:id")
        if attr and not attr.value.startswith("@android:"):
            label = attr.value.split("/")[-1]
            if self.__class__.REGEX.match(label):
                prefix = label.split("_")[0]
                if prefix in NameConvention.prefixes.keys():
                    return NameConvention.prefixes[prefix].match(node.localName.split(".")[-1])
                else:
                    if not prefix in NameConvention.undefined_prefixes:
                        NameConvention.undefined_prefixes.append(prefix)
                    return False
            return False
        return True


class NonRessourceAttributes(SimpleWidget):
    ATTRIBUTES=[]
    VALUE_WHITELIST=[]
    def check_contents(self, node):
        for attribute, value in node.attributes.items():
            if attribute in self.__class__.ATTRIBUTES:
                if not (value.startswith("@") or value == "0dp" or value in self.__class__.VALUE_WHITELIST):
                    return False
        return True

class DirectUseOfAndroid(SimpleWidget):
    ignore = ["android:id@TabWidget",]
    def check_contents(self, node):
        for attribute, value in node.attributes.items():
            for i in DirectUseOfAndroid.ignore:
                attr, name = i.split("@")
                if attr == attribute  and name == node.localName:
                    return True
            if value.startswith("@android:"):
                return False
        return True

class StyleRedundancy(Parser):
    def __init__(self, filename):
        self.tree = xml.etree.ElementTree.parse(filename)
        self.styles={}
    def matches(self, node):
        return "style" in node.attributes.keys()
    def _popuplate_style(self, stylename, targetdict):
        stylenode = self.tree.findall("./style[@name=\"%s\"]" % stylename)
        if stylenode:
            for item in stylenode[0].findall("./item"):
                name = item.attrib.get("name")
                if not name in targetdict.keys():
                    targetdict[name] = item.text
            parent = stylenode[0].attrib.get("parent")
            if parent:
                self._popuplate_style(parent, targetdict)


    def check_contents(self, node):
        stylename = node.attributes.getNamedItem("style").value.split("/")[-1]
        if not stylename in self.styles.keys():
            stylenode =self.tree.findall("./style[@name=\"%s\"]" % stylename)
            if stylenode:
                items = {}
                self._popuplate_style(stylename, items)
                self.styles[stylename]=items
        found = False
        for attribute, value in node.attributes.items():
            if stylename in self.styles.keys():
                if attribute in self.styles[stylename].keys():
                    if value == self.styles[stylename].get(attribute):
                        found = True
        return not found


class NonRessourceDimension (NonRessourceAttributes):
    ATTRIBUTES=["android:layout_margin", "android:layout_marginBottom", "android:layout_marginTop",
                "android:layout_marginStart", "android:layout_marginEnd", "android:layout_marginLeft",
                "android:layout_marginRight", "android:layout_height", "android:layout_width",
                "android:minHeight", "android:minWidth","android:maxHeight", "android:maxWidth",
                "android:padding", "android:paddingBottom", "android:paddingTop",
                "android:paddingStart", "android:paddingEnd", "android:paddingLeft", "android:paddingRight",
                "android:elevation"]
    VALUE_WHITELIST = ["wrap_content", "match_parent"]

class NonRessourceColors (NonRessourceAttributes):
    ATTRIBUTES = ["android:background", "android:textColor"]

def nodename(n):
    ret=[]
    while True:
        if n.localName:
            s=n.localName
            if "android:id" in n.attributes.keys():
                s=s+"["+str(n.attributes.get("android:id").value).split("/")[-1]+"]"
            ret.insert(0, s)
        n=n.parentNode
        if not n:
            break
    return "/"+"/".join(ret)
def check_node(n, filename, checkers):
    ret = True
    if isinstance(n, xml.dom.minidom.Element):
        for checker in checkers:
            if checker.matches(n):
                if not checker.check_contents(n):
                    line = "  [%-30s] (%s)  %s" % (checker.__class__.__name__, filename,nodename(n))
                    if not line in written:
                        written.append(line)
                        print(line)
                    ret = False
            for child in n.childNodes:
                if not check_node(child, filename,checkers):
                    ret = False
    return ret

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        prog='resources checker',
        description='checks resources for tidyness.')
    parser.add_argument('-d', '--directory', action = 'append')
    parser.add_argument('-s', '--style', action = 'store')
    args = parser.parse_args()
    allcheckers = [MissingStyleAttribute(), NonRessourceDimension(), NonRessourceColors(), NameConvention(),
                DirectUseOfAndroid()]
    if args.style:
        allcheckers.append(StyleRedundancy(args.style))
    exitcode = 0
    for d in args.directory:
        allfiles=[i[2] for i in os.walk(d)][0]
        for f in allfiles:
            dom = parse(os.sep.join((d, f)))
            exitcode = exitcode if check_node(dom.firstChild, f, allcheckers) else 1
    if NameConvention.undefined_prefixes:
        exitcode = 1
        print ("Undefined prefixes:")
        for p in NameConvention.undefined_prefixes:
            print (p)
    sys.exit(exitcode)