//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.app.Dialog;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Bundle;
import android.view.Gravity;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.RelativeLayout;
import android.widget.ScrollView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.PopupMenu;
import androidx.fragment.app.DialogFragment;

public final class TvKeyboardFragment extends DialogFragment {
    /**
     * Public constructor is required by android.
     */
    public TvKeyboardFragment() {
        super();
        mSoftwareKeyboardFunctions = null;
    }
    private final Emulation.SoftwareKeyboardFunctions mSoftwareKeyboardFunctions;
    private EmulationActivity.IsShownListener mIsShownListener;
    private InputDevice mInputDevice = null;
    void setIsShownListener(final EmulationActivity.IsShownListener listener) {
        mIsShownListener = listener;
    }
    TvKeyboardFragment(final Emulation.SoftwareKeyboardFunctions skf) {
        super();
        mSoftwareKeyboardFunctions = skf;

        setStyle(DialogFragment.STYLE_NO_FRAME, R.style.TvKeyboardTheme);
    }

    @SuppressLint("UseRequireInsteadOfGet")

    @Override
    public void onStart() {
        super.onStart();
        if (mIsShownListener != null && getDialog() != null) {
            if (getView() != null) {
                getView().post(() -> mIsShownListener.isShown(getDialog()));
                getView().setOnGenericMotionListener((view, motionEvent) -> {
                    if (getActivity() != null) {
                        return getActivity().onGenericMotionEvent(motionEvent);
                    }
                    return false;
                });
            }
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        //noinspection ConstantConditions
        if (getActivity().findViewById(R.id.tvkeyboardspacer) != null) {
            getActivity().findViewById(R.id.tvkeyboardspacer).setVisibility(View.GONE);

            ((MonitorGlSurfaceView) getActivity().findViewById(R.id.gv_monitor))
                    .setTvMode(MonitorGlSurfaceView.FULLVISIBLE_OR_SCROLL);
        }
    }

    @NonNull
    @Override
    public View onCreateView(final @NonNull LayoutInflater inflater,
                             final @Nullable ViewGroup container,
                             final @Nullable Bundle savedInstanceState) {
        ViewGroup root = (ViewGroup) inflater.inflate(R.layout.fragment_keyboard_container,
                container, false);
        root.setFocusable(true);
        if (mSoftwareKeyboardFunctions != null) {
            KeyboardFragment f = mSoftwareKeyboardFunctions.createKeyboardFragment(
                    ((EmulationActivity) requireActivity()).getSoftkeyboardType());
            f.setTriggeringInputDevice(mInputDevice);
            getChildFragmentManager()
                    .beginTransaction()
                    .add(R.id.root, f, EmulationActivity.KEYBOARDFRAGMENT)
                    .commit();
            f.setDismissListener(() -> {
                mSoftwareKeyboardFunctions.onKeyboardFragmentDismiss(f);
                requireActivity().getSupportFragmentManager()
                        .beginTransaction().remove(this).commit();
            });
        } else {
            dismiss();
        }
        return root;
    }


    @Override
    public void onResume() {
        super.onResume();
        Window window = requireDialog().getWindow();
        if (window != null) {
            WindowManager.LayoutParams wlp = window.getAttributes();
            wlp.gravity = Gravity.BOTTOM;
            wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
            window.setAttributes(wlp);
            window.getDecorView().setBackgroundColor(
                    getResources().getColor(android.R.color.transparent));
        }
    }

    void setInputDevice(final InputDevice mActiveInputDevice) {
        mInputDevice = mActiveInputDevice;
    }

    static class TvKeyboardShownListener implements EmulationActivity.IsShownListener {
        private static final String FRAGMENT_SCREENPART = "FRAGMENT_SCREENPART";
        private final EmulationActivity mActivity;
        private final Runnable mRunOnClose;

        TvKeyboardShownListener(final EmulationActivity ea, final Runnable runOnClose) {
            mActivity = ea;
            mRunOnClose = runOnClose;
        }
        private PopupMenu mPm = null;
        private PopupMenu createPopup(final Dialog d) {
            int[] screenPosMenus = new int[]{R.id.show_screen_bottom,
                    R.id.show_screen_top, R.id.zoom_out, R.id.show_screen_center};
            if (mPm == null) {
                mPm = new PopupMenu(mActivity,
                        d.findViewById(R.id.bn_screen_part_chooser));
                mPm.inflate(R.menu.menu_screensize_tv);
                final MonitorGlSurfaceView gl = mActivity.findViewById(R.id.gv_monitor);
                mPm.getMenu().findItem(R.id.hide_keyboard)
                        .setOnMenuItemClickListener(menuitem -> {
                            mRunOnClose.run();
                            gl.setTvMode(MonitorGlSurfaceView.FULLVISIBLE_OR_SCROLL);
                            return false;
                        });
                for (int id : screenPosMenus) {
                    mPm.getMenu().findItem(id).setOnMenuItemClickListener(menuitem -> {
                        for (int i : screenPosMenus) {
                            mPm.getMenu().findItem(i).setChecked(i == menuitem.getItemId());
                        }
                        ScrollView sv = mActivity.findViewById(R.id.scroll);

                        if (menuitem.getItemId() == R.id.show_screen_bottom) {
                            gl.setTvMode(MonitorGlSurfaceView.FULLVISIBLE_OR_SCROLL);
                            gl.post(() -> sv.fullScroll(ScrollView.FOCUS_DOWN));
                        }
                        if (menuitem.getItemId() == R.id.show_screen_top) {
                            gl.setTvMode(MonitorGlSurfaceView.FULLVISIBLE_OR_SCROLL);
                            gl.post(() -> sv.fullScroll(ScrollView.FOCUS_UP));
                        }
                        if (menuitem.getItemId() == R.id.show_screen_center) {
                            gl.setTvMode(MonitorGlSurfaceView.FULLVISIBLE_OR_SCROLL);
                            gl.post(() -> sv.scrollTo(0, (gl.getHeight() - sv.getHeight()) / 2));
                        }

                        if (menuitem.getItemId() == R.id.zoom_out) {
                            gl.setTvMode(MonitorGlSurfaceView.ZOOM_OUT);
                        }
                        mPm.dismiss();
                        return false;
                    });
                }
            }
            return mPm;
        }
        @Override
        public void isShown(final Dialog d) {
            Rect r1 = new Rect();
            Rect r2 = new Rect();
            if (d.getWindow() != null) {
                d.getWindow().getDecorView().getGlobalVisibleRect(r1);
                mActivity.findViewById(R.id.scroll).getGlobalVisibleRect(r2);
                View v = mActivity.findViewById(R.id.tvkeyboardspacer);
                RelativeLayout.LayoutParams lp = (RelativeLayout.LayoutParams) v.getLayoutParams();
                Point displaySize = new Point();
                mActivity.getWindowManager().getDefaultDisplay().getSize(displaySize);
                lp.height = r1.height();
                v.setLayoutParams(lp);
                v.setVisibility(View.VISIBLE);
                if (r1.intersect(r2)) {

                    d.findViewById(R.id.bn_screen_part_chooser).setVisibility(View.VISIBLE);
                    d.findViewById(R.id.bn_screen_part_chooser).setOnClickListener(view -> {
                        boolean paused = mActivity.getViewModel().getEmulation().isPaused();
                        PopupMenu pm = createPopup(d);
                        pm.setOnDismissListener(popupMenu ->
                                mActivity.getViewModel().getEmulation().setPaused(paused));
                        mActivity.getViewModel().getEmulation().setPaused(true);
                        pm.show();
                    });
                    d.findViewById(R.id.bn_screen_part_chooser).setOnKeyListener(
                            (view, i, keyEvent) -> {
                        if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(
                                keyEvent.getKeyCode())
                                && keyEvent.getKeyCode() != KeyEvent.KEYCODE_MENU
                                && keyEvent.getKeyCode() != KeyEvent.KEYCODE_DPAD_CENTER
                                && keyEvent.getAction() == KeyEvent.ACTION_UP) {
                            EmulationDialogFragment f = new EmulationDialogFragment();
                            PopupMenu pm = createPopup(d);
                            f.setMenu(pm.getMenu(), mActivity.isTv());
                            mActivity.getSupportFragmentManager().beginTransaction()
                                    .add(f, FRAGMENT_SCREENPART).commit();
                        }
                        if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(keyEvent.getKeyCode())
                                && keyEvent.getAction() == KeyEvent.ACTION_UP) {
                            mRunOnClose.run();

                        }
                        return false;
                    });
                }
            }
        }
    }
}
