//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.View;

/**
 * A Checkbox that can interact with other Views in an {@link SettingsActivity}.
 */
public final class MappedCheckBox extends androidx.appcompat.widget.AppCompatCheckBox {
    /**
     * Simple constructor to use when creating a MappedCheckBox from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public MappedCheckBox(final Context context) {
        super(context);
        init(null, 0);
    }
    /**
     * Constructor that is called when inflating a MappedCheckBox from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public MappedCheckBox(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(attrs, 0);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public MappedCheckBox(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(attrs, defStyle);
    }

    private String mMapTrue = "true";
    private String mMapFalse = "false";
    private int mDependendantId = View.NO_ID;

    String getMappingValue(final boolean val) {
        return val ? mMapTrue : mMapFalse;
    }

    void updateDependantVisibility() {
        if (mDependendantId != View.NO_ID) {
            View target = getRootView().findViewById(mDependendantId);
            if (target != null) {
                target.setVisibility(isChecked() ? View.VISIBLE : View.GONE);
            }
        }
    }

    private void init(final AttributeSet attrs, final int defStyle) {
        // Load attributes
        //noinspection resource
        final TypedArray a = getContext().obtainStyledAttributes(
                attrs, R.styleable.MappedCheckBox, defStyle, 0);

        mMapTrue = a.getString(
                R.styleable.MappedCheckBox_trueval);
        mMapFalse = a.getString(
                R.styleable.MappedCheckBox_falseval);
        mDependendantId = a.getResourceId(R.styleable.MappedCheckBox_dependant, View.NO_ID);
        a.recycle();
    }
}
