//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.appwidget.AppWidgetManager;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;

import androidx.constraintlayout.helper.widget.Flow;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;

import java.util.LinkedList;
import java.util.List;


/**
 * The configuration screen for the {@link LaunchTileWidget LaunchTileWidget} AppWidget.
 */
public final class LaunchTileWidgetConfigureActivity extends BaseActivity {

    private static final String PREFS_NAME = "de.rainerhock.eightbitwonders.LaunchTileWidget";
    private static final String PREF_PREFIX_KEY = "appwidget_";
    private int mAppWidgetId = AppWidgetManager.INVALID_APPWIDGET_ID;

    /**
     * Activity handle configuration of app widgets.
     */
    public LaunchTileWidgetConfigureActivity() {
        super();
    }

    @Override
    public void onCreate(final Bundle icicle) {
        super.onCreate(icicle);
        setContentView(R.layout.activity_widget_config);
        ViewGroup vg = findViewById(R.id.usercontent);
        Flow flow = findViewById(R.id.fl_usercontent);
        List<EmulationConfiguration> confs = new LinkedList<>();
        confs.addAll(getMachineConfigurations());
        confs.addAll(getAssetConfigurations(getResources().getAssets(), "packages", null));
        confs.addAll(getLocalConfigurations());
        boolean first = true;
        for (EmulationConfiguration conf: confs) {
            TileView tile = TileView.createWidget(getLayoutInflater(), vg, conf);
            if (tile != null) {
                tile.setId(View.generateViewId());
                vg.addView(tile);
                flow.addView(tile);
                if (first) {
                    requestFocusForFirstTile(tile);
                    first = false;
                }
                final String source;
                if (conf.getLocalPath() != null) {
                    source = conf.getLocalPath().getAbsolutePath();
                } else {

                    source = conf.getEmulatorId();
                }
                tile.setOnClickListener(view -> {
                    final Context context = LaunchTileWidgetConfigureActivity.this;
                    SharedPreferences.Editor prefs = context.getApplicationContext()
                            .getSharedPreferences(PREFS_NAME, 0).edit();
                    prefs.putString(PREF_PREFIX_KEY + mAppWidgetId, source);
                    prefs.apply();

                    // It is the responsibility of the configuration activity to update the widget
                    AppWidgetManager appWidgetManager = AppWidgetManager.getInstance(context);
                    LaunchTileWidget.updateAppWidget(context, appWidgetManager, mAppWidgetId);

                    // Make sure we pass back the original appWidgetId
                    Intent resultValue = new Intent();
                    resultValue.putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID, mAppWidgetId);
                    setResult(RESULT_OK, resultValue);
                    finish();
                });
            }
        }

        // Set the result to CANCELED.  This will cause the widget host to cancel
        // out of the widget placement if the user presses the back button.
        setResult(RESULT_CANCELED);

        // Find the widget id from the intent.
        Intent intent = getIntent();
        Bundle extras = intent.getExtras();
        if (extras != null) {
            mAppWidgetId = extras.getInt(
                    AppWidgetManager.EXTRA_APPWIDGET_ID, AppWidgetManager.INVALID_APPWIDGET_ID);
        }

        // If this activity was started with an intent without a  widget ID, finish with an error.
        if (mAppWidgetId == AppWidgetManager.INVALID_APPWIDGET_ID) {
            finish();
        }
    }
}
