//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import java.util.List;

/**
 * UI-Elemenents representing the status of disk-or tape-drives have to implement this interface.
*/
public interface DriveStatusListener {

    /**
     * Values for the state of a tape-device (i.e. the button pressed down).
     */
    enum TapedriveState { STOP, START, FORWARD, REWIND, RECORD
    }

    /**
     * Show/update the currently attached disk-drives.
     * @param viewModel each member of the list represents a device, it visualized by its
     *                   toString() method
     */
    void updateDiskdriveList(EmulationViewModel viewModel);

    /**
     * Show a disk-drive as active (or not).
     * @param drive the drive
     * @param active if the drive is active or not
     */
    void setDiskdriveState(Object drive, boolean active);

    /**
     * Show/update the currently attached tape-drives.
     * @param tapedrives each member of the list represents a device, it visualized by its
     *                  toString() method.
     */
    void updateTapedriveList(List<?> tapedrives);

    /**
     * Show the current state of a tape-drive.
     * @param tape the drive, must be a member of the last result
     *      *              from {{@link #updateTapedriveList(List)}}
     * @param state one of the states in {{@link TapedriveState}} (i.e. the button pressed down)
     */
    void setTapedriveState(Object tape, TapedriveState state);
    /**
     * Show the motor of a tape-drive running (or not).
     * @param tape the drive, must be a member of the last result
     *      *              from {{@link #updateTapedriveList(List)}}
     * @param active if motor is active
     */
    void setTapedriveMotor(Object tape, boolean active);
    /**
     * Show the counter of a tape-drive running (or not).
     * @param tape the drive, must be a member of the last result
     *      *              from {{@link #updateTapedriveList(List)}}
     * @param tapeCount current counter
     */
    void setTapedriveCounter(Object tape, int tapeCount);

}
