package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.webkit.WebView;

import androidx.annotation.Keep;
import androidx.annotation.StringRes;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

final class ArchiveOrgAccess {
    private ArchiveOrgAccess() { }
    static final List<Class<? extends BrowserActivity.WebContentAdapter>> CONTENT_ADAPTERS
            = new LinkedList<Class<? extends BrowserActivity.WebContentAdapter>>() {{
        add(ArchiveOrgC64LibraryAdapter.class);
        add(ArchiveOrgPetLibraryAdapter.class);
        add(ArchiveOrgVic20CommunitySoftware.class);
    }};

    abstract static class ArchiveOrgAdapter implements BrowserActivity.WebContentAdapter {
        public @StringRes int getLicenseTextId() {
            return R.string.archive_org_exception;
        }
        abstract @StringRes  int getTitleTextId();
        @Override
        public final String getTitle(final Context ctx) {
            return ctx.getResources().getString(getTitleTextId());
        }
        protected final void addLocalfiles(final Context ctx, final EmulationConfiguration conf) {
            InputStream is = ctx.getResources()
                    .openRawResource(R.raw.speed_up_loading);
            try {
                byte[] data = new byte[is.available()];
                if (is.read(data) > 0) {
                    FileOutputStream os = new FileOutputStream(
                            conf.getFilepath("vice-actions.js"));
                    os.write(data);
                    os.close();
                }
                is.close();

            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    abstract static class ArchiveOrgLibraryAdapter extends ArchiveOrgAdapter {
        @Override
        public boolean allowSubdirsOnly() {
            return true;
        }

        private String mEmulatorCheckScript = null;
        private String mEmulatorPropertiesScript = null;

        @Override
        public void init(final WebView webView) {
            try {
                InputStream is = webView.getContext().getResources()
                        .openRawResource(R.raw.scrape_check_archive_org_for_vice);
                byte[] data = new byte[is.available()];

                //noinspection ResultOfMethodCallIgnored
                is.read(data);
                is.close();
                mEmulatorCheckScript = new  String(data);

                is = webView.getContext().getResources()
                        .openRawResource(R.raw.scrape_archive_org_assetlist);
                data = new byte[is.available()];

                //noinspection ResultOfMethodCallIgnored
                is.read(data);
                is.close();
                mEmulatorPropertiesScript = new String(data);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        @Override
        public String getEmulatorCheckScript() {
            return mEmulatorCheckScript;
        }
        @Override
        public String getEmulatorPropertiesScript() {
            return mEmulatorPropertiesScript;
        }
        @Override
        public int handleAdditionalURL(final String url, final boolean allowPayments) {
            return allowPayments || url.startsWith("https://archive.org/details")
                    ? BrowserActivity.WebContentAdapter.NAVIGATION_OK
                    : BrowserActivity.WebContentAdapter.NAVIGATION_DENY;
        }
        @Override
        public ConfigurationFactory.StreamConfiguration createConfiguration(
                final BaseActivity activity, final JSONObject data, final boolean keep) {
            try {
                Map<String, String> emuConfig = new LinkedHashMap<>();
                Map<String, URL> assets = new LinkedHashMap<>();
                Map<String, String> config = new LinkedHashMap<>();

                String assetList = data.get("__assets__").toString();
                for (String line : assetList.split("\n")) {
                    String[] s = line.split("\t");
                    if (s.length == 2) {
                        try {
                            assets.put(s[0], new URL(s[1]));
                        } catch (MalformedURLException e) {
                            // so what
                        }
                    }
                }
                String emulator =  data.get("Emulator").toString();
                String emulatorId = null;
                if (Arrays.asList("vice-resid", "vice-cassette").contains(
                        emulator)) {
                    emulatorId = "C64";
                }
                if (emulator.startsWith("vice-pet")) {
                    emulatorId = "PET";
                }
                if (emulatorId != null) {
                    String[] parts = data.get("Emulator_start").toString().split(":", 2);
                    String file = parts[0];
                    String prg;
                    if (parts.length == 2) {
                        prg = parts[1];
                    } else {
                        prg = "";
                    }
                    if (assets.containsKey(file) && assets.get(file) != null) {
                        emuConfig.put("AutostartWarp", "1");
                        if (file.toLowerCase(Locale.getDefault()).endsWith(".prg")) {
                            emuConfig.put("AutostartPrgMode", "1");
                        } else {
                            emuConfig.put("__openfiles__", "8 "
                                    + Objects.requireNonNull(assets.get(file)));
                        }
                        URL autostart = assets.get(file);
                        if (autostart != null) {
                            emuConfig.put("__autostart_file__", autostart.toString());
                            if (prg != null && !prg.isEmpty()) {
                                emuConfig.put("__autostart_prg__", prg);
                            }
                        }
                        if (assets.containsKey("vice.conf") && emulator.startsWith("vice-pet")) {
                            String petmodel = "4032";
                            if (emulator.startsWith("vice-pet-")) {
                                String[] split = emulator.split("-");
                                if (split.length == 3) {
                                    petmodel = split[2];
                                }
                            }
                            for (DownloaderFactory.HttpRequest req : activity.getHttpRequests()) {
                                try {
                                    URL url = assets.get("vice.conf");
                                    DownloaderFactory.HttpResult res = req.execute(url);
                                    if (res.getResultcode() >= HttpURLConnection.HTTP_OK
                                            && res.getResultcode()
                                                < HttpURLConnection.HTTP_MULT_CHOICE
                                            && res.getBody() != null) {
                                        BufferedReader reader = new BufferedReader(
                                                new InputStreamReader(
                                                        new ByteArrayInputStream(res.getBody())));
                                        while (true) {
                                            try {
                                                String line = reader.readLine();
                                                if (line != null && line.startsWith("Model=")) {
                                                    petmodel = line.split("=")[1];
                                                } else {
                                                    break;
                                                }
                                            } catch (IOException e) {
                                                break;
                                            }
                                        }

                                    }
                                } catch (IOException e) {
                                    if (BuildConfig.DEBUG) {
                                        throw new RuntimeException(e);
                                    }
                                }

                            }
                            emuConfig.put("Model", petmodel);
                        }
                    }
                    String name = data.get("__title__").toString();
                    URL pageurl;
                    try {
                        pageurl = new URL(data.get("__url__").toString());
                        URL imageurl = assets.get("__ia_thumb.jpg");
                        assets.remove("__ia_thumb.jpg");
                        ConfigurationFactory.StreamConfiguration ret =  ConfigurationFactory
                                .createConfiguration(activity, name, pageurl, imageurl, emulatorId,
                                        config, emuConfig, assets, keep);
                        addLocalfiles(activity, ret);
                        Useropts opts = new Useropts();
                        opts.setCurrentEmulation(activity, ret.getEmulatorId(), ret.getId());
                        for (String key : emuConfig.keySet()) {
                            opts.setValue(Useropts.Scope.CONFIGURATION, key, emuConfig.get(key));
                        }
                        return ret;
                    } catch (MalformedURLException e) {
                        return null;
                    }
                }


            } catch (JSONException e) {
                return null;
            }
            return null;
        }

    }

    @Keep
    static class ArchiveOrgPetLibraryAdapter extends ArchiveOrgLibraryAdapter
            implements BrowserActivity.WebContentAdapter {
        @Override
        int getTitleTextId() {
            return R.string.software_library_pet;
        }

        @Override
        public String getRootURL() {
            return "https://archive.org/details/softwarelibrary_pet";
        }

        @Override
        public List<String> getEmulatorId() {
            return Collections.singletonList("PET");
        }

    }

    @Keep
    static class ArchiveOrgC64LibraryAdapter extends ArchiveOrgLibraryAdapter
            implements BrowserActivity.WebContentAdapter {

        @Override
        int getTitleTextId() {
            return R.string.software_library_c64;
        }

        @Override
        public String getRootURL() {
            return "https://archive.org/details/softwarelibrary_c64";
        }

        @Override
        public List<String> getEmulatorId() {
            return Collections.singletonList("C64");
        }


    }

    @Keep
    static class ArchiveOrgVic20CommunitySoftware extends ArchiveOrgAdapter
            implements BrowserActivity.WebContentAdapter {

        private String mEmulatorCheckScript = null;
        private String mEmulatorPropertiesScript;

        @Override
        public void init(final WebView webView) {
            try {
                InputStream is = webView.getContext().getResources()
                        .openRawResource(R.raw.scrape_vic20);
                byte[] data = new byte[is.available()];

                //noinspection ResultOfMethodCallIgnored
                is.read(data);
                is.close();
                mEmulatorCheckScript = new String(data);

                mEmulatorPropertiesScript = "";
            } catch (IOException e) {
                throw new RuntimeException(e);
            }


        }

        @Override
        public String getRootURL() {
            return "https://archive.org/download/vic20_202209/vic20.rar/";
        }

        @Override
        public boolean allowSubdirsOnly() {
            return false;
        }

        @Override
        public String getEmulatorCheckScript() {
            return mEmulatorCheckScript;
        }

        @Override
        public String getEmulatorPropertiesScript() {
            return mEmulatorPropertiesScript;
        }
        private static final  Map<String, String> CMDLINEFLAGS =
                new LinkedHashMap<String, String>() {{
            put(".20", "-cart2");
            put(".40", "-cart4");
            put(".60", "-cart6");
            put(".a0", "-cartA");
            put(".A0", "-cartA");
            put(".b0", "-cartB");
            put(".B0", "-cartB");
            put(".crt", "-cartgeneric");
            put(".CRT", "-cartgeneric");

        }};
        @Override
        public ConfigurationFactory.StreamConfiguration createConfiguration(
                final BaseActivity activity, final JSONObject data, final boolean keep) {
            Map<String, String> emuConfig = new LinkedHashMap<>();
            Map<String, URL> assets = new LinkedHashMap<>();
            Map<String, String> config = new LinkedHashMap<>();
            StringBuilder start = new StringBuilder();
            for (DownloaderFactory.HttpRequest req : activity.getHttpRequests()) {
                try {
                    URL zipurl = new URL((String) data.get("zipurl"));
                    String imageurl = (String) data.get("imageurl");
                    DownloaderFactory.HttpResult result = req.execute(zipurl);
                    if (result.getResultcode() >= HttpURLConnection.HTTP_OK
                            && result.getResultcode() < HttpURLConnection.HTTP_MULT_CHOICE
                            && result.getBody() != null) {
                        ZipInputStream zip = new ZipInputStream(new ByteArrayInputStream(
                                result.getBody()));
                        ZipEntry ze = zip.getNextEntry();
                        boolean autostart = false;
                        boolean dataFound = false;
                        while (ze != null) {
                            assets.put(ze.getName(), new URL(zipurl + "#" + ze.getName()));
                            for (String key : CMDLINEFLAGS.keySet()) {
                                if (ze.getName().endsWith(key)) {
                                    start.append(CMDLINEFLAGS.get(key))
                                            .append(" \"").append(zipurl).append("#")
                                            .append(ze.getName()).append("\" ");
                                    dataFound = true;
                                    break;
                                }
                                String name = ze.getName().toLowerCase(Locale.ROOT);
                                if (name.endsWith(".a0") || name.endsWith(".60")) {
                                    autostart = true;
                                }
                            }
                            ze = zip.getNextEntry();
                        }
                        if (dataFound) {
                            try {
                                String name = new File(new URL(java.net.URLDecoder.decode(
                                        zipurl.toString(), StandardCharsets.UTF_8.name()))
                                        .getPath()).getName();
                                if (name.indexOf(".") > 0) {
                                    name = name.substring(0, name.lastIndexOf("."));
                                }
                                if (name.indexOf(" (SYS") > 0) {
                                    String command = name.substring(name.indexOf(" (SYS") + 2);
                                    command = command.substring(0, command.lastIndexOf(")"));
                                    name = name.substring(0, name.indexOf(" (SYS"));
                                    autostart = true;
                                    start.append(" -keybuf \"")
                                            .append(command.toLowerCase(Locale.ROOT))
                                            .append("\\n\"");
                                }
                                emuConfig.put("__start__", start.toString());
                                if (name.contains("(Japan, USA)")) {
                                    emuConfig.put("MachineVideoStandard", "2");
                                } else {
                                    emuConfig.put("MachineVideoStandard", "1");
                                }
                                config.put("canBeAutostarted", autostart ? "true" : "false");
                                ConfigurationFactory.StreamConfiguration ret =
                                        ConfigurationFactory.createConfiguration(activity, name,
                                                zipurl,
                                                imageurl.isEmpty() || imageurl.equals("undefined")
                                                ? null : new URL(imageurl),
                                        "VIC20", config, emuConfig, assets, keep);
                                addLocalfiles(activity, ret);
                                return ret;
                            } catch (UnsupportedEncodingException e) {
                                return null;
                            }
                        }
                    }
                } catch (IOException | JSONException e) {
                    // ok
                }
            }
            return null;
        }

        @Override
        public int handleAdditionalURL(final String url, final boolean allowPayments) {
            return url.equals("https://archive.org/details/vic20_202209")
                    ? NAVIGATION_OPEN_EXTERNAL : NAVIGATION_OK;
        }


        @Override
        public int getLicenseTextId() {
            return super.getLicenseTextId();
        }

        @Override
        public List<String> getEmulatorId() {
            return Collections.singletonList("VIC20");
        }

        @Override
        int getTitleTextId() {
            return R.string.software_library_vic20_cartridges;
        }


    }
}
