//
// Created by rainer on 07.03.23.
//
#include <string.h>
#include <dlfcn.h>
#include "javascript.h"
#include "javascript_callbacks.h"
#include "lib.h"
#include "charset.h"
#include "mem.h"
#include "logginghelpers.h"
#include "maincpu.h"
#include "montypes.h"

extern int is_in_vsync;

static JSValue poke(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    expected_parameters parameters[] = {
            {"address", JS_TAG_INT,1,0,65535},
            {"value",JS_TAG_UNDEFINED,0,0,0},
    };
    if (!is_in_vsync) {
        return JS_ThrowInternalError(ctx, "%s may only be called during vsync, use addTask to ensure this", __FUNCTION__ );
    }
    JSValue  ret = js_checksyntax(__FUNCTION__, ctx,argc,argv,parameters,sizeof (parameters)/sizeof(parameters[0]));
    if (!JS_IsException(ret)) {
        uint32_t address = 0;
        if (JS_ToUint32(ctx, &address, argv[0]) || address > 65535) {
            return JS_ThrowSyntaxError(ctx, "parameter address must be int in (0..65535");
        }
        char *error = NULL;
        if (JS_IsObject(argv[1]) && JS_VALUE_GET_OBJ(argv[1])) {

            size_t size = 0;
            uint8_t *buf = JS_GetArrayBuffer(ctx, &size, argv[1]);
            if (size) {
                for (int i = 0; i < size; i++) {
                    mem_store(address+i, buf[i]);
                }
            } else {
                error = "buffer may not be empty";
            }
        } else if (JS_IsNumber(argv[1])) {
            uint32_t val;
            if (!JS_ToUint32(ctx, &val, argv[1]) && val >= 0 && val <= 255) {
                mem_store(address, (uint8_t) val);
            } else {
                error = "parameter 2: value must be in range 0..255";
            }

        } else {
            error = "parameter 2 must be array or integer in range 0..255";
        }
        if (error) {
            ret = JS_ThrowSyntaxError(ctx, "%s.%s: %s.", GLOBAL_OBJECT_NAME, __FUNCTION__, error);
        } else {
            ret = JS_UNDEFINED;
        }
    }
    return ret;
}

static JSValue peek_byte(JSContext *ctx, __attribute__((unused)) JSValueConst this_val,
                    __attribute__((unused)) int argc, JSValueConst *argv) {
    expected_parameters params[] = {
            {"address", JS_TAG_INT, 1, 0, 65535},
    };
    JSValue  ret = js_checksyntax("peek",ctx, argc, argv, params, LEN(params));
    if (!JS_IsException(ret)) {
        uint32_t address=0;
        JS_ToUint32(ctx, &address, argv[0]);
        ret = JS_NewUint32(ctx, mem_read(address));
    }
    return ret;
}
static JSValue peek_array(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    uint32_t address=0;
    uint32_t len=0;
    expected_parameters params[] = {
            {"address", JS_TAG_INT, 1, 0, 65535},
            {"count",JS_TAG_INT, 0, 0, 0}
    };
    JSValue  ret = js_checksyntax("peek",ctx, argc, argv, params, LEN(params));
    if (!JS_IsException(ret)) {
        JS_ToUint32(ctx, &address, argv[0]);
        JS_ToUint32(ctx, &len, argv[1]);
        uint8_t *buf = lib_malloc(len);
        if (buf) {
            for (int i = 0; i < len; i++) {
                buf[i] = mem_read(address + i);
            }
            ret = JS_NewArrayBufferCopy(ctx, buf, len);
            lib_free(buf);
        }
    }
    return ret;

}

static JSValue charconvert(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    uint32_t mode =0;
    expected_parameters parameters[] = {
            {"original", JS_TAG_STRING,0,0,0},
            {"mode",JS_TAG_INT,1,1,6},
    };
    JSValue  ret = js_checksyntax(__FUNCTION__ , ctx, argc, argv,  parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        JS_ToUint32(ctx, &mode, argv[1]);
        const char* source=JS_ToCString(ctx, argv[0]);
        uint8_t* petscii=(uint8_t*)lib_strdup(source);
        size_t len = strlen((char*) petscii);
        charset_petconvstring((uint8_t*)petscii, 0);
        if (mode != 1) {
            int reversemode = (mode & 4) == 4 ? 1 : 0;
            for (int i = 0; i < len; i++) {
                petscii[i] = charset_petcii_to_screencode(petscii[i], reversemode);
            }
        }
        ret = JS_NewArrayBufferCopy(ctx, petscii, len);
        JS_FreeCString(ctx, (char*) source);
        lib_free(petscii);
    }
    return ret;
}
static JSValue getCharacterAddress(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv, uint16_t offset) {

    uint16_t dummy;
    uint8_t rows;
    uint8_t cols;
    int bank;
    mem_get_screen_parameter(&dummy, &rows, &cols, &bank);
    expected_parameters parameters[] = {
            {"col", JS_TAG_INT,1,0,cols-1},
            {"row",JS_TAG_INT,1,0,rows-1},
    };
    JSValue  ret = js_checksyntax(__FUNCTION__ , ctx, argc, argv,  parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t col = 0;
        uint32_t row = 0;
        JS_ToUint32(ctx, &col, argv[0]);
        JS_ToUint32(ctx, &row, argv[1]);
        ret = JS_NewUint32(ctx, offset + col + row * cols);
    }
    return ret;
}
static JSValue getScreenAddress(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    uint16_t base;
    uint8_t dummy;
    int dummy2;
    mem_get_screen_parameter(&base, &dummy, &dummy, &dummy2);
    return getCharacterAddress(ctx, this_val, argc, argv, base);
}
static JSValue getColorAddress(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    return getCharacterAddress(ctx, this_val, argc, argv, 55296);
}
static JSValue setActiveMonitor(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    expected_parameters parameters[] = {
            {"col", JS_TAG_INT, 0, 0, 0},
    };
    JSValue  ret = js_checksyntax(__FUNCTION__ , ctx, argc, argv,  parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t val = 0;
        JS_ToUint32(ctx, &val, argv[0]);
        if (js_set_monitor(val)) {
            ret = JS_UNDEFINED;
        } else {
            ret = JS_ThrowRangeError(ctx, "Cannot set monitor to %d", val);
        }
    }
    return ret;

}
const JSCFunctionListEntry machinefunctions[] = {
        JS_CFUNC_DEF("_peek_array", 2, peek_array),
        JS_CFUNC_DEF("_peek_byte", 1, peek_byte),
        JS_CFUNC_DEF("poke", 2, poke),
        JS_CFUNC_DEF("charconvert", 2, charconvert),
        JS_CFUNC_DEF("getScreenAddress", 2, getScreenAddress),
        JS_CFUNC_DEF("getColorAddress", 2, getColorAddress),
        JS_CFUNC_DEF("setActiveMonitor", 0, setActiveMonitor),
        JS_PROP_INT32_DEF("PETSCII",1,JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("SCREENCODE",2,JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("REVERSE",4,JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("REVERSE",4,JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("MONITOR_80", 0, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("MONITOR_40", 1, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("MONITOR_DEFAULT", 0, JS_PROP_CONFIGURABLE),

};
int machinefunctions_size = sizeof(machinefunctions) / sizeof(machinefunctions[0]);