//
// Created by rainer on 06.03.23.
//

#include <stddef.h>
#include <jni.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include "mem.h"
#include "javascript.h"
#include "javascript_callbacks.h"
#include "lib.h"
#include "charset.h"
#include "logginghelpers.h"
#include "machine.h"
#include "quickjs.h"
#include "cutils.h"
#include "quickjs-libc.h"
#include "libbf.h"

static int pfd[2];
static pthread_t thr;
static const char *tag = "myapp";

static char* js_dump_obj(JSContext *ctx, JSValueConst val)
{
    char *str;
    char* ret;

    str = (char*) JS_ToCString(ctx, val);
    if (str) {
        ret = lib_strdup(str);
        JS_FreeCString(ctx, str);
    } else {
        ret = lib_strdup("[exception");
    }
    return ret;
}
#define JS_EXCEPTION_HEADER "Javascript-Exception - "
static char* js_std_dump_error1(JSContext *ctx, JSValueConst exception_val)
{
    JSValue val;
    BOOL is_error;
    char* line1;
    char* line2=NULL;
    char* ret;
    is_error = JS_IsError(ctx, exception_val);
    line1 = js_dump_obj(ctx, exception_val);
    if (is_error) {
        val = JS_GetPropertyStr(ctx, exception_val, "stack");
        if (!JS_IsUndefined(val)) {
            line2 = js_dump_obj(ctx, val);
        }
        ret = lib_malloc((line1 ? strlen(line1) : 0) + (line2 ? strlen(line2) : 0)+strlen(JS_EXCEPTION_HEADER)+strlen("\n")+1);
        sprintf(ret,"%s%s%s%s",JS_EXCEPTION_HEADER,line1 ? line1 : "", line1 && line2 ? "\n" : "", line2 ? line2 : "");
        if (line1) {
            lib_free(line1);
        }
        if (line2) {
            lib_free(line2);
        }
        JS_FreeValue(ctx, val);
    } else {
        return NULL;
    }
    return ret;
}
char* js_dump_error(JSContext *ctx)
{
    char* ret;
    JSValue exception_val;

    exception_val = JS_GetException(ctx);
    ret = js_std_dump_error1(ctx, exception_val);
    JS_FreeValue(ctx, exception_val);
    return ret;
}

static void *thread_func(__attribute__((unused)) void* data)
{
    ssize_t rdsz;
    char buf[128];
    while((rdsz = read(pfd[0], buf, sizeof buf - 1)) > 0) {
        if(buf[rdsz - 1] == '\n') --rdsz;
        buf[rdsz] = 0;  /* add null-terminator */
        LOGV("%s", buf);
    }
    return 0;
}
static struct {
    int type_tag;
    const char *description;
} typenames[] = {
        {JS_TAG_BIG_DECIMAL, "big decimal"},
        {JS_TAG_BIG_INT, "big integer"},
        {JS_TAG_SYMBOL, "symbol"},
        {JS_TAG_STRING, "string"},
        {JS_TAG_INT, "int"},
        {JS_TAG_BOOL, "string"},
        {JS_TAG_NULL, "null"},
        {JS_TAG_UNDEFINED, "undefined"},
        {JS_TAG_UNINITIALIZED, "unititialized"},
        {JS_TAG_EXCEPTION, "exception"},
        {JS_TAG_FLOAT64, "float64"}
};

JSValue js_checksyntax(const char* function, JSContext* ctx, size_t argc, JSValue* argv, expected_parameters* params, size_t parameters_len) {
    if (argc != parameters_len) {
        return JS_ThrowSyntaxError(ctx, "%s.%s expects %zu parameters.", GLOBAL_OBJECT_NAME, function, parameters_len);
    }
    if (params) {
        for (int i = 0; i < parameters_len; i++) {
            int t = JS_VALUE_GET_TAG(argv[i]);
            if (t != params[i].js_tag) {
                const char *type = "????";
                for (int j = 0; j < sizeof(typenames) / sizeof(typenames[0]); j++) {
                    if (typenames[j].type_tag == params[i].js_tag) {
                        type = typenames[j].description;
                    }
                }
                if (params[i].js_tag != JS_TAG_UNDEFINED) {
                    return JS_ThrowTypeError(ctx, "%s.%s expects %s as parameter %d", GLOBAL_OBJECT_NAME, function,
                                             type,
                                             i + 1);
                }
            }
            if (params[i].js_tag == JS_TAG_INT && params[i].check_limits) {
                int32_t val = 0;
                JS_ToInt32(ctx, &val, argv[i]);
                if (val < params[i].min || val > params[i].max) {
                    return JS_ThrowRangeError(ctx,
                                              "%s.%s expects param %d in range %d..%d, got %d",GLOBAL_OBJECT_NAME,
                                              function, i + 1, params[i].min, params[i].max, val);
                }
            }
            if (params[i].js_tag == JS_TAG_STRING && params[i].check_limits) {
                const char *dummy;
                size_t len;
                dummy = JS_ToCString(ctx, argv[i]);
                len = strlen(dummy);
                JS_FreeCString(ctx, dummy);
                if (len < params[i].min || len > params[i].max) {
                    return JS_ThrowRangeError(ctx,
                                              "%s.%s expects param %d length in range %d..%d, got %zu", GLOBAL_OBJECT_NAME,
                                              function, i + 1, params[i].min, params[i].max, len);
                }
            }
        }
    }
    return JS_UNDEFINED;
}
static int start_logger()
{
    /* make stdout line-buffered and stderr unbuffered */
    setvbuf(stdout, 0, _IOLBF, 0);
    setvbuf(stderr, 0, _IONBF, 0);

    /* create the pipe and redirect stdout and stderr */

    pipe(pfd);
    dup2(pfd[1], 1);
    dup2(pfd[1], 2);

    /* spawn the logging thread */

    if(pthread_create(&thr, 0, thread_func, 0) == -1) // NOLINT(bugprone-posix-return)
        return -1;
    pthread_detach(thr);
    return 0;
}

static limb_t get_limbz(const bf_t *a, limb_t idx)
{
    if (idx >= a->len)
        return 0;
    else
        return a->tab[idx];
}


/* The rounding mode is always BF_RNDZ. Return BF_ST_INVALID_OP if there
   is an overflow and 0 otherwise. */
int bf_get_uint64(uint64_t *pres, const bf_t *a)
{
    uint64_t v;
    int ret;
    if (a->expn == BF_EXP_NAN) {
        goto overflow;
    } else if (a->expn <= 0) {
        v = 0;
        ret = 0;
    } else if (a->sign) {
        v = 0;
        ret = BF_ST_INVALID_OP;
    } else if (a->expn <= 64) {
#if LIMB_BITS == 32
        if (a->expn <= 32)
            v = a->tab[a->len - 1] >> (LIMB_BITS - a->expn);
        else
            v = (((uint64_t)a->tab[a->len - 1] << 32) |
                 get_limbz(a, a->len - 2)) >> (64 - a->expn);
#else
        v = a->tab[a->len - 1] >> (LIMB_BITS - a->expn);
#endif
        ret = 0;
    } else {
        overflow:
        v = UINT64_MAX;
        ret = BF_ST_INVALID_OP;
    }
    *pres = v;
    return ret;
}

char* script_filename = NULL;

char* get_abs_filepath(const char* filename_relative_to_script) {
    char *ret = NULL;
    if (script_filename && filename_relative_to_script) {
        if (strrchr(script_filename,'/')) {
            size_t size = strlen((script_filename) + strlen(filename_relative_to_script)) + 1;
            ret = lib_malloc(size);
            memset(ret, 0, size);
            strcpy(ret,script_filename);
            *(strrchr(ret,'/')+1)=0;
            strcat(ret,filename_relative_to_script);
        }
    }
    return ret;
}

__attribute__((used)) void js_set_filename(char* name) {
    script_filename = strdup(name);
}

static JSRuntime *rt = NULL;
static JSContext *glob_ctx = NULL;

static JSValue getAbsolutePath(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    expected_parameters params[] = {
            {"name", JS_TAG_STRING, 1, 1, 255},
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, params, LEN(params));
    if (!JS_IsException(ret)) {
        if (!script_filename) {
            ret = JS_ThrowInternalError(ctx, "JS not fully initialized.");
        } else {
            const char *filename_relative_to_script = JS_ToCString(ctx, argv[0]);
            if (strrchr(script_filename, '/')) {
                size_t size =
                        strlen(script_filename) + strlen(filename_relative_to_script) + 1;
                char* path = lib_malloc(size);
                memset(path, 0, size);
                strcpy(path, script_filename);
                *(strrchr(path, '/') + 1) = 0;
                strcat(path, filename_relative_to_script);
                ret = JS_NewString(ctx, path);
                lib_free(path);
            }
        }
    }
    return ret;
}

const JSCFunctionListEntry basefunctions[] = {
        JS_CFUNC_DEF("_getAbsolutePath", 1, getAbsolutePath)
};
int basefunctions_size = sizeof(basefunctions) / sizeof(basefunctions[0]);
extern JSCFunctionListEntry machinefunctions[];
extern int machinefunctions_size;
extern JSCFunctionListEntry triggerfunctions[];
extern int triggerfunctions_size;
extern JSCFunctionListEntry uifunctions[];
extern int uifunctions_size;
extern JSCFunctionListEntry developerfunctions[];
extern int developerfunctions_size;
extern JSCFunctionListEntry resourcefunctions[];
extern int resourcefunctions_size;
extern JSCFunctionListEntry environmentfunctions[];
extern int environmentfunctions_size;


static JSValue named_obj;
static void js_add_helpers(JSContext *ctx) {
    struct {
        char* name;
        char* code;
    } functions[] = {
            {"getLocalText", "(data, textId) => {\n"
                             "  __obj__= %s;\n"
                             "  if (data.has(textId)) {\n"
                             "    v = data.get(textId);\n"
                             "    res = new Map();\n"
                             "    Object.keys(v).forEach((key) => {\n"
                             "      res.set(key, v[key]);\n"
                             "    });\n"
                             "    if (res.has(__obj__._getSystemLanguage())) {\n"
                             "      return res.get(__obj__._getSystemLanguage());"
                             "    }"
                             "    if (res.has('en')) {\n"
                             "      return res.get('en');"
                             "    }"
                             "  }"
                             "  return null;"
                             "}"

            },

            {"loadTextRessources", "() => {\n"
                                   "__obj__ = %s\n;"
                                   "return __obj__._loadJson(__obj__._getAbsolutePath('text-resources.json'));"
                                   "}"},
            {"peek","(a,b = null) => {\n"
                    "  __obj__= %s;"
                    "  if (b == null) {"
                    "    return __obj__._peek_byte(a);"
                    "  } else {"
                    "    return __obj__._peek_array(a,b);"
                    "  }"
                    "}"
            },
            {"addTask", "(a, b=0) => {\n"
                        "  __obj__= %s;"
                        "  return __obj__._addtask2_ (a,b);"
                        "}"
            },
            {"_loadJson", "(path) => {\n"
                             "  __obj__=%s;\n"
                             "  __obj__.log (__obj__.LOG_VERBOSE, 'creating data');"
                             "  data = new Map();\n"
                             "  __obj__.log (__obj__.LOG_VERBOSE, 'loading from' + path);"
                             "  content = std.loadFile(path);\n"
                             "  if (content) {\n"
                             "  __obj__.log (__obj__.LOG_VERBOSE, 'parsing ' + content);"
                             "    json = std.parseExtJSON(content);\n"
                             "  __obj__.log (__obj__.LOG_VERBOSE, 'looping');"
                             "    Object.keys(json).forEach((key) => {\n"
                             "      data.set(key, json[key]);\n"
                             "    });\n"
                             "  }\n"
                             "  __obj__.log (__obj__.LOG_VERBOSE, 'returning ' + data);"
                             "  return data;\n"
                             "}"
            },
            {"setOption", "(key, value) => {\n"
                          "  __obj__ = %s;"
                          "  data = __obj__._loadJson(__obj__._getAbsolutePath('options.json'));\n"
                          "  data.set (key, value);\n"
                          "  content = JSON.stringify(Object.fromEntries(data));\n"
                          "  f=std.open(__obj__._getAbsolutePath('options.json'), 'w');\n"
                          "  f.puts(content);\n"
                          "  f.close();\n"
                          "}"
            },
            {"getOption", "(key) => {\n"
                          "  __obj__ = %s;"
                          "  return __obj__._loadJson(__obj__._getAbsolutePath('options.json')).get(key);"
                          "}"
            },
            {"hasOption", "(key) => {\n"
                          "  __obj__ = %s;"
                          "  __obj__.log (__obj__.LOG_VERBOSE, 'calling loadJson');"
                          "  ret =  __obj__._loadJson(__obj__._getAbsolutePath('options.json')).has(key);"
                          "  __obj__.log (__obj__.LOG_VERBOSE, 'finished loadJson');"
                          "  return ret;"
                          "}"

            }

    };
    JSValue global_obj;
    global_obj = JS_GetGlobalObject(ctx);
    named_obj = JS_NewObject(ctx);
    JS_SetPropertyFunctionList(ctx, named_obj, developerfunctions, developerfunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, resourcefunctions, resourcefunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, uifunctions, uifunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, triggerfunctions, triggerfunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, machinefunctions, machinefunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, basefunctions, basefunctions_size);
    JS_SetPropertyFunctionList(ctx, named_obj, environmentfunctions, environmentfunctions_size);
    JS_SetPropertyStr(ctx, global_obj, GLOBAL_OBJECT_NAME, named_obj);
    for (int i = 0; i < sizeof(functions)/sizeof(functions[0]); i++) {
        char* code = lib_malloc(strlen(functions[i].code)+strlen(GLOBAL_OBJECT_NAME));
        sprintf(code, functions[i].code,GLOBAL_OBJECT_NAME);
        char* name = lib_malloc(strlen("<>")+strlen(functions[i].name)+1);
        sprintf(name,"<%s>", functions[i].name);
        JSValue val = JS_Eval(ctx, code, strlen(functions[i].code),name,0);
        lib_free(name);
        lib_free(code);
        if (JS_IsException(val)) {
            char* exception = js_dump_error(ctx);
            LOGE("%s", exception);
            lib_free(exception);

        } else {
            JS_SetPropertyStr(ctx, named_obj, functions[i].name, val);
        }

    }
    JS_FreeValue(ctx, global_obj);
}

void js_init() {

    if (script_filename) {
        LOGV("entering js_init");
        if (!rt) {
            rt = JS_NewRuntime();
        }
        JSContext* ctx=NULL;
        if (rt) {
            ctx = JS_NewContext(rt);
        }
#ifdef REDIRECT_STDOUT_TO_LOGGING
        start_logger();
#endif
        JSValue val;
        if (ctx) {
            glob_ctx = ctx;
            js_std_init_handlers(rt);
            js_init_module_std(ctx, "std");
            js_init_module_os(ctx, "os");
            js_add_helpers(ctx);
            const char *str = "import * as std from 'std';\n"
                              "import * as os from 'os';\n"
                              "globalThis.std = std;\n"
                              "globalThis.os = os;\n";
            JSValue helper = JS_Eval(ctx, str, strlen(str), "<init>", JS_EVAL_TYPE_MODULE);
            js_init_constants();
            size_t buf_len = 0;
            uint8_t *buf = js_load_file(ctx, &buf_len, script_filename);
            val = JS_Eval(ctx, (char*) buf, buf_len, script_filename, 0);
            if (JS_IsException(val)) {
                char* exception = js_dump_error(ctx);
                LOGE("%s", exception);
                lib_free(exception);
            }
            JS_FreeValue(ctx, helper);
            JS_FreeValue(ctx, val);
        }

        JS_PreventExtensions(ctx, named_obj);
        LOGV("leaving js_init");
    }
}
#define SET_ATTRIBUTE_TEMPLATE "%s.%s=%d;\n"

__attribute__((used)) int js_set_constant (char* name, int value) {

    char* code = lib_malloc(strlen(name)+strlen(SET_ATTRIBUTE_TEMPLATE)+19+1);
    sprintf(code, SET_ATTRIBUTE_TEMPLATE, GLOBAL_OBJECT_NAME, name, value);
    JSValue val = JS_Eval(glob_ctx, code, strlen(code),"<intern>",0);
    if (JS_IsException(val)) {
        char *exception = js_dump_error(glob_ctx);
        LOGE("%s", exception);
        lib_free(exception);
    }
    lib_free(code);
    return 0;
}
void tasklist_cleanup();
void js_reset() {
    tasklist_cleanup();
    if (glob_ctx) {
        JS_FreeContext(glob_ctx);
        glob_ctx = NULL;
    }

    if (rt) {
        JS_FreeRuntime(rt);
        rt = NULL;
    }
    js_init();

}
void js_cleanup() {
    LOGV("entering js_cleanup");
    if (script_filename) {
        free(script_filename);
        script_filename = NULL;
    }
    tasklist_cleanup();
    if (glob_ctx) {
        JS_FreeContext(glob_ctx);
        glob_ctx = NULL;
    }

    if (rt) {
        JS_FreeRuntime(rt);
        rt = NULL;
    }

    LOGV("leaving js_cleanup");
}

