//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------



#include <stdlib.h>
#include <jni.h>
#include <stdarg.h>
#include "de_rainerhock_eightbitwonders_vice_ViceEmulation.h"
#include "logginghelpers.h"
#include "main.h"
#include "joy.h"
#include <stdarg.h>
#include <string.h>
#include <lib.h>
#include <pthread.h>
#include <setjmp.h>
#include "jnihelpers.h"
#include "snapshot.h"
#include "machine.h"
#include "native_signals.h"
#include "native_threading.h"
#include "temporary_snapshot.h"
#include "javascript.h"
#include "archdep.h"
#include "timemachine.h"

extern timemachine* active_timemachine;


static volatile int terminate;

extern jmp_buf  jumpbuffer;

static jobject currentinstance=NULL;
static JavaVM *vm=NULL;

t_shared_data* g_shareddata=NULL;

JNIEnv* getAactiveenv()
{
	if (pthread_self() == g_shareddata->mainthread) {
		return g_shareddata->mainenv;
	} else if (pthread_self() == g_shareddata->workerthread) {
		return g_shareddata->workerenv;
	}
    return NULL;
}

void DeleteLocalRef(jobject ref)
{
	JNIEnv* env = getAactiveenv();
    if (env) {
        (*env)->DeleteLocalRef(env, ref);
    }
}
jobject CurrentActivity()
{
	return currentinstance;
}

#ifdef DEBUG

__attribute__((used)) void logGetname(jobject obj)
{

	jclass cls = GetObjectClass(obj);
	jmethodID mid = GetMethodID(cls, "getClass", "()Ljava/lang/Class;");
	jobject clsObj =CallObjectMethod(obj, mid);
	// Now get the class object's class descriptor
	cls = GetObjectClass(clsObj);

	// Find the getName() method on the class object
	mid = GetMethodID(cls, "getName", "()Ljava/lang/String;");

	// Call the getName() to get a jstring object back
	jstring strObj = (jstring)CallObjectMethod(clsObj, mid);

	// Now get the c string from the java jstring object
	const char* str = GetStringUTFChars(strObj, NULL);

	// Print the class name
	LOGD("class:    %s", str);

	// Release the memory pinned char array
	ReleaseStringUTFChars(strObj, str);
}

__attribute__((used)) void logToString(jobject obj)
{
	jclass cls=GetObjectClass(obj);
	jmethodID mToString=GetMethodID(cls,"toString", "()Ljava/lang/String;");
	jstring retToString=mToString ? (jstring)CallObjectMethod(obj,mToString) : NULL;
	if (retToString)
	{
		jboolean isCopy;
		const char* s=GetStringUTFChars(retToString,&isCopy);
		LOGD("toString: %s",s);
		ReleaseStringUTFChars(retToString,s);

	}
}
#endif
int unhandledExceptionOccurred() {
	JNIEnv *currentenv = getAactiveenv();
	int ret = 0;
    if (currentenv && (*currentenv)->ExceptionCheck(currentenv)) {
        jthrowable *e = (*currentenv)->ExceptionOccurred(currentenv);
        if (e != NULL) {
            (*currentenv)->ExceptionClear(currentenv);
            jclass exception = (*currentenv)->FindClass(currentenv, "java/lang/Exception");
            if ((*currentenv)->IsInstanceOf(currentenv, e, exception)) {
                (*currentenv)->ExceptionDescribe(currentenv);
                (*currentenv)->ExceptionClear(currentenv);
                jmethodID mth = (*currentenv)->GetMethodID(currentenv,
                                                           (*currentenv)->GetObjectClass(
                                                                   currentenv,
                                                                   currentinstance),
                                                           "continueAfterExceptionFromJni",
                                                           "(Ljava/lang/Exception;)Z");
                ret = (*currentenv)->CallBooleanMethod(currentenv, currentinstance, mth, e) ? 0
                                                                                            : 1;
                (*currentenv)->DeleteLocalRef(currentenv, e);
            }
        }
    }
	return ret;

}
void CallVoidMethod(jobject obj , jmethodID mid, ...)
{
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{

		va_start(myargs, mid); // NOLINT(bugprone-sizeof-expression)
		(*currentenv)->CallVoidMethodV(currentenv,obj,mid,myargs);
		va_end(myargs);
	}
	unhandledExceptionOccurred();

}
jint CallIntMethod(jobject obj , jmethodID mid, ...)
{
	jint ret=0;
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{

		va_start(myargs, mid); // NOLINT(bugprone-sizeof-expression)
		ret=(*currentenv)->CallIntMethodV(currentenv,obj,mid,myargs);
		va_end(myargs);
	}
	return unhandledExceptionOccurred() ? 0 : ret;
}

__attribute__((unused)) jlong CallLongMethod(jobject obj , jmethodID mid, ...)
{
	jlong ret=0;
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{

		va_start(myargs, mid); // NOLINT(bugprone-sizeof-expression)
		ret=(*currentenv)->CallLongMethodV(currentenv,obj,mid,myargs);
		va_end(myargs);
	}
	return unhandledExceptionOccurred() ? 0 : ret;
}

jboolean CallBooleanMethod(jobject obj , jmethodID mid, ...)
{
	jboolean ret=JNI_FALSE;
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{

		va_start(myargs, mid); // NOLINT(bugprone-sizeof-expression)
		ret=(*currentenv)->CallBooleanMethodV(currentenv,obj,mid,myargs);
		va_end(myargs);

		if ((*currentenv)->ExceptionCheck(currentenv))
		{
			LOGE("Exception!");
		}
	}
	return ret;
}

jobject CallObjectMethod(jobject obj , jmethodID mid, ...)
{
	jobject ret=NULL;
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv) {

		va_start(myargs, mid); // NOLINT(bugprone-sizeof-expression)
		ret = (*currentenv)->CallObjectMethodV(currentenv, obj, mid, myargs);
		va_end(myargs);
		if ((*currentenv)->ExceptionCheck(currentenv)) {
			LOGE("Exception!");
		}
		return unhandledExceptionOccurred() ? NULL : ret;
	}
	return NULL;
}

jclass GetObjectClass(jobject obj)
{
	JNIEnv* currentenv= getAactiveenv();
	return unhandledExceptionOccurred() ? NULL : currentenv && !unhandledExceptionOccurred() ? (*currentenv)->GetObjectClass(currentenv, obj) : NULL;
}
jstring NewStringUTF(const char* bytes)
{
	JNIEnv* currentenv= getAactiveenv();
	return currentenv && !unhandledExceptionOccurred() ? (*currentenv)->NewStringUTF(currentenv, bytes) : NULL;
}
const char* GetStringUTFChars(jstring string, jboolean* isCopy)
{
	JNIEnv* currentenv= getAactiveenv();
	return currentenv &&!unhandledExceptionOccurred() ? (*currentenv)->GetStringUTFChars(currentenv, string, isCopy) : NULL;
}
void ReleaseStringUTFChars(jstring string, const char* utfchars)
{
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{
		(*currentenv)->ReleaseStringUTFChars(currentenv,string,utfchars);
	}
}

jmethodID GetMethodID(jclass clsid, const char* name, const char* sig)
{
	JNIEnv* currentenv= getAactiveenv();
	return currentenv && !unhandledExceptionOccurred() ? (*currentenv)->GetMethodID(currentenv, clsid, name, sig) : NULL;
}
void SetByteArrayRegion(jbyteArray array, jsize start, jsize len, void* buf)
{
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{
		(*currentenv)->SetByteArrayRegion(currentenv,array,start,len,buf);
	}

}
jbyteArray NewByteArray(jsize size)
{
	jbyteArray ret = NULL;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{
		ret = (*currentenv)->NewByteArray(currentenv,size);
	}
	return ret;

}

__attribute__((unused)) jobject NewObject(jclass cls, jmethodID mth, ...)
{
	jobject ret=NULL;
	va_list myargs;
	JNIEnv* currentenv= getAactiveenv();
	if (currentenv)
	{

		va_start(myargs, mth); // NOLINT(bugprone-sizeof-expression)
		ret=(*currentenv)->NewObjectV(currentenv,cls,mth,myargs);
		va_end(myargs);
	}
	return unhandledExceptionOccurred() ? NULL : ret;
}

void stop()
{
	terminate=1;
}
int run(JNIEnv* env,t_shared_data* shared_data)
{
	char** argv=NULL;
	int ret=sigsetjmp(jumpbuffer,0);
	if (!ret) {
		currentinstance = (*env)->NewGlobalRef(env, shared_data->instance);
		g_shareddata=shared_data;
		if (!vm) {
			vm = lib_malloc(sizeof(JavaVM));
		}
		(*env)->GetJavaVM(env, &vm);
		LOGD("vm is %p", vm);
		shared_data->workerenv = env;
		shared_data->workerthread = pthread_self();
		LOGD("activeenv=%p", shared_data->workerenv);
		LOGD("entering main_program");
		terminate = 0;
		if (!shared_data->is_under_test) {
            LOGV("Starting special signal handling.");
			start_handling_signals(shared_data);
		} else {
            LOGV("Debugging, no special signal handling.");
        }
        jclass  clz = (*env)->GetObjectClass(env, shared_data->instance);
		jmethodID mth_initscripting=(*env)->GetMethodID(env,clz,"initscripting","()V");
        jmethodID mth_getargc=(*env)->GetMethodID(env, clz, "getArgcFromPresettings","()I");
        jmethodID mth_getargv=(*env)->GetMethodID(env, clz, "getArgvFromPresettings", "(I)Ljava/lang/String;");
        int argc = (*env)->CallIntMethod(env,currentinstance, mth_getargc);
        argv = malloc((1+argc)*sizeof(char *));
        LOGV("cmdline-parser: argv=%p, argc = %d", argv, argc);
        for (int i = 0; i <argc; i++) {
            jstring strObj = (jstring)(*env)->CallObjectMethod(env, currentinstance, mth_getargv, i);
            if (unhandledExceptionOccurred()) {
                break;
            }
            if (strObj) {
                const char *str = GetStringUTFChars(strObj, NULL);
                argv[i] = strdup(str);
                LOGV("cmdline-parser: argv[%d] = [%s]@%p", i + 1, argv[i], argv[i]);

                // Release the memory pinned char array
                ReleaseStringUTFChars(strObj, str);
            } else {
                argv[i] = strdup("");
            }
        }
        LOGV("cmdline-parser: done");
		(*env)->CallVoidMethod(env,currentinstance, mth_initscripting);
        char** argv_copy = malloc((argc+1)*sizeof(char*));
        memcpy(argv_copy, argv, (argc+1)*sizeof(char*));
		ret = main_program(argc, argv);
        machine_shutdown();
        for (int i = 0; i < argc; i++) {
            LOGV("cmdline-parser: freeing argv_copy[%d](=%p)", i+1,argv_copy[i]);
            free(argv_copy[i]);
        }
        LOGV("cmdline-parser: freeing argv");
        free(argv);
        LOGV("cmdline-parser: freeing argv_copy");
        free(argv_copy);

	}
	else
	{
		if (ret==1) {
			if (active_timemachine && active_timemachine->write_newest_entry(active_timemachine, shared_data->recoverysnapshot) == 0) {
				shared_data->recovery_written = 1;
			}
		}
	}
	LOGD("leaving main_program (%d)", ret);
	if (!shared_data->is_under_test) {
		stop_handling_signals();
	}
	g_shareddata=NULL;
	(*env)->DeleteGlobalRef(env, currentinstance);
	return ret;
}

void callRun(JNIEnv* env, jobject runnable) {
	jclass cls = (*env)->FindClass(env, "java/lang/Runnable");
	if (cls) {
		jmethodID mth = GetMethodID(cls, "run", "()V");
		if (mth) {
			(*env)->CallVoidMethod(env, runnable, mth);
		}
	}
	(*env)->DeleteGlobalRef(env, runnable);

}
