//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


#include <ctype.h>
#include "native_threading.h"
#include <stdlib.h>
#include <dlfcn.h>
#include <string.h>
#include <videoarch.h>
#include "jni.h"
#include "logginghelpers.h"
#include "snapshot.h"

static pthread_key_t mThreadKey;

static void Android_JNI_ThreadDestroyed(void* value) {
    /* The thread is being destroyed, detach it from the Java VM and set the mThreadKey value to NULL as required */
    t_shared_data *shared_data = (t_shared_data *) value;
    if (shared_data != NULL) {
        (*(shared_data->vm))->DetachCurrentThread(shared_data->vm);
        pthread_setspecific(mThreadKey, NULL);
    }
}


jint JNI_OnLoad(__attribute__((unused)) JavaVM *vm, __unused void *reserved)
{
    if (pthread_key_create(&mThreadKey, Android_JNI_ThreadDestroyed)) {
        __android_log_print(ANDROID_LOG_ERROR, "MY_LIB", "Error initializing pthread key");
    }

    return JNI_VERSION_1_6;
}
static void* threadfunc(void* arg)
{
    t_shared_data* shared_data=(t_shared_data*)arg;
    JNIEnv* env;


    if ((*(shared_data->vm))->AttachCurrentThread(shared_data->vm,&env,NULL)==JNI_OK)
    {
        pthread_setspecific(mThreadKey, (void*) shared_data);
    }
    (*env)->GetJavaVM(env,&(shared_data->vm));


    shared_data->worker_crashed=0;
    int (*run)(__unused JNIEnv* env, __unused jobject instance);
    void (*populate_machine_specific_functions)();
    void(*set_ram)(uint8_t*);


    LOGV("dynlib for %s: %p",shared_data->libname,shared_data->dynlib);
    if (!shared_data->dynlib)
    {
        LOGE("dlopen-Error: %s",dlerror());
    }
    else {
        /*
        (*env)->CallVoidMethod(env, shared_data->instance,
                               (*env)->GetMethodID(env, (*env)->GetObjectClass(env,
                                                                               shared_data->instance),
                                                   "initvalues", "()V"));
*/
        jstring jpath=(jstring)(*env)->CallObjectMethod(env, shared_data->instance,
                               (*env)->GetMethodID(env, (*env)->GetObjectClass(env,
                                                                               shared_data->instance),
                                                   "getRecoverySnapshotPath", "()Ljava/lang/String;"));
        jboolean isCopy;
        const char* cpath=(*env)->GetStringUTFChars(env,jpath,&isCopy);
        strncpy((char*)shared_data->recoverysnapshot,cpath,sizeof(shared_data->recoverysnapshot)/sizeof(char));
        if (!isCopy)
        {
            (*env)->ReleaseStringUTFChars(env,jpath,cpath);
        }

        populate_machine_specific_functions=dlsym(shared_data->dynlib,"populate_machine_specific_functions");
        LOGV("populate_machine_specific_functions: %p", populate_machine_specific_functions);
        populate_machine_specific_functions();
        set_ram = dlsym(shared_data->dynlib_common, "set_ram");
        uint8_t* mem_ram= dlsym(shared_data->dynlib,"mem_ram");
        set_ram(mem_ram);


        run = dlsym(shared_data->dynlib_common, "run");
        int crash = 0;
        LOGV("run: %p", run);

        if (run) {
            crash = run(env, shared_data);
            if (crash==1) {
                LOGV("Crash!");
            }
            else
            {
                LOGV("Terminated");
            }
        } else {
            LOGE("dlsym-Error: %s", dlerror());
        }
        shared_data->dynlib = NULL;
        shared_data->worker_crashed = crash==1?1:0;
    }

    return NULL;

}
static int is_debugger_attached(JNIEnv* env) {
    jclass clazz = (*env)->FindClass(env,"android/os/Debug");
    jmethodID  mth =(*env)->GetStaticMethodID(env,clazz, "isDebuggerConnected" ,"()Z");
    if ((*env)->CallStaticBooleanMethod(env, clazz, mth)) {
        return 1;
    } else {
        jclass testClass = (*env)->FindClass(env, "org/junit/Test");
        if ((*env)->ExceptionCheck(env)) {
            (*env)->ExceptionClear(env);
        }
        return testClass != NULL ? 1 : 0;
    }

}

t_shared_data *shared_data=NULL;
void startthread (JNIEnv * env, jobject instance,const char* libname, void* dynlib, void* dynlib_common, JNIEnv* mainenv, pthread_t mainthread)
{
    LOGV("main thread started.");
    if (!shared_data) {
        shared_data = (t_shared_data *) malloc(sizeof(t_shared_data));
    }
    shared_data->is_alive=1;
    shared_data->dynlib=dynlib;
    shared_data->dynlib_common=dynlib_common;
    shared_data->libname=libname;
    shared_data->mainthread = mainthread;
    shared_data->mainenv = mainenv;
    shared_data->instance = (*env)->NewGlobalRef(env, instance);
    shared_data->is_under_test = is_debugger_attached(env);

    (*env)->GetJavaVM(env, &(shared_data->vm));
    void* dummy;

    if (pthread_create(&(shared_data->workerthread), NULL, &threadfunc, shared_data) == 0) {
        pthread_join(shared_data->workerthread,&dummy);
    }
    (*env)->DeleteGlobalRef(env, shared_data->instance);
    if (shared_data->worker_crashed)
    {
        if (!(*env)->ExceptionCheck(env)) {
            (*env)->ExceptionClear(env);
            jclass cls = (*env)->FindClass(env, "de/rainerhock/eightbitwonders/NativeSignalException");
            jmethodID cons = (*env)->GetMethodID(env, cls, "<init>",
                                                 "(IIIIZ)V");
            jthrowable exception = (jthrowable) (*env)->NewObject(env, cls, cons,
                shared_data->crashinfo._signo,
                shared_data->crashinfo._code,
                shared_data->crashinfo._errno,
                shared_data->crashinfo._addr,
                shared_data->recovery_written?JNI_TRUE:JNI_FALSE);
            (*env)->Throw(env,exception);

        }
    }
    shared_data->is_alive=0;

}
