//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


#include <stdint.h>
//
// Copyright 2011-2014 NimbusKit
// Originally ported from https://github.com/ingenuitas/python-tesseract/blob/master/fmemopen.c
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>
#include <unistd.h>
#include "jnihelpers.h"
#include "logginghelpers.h"
FILE *funopen(void *pFmem,
              int (*readfn)(void *, char *, int),
              int (*writefn)(void *, const char *, int),
              fpos_t (*seekfn)(void *, fpos_t, int),
              int (*closefn)(void *));

static int handle_by_java(const char* url) {
    if (strstr(url, "content://") == url) {
        return 1;
    }
    if (strstr(url, "file://") == url) {
        return 1;
    }
    return 0;
}


struct fmem {
  int pos;
  int maxpos;
  int size;
  char *buffer;
  int keep;
  int dontfree;
  char* path;
  jobject object;
  int modified;
    int bufsize;
};
typedef struct fmem fmem_t;

static fmem_t * fmem_init() {
    fmem_t *ret = (fmem_t *) malloc(sizeof(fmem_t));
    memset(ret, 0, sizeof(fmem_t));
    return ret;
}
#define MAXPOS(p) ((p)->pos>(p)->maxpos?(p)->pos:(p)->maxpos)

static int readfn(void *handler, char *buf, int size) {
  fmem_t *mem = handler;
  int available = mem->size - mem->pos;
  
  if (size > available) {
    size = available;
  }
  memcpy(buf, mem->buffer + mem->pos, sizeof(char) * size);
  mem->pos += size;
  mem->maxpos=MAXPOS(mem);
  
  return size;
}

static int writefn(void *handler, const char *buf, int size) {
  fmem_t *mem = handler;
  int available = mem->size - mem->pos;

  if (size > available) {
      mem->size += (size - available);
  }

  if (size > (mem->bufsize - mem->pos)) {
      int add = size > 1024 ? size : 1024;
      mem->buffer = realloc(mem->buffer, mem->bufsize+add);
      mem->bufsize += add;
  }
  if (mem->buffer) {
      //LOGV("memcpy (%p, %p, %d)", mem->buffer+mem->pos, buf, size);
      memcpy(mem->buffer + mem->pos, buf, sizeof(char) * size);
      //LOGV("memcpy done");
    mem->pos += size;
    mem->maxpos = MAXPOS(mem);
    return size;
  }
  return 0;

}
static int write_content(void *handler, const char *buf, int size) {
  fmem_t *mem = handler;
  mem->modified = 1;
  return writefn(handler, buf, size);


}
static fpos_t seekfn(void *handler, fpos_t offset, int whence) {
  int pos;
  fmem_t *mem = handler;

  switch (whence) {
    case SEEK_SET: {
      if (offset >= 0) {
        pos = offset;
      } else {
        pos = 0;
      }
      break;
    }
    case SEEK_CUR: {
      if (offset >= 0 || (-offset) <= mem->pos) {
        pos = mem->pos + offset;
      } else {
        pos = 0;
      }
      break;
    }
    case SEEK_END: pos = mem->size + offset;
    break;
    default: return -1;
  }

  if (pos > mem->size) {
    return -1;
  }

  mem->pos = pos;
  mem->maxpos=MAXPOS(mem);
  return (fpos_t)pos;
}
struct
{
    char path[1024] ;
    fmem_t* fp;
    void* data;
    int size;
} stored_file[20];
int stored_files_initialized=0;

static int closefn(void *handler) {
  fmem_t *mem = handler;
  if (mem->keep)
  {
    for (int i=0;i<sizeof(stored_file)/sizeof(stored_file[0]);i++)
    {
      if (stored_file[i].fp==mem)
      {
        int size = mem->maxpos;
        if (stored_file->data)
        {
          free(stored_file[i].data);
        }
        stored_file[i].data=malloc(size);
        memcpy(stored_file[i].data,mem->buffer,size);
        stored_file[i].size=size;
        break;
      }
    }
  }
  if (!mem->dontfree) {
    free(mem->buffer);
  }
  mem->buffer=NULL;
  free(handler);

  return 0;

}
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantFunctionResult"
static int close_content(void* handler)  {
  fmem_t *mem = handler;
  static jmethodID mthCloseContent=NULL;
  if (!mthCloseContent) {
    mthCloseContent = GetMethodID(GetObjectClass(CurrentActivity()), "putContentData",
                                  "(Ljava/lang/String;[B)V");
  }
  if (mem->modified) {
      jbyteArray data = NewByteArray(mem->size);
      SetByteArrayRegion(data, 0, mem->size, mem->buffer);
      NewStringUTF(mem->path);
      jstring jpath = NewStringUTF(mem->path);
      CallVoidMethod(CurrentActivity(), mthCloseContent, jpath, data);
      DeleteLocalRef(data);
      DeleteLocalRef(jpath);
  }
  free(mem->path);
  return closefn(handler);


}
#pragma clang diagnostic pop
int get_kept_data(const char* path,long* size,void** data) {
  for (int i = 0; i < sizeof(stored_file) / sizeof(stored_file[0]); i++) {
    if (strcmp(stored_file[i].path,path)==0) {
      *size = stored_file[i].size;
      *data = stored_file->data;
      return 0;
    }
  }
  return -1;
}
int clean_kept_data(const char* path) {
  for (int i = 0; i < sizeof(stored_file) / sizeof(stored_file[0]); i++) {
    if (strcmp(stored_file[i].path,path)==0) {
      free(stored_file[i].data);
      memset(stored_file + i, 0, sizeof(stored_file[i]));
      return 0;
    }
  }
  return -1;
}
FILE* fmapopen(void* buf, int size)
{
  fmem_t* mem = fmem_init();

  mem->size = size;
  mem->bufsize = size;
  mem->buffer = buf;
  mem->dontfree=1;
  return funopen(mem, readfn, writefn, seekfn, closefn);
}
FILE *fmemopen(const char* id,void *buf, int size, __unused const char *mode) {
  // This data is released on fclose.
  fmem_t* mem = fmem_init();

  mem->size = size;
  mem->bufsize = size;
  mem->buffer = buf;
  if (id)
  {
    if (strcmp(id,"__map__")!=0) {
      if (!stored_files_initialized) {
        stored_files_initialized = 1;
        memset(stored_file, 0, sizeof(stored_file));
      }
      for (int i = 0; i < sizeof(stored_file) / sizeof(stored_file[0]); i++) {
        if (stored_file[i].fp == NULL ||
            strncmp(stored_file[i].path, id, sizeof(stored_file[i].path)) == 0) {
          stored_file[i].fp = mem;
          strncpy(stored_file[i].path, id, sizeof(stored_file[i].path));
          break;
        }
      }
    }
    mem->keep=1;
  }

  // funopen's man page: https://developer.apple.com/library/mac/#documentation/Darwin/Reference/ManPages/man3/funopen.3.html
  return funopen(mem, readfn, writefn, seekfn, closefn);
}
int arch_access(const char* path, int mode)
{
    static jmethodID mthIsAssetAvailable=NULL;
    int ret;
    if (strstr(path, "asset@") == path)
    {
        if ((mode & W_OK) == 0) {
            if (!mthIsAssetAvailable) {
              mthIsAssetAvailable = GetMethodID(GetObjectClass(CurrentActivity()), "isAssetAvailable",
                                                "(Ljava/lang/String;)Z");
            }
            jstring jpath = NewStringUTF(path + strlen("asset@"));
            ret = CallBooleanMethod(CurrentActivity(), mthIsAssetAvailable, jpath) == JNI_TRUE ? 0 : 1;
            DeleteLocalRef(jpath);
        }
        else
        {
            ret=1;
        }
    }
    else if (handle_by_java(path)) {
        int java_ret;
        static jmethodID mthGetContentAccess=NULL;
        if (!mthGetContentAccess) {
            mthGetContentAccess=GetMethodID(GetObjectClass(CurrentActivity()), "getContentAccess", "(Ljava/lang/String;)I");
        }
        if (mthGetContentAccess) {
            jstring jpath = NewStringUTF(path);
            java_ret = CallIntMethod(CurrentActivity(), mthGetContentAccess,jpath);
            return (mode & java_ret) == mode ? 0 : -1;
        } else {
            return -1;
        }
    }
    else
    {
        if (strstr(path,"file:///") == path) {
            ret = access(path+strlen("file://"), mode);
        } else {
            ret = access(path, mode);
        }
    }
    return ret;
}
static FILE* fopen_from_java_method(JNIEnv* env, jmethodID  mth, char* path, int (*write)(void *handler, const char *buf, int size), int (*close)(void *))
{
  if (mth) {
    jstring jpath = (*env)->NewStringUTF(env, path);
    jbyteArray array = CallObjectMethod(CurrentActivity(), mth, jpath);
    if (array) {
      jsize size = (*env)->GetArrayLength(env, array);
      jbyte *body = (*env)->GetByteArrayElements(env, array, 0);

      fmem_t *mem = fmem_init();

      char *buf = malloc(size);
      memcpy(buf, body, size);

      mem->size = size;
      mem->bufsize = size;
      mem->buffer = buf;
      mem->dontfree = 0;
      mem->keep = 0;
      mem->path = strdup(path);

      (*env)->ReleaseByteArrayElements(env, array, body, JNI_ABORT);
      (*env)->DeleteLocalRef(env, array);
      (*env)->ReleaseStringUTFChars(env, jpath, path);
      return funopen(mem, readfn, write, seekfn, close);
    }
  }
  return NULL;
}

FILE* arch_fopen(const char *path, const char *mode)
{
  char* work=strdup(path);
  char* id=NULL;
  char* tail;
  if (strstr(path,"memmap::")==path)
  {
    void* p;
    sscanf(work+strlen("memmap::"),"%p",&p);
    tail=strstr(work+strlen("memmap::"),":");

    if (p && tail)
    {
      tail++;
      int s;
      sscanf(tail,"%d",&s); // NOLINT(cert-err34-c)
      if (s)
      {
        return fmapopen(p,s);
      }
    }

  }
  if (strstr(path, "malloc::") == path)
  {
    if (strstr(work,"@"))
    {
      id=strstr(work,"@")+strlen("@");
      *(strstr(work,"@"))=(char)0;
    }
    char* dummy_for_strtol;
    long size=strtol(work+strlen("malloc::"), &dummy_for_strtol, 10);

    if (size)
    {
      char* data=malloc(size);
      return fmemopen(id, data, size, mode);
    }
  }
  free(work);
  if (strstr(path,"asset@")==path||strstr(path,"/asset@")==path)
  {

    JNIEnv* env=getAactiveenv();
    char* asset=strdup(strstr(path,"asset@")+strlen("asset@"));
    static jmethodID mthGetAssetData=NULL;
    if (!mthGetAssetData) {
      mthGetAssetData=GetMethodID(GetObjectClass(CurrentActivity()), "getAssetData", "(Ljava/lang/String;)[B");
    }
    if (mthGetAssetData)
    {
      return fopen_from_java_method(env,mthGetAssetData,asset,writefn,closefn);
    }
    return NULL;
  }
  if (handle_by_java(path) || strstr(path, "http://") == path || strstr(path, "https://") == path) {
    JNIEnv* env=getAactiveenv();
    char* uri=strdup(path);
    static jmethodID mthGetContentData=NULL;
    if (!mthGetContentData) {
      mthGetContentData=GetMethodID(GetObjectClass(CurrentActivity()), "getContentData", "(Ljava/lang/String;)[B");
    }
    if (mthGetContentData)
    {
      return fopen_from_java_method(env, mthGetContentData, uri, write_content, close_content);
    }
    return NULL;
  }
  return fopen(path, mode);
}