//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.media.AudioAttributes;
import android.media.AudioFocusRequest;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.util.ArrayMap;
import android.util.Log;
import android.view.Gravity;
import android.view.KeyCharacterMap;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import de.rainerhock.eightbitwonders.BuildConfig;
import de.rainerhock.eightbitwonders.CommentStripper;
import de.rainerhock.eightbitwonders.DownloaderFactory;
import de.rainerhock.eightbitwonders.DriveStatusListener;
import de.rainerhock.eightbitwonders.Emulation;
import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.KeyboardFragment;
import de.rainerhock.eightbitwonders.LinkedHashMapWithDefault;
import de.rainerhock.eightbitwonders.NativeSignalException;
import de.rainerhock.eightbitwonders.R;
import de.rainerhock.eightbitwonders.SettingsFragment;
import de.rainerhock.eightbitwonders.Useropts;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.io.StringReader;
import java.nio.ByteBuffer;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@SuppressWarnings("unused")
@Keep
abstract  class
ViceEmulation implements Emulation,
        Emulation.JoystickFunctions {
    protected static final int KEYMAPPED_JOYSTICK_1 = 42;
    protected static final int KEYMAPPED_JOYSTICK_2 = 73;
    private static final int TIMEOUT_MILLIS = 250;
    private static final int FILETYPE_UNKNOWN = 0;
    private static final int FILETYPE_DISKIMAGE = 1;
    private static final int FILETYPE_TAPEIMAGE = 2;
    private static final int FILETYPE_CARTRIDGE = 3;
    private static final int FILETYPE_RAW_CARTRDIGE = 4;
    private static final int FILETYPE_SNAPSHOT = 5;
    private static final int FILETYPE_FLIPLIST = 6;
    private static final int FILETYPE_PRG = 7;
    private static final int TEMP_DIR_ATTEMPTS = 1000;
    public static final int NUMBER_OF_SNAPSHOTS = 50;
    public static final int SNAPSHOT_INTERVAL = 200;
    private static final int NO_SNAPSHOT_SAVED = -1;
    private static final int UNDEFINED = -1;
    private static final Map<Integer, Integer> BITMASKS = new HashMap<Integer, Integer>() {{
        //CHECKSTYLE DISABLE MagicNumber FOR 4 LINES
        put(2, R.string.left_shift_key);
        put(4, R.string.right_shift_key);
        put(8192, R.string.cbm_key);
        put(16384, R.string.control_key);
    }};
    private static final Pattern PATTERN_JOYSTICK_KEY =
            Pattern.compile("[0-9]{1,3}:[0-9]{1,3}:[0-9]{1,3}");
    private final String mConfigurationId;
    private boolean mNativeEmulationLoaded = false;
    private final Object mLibraryStateMonitor = new Object();
    private boolean mTerminating = false;
    private final Map<Integer, Integer> mJoystickstates = new HashMap<>();
    private GamepadFunctions mGamepadFunctions = null;
    private int mModel = UNDEFINED;
    private Runnable mRunAfterTermination = null;
    private final Map<Integer, EmulationUi.JoystickToKeysHelper> mJoystickToKeysHelper
            = new HashMap<>();


    @Keep
    protected final boolean continueAfterExceptionFromJni(final Exception e) {
        Log.e(TAG, NATIVE_CRASH, e);
        return true;
    }

    private static ViceEmulation mInstance = null;

    static ViceEmulation getInstance() {
        return mInstance;
    }
    private boolean mRestart = false;
    private void run() {
        mRestart = false;
        synchronized (mThreadLocker) {
            Log.v(TAG, String.format("Started. Thread-ID=%d", Thread.currentThread().getId()));
            try {
                synchronized (mLibraryStateMonitor) {
                    mNativeEmulationLoaded = true;
                }
                nativeStartEmulation("lib" + mNativeLibrary + ".so");
                // mUi.onEmulatorFinished();
            } catch (NativeSignalException e) {
                Log.e(TAG, "Exception in native Code", e);
                mUi.onEmulatorException(this, e);

            } finally {
                mThread = null;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    abandomAudioFocus();
                }
                synchronized (mLibraryStateMonitor) {
                    nativeCleanupEmulation();
                }
                cleanup();
                if (mRunAfterTermination != null) {
                    Log.v(TAG, "running runAfterTermination task");
                    mRunAfterTermination.run();
                }
            }

            Log.v(TAG, String.format("Finished. Thread-ID=%d", Thread.currentThread().getId()));

        }
        if (mRestart) {
            mUi.runOnUiThread(this::startThread);
        }

    }
    private final StringBuilder mSnapshotpath = new StringBuilder();
    private void cleanup() {
        if (!mSnapshotpath.toString().isEmpty()) {
            File f = new File(mSnapshotpath.toString()).getParentFile();
            if (f != null && f.exists()) {
                if (!f.delete()) {
                    Log.v(TAG, String.format("Could not delete %s", f.getParent()));
                }
            }
        }
        for (File f : mToBeDeleted) {
            if (f.exists()) {
                if (f.delete()) {
                    Log.v(TAG, "deleted " + f.getAbsolutePath());
                } else {
                    Log.e(TAG, "Could not delete " + f.getAbsolutePath());
                }
            }

        }
    }
    boolean isToggleKeyPressed(final int row, final int col) {
        return mPressedToggleKeys.contains(new Point(row, col));
    }
    private final List<Point> mPressedToggleKeys = new LinkedList<>();
    Point setToggleKeyPressed(final int row, final int col, final boolean pressed) {
        Point p = new Point(row, col);
        if (pressed && !mPressedToggleKeys.contains(p)) {
            mPressedToggleKeys.add(p);
        }
        if (!pressed) {
            mPressedToggleKeys.remove(p);
        }
        return p;
    }
    public void setKeyButtons(final List<KeyButton> allButtons) {
        addKeys(allButtons, mAllKeys, mModificatorKeys);

    }

    public static final class DirEntry implements Serializable, DataEntry {
        private static final long serialVersionUID = -6322997067208479103L;
        private final Integer mPos;
        private final String mFilename;
        private final String mType;

        DirEntry(final Integer pos, final String filename, final String type) {
            this.mPos = pos;
            this.mFilename = filename.replace('\ufffd', (char) 0).trim();
            this.mType = type.trim();

        }

        @NonNull
        public Integer getPos() {
            return mPos;
        }
        @Override
        @NonNull
        public String getText() {
            return mFilename;
        }

        @NonNull
        public String getType() {
            return mType;
        }

    }

    static final String DEFAULT_IMAGE_EXTENSION = "d64";

    static final int TYPE_DATASETTE = 1531;


    protected static final String TAG = ViceEmulation.class.getSimpleName();
    protected static final String NATIVE_CRASH = "native crash";
    private final String mNativeLibrary;
    private EmulationUi mUi;
    private Boolean mIsPaused = false;
    private final Object mPauseMonitor = new Object();

    @Override
    public final boolean isPaused() {
        return mIsPaused;
    }

    @Override
    public final void setPaused(final boolean pause) {
        setPaused(pause, () -> { });
    }
    @Override
    public final void setPaused(final boolean pause, final Runnable r) {

        if (mIsPaused != pause) {
            mIsPaused = pause;
            if (pause) {
                synchronized (mLibraryStateMonitor) {
                    if (mNativeEmulationLoaded) {
                        nativeVoidFunc("timemachine_set_paused", 1);
                        nativeVoidFunc("sound_mute");

                    }
                }
                addTask(() -> {
                    Log.v(TAG, "reaching halt.");
                    if (r != null) {
                        r.run();
                    }
                    while (mIsPaused) {
                        try {
                            synchronized (mPauseMonitor) {
                                mPauseMonitor.wait();
                                synchronized (mLibraryStateMonitor) {
                                    if (mNativeEmulationLoaded) {
                                        nativeVoidFunc("sound_unmute");
                                    }
                                }
                            }

                        } catch (InterruptedException e) {
                            Log.v(TAG, "pause interrupted", e);
                        }
                    }

                });
            } else {
                synchronized (mLibraryStateMonitor) {
                    if (mNativeEmulationLoaded) {
                        nativeVoidFunc("timemachine_set_paused", 0);
                        nativeVoidFunc("vsync_suspend_speed_eval");
                    }
                }
                synchronized (mPauseMonitor) {
                    mPauseMonitor.notifyAll();
                }
                if (r != null) {
                    addTask(r);
                }
            }
        }
    }
    private boolean mInitialized = false;
    @Keep
    protected final void uiInit() {
        mInitialized = true;
    }
    @Keep
    protected final void uiInitFinalize() {
        mInitialized = true;
        synchronized (mWaitForEmulator) {
            mWaitForEmulator.notifyAll();
        }
        getViceMachineSettingsFunctions().presetModel();
        mUi.updateDiskdriveList(updateDriveList());
        mUi.runOnUiThread(() -> mUi.onEmulatorInitialized(this));

    }
    private final List<Runnable> mTasks = new LinkedList<>();

    protected final void addTask(final Runnable r) {
        synchronized (mTasks) {
            mTasks.add(r);
        }
        nativeVoidFunc("request_presync");
    }
    protected final void addResetTask(final @NonNull  Runnable r) {
        nativeVoidFunc("add_vsync_task", r);
    }
    private Uri mCartridge = null;
    private int mCartridgeType = 0;
    private static final int DETACH_ALL_CARTRIGDES = -1;
    protected final Uri getCartridge() {
        return mCartridge;
    }
    final void setCartridge(final Integer type, final Uri uri) {
        setCartridge(type, uri, null, null);
    }
    final void setCartridge(final Integer type, final Uri uri,
                               final Runnable runOnSuccess, final Runnable runOnError) {
        mCartridgeType = type;
        mCartridge = uri;
        if (uri == null) {
            nativeVoidFunc("cartridge_detach_image", DETACH_ALL_CARTRIGDES);

        } else {
            boolean ret;
            try {
                openCartridge(type, uri, runOnSuccess, runOnError);
                Log.v(TAG, "openCartridge called");


            } catch (IOException e) {
                Log.v(TAG, String.format("Cannot open %s", uri), e);
            }

        }
    }

    private final Map<Integer, Uri> mLockedImages = new LinkedHashMap<>();
    private final Map<Integer, Uri> mAttachedImages = new LinkedHashMap<>();

    private boolean setImage(final AttachDriveFunction attach, final Runnable detachFunction,
                             final Integer drive, final Uri uri, final boolean readonly) {
        if (uri != null) {
            if (attach.attach()) {
                mAttachedImages.put(drive, uri);
                if (!readonly) {
                    mLockedImages.put(drive, uri);
                }
                return true;
            }
            return false;
        } else {
            detachFunction.run();
            mAttachedImages.remove(drive);
            mLockedImages.remove(drive);
            return true;
        }

    }

    private interface AttachDriveFunction {
        boolean attach();
    }

    final boolean setTapeImage(final Integer drive, final Uri uri, final boolean readonly) {
        return setImage(
                () -> nativeIntFunc("tape_image_attach", drive, uri.toString()) == 0,
                () -> {
                    nativeVoidFunc("tape_image_detach", drive);
                    mAttachedTapes.remove(drive);
                },
                drive, uri, readonly);

    }

    final boolean setDriveImage(final Integer drive, final Uri uri, final boolean readonly) {
        return setImage(
                () -> nativeIntFunc(
                        "file_system_attach_disk", drive, 0, uri.toString()) == 0,
                () -> nativeVoidFunc(
                        "file_system_detach_disk", drive, 0),
                drive, uri, readonly);

    }

    @Override
    public final void setEmulationUI(final EmulationUi ui) {
        mUi = ui;
        Translator.init(ui.getContext());
    }

    protected final EmulationUi getEmulationActivity() {
        return mUi;
    }

    private final String mEmulatorId;

    @Override
    public final String getEmulatorId() {
        return mEmulatorId;
    }

    private static final Map<String, String> LIBRARIES = new HashMap<String, String>() {
        private static final long serialVersionUID = -2940022869728865754L;

        {
            put("C64", "x64");
            put("VIC20", "xvic20");
            put("PET", "xpet");
            put("C128", "x128");

        }
    };
    private final EmulationConfiguration mConf;
    protected EmulationConfiguration getConf() {
        return mConf;
    }
    ViceEmulation(final EmulationUi ui,
                  final EmulationConfiguration conf) {
        super();

        mNativeLibrary = LIBRARIES.get(conf.getEmulatorId());
        mEmulatorId = conf.getEmulatorId();
        mUi = ui;
        mConfigurationId = conf.getId();

        mLockedImages.clear();
        mTasks.clear();

        String openfiles = conf.getPresettings().get("__openfiles__");
        addTask(() -> {
            String key = null;
            if (openfiles != null) {
                for (String s : openfiles.split(" ")) {
                    if (key == null) {
                        key = s;
                    } else {
                        Uri uri = Uri.parse(s);
                        if (uri != null) {
                            String path;
                            if (uri.getScheme() != null) {
                                path = uri.toString();
                            } else {
                                path = uri.getPath();
                            }
                            final int drive = Integer.parseInt(key);
                            disableLogging();
                            switch (nativeIntFunc("get_filetype", path)) {
                                case FILETYPE_DISKIMAGE:
                                    nativeIntFunc("file_system_attach_disk",
                                            drive, 0, path);
                                    break;
                                case FILETYPE_TAPEIMAGE:
                                    nativeIntFunc("tape_image_attach",
                                            drive, path);
                                    break;
                                default:
                                    if (path != null && path.toLowerCase(Locale.getDefault())
                                            .endsWith(".prg")) {
                                        nativeVoidFunc("autostart_prg", path, 1);
                                    }
                                    break;
                            }
                            enableLogging();
                            mAttachedImages.put(Integer.valueOf(key), uri);
                            mLockedImages.put(Integer.valueOf(key), uri);
                        }
                        key = null;
                    }
                }
            }
        });
        mConf = conf;
        ViceEmulation.mInstance = this;

    }

    public final String getConfigurationId() {
        return mConfigurationId;
    }

    @Override
    public final void terminate(final Runnable runAfterTermination) {
        setPaused(false);
        mRunAfterTermination = runAfterTermination;
        if (!mTerminating) {
            mTerminating = true;
            addTask(() -> {
                mTasks.clear();
                Log.v(TAG, "Termination started");
                synchronized (mLibraryStateMonitor) {
                    mNativeEmulationLoaded = false;
                    nativeVoidFunc("stop_emulation");
                    Log.v(TAG, "stop_emulation_called.");
                }
                for (int drive : mLockedImages.keySet()) {
                    Uri uri = mLockedImages.get(drive);
                    if (uri != null) {
                        disableLogging();
                        switch (nativeIntFunc("get_filetype", uri.toString())) {
                            case FILETYPE_DISKIMAGE:
                                nativeVoidFunc("file_system_detach_disk", drive, 0);
                                break;
                            case FILETYPE_TAPEIMAGE:
                                nativeVoidFunc("tape_image_detach", drive);
                                break;
                            default:
                                break;
                        }
                        enableLogging();
                    }
                }
            });
            Log.v(TAG, "Termination requested");
            nativeVoidFunc("sound_mute");
            setPaused(false);
        }
     }
    @Keep
    protected final String getRecoverySnapshotPath() {
        String path = mUi.getContext().getCacheDir().getAbsolutePath()
                + File.separator + ".recovery" + File.separator;
        File f = new File(path);
        if (!f.isDirectory()) {
            //noinspection ResultOfMethodCallIgnored
            f.mkdirs();
        }
        return path + ".recovery.vsf";
    }
    private boolean mSkipKeyEvent = false;

    @Keep
    protected final void handleNextKey() {
        if (!mSkipKeyEvent) {
            Runnable r = null;
            if (!mKeyActions.isEmpty()) {
                synchronized (mKeyActions) {
                    r = mKeyActions.get(0);
                    mKeyActions.remove(0);
                }
            }
            if (r != null) {
                r.run();
            }
        }
        mSkipKeyEvent = !mSkipKeyEvent;
    }
    @Keep
    protected final boolean presync() {
        Runnable task;
        List<Runnable> keys;
        do {
            synchronized (mTasks) {
                try {
                    if (mTasks.isEmpty()) {
                        task = null;
                    } else {
                        task = mTasks.get(0);
                        mTasks.remove(0);
                    }
                } catch (Exception e) {
                    task = null;
                }
            }
            if (task != null) {
                task.run();
            }
        } while (task != null);
        return true;
    }
    @Keep
    protected final void onCanvasSizeChanged(final int canvasId, final int canvasWidth,
                                             final int canvasHeight,
                                             final float pixelAspectRatio,
                                             final float screenAspectRatio) {
        mUi.onCanvasSizeChanged(canvasId, canvasWidth, canvasHeight,
                pixelAspectRatio, screenAspectRatio);
    }
    @Keep
    protected final void onCanvasContentChanged(final int displayId, final Bitmap bmp) {

        if (mUi != null) {
            mUi.setBitmap(displayId, bmp);
        }
    }
    protected void initscripting() {
        String path = mConf.getFilepath("vice-actions.js");
        if (new File(path).exists()) {
            nativeVoidFunc("js_set_filename", path);
        }

    }

    private static final int MAX_SNAPSHOT_LEVELS = 9;
    @Keep
    protected final void initvalues() {

        nativeVoidFunc("timemachine_init", NUMBER_OF_SNAPSHOTS, SNAPSHOT_INTERVAL);
        nativeVoidFunc("set_tempdir", mUi.getContext().getCacheDir().getAbsolutePath());
    }
    @Keep
    protected final String getUseropts() {
        StringBuffer sb = new StringBuffer();
        getViceMachineSettingsFunctions().getConfigHeader(sb);
        getViceMachineSettingsFunctions().getUseropts(mUi.getCurrentUseropts(), sb);
        return sb.toString();

    }
    @Keep
    protected final void postMessage(final String msg) {
        mUi.runOnUiThread(() -> {
            Toast toast = Toast.makeText(mUi.getContext(), msg, Toast.LENGTH_LONG);
            toast.setGravity(Gravity.TOP, 0, 0);
            toast.show();
        });

    }
    @Keep
    protected final void postError(final String msg) {
        postMessage(msg);
    }

    @Keep
    protected final String getUserConfigPath() {
        String ret = mUi.getContext().getFilesDir().getAbsolutePath() + File.separator + "config";
        File f = new File(ret);
        if (!f.isDirectory()) {
            //noinspection ResultOfMethodCallIgnored
            f.mkdirs();
        }
        return ret;
    }


    @Keep
    protected final String getHomePath() {
        return mUi.getContext().getFilesDir().getAbsolutePath();
    }
    @Keep
    protected final boolean isAssetAvailable(final String pathInAssetsDir) {
        return DownloaderFactory.exists(mUi.getContext(), pathInAssetsDir);
    }
    @Keep
    protected final String getDefaultSysfilePathlist(final String emuId) {
        StringBuilder ret = new StringBuilder();
        for (String folder : Arrays.asList(emuId, "DRIVES", "PRINTER")) {
            if (ret.length() > 0) {
                ret.append(File.pathSeparator);
            }
            ret.append("asset@VICE_RESOURCES").append(File.separator).append(folder);
        }
        return ret.toString();
    }
    private static native int nativeGetFiletype(String library, String function, String path);
    private static final int KBYTE = 1024;

    protected final native void nativeSetUiThread();
    protected final native void nativeStartEmulation(String emu) throws NativeSignalException;

    protected final native void nativeCleanupEmulation();
    protected final native byte[] nativeBytearrayFunc2(String funcname,
                                                       int i,
                                                       boolean freeNativeReturnvalue);
    protected final native byte[] nativeBytearrayFunc(String funcname,
                                                      boolean freeNativeReturnvalue);

    protected final native void nativeVoidFunc(String funcname);

    protected final native void nativeVoidFunc(String funcname, int val1, int val2);

    protected final native void nativeVoidFunc(String funcname, float val1, float val2);

    protected final native void nativeVoidFunc(String funcname, int val1, int val2, int val3);

    protected final native void nativeVoidFunc(String funcname, int val);

    private native void nativeVoidFunc(String funcname, int val, Runnable runnable);

    protected final native void nativeVoidFunc(String string, String val);
    protected final native void nativeVoidFunc(String string, Object val);
    protected final native void nativeVoidFunc(String funcname, String val,
                                               Runnable runnable1, Runnable runnable2);

    protected final native void nativeVoidFunc(String funcname, String val1, Object val2);

    protected final native int nativeIntFunc(String funcname);

    protected final native int nativeIntFunc(String funcname, int val, String val2);

    protected final native int nativeIntFunc(String funcname, int val1, int val2, String val3);

    protected final native int nativeIntFunc(String funcname, String val, int val2);

    protected final native int nativeIntFunc(String funcname,
                                       String val, int val2, int val3, int val4);

    protected final native int nativeIntFunc(String funcname, int val);

    protected final native int nativeIntFunc(String funcname,
                                       String s1, String s2, int i1, int i2);

    protected final native int nativeIntFunc(String funcname,
                                       String s1, String s2, int i);

    protected final native int nativeIntFunc(String funcname,
                                       String s1, String s2, String s3, int i1);

    protected final native int nativeIntFunc(String funcname, String val);

    protected final native String nativeStringFunc(String funcname, int i);


    protected final native String nativeStringFunc(String funcname,
                                             String s1, String s2);
    protected final native int nativeIntFunc(String functname, int val1, String val2,
                                             Runnable runnable);

    void keypressed(final int buttonId, final int row, final int column, final int shift) {
        synchronized (mKeyActions) {
            mKeyActions.add(() -> {
                if (row < 0) {
                    nativeVoidFunc("keyboard_set_keyarr_any",
                            row, column, 1);
                } else {
                    nativeVoidFunc("keyboard_rawkey_pressed",
                            row, column, shift);
                }

            });
        }
    }

    void keyreleased(final int buttonId, final int row, final int column, final int shift) {
        synchronized (mKeyActions) {
            mKeyActions.add(() -> {
                if (row < 0) {
                    nativeVoidFunc("keyboard_set_keyarr_any",
                            row, column, 0);
                } else {
                    nativeVoidFunc("keyboard_rawkey_released",
                            row, column, shift);
                }

            });
        }
    }


    private final List<Runnable> mKeyActions = new ArrayList<>();
    private static final int BIT_NORTH = 1;
    private static final int BIT_SOUTH = 2;
    private static final int BIT_WEST = 4;
    private static final int BIT_EAST = 8;
    private static final int BIT_FIRE = 16;
    private final Set<String> mJoystickNorth = new HashSet<>();
    private final Set<String> mJoystickSouth = new HashSet<>();
    private final Set<String> mJoystickEast = new HashSet<>();
    private final Set<String> mJoystickWest = new HashSet<>();
    private final Set<String> mJoystickFire = new HashSet<>();
    private final Map<Integer, InputDeviceType> mLastInputDeviceType = new LinkedHashMap<>();
    private final Object mSynchronizeReadResources = new Object();
    private void readResourcesFromString(final String setting) {
        synchronized (mSynchronizeReadResources) {
            /*
            if (mThread == null || Thread.currentThread().getId() == mThread.getId()) {
                nativeIntFunc("read_resources_from_string", setting);
            } else {
                addTask(() -> nativeIntFunc("read_resources_from_string", setting));
            }

             */
            addTask(() -> nativeIntFunc("read_resources_from_string", setting));
        }
    }
    @Override
    public void setActiveInputDeviceType(final int port, final InputDeviceType newType) {
        final int inputval;
        if (newType != null) {
            switch (newType) {
                case MOUSE:
                    inputval = 3;
                    break;

                case PADDLE:
                    inputval = 2;
                    break;
                case DSTICK:
                    inputval = 1;
                    break;
                default:
                    inputval = 0;
                    break;

            }
        } else {
            inputval = 0;
        }
        final String setting = String.format(Locale.getDefault(),
                "[%s]\nJoyPort%dDevice=%d", getEmulatorId(), port, inputval);
        // here calling the native function in the ui thread is required and not dangerous.
        disableLogging();
        nativeIntFunc("read_resources_from_string", setting);
        enableLogging();
        mLastInputDeviceType.put(port, newType);
        addTask(() -> addTask(() -> {
            nativeVoidFunc("mousebutton_pressed", 0, 0);
            nativeVoidFunc("mouse_move", 0f, 0f);
        }));
    }
    private void addKeys(final List<KeyButton> keys,
                         final List<AdditionalKey> all,
                         final List<AdditionalKey> modificators) {
        all.clear();
        modificators.clear();
        List<Integer> ignoreKeys = getDynamicKeysToHide();
        for (KeyButton b : keys) {
            if ((b.getVisibility() == View.VISIBLE)
                    && (ignoreKeys == null || !ignoreKeys.contains(b.getId()))) {
                all.add(new ViceKey(b));
                if (!b.isShiftLock()) {
                    for (int mask : BITMASKS.keySet()) {
                        if ((b.getShiftState() & mask) == mask) {
                            //noinspection DataFlowIssue
                            modificators.add(new ViceKey(b, b.getResources()
                                    .getString(BITMASKS.get(mask))));
                        }
                    }
                }
            }
        }
    }
    private final List<AdditionalKey> mModificatorKeys = new LinkedList<>();
    private final List<AdditionalKey> mAllKeys = new LinkedList<>();

    private class RawkeyFunctions implements SoftkeyFunctions {

        private void initKeys() {
            ViceKeyboardFragment f = (ViceKeyboardFragment) getSoftwareKeyboardFunctions()
                    .createKeyboardFragment(Emulation.SOFTKEYBOARD_ALWAYS_COMPACT);
            View v = mUi.createInternalKeyboardFragmentView(f);
            f.setController((ViewGroup) v, f);
            mUi.removeInternalKeyboardFragmentView(v);

        }
        @NonNull
        @Override
        public List<AdditionalKey> getKeys() {
            if (mAllKeys.isEmpty()) {
                initKeys();
            }
            return mAllKeys;
        }
        @NonNull
        @Override
        public List<AdditionalKey> getModificatorKeys() {
            if (mModificatorKeys.isEmpty()) {
                initKeys();
            }
            return mModificatorKeys;

        }
        private final Map<String, Runnable> mPresskey = new LinkedHashMap<>();
        private final Map<String, Runnable> mReleasekey = new LinkedHashMap<>();

        @Override
        public void pressKey(final String id) {
            if (id != null && !mPresskey.containsKey(id)) {
                if (PATTERN_JOYSTICK_KEY.matcher(id).matches()) {
                    String[] parts = id.split(":");

                    mPresskey.put(id, () -> keypressed(0,
                            Integer.parseInt(parts[0]), Integer.parseInt(parts[1]),
                            Integer.parseInt(parts[2])));
                } else {
                    mPresskey.put(id, () -> { });
                }
            }
            Runnable r = mPresskey.get(id);
            if (r != null) {
                mKeyActions.add(r);
            }
        }

        @Override
        public void releaseKey(final String id) {
            if (id != null && !mReleasekey.containsKey(id)) {
                if (PATTERN_JOYSTICK_KEY.matcher(id).matches()) {
                    String[] parts = id.split(":");

                    mReleasekey.put(id, () -> keyreleased(0,
                            Integer.parseInt(parts[0]), Integer.parseInt(parts[1]),
                            Integer.parseInt(parts[2])));
                } else {
                    mReleasekey.put(id, () -> { });
                }
            }
            Runnable r = mReleasekey.get(id);
            if (r != null) {
                mKeyActions.add(r);
            }

        }

    }
    private final RawkeyFunctions mRawkeyFunctions = new RawkeyFunctions();
    @Override
    public SoftkeyFunctions getJoystickToKeyboardMapper(final Integer port) {
        if (port == KEYMAPPED_JOYSTICK_1 || port == KEYMAPPED_JOYSTICK_2) {
            return mRawkeyFunctions;
        } else {
            return null;
        }
    }
    @Override
    public void onJoystickChanged(final String joyId, final InputDeviceType type,
                                  final int portnumber,
                                  final boolean isAbsoluteMovement,
                                  final float xvalue, final float yvalue,
                                  final Set<Emulation.InputButton> buttons) {
        if (getJoystickports().containsKey(portnumber)) {
            if (portnumber == KEYMAPPED_JOYSTICK_1 || portnumber == KEYMAPPED_JOYSTICK_2) {
                if (!mJoystickToKeysHelper.containsKey(portnumber)) {
                    mJoystickToKeysHelper.put(portnumber, getEmulationActivity()
                            .createJoystickToKeysHelper(
                                    portnumber, getJoystickToKeyboardMapper(portnumber)));
                }
                Objects.requireNonNull(mJoystickToKeysHelper.get(portnumber))
                        .onJoystickChanged(
                                yvalue < 0, yvalue > 0, xvalue > 0, xvalue < 0, buttons);
            } else {

                if (type != mLastInputDeviceType.get(portnumber)) {
                    setActiveInputDeviceType(portnumber, type);
                }
                if (type == InputDeviceType.DSTICK) {
                    int val = 0;
                    if (yvalue < 0) {
                        mJoystickNorth.add(joyId);
                    } else {
                        mJoystickNorth.remove(joyId);
                    }
                    if (!mJoystickNorth.isEmpty()) {
                        val |= BIT_NORTH;
                    }
                    if (yvalue > 0) {
                        mJoystickSouth.add(joyId);
                    } else {
                        mJoystickSouth.remove(joyId);
                    }
                    if (!mJoystickSouth.isEmpty()) {
                        val |= BIT_SOUTH;
                    }
                    if (xvalue < 0) {
                        mJoystickWest.add(joyId);
                    } else {
                        mJoystickWest.remove(joyId);
                    }
                    if (!mJoystickWest.isEmpty()) {
                        val |= BIT_WEST;
                    }
                    if (xvalue > 0) {
                        mJoystickEast.add(joyId);
                    } else {
                        mJoystickEast.remove(joyId);
                    }
                    if (!mJoystickEast.isEmpty()) {
                        val |= BIT_EAST;
                    }
                    if (buttons.isEmpty()) {
                        mJoystickFire.remove(joyId);
                    } else {
                        mJoystickFire.add(joyId);
                    }
                    if (!mJoystickFire.isEmpty()) {
                        val |= BIT_FIRE;
                        if (mTestCrash) {
                            mTestCrash = false;
                            addTask(() -> nativeVoidFunc("request_test_crash"));
                        }
                    }
                    mJoystickstates.put(portnumber, val);
                    if (!isTerminating()) {
                        nativeVoidFunc("arch_joystick_set_value_absolute", portnumber, val);
                    }
                }
            }
            if (type == InputDeviceType.MOUSE || type == InputDeviceType.PADDLE) {
                if (isAbsoluteMovement) {
                    nativeVoidFunc("mouse_move", xvalue, yvalue);
                } else {
                    nativeVoidFunc("set_mouse_movement", xvalue, yvalue);
                }
                nativeVoidFunc("mousebutton_pressed",
                        buttons.contains(InputButton.ANALOG_PRIMARY) ? 1 : 0,
                        buttons.contains(InputButton.ANALOG_SECONDARY) ? 1 : 0);

            }
        }
    }

    @Override
    public boolean isSimulatingKeystrokes(final int port) {
        return port == KEYMAPPED_JOYSTICK_1 || port == KEYMAPPED_JOYSTICK_2;
    }

    @Override
    public boolean isExtentedJoystick(final int port) {
        return nativeIntFunc("joyport_get_port_active", port - 1) == 0;
    }

    @Override
    public List<InputDeviceType> getAvailableInputDevicetypes(final int port) {
        final List<InputDeviceType> ret = new LinkedList<>();
        ret.add(InputDeviceType.DSTICK);
        if (nativeIntFunc("joyport_get_port_active", port - 1) != 0) {
            ret.add(InputDeviceType.MOUSE);
            ret.add(InputDeviceType.PADDLE);
        }
        return ret;
    }

    public final synchronized LinkedList<DirEntry>  getImageContent(final String uri) {
        if (nativeIntFunc("imagecontentslist_create", uri) == 0) {
            int index = 1;
            LinkedList<DirEntry> ret = new LinkedList<>();
            while (nativeIntFunc("imagecontentslist_nextEntry") == 1) {
                ret.add(new DirEntry(index,
                        new String(nativeBytearrayFunc("imagecontentlist_getFileName", false)),
                        new String(nativeBytearrayFunc("imagecontentlist_getFileType", false))));
                index++;
            }
            return ret;
        }
        return null;
    }

    private static final  int AUTOSTART_MODE_RUN = 0;
    private static final int DEFAULT_DISKDEVICENUMBER = 8;
    private static final int DEFAULT_TAPEDEVICENUMBER = 1;
    public final void autostart(final Uri uri, final DirEntry entry,
                          final Runnable onSuccess, final Runnable onError) {
        setPaused(false);
        addTask(() -> {
            if (nativeIntFunc("autostart_autodetect",
                    uri.toString(), null, entry.getPos(), AUTOSTART_MODE_RUN) == 0) {
                switch (nativeIntFunc("get_filetype", uri.toString())) {
                    case FILETYPE_DISKIMAGE:
                        mAttachedImages.put(DEFAULT_DISKDEVICENUMBER, uri);
                        break;
                    case FILETYPE_TAPEIMAGE:
                        mAttachedImages.put(DEFAULT_TAPEDEVICENUMBER, uri);
                        break;
                    default:
                        break;
                }
                if (onSuccess != null) {
                    mUi.runOnUiThread(onSuccess);
                }
            } else {
                if (onError != null) {
                    mUi.runOnUiThread(onError);
                }
            }
        });
    }

    private boolean saveSnapshot(final String filepath) {
        nativeVoidFunc("timemachine_tick");
        return nativeIntFunc("timemachine_store_last_entry", filepath) == 0;
    }
    protected @Nullable ArrayList<Integer> getFixedKeysToHide() {
        return null;
    }
    protected @Nullable ArrayList<Integer> getFixedKeysToShow() {
        return null;
    }
    protected @Nullable ArrayList<Integer> getDynamicKeysToHide() {
        return null; }
    protected @Nullable ArrayList<Integer> getDynamicKeysToShow() {
        return null;
    }
    protected @Nullable ArrayList<Integer> getKeysToHighlight() {
        return null;
    }
    protected @Nullable ArrayList<Integer> getKeysToDeHighlight() {
        return null;
    }

    @Override
    public final JoystickFunctions getJoystickFunctions() {
        return this;
    }
    protected abstract int getCompactKeyboardLayoutId(int model);
    protected abstract int getExactKeyboardLayoutId(int model);
    @Override
    public final SoftwareKeyboardFunctions getSoftwareKeyboardFunctions() {
        return new SoftwareKeyboardFunctions() {
            @Override
            public KeyboardFragment createKeyboardFragment(final int mode) {
                KeyboardFragment f = new ViceKeyboardFragment();
                Bundle b = new Bundle();
                int model = Integer.parseInt(mUi.getCurrentUseropts().getStringValue("Model", "6"));
                if (mode == Emulation.SOFTKEYBOARD_ALWAYS_COMPACT) {
                    b.putInt("layout_id", getCompactKeyboardLayoutId(model));
                } else {
                    b.putInt("layout_id", getExactKeyboardLayoutId(model));
                }
                if (getFixedKeysToHide() != null) {
                    b.putIntegerArrayList("keys_to_hide", getFixedKeysToHide());
                }
                if (getFixedKeysToShow() != null) {
                    b.putIntegerArrayList("keys_to_show", getFixedKeysToShow());
                }
                if (getDynamicKeysToHide() != null) {
                    b.putIntegerArrayList("dynamic_keys_to_hide", getDynamicKeysToHide());
                }
                if (getDynamicKeysToShow() != null) {
                    b.putIntegerArrayList("dynamic_keys_to_show", getDynamicKeysToShow());
                }
                if (getKeysToHighlight() != null) {
                    b.putIntegerArrayList("keys_to_highlight", getKeysToHighlight());
                }
                if (getKeysToDeHighlight() != null) {
                    b.putIntegerArrayList("keys_to_dehighlight", getKeysToDeHighlight());
                }
                f.setArguments(b);
                return f;
            }

            @Override
            public boolean supportsDifferentLayouts() {
                return getCompactKeyboardLayoutId(mModel) != getExactKeyboardLayoutId(mModel);
            }

            @Override
            public void onKeyboardFragmentDismiss(final KeyboardFragment f) {
                ViceKeyboardFragment vkf = (ViceKeyboardFragment) f;
                vkf.reset();
            }
        };
    }
    private  boolean mIsInWarpmode = false;
    @Override
    public final SpeedUpFunctions getSpeedUpFunctions() {
        return new SpeedUpFunctions() {
            @Override
            public boolean getMaximumSpeedMode() {
                return mIsInWarpmode;
            }

            @Override
            public void setMaximumSpeedMode(final boolean value) {
                if (isPaused()) {
                    nativeIntFunc("resources_set_int", "WarpMode", value ? 1 : 0);
                    mIsInWarpmode = value;
                } else {
                    addTask(() -> {
                        nativeIntFunc("resources_set_int", "WarpMode", value ? 1 : 0);
                        mIsInWarpmode = value;
                    });
                }

            }
        };
    }

    private void populateCartridgeMap(final LinkedHashMap<String, Integer> map,
                                      final String getKeyFuncname) {
        int count = nativeIntFunc("get_cartridge_count");
        for (int i = 0; i < count; i++) {
            String key = nativeStringFunc("get_cartridge_name", i);
            Integer id = nativeIntFunc(getKeyFuncname, i);
            map.put(key, id);
        }

    }

    private LinkedHashMap<String, Integer> getCartridgesIds() {
        LinkedHashMap<String, Integer> ret = new LinkedHashMap<>();
        populateCartridgeMap(ret, "get_cartridge_id");
        return ret;
    }

    protected LinkedHashMap<String, Integer> getCartrigeFilters() {
        return null;
    }

    private LinkedHashMap<String, Integer> getCartridgesFlags() {
        LinkedHashMap<String, Integer> ret = new LinkedHashMap<>();
        populateCartridgeMap(ret, "get_cartridge_flags");
        return ret;
    }


    protected abstract boolean isCartridge(Uri uri);

    abstract int getCartridgeAutodetectFlag();
    private boolean isMatchingExtension(final Uri uri, final String extension) {
        return mUi.getFileName(uri).toLowerCase(Locale.ROOT)
                .endsWith(extension.toLowerCase(Locale.ROOT));
    }
    private int guessFileType(final Uri uri) {
        int filetype;
        if (isCartridge(uri)) {
            filetype = FILETYPE_CARTRIDGE;
        } else {
            disableLogging();
            filetype = nativeIntFunc("get_filetype", uri.toString());
            enableLogging();
        }
        if (filetype == FILETYPE_UNKNOWN && isMatchingExtension(uri, ".bin")) {
            filetype = FILETYPE_RAW_CARTRDIGE;
        }
        for (String ext : Arrays.asList(".vfl", ".lst")) {
            if (filetype == FILETYPE_UNKNOWN && isMatchingExtension(uri, ext)) {
                filetype = FILETYPE_FLIPLIST;
                break;
            }
        }
        if (filetype == FILETYPE_UNKNOWN && isMatchingExtension(uri, ".prg")) {
            filetype = FILETYPE_PRG;
        }
        return filetype;
    }

    static class ViceKey implements AdditionalKey {
        private static final StringBuffer STRING_BUFFER = new StringBuffer();
        private final String mString;
        private final String mId;

        ViceKey(final KeyButton b) {
            this(b, b.getMainText());
        }
        ViceKey(final KeyButton b, final String text) {
            mString = text;
            synchronized (STRING_BUFFER) {
                STRING_BUFFER.setLength(0);
                STRING_BUFFER.append(b.getRow()).append(":")
                        .append(b.getCol()).append(":")
                        .append(b.getShiftState());
                mId = STRING_BUFFER.toString();
            }


        }
        @Override
        public String getString() {
            return mString;
        }

        @Override
        public String getId() {
            return mId;
        }

    }

    class ViceFileFunctions implements FileFunctions {
        @Override
        public Intent getIntent(final Uri uri, final String title) {
            int filetype = guessFileType(uri);
            switch (filetype) {
                case FILETYPE_FLIPLIST:
                    return openFliplist(uri, title);

                case FILETYPE_CARTRIDGE:
                    mCartridge = uri;
                    try {
                        openCartridge(getCartridgeAutodetectFlag(), uri, null, null);
                        return null;

                    } catch (IOException e) {
                        Log.v(TAG, "Could not delete tmpSnapshot", e);
                    }
                    break;
                case FILETYPE_UNKNOWN:
                    break;
                case FILETYPE_PRG:
                    openPrgFile(uri);
                    return null;
                case FILETYPE_SNAPSHOT:
                    if (title.endsWith("__crashtest__.vsf")) {
                        mTestCrash = true;
                        Toast.makeText(mUi.getContext(),
                                "Fire button for Crash Test",
                                Toast.LENGTH_LONG)
                                .show();
                    }
                    final StringBuffer sb = new StringBuffer();
                    Map<String, String> settings = getViceMachineSettingsFunctions()
                            .getDeviceSpecificDefaultValues();
                    if (!settings.isEmpty()) {
                        getViceMachineSettingsFunctions().getConfigHeader(sb);
                        for (String key : settings.keySet()) {
                            if (!key.equals("keymapping")) {
                                sb.append(key).append("=");
                                if (key.equals("SidEngine")) {
                                    int i = getIntegerResource(key, Integer.MIN_VALUE);
                                    if (i != Integer.MIN_VALUE) {
                                        sb.append(i).append("\n");
                                    } else {
                                        String s = getStringResource(key, null);
                                        if (s != null) {
                                            sb.append(s);
                                        } else {
                                            sb.append(0);
                                        }
                                    }
                                } else {
                                    sb.append(settings.get(key));
                                }
                                sb.append(settings.get(key)).append("\n");
                            }
                        }
                    }
                    if (!sb.toString().isEmpty()) {
                        addTask(() ->
                                nativeVoidFunc("attach_snapshot_and_callback", uri.toString(),
                                        (Runnable) () -> {
                                            Log.v(TAG,
                                                    "read_resources_from_string called with \n"
                                                            + sb);
                                            readResourcesFromString(sb.toString());
                                        }));
                    }
                    return null;
                default:

                    Intent i = new Intent(mUi.getContext(), ViceFileActionActivity.class);
                    if (filetype == FILETYPE_DISKIMAGE || filetype == FILETYPE_TAPEIMAGE) {
                        LinkedList<DirEntry> imagecontent = getImageContent(uri.toString());
                        if (imagecontent != null && !imagecontent.isEmpty()) {
                            i.putExtra("title", title);
                            i.putExtra("imagecontent", imagecontent);
                        }
                        i.putExtra("readonly", (getContentAccess(uri.toString()) & 2) != 2);
                    }
                    if (filetype == FILETYPE_RAW_CARTRDIGE) {
                        LinkedHashMap<String, Integer> filters = getCartrigeFilters();
                        if (filters != null) {
                            i.putExtra("cartridge-filters", filters);
                        }
                        i.putExtra("cartridges", getCartridgesIds());
                        i.putExtra("cartridge-flags", getCartridgesFlags());
                    }
                    i.putExtra("file", uri.toString());
                    i.putExtra("type", filetype);
                    i.putExtra("title", title);
                    return i;
            }
                Intent i = mUi.newIntentForUi();
            i.putExtra("configuration", mConf);
            i.putExtra(
                    "errormessage",
                    mUi.getContext().getResources().getString(
                            R.string.could_not_open_file, title));
            i.putExtra(
                    "retry", true);
            return i;
        }

        private void openPrgFile(final Uri uri) {
            addTask(() -> {
                InputStream is;
                try {
                    if (uri.getScheme() != null) {
                        is = mUi.getContext().getContentResolver().openInputStream(uri);
                    } else {

                        //noinspection IOStreamConstructor
                        is = new FileInputStream(uri.getPath());
                    }
                    String folder = mUi.getContext().getCacheDir()
                            + File.separator + "autostart";
                    if (is != null) {
                        byte[] data = new byte[is.available()];
                        //noinspection ResultOfMethodCallIgnored
                        is.read(data);
                        is.close();
                        //noinspection ResultOfMethodCallIgnored
                        new File(folder).mkdirs();
                        File tmp = new File(folder, mUi.getFileName(uri));
                        @SuppressWarnings("IOStreamConstructor")
                        OutputStream os = new FileOutputStream(tmp);
                        os.write(data);
                        nativeIntFunc("resources_set_int", "AutostartPrgMode", 1);
                        nativeIntFunc("autostart_prg", tmp.getAbsolutePath(), 0);
                    }
                } catch (IOException e) {
                    Context ctx = mUi.getContext();
                    mUi.runOnUiThread(() -> Toast.makeText(ctx,
                            ctx.getResources().getString(R.string.could_not_open_file,
                                    mUi.getFileName(uri)), Toast.LENGTH_SHORT).show());
                }

            });
        }

        @Nullable
        private Intent openFliplist(final Uri uri, final String title) {
            try {
                boolean dialog = false;
                boolean error = false;
                Map<Integer, List<Uri>> mappedFliplist = readFliplist(uri, title);
                if (mappedFliplist != null) {
                    Set<Uri> newFliplist = new HashSet<>();
                    for (int drive : mappedFliplist.keySet()) {
                        List<Uri> files = mappedFliplist.get(drive);
                        if (files != null) {
                            newFliplist.addAll(files);
                            dialog |= files.size() > 1;
                            if (!files.isEmpty()) {
                                if (!setDriveImage(drive, files.get(0), false)) {
                                    error = true;
                                }
                            }
                        }
                    }
                    if (!newFliplist.isEmpty()) {
                        mFliplist.clear();
                        mFliplist.addAll(newFliplist);
                        mConf.onEmulatorFliplistChanged(mFliplist);
                        if (dialog || error) {
                            setPaused(true);
                            mUi.showFliplistDialog();
                            return new Intent("Pause");
                        }
                    }
                    return null;
                } else {
                    Intent i = mUi.newIntentForUi();
                    i.putExtra("configuration", mConf);
                    i.putExtra(
                            "errormessage",
                            mUi.getContext().getResources().getString(
                                    R.string.denied_for_security_reasons));
                    i.putExtra(
                            "retry", true);
                    return i;

                }


            } catch (IOException e) {
                String template = mUi.getContext()
                        .getString(R.string.cannot_open_snapshot_file,
                                new File(uri.toString()).getName());
                Toast.makeText(mUi.getContext(), template, Toast.LENGTH_LONG).show();
                return null;
            }
        }

        private String replaceLast(final String string, final String from, final String to) {
            int lastIndex = string.lastIndexOf(from);
            if (lastIndex < 0) {
                return string;
            }
            String tail = string.substring(lastIndex).replaceFirst(from, to);
            return string.substring(0, lastIndex) + tail;
        }
        @SuppressWarnings("ConstantConditions")
        private Map<Integer, List<Uri>> readFliplist(final Uri uri,
                                                        final String basename)
                throws IOException {
            Map<Integer, List<Uri>> mappedFiles = new ArrayMap<>();
            File f = mUi.tryToGetAsFile(uri);
            if (f == null) {
                return null;
            }
            BufferedReader br = new BufferedReader(new FileReader(f));
            String line;
            int currentdrive = DRIVENUMBERS.get(0);
            StringBuilder contents = new StringBuilder();

            while ((line = br.readLine()) != null) {
                contents.append(line).append(System.lineSeparator());
            }
            br = new BufferedReader(new StringReader(
                    CommentStripper.stripCStyleComments(contents.toString())));
            while ((line = br.readLine()) != null) {
                if (!line.isEmpty()
                        && !line.startsWith("#")
                        && !(line.startsWith(";") && !line.startsWith(";UNIT "))) {
                    if (line.startsWith(";UNIT ")) {
                        String[] parts = line.split(" ", 2);
                        if (parts.length == 2) {
                            currentdrive = Integer.parseInt(parts[1]);
                        }
                    } else {

                        File otherfile = new File(f.getParent(), line);
                        if (otherfile.canRead()) {
                            if (!mappedFiles.containsKey(currentdrive)) {
                                mappedFiles.put(currentdrive, new ArrayList<>());
                            }
                            mappedFiles.get(currentdrive).add(Uri.fromFile(
                                    new File(otherfile.getAbsolutePath())));
                        }
                    }
                }
            }
            for (int drive : mappedFiles.keySet()) {
                switch (mappedFiles.get(drive).size()) {
                    case 0:
                        break;
                    case 1:
                        if (setDriveImage(drive, (mappedFiles.get(drive).get(0)), false)) {
                            mFliplist.add(mappedFiles.get(drive).get(0));
                        } else {
                            mappedFiles.get(drive).clear();
                        }
                        break;
                    default:
                        mFliplist.addAll(mappedFiles.get(drive));
                }
            }

            return mappedFiles;
        }

        @Override
        public AvailabityChangingRunnableFunctions getDetachFunctions() {
            return new AvailabityChangingRunnableFunctions() {
                @Override
                public List<MenuFeature> getFunctions() {
                    //CHECKSTYLE DISABLE MagicNumber FOR 16 LINES
                    mDetachlabels.put(1, mUi.getContext().getString(
                            R.string.IDMS_DETACH_TAPE_IMAGE));
                    mDetachlabels.put(8, mUi.getContext().getString(
                            R.string.IDMS_DETACH_DISK_IMAGE)
                            + ": " + mUi.getContext().getString(
                            R.string.IDMS_DRIVE_8));
                    mDetachlabels.put(9, mUi.getContext().getString(
                            R.string.IDMS_DETACH_DISK_IMAGE)
                            + ": " + mUi.getContext().getString(
                            R.string.IDMS_DRIVE_9));
                    mDetachlabels.put(10, mUi.getContext().getString(
                            R.string.IDMS_DETACH_DISK_IMAGE)
                            + ": " + mUi.getContext().getString(
                            R.string.IDMS_DRIVE_10));
                    mDetachlabels.put(11, mUi.getContext().getString(
                            R.string.IDMS_DETACH_DISK_IMAGE)
                            + ": " + mUi.getContext().getString(
                            R.string.IDMS_DRIVE_11));
                    LinkedList<MenuFeature> ret = new LinkedList<>();
                    for (int drive : mDetachlabels.keySet()) {
                        if (mAttachedImages.containsKey(drive)) {
                            Runnable r = getDetachRunnable(drive);
                            ret.add(new MenuFeature() {
                                @Override
                                public String getName() {
                                    return mDetachlabels.get(drive);
                                }

                                @Override
                                public int getIconResource() {
                                    return MenuFeature.NO_ICON;
                                }

                                @Override
                                public Runnable getRunnable() {
                                    return r;
                                }
                            });
                        }
                    }
                    if (mCartridge != null) {
                        ret.add(new MenuFeature() {
                            @Override
                            public String getName() {
                                return mUi.getContext().getString(R.string.IDMS_DETACH_CART_IMAGE);
                            }

                            @Override
                            public int getIconResource() {
                                return MenuFeature.NO_ICON;
                            }

                            @Override
                            public Runnable getRunnable() {
                                return () -> {
                                    setCartridge(0, null);
                                    setPaused(false);
                                };
                            }
                        });
                    }
                    return ret;
                }

                @Override
                public boolean isAvailableNow() {
                    for (int drive : mDetachlabels.keySet()) {
                        if (mLockedImages.containsKey(drive)) {
                            return true;
                        }
                        if (mCartridge != null) {
                            return true;
                        }
                    }
                    return false;
                }
            };
        }
    }

    @NonNull
    private Runnable getDetachRunnable(final int drive) {
        Runnable r;
        if (drive == 1) {
            r = () -> {
                addTask(() -> setTapeImage(drive, null, false));
                mUi.updateTapedriveList(new ArrayList<>());
                setPaused(false);
            };
        } else {
            r = () -> {
                addTask(() -> setDriveImage(drive, null, false));
                setPaused(false);
            };
        }
        return r;
    }

    private boolean mLastCartridgeAttachOk = true;
    private void openCartridge(final int type, final Uri uri, final Runnable runOnSuccess,
                               final Runnable runOnError) throws IOException {
        File tmpSnapshot;
        tmpSnapshot = File.createTempFile("snapshot", null);
        String tmpSnapshotName = tmpSnapshot.getAbsolutePath();
        if (saveSnapshot(tmpSnapshotName)) {
            addResetTask(() -> {
                if (mLastCartridgeAttachOk) {
                    if (runOnSuccess != null) {
                        runOnSuccess.run();
                    }

                    if (!tmpSnapshot.delete()) {
                        Log.v(TAG, String.format("Could not delete %s", tmpSnapshot.getName()));
                    }
                } else {
                    addTask(() -> {
                        nativeVoidFunc("attach_snapshot_and_callback", tmpSnapshotName,
                                (Runnable) () -> {
                                    if (!tmpSnapshot.delete()) {
                                        Log.v(TAG, "Could not delete tmpSnapshot");
                                    }
                                });
                        setPaused(true);
                    });
                    setPaused(false);
                    if (runOnError != null) {
                        runOnError.run();
                    } else if (runOnSuccess != null) {
                        runOnSuccess.run();
                    }
                    //mLastCartridgeAttachOk = true;
                }

            });
            mLastCartridgeAttachOk = nativeIntFunc("cartridge_attach_image",
                    type, uri.toString()) == 0;
            Log.v(TAG, "cartridge_attach_image (" + type + ", " + uri + ") returned "
                    + mLastCartridgeAttachOk);
        }
    }
    @Override
    public FileFunctions getFileFunctions() {
        return new ViceFileFunctions();
    }

    private boolean mTestCrash = false;
    private void setJoystickAfterResume() {
        for (int i = 0; i < getJoystickCount(); i++) {
            if (mJoystickstates.containsKey(i) && mJoystickstates.get(i) != null) {
                //noinspection ConstantConditions
                nativeVoidFunc("arch_joystick_set_value_absolute", i, mJoystickstates.get(i));
            } else {
                nativeVoidFunc("joystick_clear", i);
                nativeVoidFunc("joystick_clear", i);
            }
            mLastInputDeviceType.put(i, null);
        }
    }

    @Override
    public AnyRunnableFunctions getResetFunctions() {
        return () -> {
            LinkedList<MenuFeature> ret = new LinkedList<>();
            ret.add(new MenuFeature() {
                @Override
                public String getName() {
                    return mUi.getContext().getString(R.string.IDS_MP_RESET)
                            + ": " + mUi.getContext().getString(R.string.IDS_MI_RESET_HARD);
                }

                @Override
                public int getIconResource() {
                    return MenuFeature.NO_ICON;
                }

                @Override
                public Runnable getRunnable() {
                    return () -> {
                        addTask(() ->
                                ViceEmulation.this.nativeVoidFunc(
                                        "machine_trigger_reset", 1));
                        if (mCartridge != null) {
                            setCartridge(mCartridgeType, mCartridge);
                        }
                        setPaused(false);
                    };
                }
            });
            ret.add(new MenuFeature() {
                @Override
                public String getName() {
                    return mUi.getContext().getString(R.string.IDS_MP_RESET)
                            + ": " + mUi.getContext().getString(R.string.IDS_MI_RESET_SOFT);
                }

                @Override
                public int getIconResource() {
                    return MenuFeature.NO_ICON;
                }

                @Override
                public Runnable getRunnable() {
                    return () -> {
                        addTask(() ->
                                ViceEmulation.this.nativeVoidFunc(
                                        "machine_trigger_reset", 0));
                        setPaused(false);
                    };
                }

            });
            if (mConf.getPresettings().get("__start__") != null
                    || mConf.getPresettings().get("__autostart_file__") != null) {
                ret.add(new MenuFeature() {
                    @Override
                    public String getName() {
                        return String.format(mUi.getContext().getString(R.string.restart_conf),
                                mConf.getName());
                    }
                    @Override
                    public int getIconResource() {
                        return MenuFeature.NO_ICON;
                    }

                    @Override
                    public Runnable getRunnable() {
                        return () -> terminate(() -> {
                            mTerminating = false;
                            mRestart = true;
                        });
                    }
                });
            }
            return ret;
        };
    }
    private final LinkedHashMap<Integer, String> mDetachlabels = new LinkedHashMap<>();


    @Override
    public final FileCreationFunction getFileCreationFunction() {
        return new FileCreationFunction() {
            @Override
            public Intent getNewFilePropertiesIntent(final Context ctx) {
                return new Intent(ctx, ViceCreateImageActivity.class);
            }

            @Override
            public boolean createFile(final ContentResolver contentResolver, final Uri uri,
                                      final Serializable parameters) {
                @SuppressWarnings("unchecked")
                HashMap<String, String> data = (HashMap<String, String>) parameters;
                @SuppressWarnings("ConstantConditions")
                int type = Integer.parseInt(data.get("type"));
                if (type == TYPE_DATASETTE) {
                    return nativeIntFunc("tape_image_create",
                            uri.toString(), TYPE_DATASETTE) == 0;
                } else {
                    return nativeIntFunc("create_emptydisk",
                            uri.toString(), data.get("name"), data.get("id"), type) == 0;

                }
            }
        };
    }

    @Override
    public final TimeMachineFunctions getTimeMachineFunctions() {
        return new TimeMachineFunctions() {

            @Override
            public int getNewestMoment() {
                return nativeIntFunc("newest_moment_elapsed_time");
            }

            @Override
            public int getOldestMoment() {
                return Math.min(nativeIntFunc("oldest_moment_elapsed_time"),
                        NUMBER_OF_SNAPSHOTS * SNAPSHOT_INTERVAL);
            }

            @Override
            public void travelTo(final int age) {
                addTask(() ->
                        nativeVoidFunc("timemachine_travel",
                                age, () -> setJoystickAfterResume()));
                mUi.setBitmap(EmulationUi.CURRENT_DISPLAY, getScreenPreview(age));
            }
            private Bitmap mScreenPreview = null;
            @Override
            public Bitmap getScreenPreview(final int interval) {
                byte[] pixels = nativeBytearrayFunc2("timemachine_get_pixels", interval, false);
                int w = nativeIntFunc("timemachine_get_width", interval);
                int h = nativeIntFunc("timemachine_get_height", interval);
                int d = nativeIntFunc("timemachine_get_depth", interval);
                Bitmap.Config config;
                switch (d) {
                    //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
                    case 16:
                        config = Bitmap.Config.RGB_565;
                        break;
                    //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
                    case 32:
                        //noinspection UnusedAssignment
                        config = Bitmap.Config.ARGB_8888;
                    default:
                        return null;
                }
                if (mScreenPreview == null || mScreenPreview.getWidth() != w
                        || mScreenPreview.getHeight() != h
                        || mScreenPreview.getConfig() != config) {
                    if (mScreenPreview != null) {
                        mScreenPreview.recycle();
                    }
                    mScreenPreview = Bitmap.createBitmap(w, h, config);
                }

                ByteBuffer buffer = ByteBuffer.wrap(pixels);
                mScreenPreview.copyPixelsFromBuffer(buffer);
                return mScreenPreview;
            }
        };
    }

    private Thread mThread;
    private final Object mThreadLocker = new Object();
    private final Object mWaitForEmulator = new Object();
    protected abstract ViceMachineSettingsFunctions getViceMachineSettingsFunctions();
    @Override
    public MachineSettingsFunctions getMachineSettingsFunction() {
        return getViceMachineSettingsFunctions();
    }
    private final List<File> mToBeDeleted = new LinkedList<>();
    private void deleteOnEmulatorFinish(final File f) {
        f.deleteOnExit();
        mToBeDeleted.add(f);
    }
    protected boolean checkRequiredFiles() {
        for (String s:Arrays.asList("kernal", "basic", "chargen")) {
            String required = "VICE_RESOURCES/" + mEmulatorId + "/" + s;
            if (!DownloaderFactory.exists(mUi.getContext(), required))  {
                mUi.onEmulatorException(this,
                        new Exception(mUi.getContext().getString(R.string.missing_files)));
                return false;
            }
        }
        return true;

    }
    @NonNull
    protected List<String> extractSettingsToCmdLine(final Useropts opts) {
        LinkedList<String> ret = new LinkedList<>();
        Map<String, String> mapping = getViceMachineSettingsFunctions()
                .getCmdLineParameterMappings();
        Map<String, String> data = new HashMap<>();
        data.putAll(getViceMachineSettingsFunctions().getDeviceSpecificDefaultValues());
        data.putAll(mConf.getPresettings());
        for (String key : data.keySet()) {
            String val = data.get(key);
            if (mapping.containsKey(key) && val != null) {
                ret.add(mapping.get(key));
                if (opts.getStringKeys().contains(key)) {
                    val = opts.getStringValue(key, val);
                } else {
                    opts.setValue(Useropts.Scope.CONFIGURATION, key, val);
                }
                ret.add(getCmdLineParameterValue(key, val));
            }
        }
        return ret;

    }
    private void setArgs() {
        String start = mConf.getPresettings().get("__start__");
        mArgsFromPresettings.clear();
        mArgsFromPresettings.add("ViceDroid");
        mArgsFromPresettings.addAll(extractSettingsToCmdLine(mUi.getCurrentUseropts()));
        if (getInitialSnapshot() == null) {
            if (start != null) {
                List<String> args = new ArrayList<>();
                Matcher m = Pattern.compile("([^\"]\\S*|\".+?\")\\s*").matcher(start);

                while (m.find()) {
                    String g = m.group(1);
                    if (g != null) {
                        args.add(g.replace("\"", ""));
                    }
                }
                if (args.size() != 1 || !args.get(0).endsWith(".vsf")) {
                    mArgsFromPresettings.addAll(args);
                }
            }
            String image = mConf.getPresettings().get("__autostart_file__");
            if (image != null) {
                String prg = mConf.getPresettings().get("__autostart_prg__");
                if (prg != null) {
                    mArgsFromPresettings.add(String.format("\"%s:%s\"", image, prg));
                } else {
                    mArgsFromPresettings.add(String.format("\"%s\"", image));

                }
            }
        }

    }
    @Override
    public final void startThread() {

        setEmulationUI(mUi);
        if (checkRequiredFiles()) {
            final String snapPath;
            nativeSetUiThread();
            File restart = mUi.getDirtyClosedSaveState(mConf);
            if (restart != null && restart.exists()) {

                Log.v(TAG, "old snapshot read from " + restart.getAbsolutePath() + ": "
                        + new SimpleDateFormat("dd-MM-yyyy HH-mm-ss", Locale.US).format(
                        new Date(restart.lastModified())));

                try {
                    File tmpSnap = File.createTempFile("__restore__", ".vsf");
                    FileInputStream fis = new FileInputStream(restart);
                    byte[] data = new byte[fis.available()];
                    if (fis.read(data) > 0) {
                        FileOutputStream fos = new FileOutputStream(tmpSnap);
                        fos.write(data);
                        fos.close();
                        snapPath = tmpSnap.getAbsolutePath();
                        deleteOnEmulatorFinish(tmpSnap);
                        if (!restart.delete()) {
                            deleteOnEmulatorFinish(restart);
                        }
                    } else {
                        snapPath = null;
                    }
                    fis.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            } else {
                Log.v(TAG, "no old snapshot found");
                snapPath = null;
            }
            if (snapPath != null) {
                mArgsFromPresettings.clear();
                mArgsFromPresettings.add("ViceDroid");
                mArgsFromPresettings.add(snapPath);

            } else {
                setArgs();
            }
            mThread = new Thread(this::run);
            if (snapPath == null) {
                String start = mConf.getPresettings().get("__start__");
                if (start != null && start.endsWith(".vsf")) {
                    startInitialSnapshot(start);
                } else {
                    startInitialSnapshot(getInitialSnapshot());
                }
            }

            mThread.setPriority(Thread.MAX_PRIORITY);
            mThread.start();
            addTask(() -> {
                if (!BuildConfig.DEBUG) {
                    nativeVoidFunc("log_set_silent", 1);
                }
            });
            if (!mInitialized) {
                try {
                    synchronized (mWaitForEmulator) {
                        mWaitForEmulator.wait();
                    }
                } catch (InterruptedException e) {
                    // ok
                }
            }

            Log.v(TAG, "recovery " + mUi.isInRecovery());
            if (mUi.isInRecovery()) {
                File f = new File(getRecoverySnapshotPath());
                if (f.canRead()) {
                    addTask(() -> {
                        int ret = nativeIntFunc("attach_and_delete_snapshot",
                                f.getAbsolutePath(), 0);
                        Log.v(TAG, String.format(
                                "attach_and_delete_snapshot %s after recovery: %d ",
                                f.getAbsolutePath(), ret));
                        mUi.runOnUiThread(() -> Toast.makeText(
                                        mUi.getContext(),
                                        mUi.getContext().getString(R.string.IDS_RESUMED),
                                        Toast.LENGTH_LONG)
                                .show());
                    });
                }
            }
        }
    }
    private boolean isTerminating() {
        return mTerminating;
    }
    @Override
    public boolean isRunning() {
        if (mThread != null) {
            return mThread.isAlive();
        }
        return false;
    }
    private final List<String> mArgsFromPresettings = new LinkedList<>();
    @Keep
    protected int getArgcFromPresettings() {
        return mArgsFromPresettings.size();
    }
    @Keep
    protected String getArgvFromPresettings(final int c) {
        if (c >= 0 && c < mArgsFromPresettings.size()) {
            String ret = mArgsFromPresettings.get(c);
            if (ret.startsWith("\"") && ret.endsWith("\"")) {
                ret = ret.substring(1, ret.length() - 1);
            }
            return ret;
        }
        return null;
    }
    private void applyInitialSnapshot(final String snapshotpath) {
        File tempfile = null;
        try {
            Runnable setUseropts = () -> addTask(()
                    -> getViceMachineSettingsFunctions()
                    .initUseropts(mUi.getCurrentUseropts()));

            tempfile = File.createTempFile("__dump__", "vsf");
            final File decrypted = tempfile;
            decrypt(snapshotpath, tempfile.getAbsolutePath(), mConf);
            setUseropts.run();
            nativeVoidFunc("attach_snapshot_and_callback",
                    decrypted.getAbsolutePath(), (Runnable) () -> {
                        setUseropts.run();
                        if (!decrypted.delete()) {
                            decrypted.deleteOnExit();
                        }
                    });
        } catch (IOException e) {
            if (tempfile != null && tempfile.exists()) {
                if (!tempfile.delete()) {
                    tempfile.deleteOnExit();
                }
            }
        }
    }
    private String getInitialSnapshot() {
        String ret = null;
        for (int i = 0; i <= MAX_SNAPSHOT_LEVELS; i++) {
            File snapshot = mUi.getInitialSnapshotPath(mConf, i);
            if (snapshot != null && snapshot.exists()) {
                ret = snapshot.getAbsolutePath();
                break;
            }
        }
        return ret;

    }
    private void startInitialSnapshot(final String snapshotPath) {
        addTask(() -> {
            addTask(()
                    -> getViceMachineSettingsFunctions()
                    .initUseropts(mUi.getCurrentUseropts()));
            if (snapshotPath != null)  {
                applyInitialSnapshot(snapshotPath);
            }
        });
        setPaused(false);
    }

    private void oboeSuspend() {
        nativeVoidFunc("oboe_suspend");
    }

    private void oboeResume() {
        nativeVoidFunc("oboe_resume");
    }

    private boolean mPlaybackDelayed = false;
    private boolean mResumeOnFocusGain = false;
    private final Object mFocusLock = new Object();
    private AudioFocusRequest mAudioFocusRequest = null;

    @RequiresApi(api = Build.VERSION_CODES.O)
    protected final void abandomAudioFocus() {
        if (mAudioFocusRequest != null) {
            ((AudioManager) mUi.getContext().getSystemService(Context.AUDIO_SERVICE))
                    .abandonAudioFocusRequest(mAudioFocusRequest);
        }
    }
    @Keep
    protected void setAudioDefaults() {
        if (android.os.Build.VERSION.SDK_INT <= Build.VERSION_CODES.O) {
            AudioManager myAudioMgr = (AudioManager) mUi.getContext()
                    .getSystemService(Context.AUDIO_SERVICE);
            String sampleRateStr = myAudioMgr.getProperty(AudioManager.PROPERTY_OUTPUT_SAMPLE_RATE);
            if (sampleRateStr != null) {
                int defaultSampleRate = Integer.parseInt(sampleRateStr);
                String framesPerBurstStr = myAudioMgr.getProperty(
                        AudioManager.PROPERTY_OUTPUT_FRAMES_PER_BUFFER);
                if (framesPerBurstStr != null) {
                    int defaultFramesPerBurst = Integer.parseInt(framesPerBurstStr);
                    nativeVoidFunc("oboe_set_default_values",
                            defaultSampleRate, defaultFramesPerBurst);
                }
            }

        }
    }
    @Keep
    protected final void getAudioFocus() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            AudioAttributes playbackAttributes = new AudioAttributes.Builder()
                    //.setUsage(AudioAttributes.USAGE_GAME)
                    //.setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION)
                    .build();
            if (mAudioFocusRequest == null) {
                mAudioFocusRequest = new AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN)
                        .setAudioAttributes(playbackAttributes)
                        .setOnAudioFocusChangeListener(focusChange -> {
                            switch (focusChange) {
                                case AudioManager.AUDIOFOCUS_GAIN:

                                    if (mPlaybackDelayed || mResumeOnFocusGain) {
                                        synchronized (mFocusLock) {
                                            mPlaybackDelayed = false;
                                            mResumeOnFocusGain = false;
                                        }
                                        oboeResume();
                                    }
                                    break;
                                case AudioManager.AUDIOFOCUS_LOSS:
                                    synchronized (mFocusLock) {
                                        mResumeOnFocusGain = false;
                                        mPlaybackDelayed = false;
                                    }
                                    oboeSuspend();
                                    break;
                                case AudioManager.AUDIOFOCUS_LOSS_TRANSIENT:
                                case AudioManager.AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK:
                                    synchronized (mFocusLock) {
                                        mPlaybackDelayed = false;
                                    }
                                    oboeSuspend();
                                    break;
                                default:
                                    break;
                            }
                        })
                        .build();
            }
            if (((AudioManager) mUi.getContext()
                    .getSystemService(Context.AUDIO_SERVICE))
                    .requestAudioFocus(mAudioFocusRequest)
                    != AudioManager.AUDIOFOCUS_REQUEST_GRANTED) {
                Log.v(TAG, "Audio Focus not granted");
            }
        }
    }
    private int getIntegerResource(final String key,
                                   @SuppressWarnings("SameParameterValue") final int defaultvalue) {
        int ret = defaultvalue;
        if (nativeIntFunc("get_resource_type", key) == 1) {
            ret = nativeIntFunc("get_int_resource", key, defaultvalue);
        }
        return ret;
    }
    private String getStringResource(final String key,
                                     @SuppressWarnings("SameParameterValue")
                                     final String defaultvalue) {
        String ret = defaultvalue;
        if (nativeIntFunc("get_resource_type", key) == 2) {
            ret = nativeStringFunc("get_int_resource", key, defaultvalue);
        }
        return ret;
    }
    int getModel() {
        return mModel;
    }

    protected class ViceMachineSettingsFunctions implements MachineSettingsFunctions {
        private final String mSetModelFuncname;
        private final List<String> mModelSpecificStringOpts = new LinkedList<>();
        private final List<String> mModelSpecificIntOpts = new LinkedList<>();
        private final List<Integer> mLayoutIds = new LinkedList<>();
        private final Map<String, String> mCmdLineParameterMappings = new LinkedHashMap<>();
        private final List<Integer> mRestrictedLayoutIds = new LinkedList<>();

        ViceMachineSettingsFunctions(final List<Integer> layoutIds,
                                     final List<Integer> restrictedLayoutIds,
                                     final Map<String, String> cmdlineParameterMapping) {
            this(null, null, null, layoutIds, restrictedLayoutIds, cmdlineParameterMapping);
        }

        ViceMachineSettingsFunctions(final String modelSetFuncname,
                                     final List<String> modelSpecificStringOpts,
                                     final List<String> modelSpecificIntOpts,
                                     final List<Integer> layoutIds,
                                     final List<Integer> restrictedLayoutIds,
                                     final Map<String, String> cmdlineParameterMapping) {
            mSetModelFuncname = modelSetFuncname;
            if (modelSpecificStringOpts != null) {
                mModelSpecificStringOpts.addAll(modelSpecificStringOpts);
            }
            if (modelSpecificIntOpts != null) {
                mModelSpecificIntOpts.addAll(modelSpecificIntOpts);
            }
            if (layoutIds != null) {
                mLayoutIds.addAll(layoutIds);
            }
            if (restrictedLayoutIds != null) {
                mRestrictedLayoutIds.addAll(restrictedLayoutIds);
            }
            if (cmdlineParameterMapping != null) {
                mCmdLineParameterMappings.putAll(cmdlineParameterMapping);
            }
        }
        private void setModelFuncname(final int newmodel) {
            synchronized (mSynchronizeReadResources) {
                int paused = nativeIntFunc("timemachine_is_paused");
                nativeVoidFunc("timemachine_set_paused", 1);
                nativeVoidFunc(mSetModelFuncname, newmodel);
                nativeVoidFunc("timemachine_set_paused", paused);
            }

        }
        protected String getDeviceSpecificModelDefault() {
            return "0";
        }
        void presetModel() {
            if (mSetModelFuncname != null) {
                int newmodel = Integer.parseInt(mUi.getCurrentUseropts().getStringValue("Model",
                        getDeviceSpecificModelDefault()));
                setModelFuncname(newmodel);
                synchronized (mSynchronizeReadResources) {
                    nativeVoidFunc(mSetModelFuncname, newmodel);
                }
            }
        }
        void addModelSpecificSetting(final StringBuffer sb, final int model) {
            mModel = model;
            setModelFuncname(mModel);
            synchronized (mSynchronizeReadResources) {
                nativeVoidFunc(mSetModelFuncname, model);
            }
            //noinspection ConstantConditions
            if (mModelSpecificStringOpts != null) {
                for (String s : mModelSpecificStringOpts) {
                    int val = nativeIntFunc("get_int_resource", s, View.NO_ID);
                    if (val != View.NO_ID) {
                        sb.append(s).append("=").append(val).append("\n");
                    }

                }
            }
            //noinspection ConstantConditions
            if (mModelSpecificIntOpts != null) {

                for (String s : mModelSpecificIntOpts) {
                    String val = nativeStringFunc("get_string_resource", s, "");
                    if (val != null) {
                        if (!val.isEmpty()) {
                            sb.append(s).append("=").append(val).append("\n");
                        }
                    }
                }
            }
        }
        private void getMachineSpecificUseropts(final Useropts useropts, final StringBuffer sb) {
            if (mSetModelFuncname != null) {

                //if (!useropts.getStringKeys().contains("Model")) {
                    int newmodel = Integer.parseInt(useropts.getStringValue("Model",
                            getDeviceSpecificModelDefault()));
                    addModelSpecificSetting(sb, newmodel);

                    useropts.setValue(Useropts.Scope.EMULATOR, "Model",
                            String.valueOf(newmodel));
                //}
            }

        }
        private String getUseropts(final Useropts useropts) {
            StringBuffer sb = new StringBuffer();
            getConfigHeader(sb);
            getMachineSpecificUseropts(useropts, sb);
//            Log.v(TAG, "read_resources_from_string called with \n" + sb);
            readResourcesFromString(sb.toString());
            getUseropts(useropts, sb);
            if (sb.indexOf("=") > 0) {
                return sb.toString();
            } else {
                return null;
            }
        }
        private void getConfigHeader(final StringBuffer sb) {
            sb.append("[").append(getEmulatorId()).append("]\n");
        }
        private void getUseropts(final Useropts useropts, final StringBuffer sb) {
            List<String> useroptsSetFromFile = new LinkedList<>();
            for (String key : useropts.getStringKeys()) {
                if (isUseroptValid(key) && isStringUseroptChanged(useropts, key)) {
                    useroptsSetFromFile.add(key);
                    String useroptsVal = useropts.getStringValue(key, null);
                    if (useroptsVal != null && !mModelSpecificStringOpts.contains(key)) {
                        sb.append(key).append("=").append(useroptsVal).append("\n");
                    }
                }
            }

            for (String key : useropts.getIntegerKeys()) {
                if (isIntUseroptChanged(useropts, key)) {
                    useroptsSetFromFile.add(key);
                    int useroptsVal = useropts.getIntegerValue(key, View.NO_ID);
                    if (useroptsVal != View.NO_ID) {
                        sb.append(key).append("=").append(useroptsVal).append("\n");
                    }
                }
            }
            for (Map<String, String> devicevalues
                    : Arrays.asList(
                    getDeviceSpecificDefaultValues(),
                    getGlobalDefaultValues(),
                    mConf.getPresettings())) {
                if (devicevalues != null) {
                    writevalues(devicevalues, useropts, sb, useroptsSetFromFile,
                            Useropts.Scope.EMULATOR);
                }
                writevalues(mConf.getPresettings(), useropts, sb, useroptsSetFromFile,
                        Useropts.Scope.CONFIGURATION);
            }

        }

        private void writevalues(final Map<String, String> devicevalues,
                                 final Useropts useropts,
                                 final StringBuffer sb,
                                 final List<String> useroptsSetFromFile,
                                 final Useropts.Scope scope) {
            for (String key : devicevalues.keySet()) {
                if (!useroptsSetFromFile.contains(key)) {
                    String value = devicevalues.get(key);
                    if (isUseroptValid(key) && isStringUseroptChanged(useropts, key)) {
                        sb.append(key).append("=").append(value).append("\n");
                        useropts.setValue(scope, key, value);
                    }
                }
            }
        }

        private boolean isIntUseroptChanged(final Useropts useropts, final String key) {
            if (nativeIntFunc("get_resource_type", key) != 0) {
                return nativeIntFunc("get_int_resource", key, Integer.MIN_VALUE)
                    != useropts.getIntegerValue(key, View.NO_ID);
            }
            return false;
        }
        private boolean isUseroptValid(final String key) {
            return nativeIntFunc("resources_query_type", key) >= 0;

        }
        private boolean isStringUseroptChanged(final Useropts useropts, final String key) {
            boolean change;
            if (nativeIntFunc("get_resource_type", key) == 1) {
                int i = nativeIntFunc("get_int_resource", key, Integer.MIN_VALUE);
                change = !String.valueOf(i).equals(useropts.getStringValue(key, null));

            } else if (nativeIntFunc("get_resource_type", key) == 2) {
                String s = nativeStringFunc("get_string_resource", key, "");
                if (s != null) {
                    change = !s.equals(useropts.getStringValue(key, null));
                } else {
                    change = useropts.getStringValue(key, null) != null;
                }

            } else {
                change = true;
            }
            return change;
        }

        public void initUseropts(final Useropts useropts) {
            String s = getUseropts(useropts);
            Log.v(TAG, "read_resources_from_string called with \n" + s);
            addTask(() -> readResourcesFromString(s));
        }
        List<String> getResetCausingUseropts() {
            return Arrays.asList("VICIIBorderMode", "VICBorderMode");
        }
        @Override
        public void applyUseropts(final Useropts opts, final Runnable runAfter) {
            if (!opts.getStringValue("Model", String.valueOf(mModel))
                    .equals(String.valueOf(mModel))) {
                mModel = Integer.parseInt(opts.getStringValue("Model", String.valueOf(mModel)));
            }
            mJoystickToKeysHelper.clear();
            addTask(() -> {
                boolean repairReset = false;
                for (String key:getResetCausingUseropts()) {
                    if (opts.getStringKeys().contains(key) || opts.getIntegerKeys().contains(key)) {
                        repairReset = true;
                        break;
                    }
                }
                if (repairReset) {
                    try {
                        File tmpFile = File.createTempFile("snapshot", null);
                        String path = tmpFile.getAbsolutePath();
                        Runnable runForopts = () -> initUseropts(opts);
                        if (saveSnapshot(path))  {
                            //nativeIntFunc("read_resources_from_string",getUseropts(opts));
                            initUseropts(opts);
                            addResetTask(() -> addTask(() -> addTask(() ->
                                nativeVoidFunc("attach_snapshot_and_callback", path,
                                        (Runnable) () -> {
                                            initUseropts(opts);
                                            //noinspection ResultOfMethodCallIgnored
                                            new File(path).delete();
                                            mUi.updateDiskdriveList(updateDriveList());
                                        })
                                )
                            ));
                        }
                    } catch (IOException e) {
                        // ok
                    }
                } else {
                    initUseropts(opts);
                    try {
                        mUi.updateDiskdriveList(updateDriveList());
                    } catch (Exception e) {
                        Log.v(TAG, "Exception in updateDiskdriveList", e);
                    }
                }
                if (runAfter != null) {
                    runAfter.run();
                }
            });
        }
        protected Map<String, String> getDeviceSpecificDefaultValues() {
            Map<String, String> ret = new LinkedHashMap<>();
            //"keymapping"

            try {
                boolean keymappingExists = false;
                List<String> files = Arrays.asList(Objects.requireNonNull(
                        mUi.getContext().getAssets()
                                .list("VICE_RESOURCES/" + getEmulatorId())));
                String currentMapping = mUi.getCurrentUseropts().getStringValue("keymapping", null);
                if (currentMapping != null) {
                    if (files.contains(currentMapping)) {
                        keymappingExists = true;
                    }
                }
                if (!keymappingExists) {
                    for (String file : files) {
                        if (file.equals("gtk3_sym_" + Locale.getDefault().getLanguage()
                                + ".vkm")) {
                            ret.put("keymapping", file);
                            mUi.getCurrentUseropts().setValue(Useropts.Scope.EMULATOR,
                                    "keymapping", file);
                        }
                    }
                    if (!ret.containsKey("keymapping")) {
                        for (String file : files) {
                            if (file.equals("gtk3_pos_" + Locale.getDefault().getCountry()
                                    + ".vkm")) {
                                ret.put("keymapping", file);
                                mUi.getCurrentUseropts().setValue(Useropts.Scope.EMULATOR,
                                        "keymapping", file);
                            }
                        }
                    }
                }
                ret.put("SidEngine", "0");
            } catch (IOException e) {
                // Make Compiler smile
            }
            return ret;

        }

        final Map<String, String> getGlobalDefaultValues() {
            LinkedHashMap<String, String> ret = new LinkedHashMap<>();
            ret.put("JAMAction", "0");
            ret.put("SidEngine", "0");
            return ret;
        }

        @Override
        public List<Integer> getLayouts(final boolean isMachineFixed) {
            return isMachineFixed ? mRestrictedLayoutIds : mLayoutIds;
        }

        public Map<String, String> getCmdLineParameterMappings() {
            return mCmdLineParameterMappings;
        }

    }
    @Keep
    protected void onModelChanged() {
        Log.v(TAG, "onModelChanged");
    }
    protected String getCmdLineParameterValue(final String key, final String val) {
        return val;

    }
    @Keep
    protected final int getContentAccess(final String uri) {
        return mUi.getContentAccess(uri);
    }
    @Keep
    protected void putContentData(final String uri, final byte[] data) {
        mUi.putContentData(uri, data);
    }
    @Keep
    protected final byte[] getContentData(final String uri) {
        return mUi.getContentData(uri);
    }
    @Keep
    protected final byte[] getAssetData(final String assetname) {
        try {
            InputStream is = DownloaderFactory.open(mUi.getContext(), assetname);
            int available = is.available();
            byte[] ret = new byte[available];
            if (is.read(ret) > 0) {
                is.close();
                return ret;
            }
            is.close();
            return null;

        } catch (IOException e) {
            return null;
        }
    }
    @Keep
    protected final void uiDisplayDriveLed(final int d, final int pwm) {
        Log.v(TAG, "uiDisplayDriveLed (" + d + ", " + pwm + ")");
        mUi.setDiskdriveState(d, pwm != 0);
    }
    @Keep
    protected final void uiDisplayTapemotorStatus(final int val) {
        mUi.runOnUiThread(() -> mUi.setTapedriveMotor(1, val != 0));
    }
    @Keep
    protected LinkedList<Integer> updateDriveList() {
        LinkedList<Integer> drives = new LinkedList<>();
        for (int drive : DRIVENUMBERS) {
            String resource = String.format("Drive%sType", drive);
            String useroptVal = mUi.getCurrentUseropts().getStringValue(resource, "");
            int drivetype;
            if (!useroptVal.isEmpty()) {
                drivetype = Integer.parseInt(useroptVal);
            } else {
                drivetype = ViceEmulation.getInstance().nativeIntFunc(
                        "get_int_resource", resource, DRIVETYPE_UNKNOWN);
            }
            if (drivetype  > 0) {
                //noinspection ConstantConditions
                drives.add(drive);
            }
        }
        return drives;

    }
    private static final int DATASETTE_CONTROL_STOP = 0;
    private static final int DATASETTE_CONTROL_START = 1;
    private static final int DATASETTE_CONTROL_FORWARD = 2;
    private static final int DATASETTE_CONTROL_REWIND = 3;
    private static final int DATASETTE_CONTROL_RECORD = 4;
    private static final int DATASETTE_CONTROL_RESET = 5;
    private static final int DATASETTE_CONTROL_RESET_COUNTER = 6;

    private static final  Map<Integer, DriveStatusListener.TapedriveState> TAPESTATE
            = new LinkedHashMap<Integer, DriveStatusListener.TapedriveState>() {{
                    put(DATASETTE_CONTROL_STOP, DriveStatusListener.TapedriveState.STOP);
                    put(DATASETTE_CONTROL_START, DriveStatusListener.TapedriveState.START);
                    put(DATASETTE_CONTROL_FORWARD, DriveStatusListener.TapedriveState.FORWARD);
                    put(DATASETTE_CONTROL_REWIND, DriveStatusListener.TapedriveState.REWIND);
                    put(DATASETTE_CONTROL_RECORD, DriveStatusListener.TapedriveState.RECORD);
                    put(DATASETTE_CONTROL_RESET, DriveStatusListener.TapedriveState.STOP);
                    put(DATASETTE_CONTROL_RESET_COUNTER, DriveStatusListener.TapedriveState.STOP);
                }};
    @Keep
    protected final void uiDisplayTapecontrolStatus(final int val) {

        mUi.runOnUiThread(() -> mUi.setTapedriveState(1, TAPESTATE.get(val)));
    }
    private static final int FAKE_TAPECOUNTER = 1000;
    @Keep
    protected final void uiDisplayTapecounter(final int val) {
        boolean wasEmpty = mAttachedTapes.isEmpty();
        if (val < FAKE_TAPECOUNTER) {
            //noinspection UnnecessaryBoxing
            mAttachedTapes.add(Integer.valueOf(1));
        } else {
            mAttachedTapes.remove(Integer.valueOf(1));
        }
        if (wasEmpty != mAttachedTapes.isEmpty()) {
            mUi.runOnUiThread(() -> mUi.updateTapedriveList(mAttachedTapes));
        }
        if (!mAttachedTapes.isEmpty()) {
            mUi.runOnUiThread(() -> mUi.setTapedriveCounter(1, val));
        }
    }

    private final List<Object> mAttachedTapes = new LinkedList<>();
    private Runnable runAndContinue(final Runnable cmd) {
        return () -> {
            addTask(cmd);
            setPaused(false);
        };
    }
    @Override
    public final TapeDeviceFunctions getTapeDeviceFunctions() {

        return new TapeDeviceFunctions() {
            private Runnable runCommand(final int command) {
                return runAndContinue(() -> addTask(() -> addTask(()
                        -> nativeVoidFunc("datasette_control", command))));
            }
            @Override
            public Runnable getStopCommand() {
                return runCommand(DATASETTE_CONTROL_STOP);
            }
            @Override
            public Runnable getPlayCommand() {
                return runCommand(DATASETTE_CONTROL_START);
            }

            @Override
            public Runnable getRewindCommand() {
                return runCommand(DATASETTE_CONTROL_REWIND);
            }

            @Override
            public Runnable getForwardCommand() {
                return runCommand(DATASETTE_CONTROL_FORWARD);
            }

            @Override
            public Runnable getRecordCommand() {
                return runCommand(DATASETTE_CONTROL_RECORD);
            }

            @Override
            public boolean isAvailableNow() {
                return !mAttachedTapes.isEmpty();
            }
        };
    }
    @Keep
    protected int getGtkKeycode(final String keyname) {
        return GtkKeySymbols.getKeycode(keyname);
    }
    private final List<Integer> mStoredKeys = new LinkedList<>();
    @Keep
    protected void clearStoredKeys() {
        mStoredKeys.clear();
    }
    @Keep
    protected void storedKey(final int keycode) {
        mStoredKeys.add(keycode);
    }
    private int keyEventToKeyNum(final KeyEvent event) {
        KeyEvent e = new KeyEvent(event.getDownTime(), event.getEventTime(), event.getAction(),
                event.getKeyCode(), event.getRepeatCount(),
                event.getMetaState() & ~(KeyEvent.META_CTRL_RIGHT_ON
                        | KeyEvent.META_CTRL_LEFT_ON
                        | KeyEvent.META_CTRL_ON),
                event.getDeviceId(), event.getScanCode(), event.getFlags(), event.getSource());

        int ret = e.getUnicodeChar();
        if ((ret & KeyCharacterMap.COMBINING_ACCENT) != 0) {
            ret = new KeyEvent(e.getDownTime(),
                    e.getEventTime(),
                    e.getAction(),
                    e.getKeyCode(),
                    e.getRepeatCount(),
                    0,
                    e.getDeviceId(),
                    e.getScanCode(),
                    e.getFlags(),
                    e.getSource()).getUnicodeChar() & KeyCharacterMap.COMBINING_ACCENT_MASK;
        }
        if (ret == 0) {
            ret = AndroidGtkKeyMappings.getUnicodeNonChar(e.getKeyCode());
        } else {
            ret = AndroidGtkKeyMappings.getGtkAdaptedKey(e.getKeyCode(), ret);
        }
        return ret;
    }
    private final LinkedHashMapWithDefault<Integer, Integer> mPressedKeys
            = new LinkedHashMapWithDefault<>(KeyEvent.KEYCODE_UNKNOWN);
    private final List<HardwareKeyboardFunctions.KeyboardMapping> mKeyboardMappings =
            new LinkedList<>();
    private final Pattern mPatternPos =
            Pattern.compile("gtk3_pos_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);

    private final Pattern mPatternSym =
            Pattern.compile("gtk3_sym_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);
    private static final  Map<String, String> CORRECTED_LANGUAGE_CODES =
            new HashMap<String, String>() {{
               put("se", "sv");
               put("be", "nl");
            }};
    String getLanguageDisplayname(final String language) {
        String ret;
        if (language == null) {
            ret = "en";
        } else {
            ret = language;
        }
        if (CORRECTED_LANGUAGE_CODES.containsKey(ret)) {
            ret = CORRECTED_LANGUAGE_CODES.get(ret);
        }
        for (Locale l: Locale.getAvailableLocales()) {
            if (l.getLanguage().equals(ret)) {
                return l.getDisplayLanguage();
            }
        }
        return ret;
    }
    void keyboardKeyClear() {
        nativeVoidFunc("keyboard_key_clear");
    }
    protected boolean isMatchingVirtualKeyboard(final int model, final String file) {
        return mPatternPos.matcher(file).matches() || mPatternSym.matcher(file).matches();
    }
    protected String getVirtualKeyboardName(final String file) {
        String ret = null;
        if (mPatternPos.matcher(file).matches()) {
            ret = getLanguageDisplayname(file.replace("gtk3_pos_", "")
                    .replace(".vkm", "")) + ", "
                    + mUi.getContext().getString(R.string.matching_position);

        } else if (mPatternSym.matcher(file).matches()) {
            ret = getLanguageDisplayname(file.replace("gtk3_sym_", "")
                    .replace(".vkm", "")) + ", "
                    + mUi.getContext().getString(R.string.matching_symbol);
        }
        return ret;

    }
    static class ViceKeyboardMapping implements Serializable,
            HardwareKeyboardFunctions.KeyboardMapping {
        private final String mId;
        private final String mName;

        ViceKeyboardMapping(final String id, final String name) {
            mId = id;
            mName = name;
        }
        @Override
        public String getId() {
            return mId;
        }

        @Override
        public String getName() {
            return mName;
        }
        @Override
        public int compareTo(final HardwareKeyboardFunctions.KeyboardMapping keyboardMapping) {
            return getName().compareTo(keyboardMapping.getName());
        }
    }
    @Override
    public HardwareKeyboardFunctions getHardwareKeyboardFunctions() {
        return new HardwareKeyboardFunctions() {
            @Override
            public List<KeyboardMapping> getMappings() {
                mKeyboardMappings.clear();
                try {
                    for (String file: Objects.requireNonNull(mUi.getContext().getAssets().list(
                            "VICE_RESOURCES/" + getEmulatorId()))) {
                        if (isMatchingVirtualKeyboard(mModel, file)) {
                            String label = getVirtualKeyboardName(file);
                            if (label != null) {
                                mKeyboardMappings.add(new ViceKeyboardMapping(file, label));
                            }
                        }
                    }
                } catch (IOException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    }
                }
                Collections.sort(mKeyboardMappings);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    mKeyboardMappings.sort(Comparator.comparing(KeyboardMapping::getName));
                }
                return mKeyboardMappings;
            }
            @Override
            public boolean onKeyDown(final KeyEvent keyevent) {
                if (isRunning()) {
                    int keycode = keyEventToKeyNum(keyevent);
                    if (mStoredKeys.contains(keycode)
                            && keycode != GtkKeySymbols.KEYSYM_VoidSymbol) {
                        if (!mPressedKeys.containsKey(keyevent.getKeyCode())) {
                            mPressedKeys.put(keyevent.getKeyCode(), keycode);
                            synchronized (mKeyActions) {
                                mKeyActions.add(() -> nativeVoidFunc("keyboard_key_pressed",
                                        keycode));
                            }
                        }
                        return true;
                    }
                }
                return false;
            }
            @Override
            public boolean onKeyUp(final KeyEvent keyevent) {
                if (isRunning()) {
                    if (mPressedKeys.containsKey(keyevent.getKeyCode())) {
                        int keycode = mPressedKeys.get(keyevent.getKeyCode());
                        mPressedKeys.remove(keyevent.getKeyCode());
                        synchronized (mKeyActions) {
                            mKeyActions.add(() ->
                                    nativeVoidFunc("keyboard_key_released", keycode));

                        }
                        return true;
                    }
                }
                return false;

            }

            @Override
            public SettingsFragment getSettingsFragment() {
                return createHardwareKeyboardSettingsFragment();
            }
        };
    }
    private File createTempDir() {

        File baseDir = getEmulationActivity().getContext().getCacheDir();
        String baseName = System.currentTimeMillis() + "-";
        for (int counter = 0; counter < TEMP_DIR_ATTEMPTS; counter++) {
            File tempDir = new File(baseDir, baseName + counter);
            if (tempDir.mkdir()) {
                return tempDir;
            }
        }
        return null;
    }
    private interface CryptDecrypt {
        byte[] action(byte[] input);

    }
    private void crypt(final String sourcepath, final String destinationpath,
                       final CryptDecrypt cryptecrypt) throws IOException {
        FileInputStream fis = new FileInputStream(sourcepath);
        byte[] src = new byte[fis.available()];
        //noinspection ResultOfMethodCallIgnored
        fis.read(src);
        byte[] dest = cryptecrypt.action(src);
        fis.close();
        FileOutputStream fos = new FileOutputStream(destinationpath);
        fos.write(dest);
        fos.close();
    }
    private void encrypt(final String sourcepath, final String destinationpath,
                         final EmulationUi.Encryptor enc) throws IOException {
        crypt(sourcepath, destinationpath, enc::encrypt);
    }
    private void decrypt(final String sourcepath, final String destinationpath,
                         final EmulationUi.Encryptor enc) throws IOException {
        crypt(sourcepath, destinationpath, enc::decrypt);
    }
    @NonNull
    private Map<String, String> readUseropts(final Useropts opts) {
        Map<String, String> ret = new LinkedHashMap<>();
        for (String key : opts.getStringKeys()) {
            if (nativeIntFunc("get_resource_type", key) != 0
                    && opts.getStringValue(key, null) != null) {
                ret.put(key, opts.getStringValue(key, null));
            }
        }
        for (String key : opts.getIntegerKeys()) {
            if (nativeIntFunc("get_resource_type", key) != 0
                    && opts.getIntegerValue(key, Integer.MAX_VALUE) != Integer.MAX_VALUE) {
                ret.put(key, String.valueOf(opts.getIntegerValue(key, 0)));
            }
        }
        return ret;
    }
    @Override
    public Runnable getInitialSnapshotStorer() {
        if (jsGetInitialSnapshotLevel() != NO_SNAPSHOT_SAVED) {
            return () -> jsStoreInitialSnapshot(0);
        }
        return null;
    }
    @Override
    public PackCurrentStateFunctions getPackCurrentStateFunctions() {
        return new PackCurrentStateFunctions() {
            @Override
            public Map<String, String> getProperties() {
                Map<String, String> ret = readUseropts(mUi.getCurrentUseropts());
                readAttachments(ret);
                ret.put("__start__", "snapshot.vsf");
                return ret;
            }

            @Override
            public Runnable getDumpContentsRunnable(final int milisecondsAgo,
                                                    @NonNull final EmulationUi.Encryptor enc) {
                if (mSnapshotpath.length() == 0) {
                    File tmpdir = createTempDir();
                    if (tmpdir != null) {
                        mSnapshotpath.append(tmpdir.getPath()).append("/snapshot.vsf");
                    }
                }
                return () -> {
                    try {
                        final File tempFile = File.createTempFile("__dump__", "");
                        if (milisecondsAgo == 0) {
                            Runnable r = () -> {

                                try {
                                    encrypt(tempFile.getAbsolutePath(), mSnapshotpath.toString(),
                                            enc);
                                    synchronized (mSnapshotpath) {
                                        mSnapshotpath.notifyAll();
                                    }
                                } catch (IOException e) {
                                    // ok
                                }
                            };

                            addTask(() -> nativeVoidFunc("store_reduced_snapshot",
                                    tempFile.toString(), r));

                            setPaused(false);
                            try {
                                synchronized (mSnapshotpath) {
                                    mSnapshotpath.wait(TIMEOUT_MILLIS);
                                }
                            } catch (InterruptedException e) {
                                // ok
                            }
                        } else {
                            writeDump(milisecondsAgo, new File(mSnapshotpath.toString()), enc);
                        }
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                    setPaused(true);
                };
            }

            @Override
            public boolean writeDump(final int milisecondsAgo, final File target,
                                     @NonNull final EmulationUi.Encryptor enc) {
                try {
                    final File f = File.createTempFile("_dump_", "");
                    if (milisecondsAgo < 0) {
                        createSnapshot(target);
                        return true;
                    } else {
                        if (nativeIntFunc("timemachine_store_entry", milisecondsAgo,
                                f.getAbsolutePath()) == 0) {
                            encrypt(f.getAbsolutePath(), target.getAbsolutePath(), enc);
                            return true;
                        }
                    }
                    return false;

                } catch (IOException e) {
                    throw new RuntimeException(e);
                }

            }

            @Override
            public boolean loadSnapshot(final String filename,
                                        final @NonNull EmulationUi.Encryptor enc) {
                try {
                    final File tempfile = File.createTempFile("__dump__", "");
                    decrypt(filename, tempfile.getAbsolutePath(), enc);
                    addTask(() -> nativeVoidFunc("attach_snapshot_and_callback",
                            tempfile.getAbsolutePath(), (Runnable) () -> {
                                setJoystickAfterResume();
                                //noinspection ResultOfMethodCallIgnored
                                tempfile.delete();
                            }));

                    return true;

                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            @Override
            public Runnable getImportContentsRunnable(final File source,
                                                      final @NonNull EmulationUi.Encryptor enc) {
                if (mSnapshotpath.length() == 0) {
                    File tmpdir = createTempDir();
                    if (tmpdir != null) {
                        mSnapshotpath.append(tmpdir.getPath()).append("/snapshot.vsf");
                    }
                }
                    return () -> {
                        try {
                            decrypt(source.getAbsolutePath(), mSnapshotpath.toString(), enc);
                        } catch (IOException e) {
                            if (BuildConfig.DEBUG) {
                                throw new RuntimeException(e);
                            }
                        }
                    };
            }


            @Override
            public List<Uri> getAttachedFiles() {
                List<Uri> ret = new LinkedList<>();

                if (mSnapshotpath.length() > 0) {
                    ret.add(Uri.fromFile(new File(mSnapshotpath.toString())));
                }
                for (int drive:mAttachedImages.keySet()) {
                    if (mAttachedImages.get(drive) != null) {
                        ret.add(mAttachedImages.get(drive));
                    }
                }
                return ret;
            }

            @Override
            public boolean showFiletoUser(final Uri uri) {
                return !uri.toString().endsWith("snapshot.vsf");
            }
        };
    }

    private void readAttachments(final Map<String, String> properties) {
        StringBuilder builder = new StringBuilder();
        for (int drive:mAttachedImages.keySet()) {
            String name;
            if (mAttachedImages.get(drive) != null) {
                Uri uri = mAttachedImages.get(drive);
                if (uri != null) {
                    builder.append(drive).append(" ").append(uri).append(" ");
                }
            }
            if (!mLockedImages.containsKey(drive)) {
                properties.put(String.format(Locale.getDefault(),
                        "AttachDevice%dReadonly", drive), "");
            }
            properties.put("__openfiles__", builder.toString().trim());
        }
    }

    private final Set<Uri> mFliplist = new HashSet<>();
    private static final  Map<Integer, Integer> DRIVELABELS = new HashMap<Integer, Integer>() {{
        //CHECKSTYLE DISABLE MagicNumber FOR 4 LINES
        put(8, R.string.IDMS_DRIVE_8);
        put(9,  R.string.IDMS_DRIVE_9);
        put(10, R.string.IDMS_DRIVE_10);
        put(11, R.string.IDMS_DRIVE_11);

    }};
    private static final int DRIVETYPE_UNKNOWN = -1;
    static final List<Integer> DRIVENUMBERS = new ArrayList<>(Arrays.asList(8, 9, 10, 11));
    @Override
    public FliplistFunctions getFliplistFunctions(final Set<Uri> fliplistFromConfig) {
        if (fliplistFromConfig == null && mFliplist.isEmpty()) {
            return null;
        }
        return new FliplistFunctions() {
            @Override
            public Map<Integer, String> getTargetDevices() {
                Map<Integer, String> ret = new HashMap<>();
                Useropts opts = mUi.getCurrentUseropts();
                for (int drive : DRIVENUMBERS) {
                    String resource = String.format("Drive%sType", drive);
                    String useroptVal = opts.getStringValue(resource, "0");
                    int drivetype = DRIVETYPE_UNKNOWN;
                    if (!useroptVal.equals("0") && !useroptVal.isEmpty()) {
                        drivetype = Integer.parseInt(useroptVal);
                    }
                    if (drivetype == DRIVETYPE_UNKNOWN) {
                        drivetype = ViceEmulation.getInstance().nativeIntFunc(
                                "get_int_resource", resource, DRIVETYPE_UNKNOWN);
                    }
                    if (drivetype  > 0) {
                        //noinspection ConstantConditions
                        ret.put(drive, mUi.getContext().getString(DRIVELABELS.get(drive)));
                    }


                }
                return ret;
            }
            @Override
            public Set<Uri> getFilesInList() {
                return mFliplist;
            }
            @Override
            public boolean attach(final Integer device, final Uri uri) {
                for (int i: DRIVENUMBERS) {
                    if (i != device && uri.equals(mAttachedImages.get(i))) {
                        setDriveImage(i, null, false);
                    }
                }
                return setDriveImage(device, uri, false);
            }

            @Override
            public Uri getAttachedImage(final int device) {
                if (mAttachedImages.containsKey(device)) {
                    if (mAttachedImages.get(device) != null) {
                        return mAttachedImages.get(device);
                    }
                }
                return null;
            }

            @Override
            public boolean canBePart(final Uri image) {
                String path = image.getPath();
                if (path != null) {
                    int i = path.lastIndexOf('.');
                    if (i > 0) {
                        String extension = image.getPath().substring(i + 1);
                        for (String entry: mUi.getContext().getResources()
                                .getStringArray(R.array.image_extensions)) {
                            String[] split = entry.split(":");
                            if (split.length >= 2
                                    && !split[0].equals("1531")
                                    && split[1].toLowerCase(Locale.ROOT)
                                    .equals(extension.toLowerCase(Locale.ROOT))) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            }
        };
    }
    private String getInstanceStatePath() {
        return mUi.getContext().getCacheDir().getAbsolutePath()
                + File.separator + "instancestate_" + mConf.getEmulatorId() + "_" + mConf.getId();
    }
    @Override
    public SoundFunctions getSoundFunctions() {
        return new SoundFunctions() {
            @Override
            public void mute() {
                addTask(() -> nativeVoidFunc("sound_mute"));
            }

            @Override
            public void unmute() {
                addTask(() -> nativeVoidFunc("sound_unmute"));
            }

            @Override
            public void callOnAudioPlaying(final Runnable r) {
                nativeVoidFunc("set_audio_test_callback", r);
            }
        };
    }
    private static final int TIMEOUI = 250;
    @Override
    public boolean createStateForNextRestart(final File f) {
        if (f.exists() && !f.delete()) {
            Log.e(TAG, "Cannot delete " + f.getAbsolutePath());
        }
        String filepath = f.getAbsolutePath();
        boolean ret = saveSnapshot(filepath);
        Log.v(TAG, "snapshot saved to " + filepath + ": "
                + new SimpleDateFormat("dd-MM-yyyy HH-mm-ss", Locale.US).format(
                new Date(new File(filepath).lastModified())));
        return ret;
    }

    protected String getViceTranslatedText(final String resIdAsString) {
        return Translator.gettext(resIdAsString);
    }
    private interface SoftkeyDescription {
        int getRow();
        int getCol();

        int getShiftState();
    }
    private final Map<String, SoftkeyDescription> mSoftkeys = new LinkedHashMap<>();
    @Override
    public SoftkeyFunctions getSoftkeyFunctions() {
        return new SoftkeyFunctions() {
            public void pressKey(final String id) {
                SoftkeyDescription skd = mSoftkeys.get(id);
                if (skd != null) {
                    mKeyActions.add(() -> nativeVoidFunc("keyboard_rawkey_pressed",
                            skd.getRow(), skd.getCol(), skd.getShiftState()));
                } else {
                    mRawkeyFunctions.pressKey(id);
                }
            }

            @Override
            public void releaseKey(final String id) {
                SoftkeyDescription skd = mSoftkeys.get(id);
                if (skd != null) {
                    final int r = skd.getRow();
                    final int c = skd.getCol();
                    final int s = skd.getShiftState();
                    mKeyActions.add(() ->
                            nativeVoidFunc("keyboard_rawkey_released", r, c, s));
                } else {
                    mRawkeyFunctions.releaseKey(id);
                }
            }

            @Override
            @NonNull
            public List<AdditionalKey> getKeys() {
                return mAllKeys;
            }
            @Override
            @NonNull
            public List<AdditionalKey> getModificatorKeys() {
                return mModificatorKeys;
            }
        };
    }
    @Keep
    protected  int getJoystickCount() {
        return getJoystickports().size();
    }
    @Keep
    protected void createSoftkey(final String id, final String text,
                                 final int row, final int col, final int shiftstate) {
        if (!mSoftkeys.containsKey(id)) {
            mSoftkeys.put(id, new SoftkeyDescription() {
                @Override
                public int getRow() {
                    return row;
                }

                @Override
                public int getCol() {
                    return col;
                }

                @Override
                public int getShiftState() {
                    return shiftstate;
                }
            });
            mUi.runOnUiThread(() -> mUi.createSoftkey(id, text));
        }
    }
    @Keep
    protected void destroySoftkey(final String id) {
        if (mSoftkeys.containsKey(id)) {
            mUi.runOnUiThread(() -> {
                mUi.destroySoftkey(id);
                mSoftkeys.remove(id);
            });
        }
    }
    void enableSoftkey(final String id, final boolean enabled) {
        if (mSoftkeys.containsKey(id)) {
            mUi.runOnUiThread(() -> mUi.enableSoftkey(id, enabled));
        }
    }
    private static boolean mAudioFailed = false;
    protected void onAudioFailure() {
        mAudioFailed = true;
    }
    protected boolean isAudioFailing() {
        return mAudioFailed;
    }
    @Keep
    protected void jsInitConstants() {
        mUi.getConstants((label, value) -> nativeIntFunc("js_set_constant", label, value));
        nativeIntFunc("js_set_constant", "JOYSTICK_NONE", 0);
        nativeIntFunc("js_set_constant", "JOYSTICK_NORTH", ViceEmulation.BIT_NORTH);
        nativeIntFunc("js_set_constant", "JOYSTICK_SOUTH", ViceEmulation.BIT_SOUTH);
        nativeIntFunc("js_set_constant", "JOYSTICK_EAST", ViceEmulation.BIT_EAST);
        nativeIntFunc("js_set_constant", "JOYSTICK_WEST", ViceEmulation.BIT_WEST);
        nativeIntFunc("js_set_constant", "JOYSTICK_FIRE", ViceEmulation.BIT_FIRE);
        nativeIntFunc("js_set_constant", "NO_SNAPSHOT_SAVED", ViceEmulation.NO_SNAPSHOT_SAVED);

    }
    @Keep
    protected boolean jsGetDeviceFeature(final int feature) {
        return mUi.getDeviceFeature(feature);
    }
    @Keep
    protected boolean jsIsKeycodeSupported(final int keycode) {
        return mUi.isKeycodeAvailable(keycode);
    }
    @Keep
    protected void jsRequestVirtualKeyboard(final int screenpart) {
        mUi.showVirtualKeyboard(screenpart);
    }
    @Keep
    protected void jsRestoreVirtualKeyboard() {
        mUi.restoreVirtualKeyboardVisibility();
    }

    @Keep
    protected void jsNotifyTasksDeleted() {
        mGamepadFunctions = null;
    }
    @Keep
    protected void jsNotifyGamepadTaskCreated() {
        mGamepadFunctions = new GamepadFunctions() {
            @Override
            public boolean onButtonPressed(final int keycode) {
                return nativeIntFunc("js_gamepad_pressed_check", keycode) == 1;
            }

            @Override
            public boolean onButtonReleased(final int keycode) {
                return nativeIntFunc("js_gamepad_released_check", keycode) == 1;
            }
        };
    }
    @Override
    public GamepadFunctions getGamepadFunctions() {
        return mGamepadFunctions;
    }
    protected String getSystemLanguage() {
        return mUi.getSystemLanguage();
    }
    @Keep
    protected void jsStoreInitialSnapshot(@SuppressWarnings("SameParameterValue") final int level) {
        File target = mUi.getInitialSnapshotPath(mConf, level);
        createSnapshot(target);
    }

    private void createSnapshot(final @NonNull  File target) {
        try {

            File tmpFile = File.createTempFile("__dump__", "");
            nativeVoidFunc("timemachine_store_new_entry", tmpFile.getAbsolutePath(),
            () -> {
                try {
                    encrypt(tmpFile.getAbsolutePath(),
                            target.getAbsolutePath(), mConf);
                    if (!tmpFile.delete()) {
                        tmpFile.deleteOnExit();
                    }
                } catch (IOException e) {
                    Log.v("initialSnapshot", "exception on conversion", e);
                }
            }, () -> Log.v("initialSnapshot", "writing failed"));
        } catch (IOException e) {
            // ok
        }
    }
    @Keep
    protected String getSnapshotForStreamRestart() {
        final int level = jsGetInitialSnapshotLevel();
        if (level != NO_SNAPSHOT_SAVED) {
            File f = mUi.getInitialSnapshotPath(mConf, level);
            if (f != null && f.canRead()) {
                try {
                    File tempfile = File.createTempFile("__dump__", "vsf");
                    decrypt(f.getAbsolutePath(), tempfile.getAbsolutePath(), mConf);
                    deleteOnEmulatorFinish(tempfile);
                    return tempfile.getAbsolutePath();
                } catch (IOException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return null;
    }
    @Keep
    protected int jsGetInitialSnapshotLevel() {

        for (int i = 0; i <= MAX_SNAPSHOT_LEVELS; i++) {
            File snapshot = mUi.getInitialSnapshotPath(mConf, i);
            if (snapshot != null && snapshot.exists()) {
                try {
                    File tempfile = File.createTempFile("__check__", "vsf");
                    decrypt(snapshot.getAbsolutePath(), tempfile.getAbsolutePath(), mConf);
                    if (nativeIntFunc("check_snapshot", tempfile.getAbsolutePath()) == 0) {
                        return i;
                    }
                    if (!tempfile.delete()) {
                        tempfile.deleteOnExit();
                    }
                } catch (IOException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    } else {
                        return NO_SNAPSHOT_SAVED;
                    }
                }
            }
        }
        return NO_SNAPSHOT_SAVED;
    }
    private static final int FULLQUALITY = 100;
    @Keep
    protected void jsStoreTestScreenshot(final String filename) {
        mUi.storeJunitScreenshot(mUi.getCurrentBitmap(), filename);
    }
    protected final SettingsFragment createHardwareKeyboardSettingsFragment() {
        return new ViceKeyboardFragment.KeyboardSettingsFragment();
    }
    protected boolean isMatchingKeyboardMapping(final EmulationUi.SettingSpinnerElement e,
                                                final String activeElementId) {
        return e.getId().equals(activeElementId);
    }

    @Nullable
    @Override
    public DualMonitorFunctions getDualMonitorFunctions() {
        return null;
    }
    int[] hideKeysAfterKeypress(final int row, final int col) {
        return null;
    }
    int[] showKeysAfterKeypress(final int row, final int col) {
        return null;
    }
    boolean toggleKeyset(final int row, final int col) {
        return false;
    }
    @Keep
    protected boolean jsSetMonitor(final int monitor) {
        DualMonitorFunctions dmf = getDualMonitorFunctions();
        if (dmf != null && dmf.getDisplays().containsKey(monitor)) {
            dmf.setActiveMonitor(monitor);
            mUi.runOnUiThread(() -> getEmulationActivity().setVisibleMonitor(monitor));
            return true;
        }
        return false;
    }

    @Override
    public int getAutofireFrequency(final int port) {
        return 255;
    }
    protected void fixJoystickNames(final HashMap<Integer, String> values) {
        for (int key : values.keySet()) {
            String value = values.get(key);
            if (value != null) {
                value = value.replace("Joystick in", "Controller in");
                values.put(key, value);
            }
        }
    }
    protected void disableLogging() {
        if (BuildConfig.DEBUG) {
            nativeVoidFunc("log_set_silent", 1);
        }
    }
    protected void enableLogging() {
        if (BuildConfig.DEBUG) {
            nativeVoidFunc("log_set_silent", 0);
        }
    }
}
