//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.ShareEmulationActivity.addSavestatesForZip;
import static de.rainerhock.eightbitwonders.ShareEmulationActivity.writeSaveStates;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Bundle;
import android.util.AttributeSet;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.PopupMenu;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.constraintlayout.helper.widget.Flow;
import androidx.core.content.res.ResourcesCompat;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

public final class TileView extends LinearLayout {
    private String mName;
    private boolean mBuiltin = false;
    void setBuiltin(final boolean value) {
        mBuiltin = value;
    }
    private boolean mCopy = false;
    boolean isCopy() {
        return mCopy;
    }
    void setCopy(final boolean value) {
        mCopy = value;
    }
    boolean isBuiltin() {
        return mBuiltin;
    }
    private void exportConfiguration(@NonNull final BaseActivity activity,
                                     @NonNull final EmulationConfiguration config,
                                     final Map<String, String> saveStateData,
                                     final Useropts useropts) {
        if (config.getLocalPath() != null) {
            File[] files = config.getLocalPath().listFiles();
            if (files != null) {
                //noinspection RegExpRedundantEscape
                File out = new File(TileView.this.getContext().getCacheDir(),
                        config.getName().trim()
                                .replaceAll("[^a-zA-Z\\d\\.\\-]", "_")
                                + ".8bw");
                try {
                    //noinspection IOStreamConstructor
                    ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(out));
                    if (saveStateData != null) {
                        Map<String, String> m = new HashMap<>();
                        addSavestatesForZip(new File(activity.getSnapshotPath(config)
                                .getAbsolutePath()), ".", m);
                        writeSaveStates(saveStateData, zos);
                    }
                    if (useropts != null) {
                        useropts.setCurrentEmulation(getContext(), config.getEmulatorId(),
                                config.getId());
                        StringWriter s = new StringWriter();
                        useropts.yamlify(Useropts.Scope.CONFIGURATION, s);
                        zos.putNextEntry(new ZipEntry("__settings__.yml"));
                        zos.write(s.toString().getBytes(StandardCharsets.UTF_8));
                        zos.closeEntry();
                    }
                    for (File in : files) {
                        if (in.isFile() && in.canRead()) {
                            //noinspection IOStreamConstructor
                            InputStream is = new FileInputStream(in);
                            byte[] data = new byte[is.available()];
                            if (is.read(data) >= 0) {
                                ZipEntry ze = new ZipEntry(in.getName());
                                zos.putNextEntry(ze);
                                zos.write(data);
                                zos.closeEntry();
                            }
                            is.close();
                        }
                    }
                    zos.close();
                    activity.shareFile(out, config.getName());
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }
    void setPosition(final int pos) {
        String name;
        if (pos > 1) {
            name = String.format(Locale.getDefault(), "%s (%d)", mName, pos);
        } else {
            name = mName;
        }
        ((TextView) findViewById(R.id.tv_name)).setText(name);
    }
    static class CopiedConfiguration implements EmulationConfiguration {
        private final EmulationConfiguration mSource;

        CopiedConfiguration(final EmulationConfiguration source) {
            mSource = source;

        }
        @Override
        public void adaptJoysticks(final BaseActivity activity,
                                   final Emulation.JoystickFunctions joystickfunctions,
                                   final List<Joystick> availableJoysticks,
                                   final Map<Integer, PreferredJoystickType>
                                                   preferredJoysticktypes) {
            mSource.adaptJoysticks(activity, joystickfunctions, availableJoysticks,
                    preferredJoysticktypes);
        }

        @Override
        public boolean getTouchJoystickDiagonalsLocked() {
            return mSource.getTouchJoystickDiagonalsLocked();
        }

        @Override
        public Map<Integer, PreferredJoystickType> getVirtualJoystickTypes() {
            return mSource.getVirtualJoystickTypes();
        }

        @Override
        public Map<Integer, Emulation.InputDeviceType> getInputDeviceTypes() {
            return mSource.getInputDeviceTypes();
        }

        @NonNull
        @Override
        public String getId() {
            return mSource.getId();
        }

        @Override
        public Bitmap getBitmap() {
            return mSource.getBitmap();
        }

        @NonNull
        @Override
        public String getEmulatorId() {
            return mSource.getEmulatorId();
        }

        @Override
        public Orientation getBestOrientation() {
            return mSource.getBestOrientation();
        }

        @Override
        public void apply(final EmulationActivity ui, final Emulation emu) {
            mSource.apply(ui, emu);

        }

        @Override
        public Map<String, String> getPresettings() {
            return mSource.getPresettings();
        }

        @Override
        public DownloaderFactory.Downloader getAdditionalDownloader(final BaseActivity context) {
            return mSource.getAdditionalDownloader(context);
        }

        @Override
        public String getName() {
            return mSource.getName();
        }

        @Override
        public boolean isBareMachine() {
            return mSource.isBareMachine();
        }

        @Override
        public boolean isLocallyGenerated() {
            return mSource.isLocallyGenerated();
        }

        @Override
        public boolean isImported() {
            return mSource.isImported();
        }

        @Override
        public boolean isStream() {
            return true;
        }

        @Override
        public URL getWebAdress() {
            return mSource.getWebAdress();
        }

        @Override
        public Runnable getFactoryResetter() {
            return mSource.getFactoryResetter();
        }

        @Override
        public Runnable getDeinstaller() {
            return null;
        }

        @Override
        public List<Integer> getUsedJoystickports(final List<Integer> allJoystickports) {
            return mSource.getUsedJoystickports(allJoystickports);
        }

        @NonNull
        @Override
        public List<Integer> getRequiredJoystickports(final List<Integer> allJoystickports) {
            return mSource.getRequiredJoystickports(allJoystickports);
        }

        @Override
        public boolean isKeyboardUsed() {
            return mSource.isKeyboardUsed();
        }

        @Override
        public Set<Uri> getFliplist() {
            return mSource.getFliplist();
        }

        @Override
        public void onEmulatorFliplistChanged(final Set<Uri> emulatorlist) {
            mSource.onEmulatorFliplistChanged(emulatorlist);
        }

        @Override
        public String getFilepath(final String basename) {
            return mSource.getFilepath(basename);
        }

        @Override
        public byte[] encrypt(final byte[] plaindata) {
            return mSource.encrypt(plaindata);
        }

        @Override
        public byte[] decrypt(final byte[] encrypted) {
            return mSource.decrypt(encrypted);
        }

        @Nullable
        @Override
        public File getLocalPath() {
            return mSource.getLocalPath();
        }

        @Override
        public String getProperty(final String key, final String defaultvalue) {
            return mSource.getProperty(key, defaultvalue);
        }
    }
    private boolean mPopupOpen = false;
    boolean isPopupOpen() {
        return mPopupOpen;
    }

    void showExtendedOption(final MainActivity activity,
                                    final EmulationConfiguration config,
                                    final boolean useDialog) {
        PopupMenu p = new PopupMenu(getContext(), this);
        p.getMenuInflater().inflate(R.menu.menu_configuration, p.getMenu());
        Menu m = p.getMenu();
        Intent intentView = new Intent(Intent.ACTION_VIEW);
        if (config.getWebAdress() != null) {
            intentView.setData(Uri.parse(config.getWebAdress().toString()));
        }
        m.findItem(R.id.mi_gotoweb).setVisible(config.getWebAdress() != null
                && intentView.resolveActivity(getContext().getPackageManager()) != null);
        m.findItem(R.id.mi_reset).setVisible(config.getFactoryResetter() != null);
        m.findItem(R.id.mi_uninstall).setVisible(config.getDeinstaller() != null);
        m.findItem(R.id.mi_share).setVisible(config.getLocalPath() != null);
        m.findItem(R.id.rename).setVisible(config.isLocallyGenerated() || config.isStream());
        Useropts opts = new Useropts(activity);
        String optsKey = "SHOW_IN_USERCONTENT_" + config.getEmulatorId() + "_" + config.getId();
        m.findItem(R.id.mi_add_to_usercontent).setVisible(
                ((View) getParent()).getId() == R.id.preinstalled
                        && !opts.getBooleanValue(optsKey, false));
        m.findItem(R.id.mi_remove_usercontent).setVisible(
                ((View) getParent()).getId() == R.id.usercontent
                        && opts.getBooleanValue(optsKey, false));
        p.setOnMenuItemClickListener(item -> {
            final int id = item.getItemId();
            if (id == R.id.mi_run) {
                activity.startEmulation(config);
                return true;
            } else if (id == R.id.mi_gotoweb) {
                Intent i = new Intent(Intent.ACTION_VIEW);
                i.setData(Uri.parse(config.getWebAdress().toString()));
                activity.startActivity(i);
                post(this::requestFocus);
                return true;
            } else if (id == R.id.mi_reset) {
                config.getFactoryResetter().run();
                requestFocus();
                return true;
            } else if (id == R.id.mi_uninstall) {
                Runnable deleteDownloadOnly = () -> {
                    if (config.getWebAdress() != null) {
                        activity.removeStreamingUrl(config.getWebAdress().toString());
                    }
                    activity.removeInitialSnapshots(config);
                    config.getDeinstaller().run();
                    ((Flow) activity.findViewById(R.id.fl_usercontent)).removeView(this);
                    ((ViewGroup) activity.findViewById(R.id.usercontent)).removeView(this);
                    activity.removeWidget(config);
                    activity.updateUsercontentUI();
                };
                Runnable deleteWithUseropts;
                deleteWithUseropts = () -> {
                    deleteDownloadOnly.run();
                    Useropts.delete(activity, config.getEmulatorId(), config.getId());
                    new BaseActivity.Deleter(activity.getSnapshotPath(config)).run();
                };
                if (config.isLocallyGenerated()) {
                    activity.showReallyDeleteDialog(deleteWithUseropts, deleteWithUseropts);
                } else {
                    activity.showReallyDeleteDialog(deleteDownloadOnly, deleteWithUseropts);
                }

                return true;
            } else if (id == R.id.rename) {
                MainActivity.RenameConfigurationFragment f =
                        new MainActivity.RenameConfigurationFragment();
                Bundle b = new Bundle();
                b.putSerializable("configuration", config);
                f.setArguments(b);
                f.showNow(activity.getSupportFragmentManager(), "");

                return true;
            } else if (id == R.id.mi_share) {
                Map<String, String> savestates = new HashMap<>();
                addSavestatesForZip(new File(activity.getSnapshotPath(config)
                        .getAbsolutePath()), ".", savestates);
                opts.setCurrentEmulation(getContext(), config.getEmulatorId(), config.getId());
                AlertDialog.Builder builder = null;
                if (opts.isDifferent(config) && !savestates.isEmpty()) {
                    boolean[] selection = new boolean[]{false, true};
                    builder = new BaseActivity.AlertDialogBuilder(getContext())
                            .setMultiChoiceItems(R.array.export_options, selection,
                                    (dialogInterface, i, b) -> selection[i] = b)
                            .setPositiveButton(R.string.share, (dialogInterface, i)
                                    -> exportConfiguration(activity, config,
                                    selection[0] ? savestates : null, selection[1] ? opts : null));
                } else if (opts.isDifferent(config)) {
                    boolean[] selection = new boolean[]{true};
                    builder = new BaseActivity.AlertDialogBuilder(getContext())
                            .setMultiChoiceItems(new String[]{
                                    getResources().getString(R.string.share_useropts)}, selection,
                                    (dialogInterface,  i, b) -> selection[0] = b)
                            .setPositiveButton(R.string.share, (dialogInterface, i)
                                    -> exportConfiguration(activity, config,
                                    null, selection[0] ? opts : null));

                } else if (!savestates.isEmpty()) {
                    boolean[] selection = new boolean[]{false};
                    builder = new BaseActivity.AlertDialogBuilder(getContext())

                            .setMultiChoiceItems(new String[]{getResources()
                                            .getString(R.string.share_savestates)}, selection,
                                    (dialogInterface, i, b) -> selection[0] = b)
                            .setPositiveButton(R.string.share, (dialogInterface, i)
                                    -> exportConfiguration(activity, config,
                                    selection[0] ? savestates : null, null));
                }
                if (builder != null) {
                    builder.setNegativeButton(R.string.cancel, ((dialogInterface, i)
                            -> dialogInterface.dismiss())).create().show();
                } else {
                    exportConfiguration(activity, config, null, null);
                }
                return true;
            } else if (id == R.id.mi_add_to_usercontent) {
                opts.setValue(Useropts.Scope.GLOBAL, optsKey, true);
                activity.addEmulationTile(new CopiedConfiguration(config), false, false);
                activity.updateUsercontentUI();
                return true;
            } else if (id == R.id.mi_remove_usercontent) {
                opts.setValue(Useropts.Scope.GLOBAL, optsKey, false);
                ((ViewGroup) getParent()).removeView(this);
                activity.updateUsercontentUI();
                return true;

            }
            return false;
        });
        if (useDialog) {
            activity.openInfoDialog(p.getMenu());
        } else {
            p.setOnDismissListener(popupMenu -> mPopupOpen = false);
            mPopupOpen = true;
            p.show();
        }
    }

    static TileView createMachinePreview(final LayoutInflater inflater,
                                         final ViewGroup container,
                                         final String name,
                                         final int machineImageResId,
                                         final int backgroundColorId) {
        return TileView.create(inflater, container, name, machineImageResId,
                null, backgroundColorId);
    }

    static TileView createConfigurationPreview(final LayoutInflater inflater,
                                               final ViewGroup container,
                                               final String name,
                                               final int machineImageResId,
                                               final Bitmap screenshot) {
        return TileView.create(inflater, container, name, machineImageResId,
                screenshot, 0);
    }
    static TileView createWidget(final LayoutInflater inflater,
                                 final ViewGroup container,
                                 final EmulationConfiguration conf) {
        Emulationslist.Description desc =
                Emulationslist.getDescription(conf.getEmulatorId());
        if (desc != null) {
            if (conf instanceof ConfigurationFactory.MachineConfiguration) {
                return TileView.createMachinePreview(inflater, container, conf.getName(),
                        desc.getImageResourceId(), container.getResources()
                                .getColor(desc.getColorResourceId()));
            } else {
                return TileView.createConfigurationPreview(inflater, container, conf.getName(),
                        desc.getImageResourceId(), conf.getBitmap());
            }
        } else {
            return null;
        }
    }
    private static TileView create(final LayoutInflater inflater,
                                   final ViewGroup container,
                                   final String name,
                                   final int machineImageResId,
                                   final Bitmap screenshot,
                                   final int color) {
        TileView ret = (TileView) inflater
                .inflate(R.layout.view_tile,
                        container, false);
        ret.mName = name;
        ((TextView) ret.findViewById(R.id.tv_name)).setText(name);
        int machineImageId = screenshot != null ? R.id.iv_machine_mall : R.id.iv_machine_large;
        if (machineImageResId != 0) {
            ((ImageView) ret.findViewById(machineImageId))
                    .setImageDrawable(
                            Objects.requireNonNull(ResourcesCompat.getDrawable(
                                    ret.getResources(), machineImageResId,
                                    ret.getContext().getTheme()
                            )));
            if (color != 0 && screenshot == null) {
                ret.findViewById(R.id.iv_bitmap).setBackgroundColor(color);
                Log.v("setBackgroundColor", "" + color);
            }
        }
        if (screenshot != null) {
            ret.setBitmap(screenshot);
            if (machineImageResId != 0) {
                ImageView machine = ret.findViewById(R.id.iv_machine_mall);
                machine.setVisibility(View.VISIBLE);
            }
        }
        return ret;
    }

    private void init() {

    }
    String getName() {
        return mName;
    }

    void setPopupOpen(final boolean val) {
        mPopupOpen = val;
    }

    interface StartFragmentListener {
        void action(boolean isJoystick, boolean showExtendedAction);
    }

    private StartFragmentListener mListener = (b1, b2) -> {
    };

    void setListener(@NonNull final StartFragmentListener listener) {
        mListener = listener;
        setOnLongClickListener(view -> {
            mListener.action(false, true);
            return true;
        });
        setOnClickListener(view -> mListener.action(false, false));
        setOnKeyListener((v, keycode, keyEvent) -> {
            if (keyEvent.getAction() == KeyEvent.ACTION_UP
                    && EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)) {
                mListener.action(keycode != KeyEvent.KEYCODE_DPAD_CENTER,
                        EmulationUi.JOYSTICK_SELECT_BUTTONS.contains(keycode));
                return true;
            }
            return false;
        });

    }

    void setBitmap(final Bitmap screenshot) {
        ImageView v = findViewById(R.id.iv_bitmap);
        v.setImageBitmap(screenshot);
    }

    @Override
    public boolean onKeyDown(final int keyCode, final KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                || EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)) {
            event.startTracking();
            return true;
        }
        if (keyCode == KeyEvent.KEYCODE_BUTTON_B) {
            return true;
        }
        return super.onKeyDown(keyCode, event);
    }

    @Override
    public boolean onKeyLongPress(final int keyCode, final KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                || EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)) {
            mListener.action(false, true);
            return true;
        }
        return super.onKeyLongPress(keyCode, event);
    }

    @Override
    public boolean onKeyUp(final int keyCode, final KeyEvent event) {
        if ((event.getFlags() & KeyEvent.FLAG_CANCELED_LONG_PRESS) == 0) {
            if (keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                    || EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)) {
                mListener.action(keyCode != KeyEvent.KEYCODE_DPAD_CENTER, false);
                return true;
            }
        }
        if (keyCode == KeyEvent.KEYCODE_MENU
                || EmulationUi.JOYSTICK_SELECT_BUTTONS.contains(keyCode)) {
            mListener.action(keyCode != KeyEvent.KEYCODE_MENU, true);
            return true;
        }
        if (keyCode == KeyEvent.KEYCODE_BUTTON_B) {
            return true;
        }
        return super.onKeyUp(keyCode, event);
    }

    /**
     * Default constructor, see {@link LinearLayout}.
     *
     * @param context see {@link LinearLayout}
     */
    public TileView(final Context context) {
        super(context);
        init();
    }


    /**
     * Default constructor, see {@link LinearLayout}.
     *
     * @param context see {@link LinearLayout}
     * @param attrs   see {@link LinearLayout}
     */
    public TileView(final Context context, @Nullable final AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    /**
     * Default constructor, see {@link LinearLayout}.
     *
     * @param context      see {@link LinearLayout}
     * @param attrs        see {@link LinearLayout}
     * @param defStyleAttr see {@link LinearLayout}
     */
    public TileView(final Context context, @Nullable final AttributeSet attrs,
                    final int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

}
