//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import androidx.annotation.Keep;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * An exception that is thrown when a signal has occured in native code.
 */
@Keep
public final class NativeSignalException extends Exception {
    private final int mSigno;
    private final int mCode;
    private final int mErrno;
    private final int mAddr;
    private final boolean mRecoveryWritten;

    /**
     * create an new Exception after a signal in native code.
     * @param signo signal-number
     * @param code from native struct siginfo
     * @param errno from native struct siginfo
     * @param addr from native struct siginfo
     * @param recoveryWritten if a recovery was written
     */
    public NativeSignalException(final int signo,
                                 final int code,
                                 final int errno,
                                 final int addr,
                                 final boolean recoveryWritten) {
        super();
        mSigno = signo;
        mCode = code;
        mErrno = errno;
        mAddr = addr;
        mRecoveryWritten = recoveryWritten;
    }
    private static final int SIGABRT = 6;
    private static final int SIGBUS = 7;
    private static final int SIGFPE = 8;
    private static final int SIGSEV = 11;
    private static final int SIGKILL = 4;
    private static final int SIGSTKFLT = 16;
    private static final int SIGTRAP = 5;
    private static final int SIGUSR2 = 12;

    private static final Map<Integer, String> SIGNALS = new LinkedHashMap<Integer, String>() {{
            put(SIGABRT, "SIGABRT");
            put(SIGBUS, "SIGBUS");
            put(SIGFPE, "SIGFPE");
            put(SIGSEV, "SIGSEV");
            put(SIGKILL, "SIGKILL");
            put(SIGSTKFLT, "SIGSTKFLT");
            put(SIGTRAP, "SIGTRAP");
            put(SIGUSR2, "SIGUSR2");
        }};

    @Override
    public String getMessage() {
        StringBuilder sb = new StringBuilder();
        sb.append("Signal ");
        sb.append(mSigno);
        if (SIGNALS.containsKey(mSigno)) {
            sb.append(" (");
            sb.append(SIGNALS.get(mSigno));
            sb.append(") ");
        }
        sb.append("received, Code=");
        sb.append(mCode);
        if (mAddr > 0) {
            sb.append(", addr=");
            sb.append(Integer.toHexString(mAddr));
        }
        if (mErrno > 0) {
            sb.append(", errno=");
            sb.append(mAddr);
        }
        if (mRecoveryWritten) {
            sb.append(", recovery written");
        }
        sb.append(".");
        return sb.toString();
    }

    /**
     * If a recovery was written, the application can relaunch and restart the emulation.
     * @return recovery successfully written
     */
    public boolean isRecoveryWritten() {
        return mRecoveryWritten;
    }
}
