//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.util.AttributeSet;

import java.util.LinkedList;
import java.util.List;

public final class JoystickWheelView extends JoystickDirectionView<WheelJoystick>
        implements TouchDisplayRelativeLayout.TouchDisplayElement {
    /**
     * Simple constructor to use when creating a JoystickWheelView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public JoystickWheelView(final Context context) {
        super(context);
    }
     /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickWheelView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }
    /**
     * Constructor that is called when inflating a JoystickWheelView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickWheelView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }


    private static final int ROTATION_CENTER = 10000 / 2;
    private static final int ROTATION_LEFT = 0;
    private static final int ROTATION_RIGHT = 10000;
        @Override
    protected void showNewDirection(final Directionvalue dv) {
        if (dv != getLastDirection()) {
            setLastDirection(dv);
            getJoystick().onChanged();
            if (isPointingEast()) {
                getDrawable().setLevel(ROTATION_RIGHT);
            } else if (isPointingWest()) {
                getDrawable().setLevel(ROTATION_LEFT);
            } else {
                getDrawable().setLevel(ROTATION_CENTER);
            }
        }
        if (dv != null) {
            setTag(dv.toString());
        } else {
            setTag(Directionvalue.CENTERED.toString());
        }
    }
    // CHECKSTYLE DISABLE MagicNumber FOR 35 LINES
    private static final  List<RangeChecker> RANGES_FROM_BOTTOMRIGHT
            = new LinkedList<RangeChecker>() {{
        add(new RangeChecker(0, 70, Directionvalue.WEST)); // 70
        add(new RangeChecker(70, 95, Directionvalue.CENTERED)); // 25
        add(new RangeChecker(95, 160, Directionvalue.EAST)); // 65
        add(new RangeChecker(95, 200, Directionvalue.EAST)); // 65
        add(new RangeChecker(200, 360, Directionvalue.WEST)); // 200

    }};

    private static final  List<RangeChecker> RANGES_FROM_BOTTOMLEFT
            = new LinkedList<RangeChecker>() {{
        add(new RangeChecker(0, 160, Directionvalue.EAST)); // 200
        add(new RangeChecker(160, 200, Directionvalue.CENTERED)); // 200
        add(new RangeChecker(200, 265, Directionvalue.WEST)); // 65
        add(new RangeChecker(265, 290, Directionvalue.CENTERED)); // 25
        add(new RangeChecker(290, 360, Directionvalue.EAST)); // 70
    }};
    private static final  List<RangeChecker> RANGES_FROM_TOPRIGHT
            = new LinkedList<RangeChecker>() {{
        add(new RangeChecker(0, 70, Directionvalue.WEST)); // 160
        add(new RangeChecker(70, 100, Directionvalue.CENTERED)); // 20
        add(new RangeChecker(100, 200, Directionvalue.EAST)); // 65
        add(new RangeChecker(200, 230, Directionvalue.CENTERED)); // 25
        add(new RangeChecker(230, 360, Directionvalue.WEST)); // 70

    }};
    private static final  List<RangeChecker> RANGES_FROM_TOPLEFT
            = new LinkedList<RangeChecker>() {{
        add(new RangeChecker(0, 130, Directionvalue.EAST)); // 160
        add(new RangeChecker(130, 160, Directionvalue.CENTERED)); // 25
        add(new RangeChecker(160, 260, Directionvalue.WEST)); // 25
        add(new RangeChecker(260, 290, Directionvalue.CENTERED)); // 20
        add(new RangeChecker(290, 360, Directionvalue.EAST)); // 160

    }};
    private static final  List<RangeChecker> RANGES_FROM_CENTER = new LinkedList<RangeChecker>() {{
        // CHECKSTYLE DISABLE MagicNumber FOR 7 LINES
        add(new RangeChecker(5, 70, Directionvalue.WEST));
        add(new RangeChecker(70, 95, Directionvalue.CENTERED));
        add(new RangeChecker(95, 160, Directionvalue.EAST));
        add(new RangeChecker(160, 200, Directionvalue.CENTERED));
        add(new RangeChecker(200, 265, Directionvalue.WEST));
        add(new RangeChecker(265, 290, Directionvalue.CENTERED));
        add(new RangeChecker(290, 355, Directionvalue.EAST));
    }};
    @Override
    protected List<RangeChecker> getRanges() {
        if (mDownX != null && mDownY != null) {
            if (mDownX > 0 && mDownY > 0) {
                //Log.v("calculateDirection", "RANGES_FROM_BOTTOMRIGHT");
                return RANGES_FROM_BOTTOMRIGHT;
            }
            if (mDownX <= 0 && mDownY > 0) {
                //Log.v("calculateDirection", "RANGES_FROM_BOTTOMLEFT");
                return RANGES_FROM_BOTTOMLEFT;
            }
            if (mDownX <= 0 && mDownY <= 0) {
                //Log.v("calculateDirection", "RANGES_FROM_TOPLEFT");
                return RANGES_FROM_TOPLEFT;
            }
            if (mDownX > 0 && mDownY <= 0) {
                //Log.v("calculateDirection", "RANGES_FROM_TOPRIGHT");
                return RANGES_FROM_TOPRIGHT;
            }
        }
        // Log.v("calculateDirection", "RANGES_FROM_CENTER");
        return RANGES_FROM_CENTER;
    }


    @Override
    protected void initcontrol() {
        setTag(Directionvalue.CENTERED.toString());
        getDrawable().setLevel(ROTATION_CENTER);
        super.initcontrol();
    }

    boolean isPointingEast() {
        return DIRECTION_IS_RIGHT.contains(getLastDirection());
    }
    boolean isPointingWest() {
        return DIRECTION_IS_LEFT.contains(getLastDirection());
    }
    private Float mDownX = null;
    private Float mDownY = null;
    private void press(final float x, final float y, final boolean firstDown) {
        super.press(x, y);
        if (firstDown) {
            mDownX  = x;
            mDownY = y;
        }

    }
    @Override
    public void moveTo(final float x, final float y) {
        press(x, y, false);
    }
    @Override
    public void press(final float x, final float y) {
        press(x, y, true);
    }

    @Override
    public void release() {
        mDownX = null;
        mDownY = null;
        super.release();
    }
}
