package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.JoystickFireView.STATES_PRESSED;
import static de.rainerhock.eightbitwonders.JoystickFireView.STATES_UNPRESSED;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.ImageButton;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.HashMap;
import java.util.Map;

public final class JoystickSecondaryView extends RelativeLayout {
    private ImageButton mButton;
    private TouchJoystick mJoystick = null;
    private TextView mText;

    /**
     * Simple constructor to use when creating a JoystickSecondaryView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public JoystickSecondaryView(final Context context) {
        super(context);
        init();
    }
    /**
     * Constructor that is called when inflating a JoystickSecondaryView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickSecondaryView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickSecondaryView(final Context context, final AttributeSet attrs,
                                 final int defStyle) {
        super(context, attrs, defStyle);
        init();
    }
    private void init() {
        LayoutInflater inflater = (LayoutInflater) getContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        inflater.inflate(R.layout.view_secondary_joystick_button, this, true);
        mRadius = getResources().getDimensionPixelSize(R.dimen.hamburger_size);
        mButton = findViewById(R.id.ib_secondary_function);
        mText = findViewById(R.id.tv_secondary_function);
        setEnabled(true);


    }
    private int mRadius = 0;
    @Override
    protected void onLayout(final boolean changed, final int left, final int top,
                            final int right, final int bottom) {
        if (changed) {
            mRadius = (bottom - top) / 2;
        }
        super.onLayout(changed, left, top, right, bottom);
    }
    private void setButtonPressed(final boolean pressed) {
        if (mJoystick != null) {
            mJoystick.setSecondaryButtonPressed(pressed);
        }
        mButton.getDrawable().setState(pressed ? STATES_PRESSED : STATES_UNPRESSED);
        setTag(pressed ? "pressed" : "unpressed");

    }
    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouchEvent(final MotionEvent event) {
        int action = event.getAction();
        if (action == MotionEvent.ACTION_DOWN) {
            setButtonPressed(Math.sqrt(((event.getX() - mRadius)
                    * (event.getX() - mRadius)
                    + (event.getY() - mRadius)
                    * (event.getY() - mRadius))) < mRadius);
        }
        if (action == MotionEvent.ACTION_UP) {
            setButtonPressed(false);
            updateText();
        }
        return true;
    }
    private static final Map<Joystick.SecondaryUsage, Integer> LABELS
            = new HashMap<Joystick.SecondaryUsage, Integer>() {{
                put(Joystick.SecondaryUsage.AUTOFIRE_TOGGLE, R.string.autofire_off_symbol);
        put(Joystick.SecondaryUsage.AUTOFIRE, R.string.autofire_symbol);
                put(Joystick.SecondaryUsage.UP, R.string.up_symbol);
        put(Joystick.SecondaryUsage.DOWN, R.string.down_symbol);
        put(Joystick.SecondaryUsage.LEFT, R.string.left_symbol);
        put(Joystick.SecondaryUsage.RIGHT, R.string.right_symbol);
        put(Joystick.SecondaryUsage.FIRE, R.string.empty);
    }};
    private void updateText() {
        int  res = R.string.empty;
        Joystick.SecondaryUsage usage = mJoystick.getSecondaryButtonUsage();
        if (usage == Joystick.SecondaryUsage.AUTOFIRE_TOGGLE
                && mJoystick.isAutofiring()) {
            res = R.string.autofire_on_symbol;
        } else {
            if (LABELS.containsKey(usage)) {
                Integer value = LABELS.get(usage);
                if (value != null) {
                    res = value;
                }
            }
        }
        mText.setText(res);
    }
    @SuppressLint("ClickableViewAccessibility")
    void setJoystick(final TouchJoystick touchJoystick) {
        mJoystick = touchJoystick;
        if (touchJoystick.getSecondaryButtonUsage() == Joystick.SecondaryUsage.FIRE) {
            setVisibility(View.GONE);
            findViewById(R.id.ib_secondary_function).setOnTouchListener((v, event) -> false);
        } else {
            setVisibility(View.VISIBLE);
        }
        updateText();
    }
    @Override
    public boolean dispatchTouchEvent(final MotionEvent event) {
        return onTouchEvent(event);
    }
}
