//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

/**
 * View for the gear-shift of a virtual-wheel joystick.
 */

public final class JoystickGearshiftView extends RelativeLayout
        implements TouchDisplayRelativeLayout.TouchDisplayElement {
    /**
     * Constructor that is called when inflating a JoystickGearshiftView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickGearshiftView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickGearshiftView(final Context context,
                                 final AttributeSet attrs,
                                 final int defStyle) {
        super(context, attrs, defStyle);
        init();
    }
    private int mFireOrientation = View.LAYOUT_DIRECTION_RTL;
    void setFireOrientation(final int val) {
        mFireOrientation = val;
        drawLever(mDirection, mFirePressed);

    }
    private boolean mFirePressed = false;
    private Direction mDirection = Direction.CENTER;

    boolean isPointingNorth() {
        return mDirection == Direction.UP;
    }
    boolean isPointingSouth() {
        return mDirection == Direction.DOWN;
    }
    boolean isButtonPressed() {
        return mFirePressed;
    }
    private static final float MIDDLE = 0.5f;
    private static final float UPPER = -0.25f;
    private static final float LOWER = 0.75f;
    @Override
    public void press(final float x, final float y) {
        boolean fire = false;
        Direction direction = Direction.CENTER;
        if (x < MIDDLE && mFireOrientation == View.LAYOUT_DIRECTION_LTR) {
            fire = true;
        } else if (x  > MIDDLE && mFireOrientation == View.LAYOUT_DIRECTION_RTL) {
            fire = true;
        }
        if (y < UPPER) {
            direction = Direction.UP;
        } else if (y > LOWER) {
            direction = Direction.DOWN;
        }
        if (direction != mDirection || fire != mFirePressed) {
            drawLever(direction, fire);
            mDirection = direction;
            mFirePressed = fire;
            if (mJoystick != null) {
                mJoystick.onChanged();
            }
        }
        String tag = direction.toString();
        if (fire) {
            tag = tag + "|FIRE";
        }
        setTag(tag);
    }

    @Override
    public void release() {
        if (Direction.CENTER != mDirection || mFirePressed) {
            drawLever(Direction.CENTER, false);
            mDirection = Direction.CENTER;
            mFirePressed = false;
            if (mJoystick != null) {
                mJoystick.onChanged();
            }
        }
        setTag(Direction.CENTER.toString());
    }

    @Override
    public void moveTo(final float x, final float y) {
        press(x, y);
    }

    private enum Direction { UP, DOWN, CENTER }
    private void drawLever(final Direction direction, final boolean fire) {
        RelativeLayout.LayoutParams lp =
                new RelativeLayout.LayoutParams(findViewById(R.id.iv_lever).getLayoutParams());
        switch (direction) {
            case UP:
                lp.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                break;
            case DOWN:
                lp.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                break;
            default:
                lp.addRule(RelativeLayout.CENTER_VERTICAL);
                break;
        }
        if (fire == (mFireOrientation == View.LAYOUT_DIRECTION_LTR)) {
            lp.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        } else {
            lp.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        }
        findViewById(R.id.iv_lever).setLayoutParams(lp);
    }

    @SuppressLint("ClickableViewAccessibility")

    private void init() {
        setTag(Direction.CENTER.toString());
        inflate(getContext(), R.layout.gearshift, this);
        drawLever(Direction.CENTER, false);
        setLongClickable(false);
    }
    private Joystick mJoystick = null;
    void setJoystick(final WheelJoystick value) {
        mJoystick = value;

    }
}
