//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.res.Configuration;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.View;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
@Keep
abstract class Joystick implements Serializable, Comparable<Joystick> {

    protected static final  Set<Emulation.InputButton> NO_PRESSED_BUTTON_KEYS
            = new HashSet<>();

    protected static final int PSEUDO_KEYCODE_FIREBUTTON = KeyEvent.getMaxKeyCode() + 1;
    protected static final int PSEUDO_KEYCODE_GEARSHIFT = PSEUDO_KEYCODE_FIREBUTTON + 1;
    protected static final int PSEUDO_KEYCODE_PADDLE_X = PSEUDO_KEYCODE_GEARSHIFT + 1;
    protected static final int PSEUDO_KEYCODE_PADDLE_Y = PSEUDO_KEYCODE_PADDLE_X + 1;
    static final Map<Integer, Integer> MAPPED_BUTTONS = new HashMap<Integer, Integer>() {{
        put(KeyEvent.KEYCODE_BUTTON_A, KeyEvent.KEYCODE_BUTTON_A);
        put(KeyEvent.KEYCODE_BUTTON_2, KeyEvent.KEYCODE_BUTTON_A);
        put(KeyEvent.KEYCODE_BUTTON_B, KeyEvent.KEYCODE_BUTTON_B);
        put(KeyEvent.KEYCODE_BUTTON_3, KeyEvent.KEYCODE_BUTTON_B);
        put(KeyEvent.KEYCODE_BUTTON_C, KeyEvent.KEYCODE_BUTTON_C);
        put(KeyEvent.KEYCODE_BUTTON_X, KeyEvent.KEYCODE_BUTTON_X);
        put(KeyEvent.KEYCODE_BUTTON_1, KeyEvent.KEYCODE_BUTTON_X);
        put(KeyEvent.KEYCODE_BUTTON_Y, KeyEvent.KEYCODE_BUTTON_Y);
        put(KeyEvent.KEYCODE_BUTTON_4, KeyEvent.KEYCODE_BUTTON_Y);
        put(KeyEvent.KEYCODE_BUTTON_L1, KeyEvent.KEYCODE_BUTTON_L1);
        put(KeyEvent.KEYCODE_BUTTON_5, KeyEvent.KEYCODE_BUTTON_L1);

        put(KeyEvent.KEYCODE_BUTTON_R1, KeyEvent.KEYCODE_BUTTON_R1);
        put(KeyEvent.KEYCODE_BUTTON_6, KeyEvent.KEYCODE_BUTTON_R1);

        put(KeyEvent.KEYCODE_BUTTON_L2, KeyEvent.KEYCODE_BUTTON_L2);
        put(KeyEvent.KEYCODE_BUTTON_7, KeyEvent.KEYCODE_BUTTON_L2);
        put(KeyEvent.KEYCODE_BUTTON_R2, KeyEvent.KEYCODE_BUTTON_R2);
        put(KeyEvent.KEYCODE_BUTTON_8, KeyEvent.KEYCODE_BUTTON_R2);
        put(KeyEvent.KEYCODE_BUTTON_THUMBL, KeyEvent.KEYCODE_BUTTON_THUMBL);
        put(KeyEvent.KEYCODE_BUTTON_THUMBR, KeyEvent.KEYCODE_BUTTON_THUMBR);
    }};
    protected static final List<Integer> SECONDARY_KEYS = Arrays.asList(
            KeyEvent.KEYCODE_BUTTON_X, KeyEvent.KEYCODE_BUTTON_B,
            KeyEvent.KEYCODE_BUTTON_Y, KeyEvent.KEYCODE_BUTTON_C,
            KeyEvent.KEYCODE_BUTTON_THUMBL, KeyEvent.KEYCODE_BUTTON_THUMBR,
            KeyEvent.KEYCODE_BUTTON_R1, KeyEvent.KEYCODE_BUTTON_L1,
            KeyEvent.KEYCODE_BUTTON_1, KeyEvent.KEYCODE_BUTTON_3, KeyEvent.KEYCODE_BUTTON_4,
            KeyEvent.KEYCODE_BUTTON_5, KeyEvent.KEYCODE_BUTTON_6);
    protected static final int PORT_NOT_CONNECTED = -1;
    static final String CURRENTDEVICETYPE = "_CURRENTDEVICETYPE_";

    private final Set<Integer> mButtonsPressed = new HashSet<>();

    private JoystickListener mListener = null;
    private Emulation.GamepadFunctions mGamepadListener = null;
    private boolean mLockDiagonals = false;
    private EmulationActivity mActivity;
    private MultiplexerJoystick mMuxer = null;
    private Emulation.InputDeviceType mCurrentDeviceType = Emulation.InputDeviceType.DSTICK;
    private SecondaryUsage mSecondaryButtonUsage = SecondaryUsage.FIRE;
    private boolean mSecondaryButtonPressed = false;
    private Integer mAutofireFrequency = 1;
    private Thread mAutofireThread = null;
    private boolean mAutofireThreadRunning = false;

    protected SecondaryUsage getSecondaryButtonUsage() {
        return mSecondaryButtonUsage;
    }

    abstract String getId();

    abstract String getHardwareDescriptor();
    static final int VIRTUAL_HARDWARE_ID = -1;
    static final int KEYBOARD_HARDWARE_ID = -2;
    static final int MULTIPLEXER_HARDWARE_ID = -3;
    abstract int getHardwareId();

    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    void connect(final EmulationActivity activity, final JoystickListener listener,
                 final Emulation.GamepadFunctions gamepadListener,
                 final MultiplexerJoystick muxer) {
        mListener = listener;
        mGamepadListener = gamepadListener;
        mActivity = activity;
        mMuxer = muxer;
        setPortnumber(mPortnumber);

    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    protected Emulation.GamepadFunctions getmGamepadListener() {
        return mGamepadListener;
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    void disconnect() {
        mListener = null;
        mMuxer = null;
        reset();
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    protected void onChanged() {

        notifyListener();
    }
    private boolean mAutofireNowPressed = false;
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    void readUseropts(final BaseActivity activity) {
        try {
            int port = activity.getCurrentUseropts()
                    .getIntegerValue("JOYSTICK_" + getId() + "_PORT", PORT_NOT_CONNECTED);
            setPortnumber(port);
        } catch (NumberFormatException e) {
            setPortnumber(PORT_NOT_CONNECTED);
        }
        String autofireFrequencyKey = "JOYSTICK_AUTOFIRE_FREQ_" + getPortnumber();
        mAutofireFrequency = activity.getCurrentUseropts().getIntegerValue(autofireFrequencyKey, 1);
        String secondaryButtonKey = "JOYSTICK_SECONDARY_BUTTON_USAGE_" + getPortnumber();
        mSecondaryButtonUsage = SecondaryUsage.valueOf(activity.getCurrentUseropts().getStringValue(
                secondaryButtonKey, "FIRE"));
        String diagonalsKey = activity.getResources()
                .getString(R.string.key_joystick_lock_diagonals) + "_" + getPortnumber();
        mLockDiagonals = activity.getCurrentUseropts().getBooleanValue(diagonalsKey, false);
        if (mActivity != null && mSecondaryButtonUsage != SecondaryUsage.AUTOFIRE_TOGGLE) {
            mActivity.getViewModel().setJoystickAutofireToggled(mPortnumber, false);
            setAutofiring(false);
        }
        mCurrentDeviceType = Emulation.InputDeviceType.valueOf(
                activity.getCurrentUseropts().getStringValue(
                        "JOYSTICK_DEVICETYPE_" + getPortnumber(),
                        String.valueOf(getCurrentDeviceType())));
    }
    private static final long HALF_A_SECOND = 500;
    void setAutofiring(final boolean value) {
        if (value) {

            if (mAutofireThreadRunning) {
                mAutofireThreadRunning = false;
                mAutofireThread.interrupt();
                mAutofireThread = null;
            }
            mAutofireNowPressed = true;
            notifyListener();
            mAutofireThread = new Thread(() -> {
                mAutofireThreadRunning = true;
                long interval = HALF_A_SECOND / mAutofireFrequency;
                while (mAutofireThreadRunning) {
                    try {
                        //noinspection BusyWait
                        Thread.sleep(interval);
                        mAutofireNowPressed = !mAutofireNowPressed;
                        notifyListener();

                    } catch (InterruptedException e) {
                        mAutofireNowPressed = false;
                        notifyListener();
                        return;
                    }
                }
            });
            mAutofireThread.start();
        } else {
            mAutofireNowPressed = false;
            notifyListener();
            if (mAutofireThreadRunning) {
                mAutofireThreadRunning = false;
                mAutofireThread.interrupt();
                mAutofireThread = null;
            }
        }
    }

    @Override
    public int compareTo(Joystick o) {
        return Integer.compare(getPortnumber(), o.getPortnumber());
    }

    boolean isAutofiring() {
        return mAutofireThreadRunning;
    }
    abstract static class InputDeviceConfigFragment extends Fragment {
        private SettingsController mSettingsController = null;

        void setSettingsController(final SettingsController sc) {
            mSettingsController = sc;
        }

        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            if (mSettingsController != null) {
                Set<View> views = SettingsActivity.addWidget(view, new HashSet<>());
                for (View v : views) {
                    if (v.getTag() != null && v.getId() != View.NO_ID) {
                        mSettingsController.addWidget(v.getId(), v);
                        mSettingsController.initWidget(v);
                    }
                }
            }

        }


    }
    abstract boolean canLockDiagonals();
    abstract boolean hasSecondaryButton();
    abstract InputDeviceConfigFragment getDstickConfigFragment();
    protected final boolean isWithLockedDiagonals() {
        if (canLockDiagonals()) {
            return mLockDiagonals;
        } else {
            return false;
        }
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    protected boolean isListenerAttached() {
        return mListener != null;
    }

    /**
     * Return all emulated devices types a real device supports.
     * This depends if the device just has buttons for the directions or analog axises
     * or the number of buttons.
     * @return List of Emulation.InputDeviceType indicating supported emulated device types.
     */
    @NonNull List<Emulation.InputDeviceType> getSupportedDeviceTypes() {
        return Collections.singletonList(Emulation.InputDeviceType.DSTICK);
    }
    final Emulation.InputDeviceType getCurrentDeviceType() {
        if (mMuxer != null && mMuxer.getPortnumber() != PORT_NOT_CONNECTED) {
            return mMuxer.getCurrentDeviceType();
        }
        return mCurrentDeviceType;
    }
    void setCurrentdevicetype(final Emulation.InputDeviceType idt) {
        mCurrentDeviceType = idt;
    }
    /**
     * Map real buttons to emulated input device buttons.
     * @param realButtons Set of Keycodes from #KeyEvent static members.
     * @return Set of matching emulation keycodes, e.g. Emulation.InputButton.FIRE
     * for fire button.
     */
    protected Set<Emulation.InputButton> mapRealButtons(final Set<Integer> realButtons) {
        return (realButtons.isEmpty() && !mAutofireNowPressed
                ? Collections.emptySet() : Collections.singleton(Emulation.InputButton.FIRE));
    }
    protected void setSecondaryButtonPressed(final boolean pressed) {
        mSecondaryButtonPressed = pressed;
        if (mSecondaryButtonUsage == SecondaryUsage.AUTOFIRE) {
            setAutofiring(pressed);
            return;
        }
        if (mSecondaryButtonUsage == SecondaryUsage.AUTOFIRE_TOGGLE && pressed) {
            boolean wasAutofiring = isAutofiring();
            setAutofiring(!isAutofiring());
            mActivity.getViewModel().setJoystickAutofireToggled(mPortnumber, !wasAutofiring);
            return;
        }
        notifyListener();
    }
    private float mergeSecondaryButtonToX(final float x) {
        if (mSecondaryButtonPressed) {
            if (mSecondaryButtonUsage == SecondaryUsage.LEFT) {
                return -FULL_AMPLITUDE;
            }
            if (mSecondaryButtonUsage == SecondaryUsage.RIGHT) {
                return FULL_AMPLITUDE;
            }
        }
        return x;
    }
    private float mergeSecondaryButtonToY(final float y) {
        if (mSecondaryButtonPressed) {
            if (mSecondaryButtonUsage == SecondaryUsage.UP) {
                return -FULL_AMPLITUDE;
            }
            if (mSecondaryButtonUsage == SecondaryUsage.DOWN) {
                return FULL_AMPLITUDE;
            }
        }
        return y;
    }

    protected
    enum SecondaryUsage { FIRE, AUTOFIRE, AUTOFIRE_TOGGLE, UP, LEFT, RIGHT, DOWN }
    enum ValueType { POSITION, MOVEMENT }
    ValueType getValueType() {
        return ValueType.POSITION;
    }
    protected final boolean notifyListener() {
        int portnumber = getPortnumber();
        Joystick handler;
        if (portnumber == Joystick.PORT_NOT_CONNECTED && mMuxer != null) {
            handler = mMuxer;
        } else {
            handler = this;
        }
        portnumber = handler.getPortnumber();
        if (isListenerAttached() && portnumber != Joystick.PORT_NOT_CONNECTED) {
            mListener.onJoystickChanged(
                    handler.getId(),
                    handler.getCurrentDeviceType(),
                    portnumber,
                    getValueType() == ValueType.POSITION,
                    mergeSecondaryButtonToX(mXValue), mergeSecondaryButtonToY(mYValue),
                    mapRealButtons(mButtonsPressed));
            return true;
        }
        return false;
    }
    private float mXValue = 0f;
    protected final void setXValue(final float value) {
        mXValue = value;
    }
    private float mYValue = 0f;
    protected final void setYValue(final float value) {
        mYValue = value;
    }
    protected static final float FULL_AMPLITUDE = 1f;
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    protected void setRealButtonsPressed(final Set<Integer> value) {
        mButtonsPressed.clear();
        mButtonsPressed.addAll(value);
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES

    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    protected final Set<Integer> getRealButtonsPressed() {
        return mButtonsPressed;
    }

    /**
     * Get current x value.
     * @return range between -1 (most left position) and +1 (most right position)
     */
    protected float getCurrentXValue() {
        return mXValue;
    }
    /**
     * Get current y value.
     * @return range between -1 (most top position) and +1 (most bottom position)
     */
    protected float getCurrentYValue() {
        return mYValue;
    }
    protected void reset() {
        mXValue = 0f;
        mYValue = 0f;
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    String getHardwareButtons() {
        StringBuilder ret = new StringBuilder();
        for (int key = 0; key < KeyEvent.getMaxKeyCode(); key++) {
            if (deviceHasButtonWithKeycode(key)) {
                if (ret.length() == 0) {
                    ret.append("[");
                } else {
                    ret.append(", ");
                }
                ret.append(KeyEvent.keyCodeToString(key));
            }
        }
        if (this instanceof VirtualJoystick) {
            if (ret.length() == 0) {
                ret.append("[FIRE");
            } else {
                ret.append(", FIRE");
            }
        }
        ret.append("]");
        return ret.toString();
    }

    private int mPortnumber = PORT_NOT_CONNECTED;

    final int getPortnumber() {
        return mPortnumber;
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    final void setPortnumber(final int number) {
        if (number != PORT_NOT_CONNECTED && canLockDiagonals() && mActivity != null) {
            Useropts opts = mActivity.getCurrentUseropts();
            mLockDiagonals = opts.getBooleanValue("JOYSTICK_LOCK_DIAGONALS_" + number,
                    mActivity.getViewModel().getConfiguration().getProperty(
                            "touchjoystick_diagonals_locked", "").equals("1"));
        }
        mPortnumber = number;
    }

    interface UnitTestClass {
        @NonNull
        List<Joystick> getTestJoysticks();
        boolean useThisJoystick(Joystick joy);
    }
    private static class StateBuffer implements JoystickListener {
        private final JoystickListener mOriginalListener;
        private final int mPortNumber;
        private final String mJoystickId;
        private final Emulation.InputDeviceType mCurrentDevicetype;
        private float mXvalue = 0f;
        private float mYvalue = 0f;
        private final Set<Emulation.InputButton> mButtons = new HashSet<>();
        private final boolean mIsaboluteMovement;

        StateBuffer(final Joystick joystick, final JoystickListener parent, final int portNumber) {
            mOriginalListener = parent;
            mPortNumber = portNumber;
            mJoystickId = joystick.getId();
            mCurrentDevicetype = joystick.getCurrentDeviceType();
            mIsaboluteMovement = joystick.getValueType() == ValueType.POSITION;

        }
        @Override
        public void onJoystickChanged(final String joyId, final Emulation.InputDeviceType type,
                                      final int portnumber,
                                      final boolean isAbsoluteMovement,
                                      final float xvalue, final float yvalue,
                                      final Set<Emulation.InputButton> buttons) {
            mXvalue = xvalue;
            mYvalue = yvalue;
            mButtons.clear();
            mButtons.addAll(buttons);
        }

        JoystickListener restoreOldState() {
            mOriginalListener.onJoystickChanged(
                    mJoystickId, mCurrentDevicetype, mPortNumber,
                    mIsaboluteMovement, mXvalue, mYvalue, mButtons);
            return mOriginalListener;
        }
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    void startStateBuffering() {
        if (getPortnumber() != PORT_NOT_CONNECTED && mListener != null) {
            mListener = new StateBuffer(this, mListener, getPortnumber());
        }

    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    boolean isBuffered() {
        return (mListener instanceof StateBuffer);
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    void readFromStateBuffer() {
        if (mListener instanceof StateBuffer) {
            mListener = ((StateBuffer) mListener).restoreOldState();
        }
    }
    //CHECKSTYLE DISABLE DesignForExtension FOR 1 LINES
    abstract boolean deviceHasButtonWithKeycode(int keycode);
    final boolean conflictsWith(final Joystick other) {
        if (other != null) {
            return (other == this)
                    || (other.getHardwareDescriptor() != null
                    && this.getHardwareDescriptor() != null
                    && other.getHardwareDescriptor().equals(this.getHardwareDescriptor()))
                    || this instanceof MultiplexerJoystick || other instanceof MultiplexerJoystick;
        } else {
            return  false;
        }
    }
    protected void onConfigurationChanged(@NonNull final Configuration newConfig) {
    }
    protected boolean isMuxerConnected() {
        return mMuxer != null
                && mMuxer.getPortnumber() != Joystick.PORT_NOT_CONNECTED
                && getSupportedDeviceTypes().contains(mMuxer.getCurrentDeviceType());
    }
}
