//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContract;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.yaml.snakeyaml.error.YAMLException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class ImportFileActivity extends EmulationLauncherActivity {
    private static final int CHUNKSIZE = 1024 * 1024;
    @Override
    protected final void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_importfile);
        openAsZip(Objects.requireNonNull(getIntent().getData()));

    }
    private void openAsZip(@NonNull final Uri data) {
        Thread t = new Thread(() -> {
            boolean ok = false;
            try {
                if (data.getScheme() != null) {
                    InputStream is = getInputStream(data);
                    if (is != null) {
                        byte[] bytes = new byte[is.available()];
                        //noinspection ResultOfMethodCallIgnored
                        is.read(bytes);
                        is.close();
                        is = new ByteArrayInputStream(bytes);

                        Map<String, ByteArrayOutputStream> contents = new HashMap<>();
                        Map<String, ByteArrayOutputStream> savestates = new HashMap<>();
                        final Properties props = getProperties(this, is, savestates, contents);
                        if (props != null && !props.isEmpty()) {

                            ok = props.containsKey("name");
                            runOnUiThread(() -> {
                                if (props.containsKey("name")) {

                                    String title = props.getProperty("name");
                                    if (title != null) {
                                        TextView tv = findViewById(R.id.tv_title);
                                        tv.setText(title);
                                        tv.setVisibility(View.VISIBLE);

                                    }
                                }
                                if (props.containsKey("url")) {
                                    Button b = findViewById(R.id.bn_website);
                                    b.setVisibility(View.VISIBLE);
                                    b.setOnClickListener(view -> {
                                        Intent intent = new Intent();
                                        intent.setAction(Intent.ACTION_VIEW);
                                        intent.setData(Uri.parse(props.getProperty("url")));
                                        startActivity(intent);
                                    });
                                }
                                Bitmap bmp = null;
                                if (props.containsKey("image")) {
                                    String imagename = props.getProperty("image");
                                    if (contents.containsKey(imagename)) {
                                        ByteArrayOutputStream baos = contents.get(
                                                props.getProperty("image"));
                                        if (baos != null) {
                                            byte[] imageData = baos.toByteArray();
                                            bmp = BitmapFactory.decodeStream(
                                                    new ByteArrayInputStream(imageData));
                                        }
                                    }
                                }
                                ImageView iv = findViewById(R.id.iv_screenshot);
                                if (bmp != null) {
                                    iv.setImageBitmap(bmp);
                                    iv.setVisibility(View.VISIBLE);
                                }
                                Button bnRun = findViewById(R.id.bn_run);
                                bnRun.setOnClickListener(view -> {
                                    //final EmulationConfiguration conf;
                                    try {
                                        final EmulationConfiguration conf = ConfigurationFactory
                                                .createOneTimeZipConfiguration(
                                                        ImportFileActivity.this, contents,
                                                        savestates);
                                        cancelRunningEmulation(null, conf.getName(),
                                                () -> startEmulation(conf), () -> {
                                                });

                                    } catch (IOException e) {
                                        if (BuildConfig.DEBUG) {
                                            throw new RuntimeException(e);
                                        } else {
                                            runOnUiThread(() -> showErrorDialog(
                                                    getResources().getString(R.string
                                                            .import_local_content),
                                                    e.getMessage()));
                                        }
                                    }
                                });
                                Button bnAdd = findViewById(R.id.bn_add);
                                bnAdd.setOnClickListener(view -> {
                                    try {
                                        storeZip(contents, props, savestates);
                                    } catch (IOException e) {
                                        if (BuildConfig.DEBUG) {
                                            throw new RuntimeException(e);
                                        }
                                    }
                                });
                            });

                        } else if (Rp9Configuration.isRp9File(
                                getContentResolver().openInputStream(data))) {
                            final Rp9Configuration rp9 = new Rp9Configuration(this, bytes);
                            if (rp9.validate()) {
                                ok = true;
                                Button bnAdd = findViewById(R.id.bn_add);
                                bnAdd.setOnClickListener(view -> {
                                    try {
                                        rp9.add();
                                    } catch (Rp9Configuration.Rp9ImportException e) {
                                        runOnUiThread(() -> showErrorDialog(getResources()
                                                        .getString(R.string.import_local_content),
                                                e.getMessage()));
                                    }
                                });
                                Button bnRun = findViewById(R.id.bn_run);
                                bnRun.setOnClickListener(view -> {
                                    try {
                                        EmulationConfiguration conf = rp9
                                                .createOneTimeConfiguration();
                                        cancelRunningEmulation(null, conf.getName(),
                                                () -> startEmulation(conf), () -> {
                                                });
                                    } catch (Rp9Configuration.Rp9ImportException e) {
                                        runOnUiThread(() -> showErrorDialog(
                                                getResources().getString(
                                                        R.string.import_local_content),
                                                e.getMessage()));
                                    }
                                });
                            }
                        }
                        is.close();
                    }
                }
            } catch (IOException e) {
                // no action required ok is already false.
            }
            if (!ok) {
                runOnUiThread(() -> {
                findViewById(R.id.bn_add).setVisibility(View.GONE);
                showErrorDialog(getTitle().toString(), getString(R.string.import_error,
                                getIntent().getData() != null
                                        && getIntent().getData().getPath() != null
                                        ? Uri.decode(
                                                new File(getIntent().getData().getPath()).getName())
                                        : getResources().getString(R.string.IDMS_FILE)),
                        this::finish);
                });
            }
        });
        t.start();
    }

    @Nullable
    private InputStream getInputStream(final @NonNull Uri data) throws FileNotFoundException {
        InputStream is;
        if (data.getScheme() != null) {
            switch (data.getScheme()) {
                case "http":
                case "https":
                    is = null;
                    for (DownloaderFactory.HttpRequest req : getHttpRequests()) {
                        try {
                            DownloaderFactory.HttpResult res = req.execute(
                                    new URL(data.toString()));
                            if (res.getResultcode()
                                    < HttpURLConnection.HTTP_MULT_CHOICE
                                    && res.getResultcode()
                                    >= HttpURLConnection.HTTP_OK
                                    && res.getBody() != null) {
                                is = new ByteArrayInputStream(res.getBody());
                                break;
                            }
                        } catch (IOException e) {
                            Log.v(getClass().getSimpleName(),
                                    "error downloading " + data, e);
                        }
                    }
                    break;
                default:
                    is = getContentResolver().openInputStream(data);
            }
        } else {
            is = null;
        }
        return is;
    }
    static Properties checkProperties(final InputStream is) throws IOException {
        ZipInputStream zis = new ZipInputStream(is);
        ZipEntry ze;

        byte[] buffer = new byte[CHUNKSIZE];
        int count;
        while ((ze = zis.getNextEntry()) != null) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            while ((count = zis.read(buffer)) >= 0) {
                baos.write(buffer, 0, count);
            }
            if (ze.getName().equals("properties")) {
                Properties p = new Properties();
                p.load(new ByteArrayInputStream(baos.toByteArray()));
                return p;
            }
        }
        return null;

    }
    static Properties getProperties(final Context context,
                                    final InputStream is,
                                    final Map<String, ByteArrayOutputStream> savestates,
                                    final Map<String, ByteArrayOutputStream> contents)
            throws IOException {
        Properties p = new Properties();
        if (is != null) {
            ZipInputStream zis = new ZipInputStream(is);
            ZipEntry ze;

            byte[] buffer = new byte[CHUNKSIZE];
            int count;
            while ((ze = zis.getNextEntry()) != null) {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                while ((count = zis.read(buffer)) >= 0) {
                    baos.write(buffer, 0, count);
                }
                if (ze.getName().startsWith("__savestates__/")) {
                    savestates.put(ze.getName()
                            .replaceFirst("__savestates__/", ""), baos);
                } else if (ze.getName().equals("__settings__.yml")) {
                    try {
                        new Useropts().readFromYaml(context, new InputStreamReader(
                                new ByteArrayInputStream(baos.toByteArray())));
                    } catch (YAMLException e) {
                        // so what, too late to do anything.
                    }
                } else {
                    contents.put(ze.getName(), baos);
                }
                if (ze.getName().equals("properties")) {
                    p = new Properties();
                    p.load(new ByteArrayInputStream(baos.toByteArray()));
                }
            }
        }
        return p;
    }

    @Override
    final ActivityResultLauncher<Intent> getEmulationContract() {
        return registerForActivityResult(
                new ActivityResultContract<Intent, Intent>() {

                    @Override
                    public Intent parseResult(final int i, final @Nullable Intent intent) {
                        return intent;
                    }

                    @NonNull
                    @Override
                    public Intent createIntent(final @NonNull Context context,
                                               final Intent intent) {
                        return intent;
                    }
                },
                result -> {
                    EmulationLauncherActivity.setEmulationRunning(false);
                    if (result != null) {
                        if (result.getStringExtra("error-message") != null) {
                            showErrorDialog(null, result.getStringExtra("error-message"));
                        }
                    }
                });
    }
}
