//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.os.Bundle;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.SeekBar;

import androidx.annotation.NonNull;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

class GameControllerJoystick extends AnalogController
        implements View.OnKeyListener, View.OnGenericMotionListener {

    private static final int CURRENT_POS = -1;
    private final float mXFuzz;
    private final float mYFuzz;

    private final ValueType mValueType;

    @Override
    ValueType getValueType() {
        return mValueType;
    }

    void setOtherControllerjoysticks(final GameControllerJoystick[] others) {
        mOtherJoysticks = new LinkedList<>();
        for (GameControllerJoystick j : others) {
            if (j != null) {
                mOtherJoysticks.add(j);
            }
        }
    }
    private static final List<Integer> PRIMARY_BUTTON_TRIGGER_AXISSES = Arrays.asList(
      MotionEvent.AXIS_LTRIGGER, MotionEvent.AXIS_RTRIGGER);


    private static final List<GameControllerJoystick> INSTANCES = new LinkedList<>();

    static GameControllerJoystick getInstance(final Context ctx,
                                              final InputDevice dev,
                                              final int xAxis, final int yAxis,
                                              final Integer[] triggerAxes,
                                              final Integer[] triggerKeys,
                                              final ValueType type) {
        for (GameControllerJoystick c : INSTANCES) {
            if (c.mDevice == dev && c.mXaxis == xAxis && c.mYaxis == yAxis) {
                return c;
            }
        }
        GameControllerJoystick c =
                new GameControllerJoystick(ctx, dev, xAxis, yAxis, triggerAxes, triggerKeys, type);
        INSTANCES.add(c);
        return c;
    }

    GameControllerJoystick(final Context ctx, final InputDevice dev,
                           final int xAxis, final int yAxis,
                           final Integer[] triggerAxes,
                           final Integer[] triggerKeys,
                           final ValueType valuetype) {
        mValueType = valuetype;
        mDevice = dev;
        mXaxis = xAxis;
        mYaxis = yAxis;
        if (dev != null) {
            mXFuzz = dev.getMotionRange(mXaxis).getFuzz();
            mYFuzz = dev.getMotionRange(mXaxis).getFuzz();
        } else {
            mXFuzz = 0;
            mYFuzz = 0;
        }

        mTriggerAxes = Arrays.asList(triggerAxes);
        mTriggerKeys = Arrays.asList(triggerKeys);

        mUseroptskeyDstickSensitivy = ctx.getResources()
                .getString(R.string.key_gamepad_sensitivity_dstick) + "_" + getId();
        switch (xAxis) {
            case MotionEvent.AXIS_X:
                mJoystickDescription =
                        String.format(" (%s)", ctx.getString(R.string.left_stick));
                break;
            case MotionEvent.AXIS_Z:
                mJoystickDescription =
                        String.format(" (%s)", ctx.getString(R.string.right_stick));
                break;
            case MotionEvent.AXIS_HAT_X:
                mJoystickDescription =
                        String.format(" (%s)", ctx.getString(R.string.dpad));
                break;
            default:
                mJoystickDescription = "";

        }

    }

    private List<GameControllerJoystick> mOtherJoysticks = null;
    private final InputDevice mDevice;
    private final int mXaxis;
    private final int mYaxis;
    private final List<Integer> mTriggerAxes;
    private final List<Integer> mTriggerKeys;
    private final String mJoystickDescription;

    @Override
    String getId() {
        return mDevice.getDescriptor() + "-" + mXaxis;
    }

    @Override
    String getHardwareDescriptor() {
        return mDevice.getDescriptor() + "-" + mXaxis;
    }

    @Override
    int getHardwareId() {
        return mDevice.getId();
    }

    @Override
    void readUseropts(final BaseActivity activity) {
        super.readUseropts(activity);
        Useropts useropts = activity.getCurrentUseropts();
        mSensitivityDstick = useropts.getIntegerValue(mUseroptskeyDstickSensitivy, activity
                .getResources().getInteger(R.integer.gamepad_sensitivity_max) / 2);
    }

    @Override
    boolean canLockDiagonals() {
        return !mTriggerAxes.isEmpty();
    }

    @Override
    boolean hasSecondaryButton() {
        for (int key :mTriggerKeys) {
            if (SECONDARY_KEYS.contains(key)) {
                return true;
            }
        }
        return false;
    }

    private final String mUseroptskeyDstickSensitivy;
    private int mSensitivityDstick = 0;

    @Override
    InputDeviceConfigFragment getDstickConfigFragment() {

        if (!mTriggerAxes.isEmpty()) {
            InputDeviceConfigFragment f = new AnalogDeviceSettingsFragment();
            Bundle b = new Bundle();
            b.putString("joystickId", getId());
            f.setArguments(b);
            return f;
        } else {
            return null;
        }
    }


    @Override
    boolean deviceHasButtonWithKeycode(final int keycode) {
        return mDevice.hasKeys(keycode)[0];

    }

    @NonNull
    @Override
    public String toString() {
        return mDevice.getName() + mJoystickDescription;
    }

    private final Set<Integer> mPressedKeys = new HashSet<>();
    protected boolean isMatchingHardwareDescriptor(final KeyEvent event) {
        return getHardwareDescriptor().equals(event.getDevice().getDescriptor() + "-" + mXaxis);
    }


    @Override
    public boolean onKey(final View v, final int keyCode, final KeyEvent event) {

        if (event.getDeviceId() == getHardwareId()) {
            if (mAllTriggerKeyCodes == null) {
                mAllTriggerKeyCodes = new LinkedList<>();
            }
            mAllTriggerKeyCodes.clear();
            mAllTriggerKeyCodes.addAll(mTriggerKeys);
            for (GameControllerJoystick other : mOtherJoysticks) {
                if (other.getPortnumber() == PORT_NOT_CONNECTED
                        || other.getPortnumber() == getPortnumber()) {
                    mAllTriggerKeyCodes.addAll(other.mTriggerKeys);
                    mOthersTriggerkeycodes.addAll(other.mTriggerKeys);
                }
            }
            if (mAllTriggerKeyCodes.contains(event.getKeyCode())) {
                if (SECONDARY_KEYS.contains(keyCode)
                        && getSecondaryButtonUsage() != SecondaryUsage.FIRE) {
                    setSecondaryButtonPressed(event.getAction() == KeyEvent.ACTION_DOWN);
                } else {
                    if (event.getAction() == KeyEvent.ACTION_DOWN) {
                        mPressedKeys.add(event.getKeyCode());
                    }
                    if (event.getAction() == KeyEvent.ACTION_UP) {
                        mPressedKeys.remove(event.getKeyCode());
                    }
                }
                if (mPrimaryAxisTriggered) {
                    mPressedKeys.add(Joystick.PSEUDO_KEYCODE_PADDLE_X);
                } else {
                    mPressedKeys.remove(Joystick.PSEUDO_KEYCODE_PADDLE_X);
                }
                if (mSecondaryAxisTriggered) {
                    mPressedKeys.add(Joystick.PSEUDO_KEYCODE_PADDLE_Y);
                } else {
                    mPressedKeys.remove(Joystick.PSEUDO_KEYCODE_PADDLE_Y);
                }

                if (mPrimaryAxisTriggered || mSecondaryAxisTriggered) {
                    mPressedKeys.add(Joystick.PSEUDO_KEYCODE_FIREBUTTON);
                } else {
                    mPressedKeys.remove(Joystick.PSEUDO_KEYCODE_FIREBUTTON);
                }
                setRealButtonsPressed(mPressedKeys);
                return notifyListener();
            }
            if (mOthersTriggerkeycodes.contains(keyCode)) {
                return false;
            }
            return mAllTriggerKeyCodes.contains(keyCode) && isMatchingHardwareDescriptor(event);
        } else {
            return false;
        }
    }


    protected float getFlatRange(final InputDevice device, final int axis, final int source) {
        if (device.getMotionRange(axis, source) != null) {
            return device.getMotionRange(axis, source).getFlat();
        }
        return 0;
    }
    protected float getCurrentValue(final MotionEvent event, final int axis, final int historyPos) {
        return historyPos < 0 ? event.getAxisValue(axis)
                : event.getHistoricalAxisValue(axis, historyPos);

    }
    private float getAxisValue(final MotionEvent event,
                               final InputDevice device,
                               final int axis,
                               final int historyPos) {
        InputDevice.MotionRange range;
        if (device != null) {
            range = device.getMotionRange(axis, event.getSource());
        } else {
            range = null;
        }
        float max;
        float min;
        if (range != null) {
            max = range.getMax();
            min = range.getMin();
        } else {
            try {
                Class<?> clz = Objects.requireNonNull(getClass().getClassLoader())
                        .loadClass("org.junit.Test");
                if (clz == null) {
                    return 0;
                }
            } catch (ClassNotFoundException e) {
                return 0;
            }
            max = 1.0f;
            min = -max;
        }
        float value = getCurrentValue(event, axis, historyPos);
        if (value != 0) {
            float flat;
            if (device != null) {
                flat = getFlatRange(device, axis, event.getSource());
            } else {
                flat = 0;
            }
            // CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
            if (value >= flat + ((max - flat) * ((70f - mSensitivityDstick) / 100f))) {
                return value;
            }
            // CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
            if (-value >= flat - ((min + flat) * ((70f - mSensitivityDstick) / 100f))) {
                return value;
            }
        }
        return 0;
    }

    private boolean mPrimaryAxisTriggered = false;
    private boolean mSecondaryAxisTriggered = false;

    private void processJoystickInput(final MotionEvent event,
                                      final int historyPos) {

        InputDevice inputDevice = event.getDevice();
        // Calculate the horizontal distance to move by
        // using the input value from one of these physical controls:
        // the left control stick, hat axis, or the right control stick.
        float x = getAxisValue(event, inputDevice, mXaxis, historyPos);
        float y = getAxisValue(event, inputDevice, mYaxis, historyPos);
        if (getCurrentDeviceType() == Emulation.InputDeviceType.DSTICK) {
            if (isWithLockedDiagonals()) {
                boolean xIsLarger = Math.abs(x) > Math.abs(y);
                x = xIsLarger ? x : 0;
                y = xIsLarger ? 0 : y;
            }
        }
        setXValue(x);
        setYValue(y);
    }
    private void readTriggerAxes(final MotionEvent event,
                                 final List<Integer> triggeraxes) {
        InputDevice inputDevice = event.getDevice();
        for (int axis : triggeraxes) {
            boolean val = getAxisValue(event, inputDevice, axis, CURRENT_POS) != 0;
            if (PRIMARY_BUTTON_TRIGGER_AXISSES.contains(axis)) {
                mPrimaryAxisTriggered = val;
            } else {
                mSecondaryAxisTriggered = val;
            }

        }
        if (mSecondaryAxisTriggered || mPrimaryAxisTriggered) {
            mPressedKeys.add(Joystick.PSEUDO_KEYCODE_FIREBUTTON);
        } else {
            mPressedKeys.remove(Joystick.PSEUDO_KEYCODE_FIREBUTTON);
        }
        setRealButtonsPressed(mPressedKeys);

    }
    private List<Integer> mAllTriggerAxes = null;
    private List<Integer> mAllTriggerKeyCodes = null;
     private final List<Integer> mOthersTriggerkeycodes = new LinkedList<>();

    @Override
    void connect(final EmulationActivity activity, final JoystickListener listener,
                 final Emulation.GamepadFunctions gamepadListener,
                 final MultiplexerJoystick muxer) {
        mAllTriggerAxes = null;
        mAllTriggerKeyCodes = null;
        super.connect(activity, listener, gamepadListener, muxer);
    }
    protected boolean isDeviceMatch(final InputDevice dev) {
        return dev == mDevice;
    }
    private static final float BASE_SPEED_FACTOR = 0.1f;

    @Override
    protected boolean showMouseUsageHint() {
        return false;
    }

    protected float getMovementFactor() {
        return BASE_SPEED_FACTOR * super.getMovementFactor();
    }

    private static final float FLOAT_TO_RANGE_VALUE = 0.1f;
    @Override
    public boolean onGenericMotion(final View v, final MotionEvent event) {
        if (isDeviceMatch(event.getDevice()) && isListenerAttached()) {
            if (mAllTriggerAxes == null) {
                mAllTriggerAxes = new LinkedList<>();
                mAllTriggerAxes.addAll(mTriggerAxes);
                for (GameControllerJoystick other : mOtherJoysticks) {
                    if (other.getPortnumber() == PORT_NOT_CONNECTED) {
                        mAllTriggerAxes.addAll(other.mTriggerAxes);
                    }
                }
            }
            float oldX = getCurrentXValue();
            float oldY = getCurrentYValue();
            Emulation.InputDeviceType type = getCurrentDeviceType();
            Set<Integer> oldbuttons = getRealButtonsPressed();

            float newX;
            float newY;

            if (type == Emulation.InputDeviceType.DSTICK) {
                for (int i = 0; i < event.getHistorySize(); i++) {
                    processJoystickInput(event, i);
                    //readTriggerAxes(event, i, mAllTriggerAxes);
                }
                processJoystickInput(event, CURRENT_POS);
                readTriggerAxes(event, mAllTriggerAxes);
                newX = getCurrentXValue();
                newY = getCurrentYValue();
            } else {
                InputDevice dev = event.getDevice();
                float x = event.getAxisValue(mXaxis);
                float y = event.getAxisValue(mYaxis);
                readTriggerAxes(event, mAllTriggerAxes);
                if (dev != null) {
                    InputDevice.MotionRange mrx = dev.getMotionRange(mXaxis, event.getSource());
                    InputDevice.MotionRange mry = dev.getMotionRange(mYaxis, event.getSource());
                    float xflat = mrx != null ? mrx.getFlat() : 0f;
                    float xrange = mrx != null ? mrx.getRange() : 2f;
                    if (xflat == 0) {
                        xflat = xrange * FLOAT_TO_RANGE_VALUE;
                    }
                    float yflat = mry != null ?  mry.getFlat() : 0f;
                    float yrange = mry != null ? mry.getRange() : 2f;
                    if (yflat == 0) {
                        yflat = yrange * FLOAT_TO_RANGE_VALUE;
                    }
                    if (Math.abs(x) > xflat) {
                        x = (x - Math.signum(x) * xflat) / (xrange - xflat);
                    } else {
                        x = 0;
                    }
                    if (Math.abs(y) > yflat) {
                        y = (y - Math.signum(y) * yflat) / (yrange - yflat);
                    } else {
                        y = 0;
                    }
                }
                float factor = getMovementFactor();
                newX = x * factor;
                newY = y * factor;
            }
            if (newX - oldX > mXFuzz || oldX - newX > mXFuzz
                    || newY - oldY > mYFuzz || oldY - newY > mYFuzz
                    || !oldbuttons.equals(getRealButtonsPressed())) {
                setXValue(newX);
                setYValue(newY);
                if (isListenerAttached()) {
                    return notifyListener();
                }
            }
            return false;
        }
        return false;
    }

    @Override
    @NonNull  List<Emulation.InputDeviceType> getSupportedDeviceTypes() {
        if (mXaxis == MotionEvent.AXIS_HAT_X) {
            return Collections.singletonList(Emulation.InputDeviceType.DSTICK);
        } else {
            return Arrays.asList(Emulation.InputDeviceType.DSTICK,
                    Emulation.InputDeviceType.MOUSE,
                    Emulation.InputDeviceType.PADDLE);

        }
    }

    public static final class AnalogDeviceSettingsFragment extends InputDeviceConfigFragment {
        AnalogDeviceSettingsFragment() {
            super();
        }

        @Override
        public View onCreateView(final LayoutInflater inflater, final ViewGroup container,
                                 final Bundle savedInstanceState) {
            View ret = inflater.inflate(R.layout.fragment_config_digital_controller,
                    container, false);
            SettingsActivity activity = ((SettingsActivity) requireActivity());
            SettingsController sc = activity.getSettingsController();
            SeekBar sb = ret.findViewById(R.id.sb_sensitivy);

            String progressTag = getResources().getString(R.string.key_gamepad_sensitivity_dstick)
                    + "_" + requireArguments().getString("joystickId");
            int progressId = View.generateViewId();
            sb.setProgress(sc.getCurrentValue(progressTag, sb.getProgress()));
            sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
                @Override
                public void onProgressChanged(final SeekBar seekBar, final int i, final boolean b) {
                    sc.storeValue(progressId, progressTag, sb.getProgress());
                }

                @Override
                public void onStartTrackingTouch(final SeekBar seekBar) {

                }

                @Override
                public void onStopTrackingTouch(final SeekBar seekBar) {

                }
            });
            return ret;
        }


    }
}
