//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static android.graphics.Bitmap.CompressFormat.PNG;

import android.annotation.SuppressLint;
import android.content.ClipData;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.OvalShape;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.SystemClock;
import android.text.Html;
import android.text.format.DateFormat;
import android.util.Log;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;

import androidx.appcompat.widget.PopupMenu;

import android.widget.ProgressBar;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.SeekBar;
import android.widget.Spinner;
import android.widget.TabHost;
import android.widget.TabWidget;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.FileProvider;
import androidx.core.content.res.ResourcesCompat;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

final class EmulationDialogFactory {


    private static Intent prepareCreatePackageIntent(final EmulationActivity ui,
                                                     final Runnable runOnPrepare,
                                                     final Bitmap screenshot) {
        Intent i = new Intent(ui, ShareEmulationActivity.class);
        ui.applyScreenOrientation(i);
        if (ui.findViewById(R.id.jv_wheel) != null) {
            i.putExtra(ShareEmulationActivity.VIRTUAL_WHEELJOYSTICK, true);
        }
        Emulation.PackCurrentStateFunctions pcsf =
                ui.getViewModel().getEmulation().getPackCurrentStateFunctions();
        runOnPrepare.run();
        if (pcsf.getProperties() != null) {
            i.putExtra("emu_props", new HashMap<>(pcsf.getProperties()));
        }
        i.putExtra("exportable_snapshotdata", ui.getSnapshotPath(
                ui.getViewModel().getConfiguration()).getAbsolutePath());
        if (ui.getViewModel().getConfiguration().getTouchJoystickDiagonalsLocked()
                || !ui.getCurrentUseropts().getStringValue(ui.getResources()
                        .getString(R.string.key_joystick_lock_diagonals), "0")
                .equals("0")) {
            i.putExtra("touch_diagonals_locked", true);
        }
        List<Uri> attachedFiles = pcsf.getAttachedFiles();
        if (attachedFiles != null) {
            i.putExtra("attached_files", new LinkedList<>(attachedFiles));
            List<String> visibleFiles = new LinkedList<>();
            for (Uri uri : attachedFiles) {
                if (pcsf.showFiletoUser(uri)) {
                    visibleFiles.add(uri.toString());
                }
            }
            i.putExtra("visible_files", new LinkedList<>(visibleFiles));

        }
        i.putExtra("keyboardvisible", ui.getKeyboardVisibility() == View.VISIBLE);
        i.putExtra("hardwarekeyboard", ui.getViewModel().isKeyboardUsed());

        Emulation.JoystickFunctions jf = ui.getViewModel().getEmulation().getJoystickFunctions();
        if (jf != null) {
            i.putExtra("joysticknames", new HashMap<>(jf.getJoystickports()));
            LinkedHashMap<Integer, Emulation.InputDeviceType> usedJoysticks = new LinkedHashMap<>();
            for (Joystick joy : ui.getAvailableJoysticks()) {
                if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                    usedJoysticks.put(joy.getPortnumber(), joy.getCurrentDeviceType());
                }
            }
            i.putExtra(ShareEmulationActivity.CONNECTED_JOYSTICKS, usedJoysticks);
        }
        if (ui.getViewModel().getEmulation().getPackCurrentStateFunctions() != null) {
            ArrayList<Uri> a = new ArrayList<>(
                    ui.getViewModel().getEmulation().getPackCurrentStateFunctions()
                            .getAttachedFiles());
            i.putExtra("attachedfiles", a);
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        screenshot.compress(PNG, FULLQUALITY, baos);
        i.putExtra("bitmap", baos.toByteArray());
        i.putExtra("configuration", ui.getViewModel().getConfiguration());
        return i;
    }

    private abstract static class CommonDialogFragment extends EmulationDialogFragment {
        @Override
        protected void setStyle() {
            setStyle(DialogFragment.STYLE_NO_TITLE, R.style.AlertDialogTheme);
        }
        protected View createUi(final EmulationActivity ui,
                               final @NonNull LayoutInflater inflater,
                               final @Nullable ViewGroup parent,
                               final @Nullable Bundle savedInstanceState) {
            throw new IllegalStateException("Override this method when creating an "
                    + getClass().getName() + " from an " + EmulationActivity.class.getName());
        }
        public View onCreateView(@NonNull final LayoutInflater inflater,
                                 @Nullable final ViewGroup parent,
                                 @Nullable final Bundle savedInstanceState) {
            if (getActivity() instanceof EmulationActivity) {
                return createUi((EmulationActivity)  getActivity(),
                        inflater, parent, savedInstanceState);
            } else {
                throw new IllegalStateException(getClass().getName() + " may only exist in "
                        + EmulationActivity.class.getName());
            }
        }
    }
    private static final int FULLQUALITY = 100;

    public static final class SavestateDialogFragment extends CommonDialogFragment {

        private static final String FRAGMENT_SAVESTATE_EXTENDED = "FRAGMENT_SAVESTATE_EXTENDED";

        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            view.findViewById(android.R.id.tabs).requestFocus();
        }

        /** @noinspection deprecation*/
        @Override
        // CHECKS---TYLE DISABLE MethodLength FOR 1 LINES
        protected View createUi(final EmulationActivity ui,
                             @NonNull final LayoutInflater inflater,
                             @Nullable final ViewGroup parent,
                             @Nullable final Bundle savedInstanceState) {
            @SuppressLint("InflateParams") ViewGroup root = (ViewGroup)
                    inflater.inflate(R.layout.dialog_savestates, null);

            byte[] buf = requireArguments().getByteArray("pixeldata");
            int bitmapwidth = requireArguments().getInt("bitmapwidth", -1);
            if (buf == null) {
                buf = new byte[1];
            }
            Bitmap currentScreenshot = BitmapFactory.decodeByteArray(buf, 0, buf.length);
            List<LinearLayout> views = getSingleSnapshotViews(root);

            List<SaveState> allStates =
                    getSaveStates(inflater.getContext(),
                            new File(Objects.requireNonNull(
                                    requireArguments().getString("rootpath"))));
            EmulationConfiguration conf = (EmulationConfiguration)
                    requireArguments().getSerializable("configuration");

            TabHost tabhost = root.findViewById(R.id.th_tabhost);
            if (tabhost != null) {
                tabhost.setup();
                for (int i = 0; i < views.size(); i++) {
                    SaveState state = allStates.get(i);
                    singleStateView(ui, views.get(i), tabhost, i, state);
                }
                final TabWidget tw = tabhost.getTabWidget();
                tabhost.setOnTabChangedListener(s -> updateButtons(ui, root, currentScreenshot,
                        allStates.get(tabhost.getCurrentTab()), () -> {
                    Log.v("-->delete<--", "" + tabhost.getChildCount());
                            updateDescription(ui, views.get(tabhost.getCurrentTab()),
                                    allStates.get(tabhost.getCurrentTab()));
                            Button viewSave = root.findViewById(R.id.bn_save);
                            viewSave.setText(R.string.save);
                        }));
                updateButtons(ui, root, currentScreenshot, allStates.get(0), () -> {
                            updateDescription(ui, views.get(0), allStates.get(0));
                            Button viewSave = root.findViewById(R.id.bn_save);
                            viewSave.setText(R.string.save);
                        });
                int w = ui.getResources().getDimensionPixelOffset(R.dimen.touchable_minimal_height);
                for (int i = 0; i < tw.getChildCount(); i++) {
                    View v = tw.getChildAt(i);
                    LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams) v.getLayoutParams();
                    lp.width = w;
                    v.setLayoutParams(lp);
                }
                tabhost.setCurrentTab(tabhost.getCurrentTab());

            }
            View viewCsf = root.findViewById(R.id.bn_current_state_functions);
            if (requireArguments().getInt("timemachine_offset", NO_TIMEMACHINE)
                    == NO_TIMEMACHINE && conf != null) {
                PopupMenu pm = createSaveStatePopup(ui, viewCsf, null,
                        null, null, currentScreenshot, conf,
                        null);
                viewCsf.setVisibility(pm != null ? View.VISIBLE : View.INVISIBLE);
                if (pm != null) {
                    BaseActivity.setViewListeners(viewCsf, pm::show, () -> {
                        EmulationDialogFragment f = new EmulationDialogFragment();
                        f.setMenu(pm.getMenu(),  false);
                        f.showNow(ui.getSupportFragmentManager(), FRAGMENT_SAVESTATE_EXTENDED);
                    }, this::dismiss);
                }
            } else {
                viewCsf.setVisibility(View.GONE);
            }
            return root;
        }
        private Bitmap getBitmap(final SaveState state) {
            Bitmap loadedBitmap = null;
            final File screenshot = state.getScreenshotFile();
            final File timestamp = state.getLabelFile();
            final File data = state.getDataFile();
            if (screenshot != null && screenshot.exists()
                    && timestamp.exists() && data.exists()) {
                try {
                    loadedBitmap = BitmapFactory.decodeStream(new FileInputStream(screenshot));
                } catch (FileNotFoundException e) {
                    // that's ok.
                }
            }
            return loadedBitmap;

        }
        private void updateDescription(final EmulationActivity ui, final View snapshotView,
                                       final SaveState state) {
            final File timestamp = state.getLabelFile();
            final Bitmap bmp = getBitmap(state);
            final ImageView viewScreenshot = snapshotView.findViewById(R.id.iv_bitmap);
            final TextView viewNoScreenShot = snapshotView.findViewById(R.id.tv_no_bitmap);
            if (bmp != null) {
                viewScreenshot.setImageBitmap(bmp);
            } else {
                if (ui.getCurrentBitmap() != null) {
                    viewScreenshot.setImageBitmap(ui.getCurrentBitmap());
                }
                viewScreenshot.setVisibility(View.INVISIBLE);
                viewNoScreenShot.setVisibility(View.VISIBLE);
            }
            TextView viewTimestamp = snapshotView.findViewById(R.id.tv_description);
            try {
                viewTimestamp.setText(
                        new BufferedReader(new FileReader(timestamp)).readLine());
            } catch (IOException ioException) {
                viewTimestamp.setText(ui.getResources()
                        .getString(R.string.available_snapshot_slot));
            }
        }
        private void singleStateView(final EmulationActivity ui,
                                     final LinearLayout snapshotView,
                                     final @NonNull TabHost tabhost,
                                     final int i, final SaveState state) {
            snapshotView.setId(View.generateViewId());
            //noinspection deprecation
            TabHost.TabSpec spec = tabhost.newTabSpec(String.valueOf(i + 1));
            @SuppressLint("InflateParams")
            ViewGroup viewTab = (ViewGroup) ui.getLayoutInflater()
                    .inflate(R.layout.view_tab_indicator, null);
            ((TextView) viewTab.findViewById(R.id.tv_title)).setText(String.valueOf(i + 1));
            if (ui.isTv()) {
                viewTab.setOnFocusChangeListener((view, b) -> {
                    if (b) {
                        int number = Integer.parseInt(((TextView) view.findViewById(R.id.tv_title))
                                .getText().toString());
                        int pos = number - 1;
                        //noinspection deprecation
                        tabhost.setCurrentTab(pos);
                    }
                });
            }
            spec.setIndicator(viewTab);
            //noinspection deprecation
            spec.setContent(snapshotView.getId());
            //noinspection deprecation
            tabhost.addTab(spec);
            updateDescription(ui, snapshotView, state);
        }
        private void updateButtons(final EmulationActivity ui,
                                   final ViewGroup rootView,
                                   final Bitmap currentScreenshot,
                                   final SaveState state,
                                   final Runnable runUpdateTab) {
            File rootFolder = state.getDataFile().getParentFile();
            final int timemachineOffset = requireArguments()
                    .getInt("timemachine_offset", NO_TIMEMACHINE);
            final File timestamp = state.getLabelFile();
            final File data = state.getDataFile();
            final File screenshot = state.getScreenshotFile();
            final Bitmap bmp = getBitmap(state);
            boolean calledFromTimemachine = timemachineOffset != NO_TIMEMACHINE;
            View viewStart = rootView.findViewById(R.id.bn_start);
            Runnable r = () -> {
                boolean ret;
                Emulation.PackCurrentStateFunctions pcsf = ui.getViewModel().getEmulation()
                        .getPackCurrentStateFunctions();
                File snapshotfile;
                Runnable deleteFile = () -> { };
                if (calledFromTimemachine) {
                    try {
                        snapshotfile = File.createTempFile("__dump__", "");
                        final File toDelete = snapshotfile;
                        //noinspection ResultOfMethodCallIgnored
                        deleteFile = toDelete::delete;
                    } catch (IOException e) {
                        snapshotfile = data;
                    }
                    // storing unencrypted is ok since file will be deleted immediately.
                    ret = pcsf != null && pcsf.writeDump(timemachineOffset,
                            snapshotfile.getAbsoluteFile(), ui.getViewModel().getConfiguration());
                } else {
                    snapshotfile = data;
                    ret = true;
                }
                ret = ret && pcsf.loadSnapshot(snapshotfile.getAbsolutePath(),
                        ui.getViewModel().getConfiguration());
                deleteFile.run();
                if (ret) {
                    if (bmp != null) {
                        ui.setBitmap(ui.getViewModel().getCurrentDisplayId(), bmp);
                    }
                    delayAfterDismiss();
                    dismiss();

                } else {
                    ui.showErrorMessage(ui.getResources().getString(
                            R.string.IDS_CANNOT_READ_SNAPSHOT_IMG));
                }
            };
            boolean dataExists = data.exists() && timestamp.exists() && !calledFromTimemachine;
            viewStart.setVisibility(dataExists ? View.VISIBLE : View.GONE);
            BaseActivity.setViewListeners(viewStart, r, this::dismiss);
            View viewMore = rootView.findViewById(R.id.bn_more);
            final EmulationConfiguration conf = (EmulationConfiguration)
                    requireArguments().getSerializable("configuration");
            if (data.exists() && timestamp.exists() && conf != null) {
                final PopupMenu pm = createSaveStatePopup(ui, viewMore, timestamp,
                        data, screenshot, bmp, conf, () -> {
                            rootView.findViewById(R.id.bn_save).requestFocus();
                            viewStart.setVisibility(View.GONE);
                            viewMore.setVisibility(View.GONE);
                            runUpdateTab.run();

                        });
                viewMore.setVisibility(pm != null && !calledFromTimemachine
                        ? View.VISIBLE : View.GONE);
                if (pm != null) {
                    BaseActivity.setViewListeners(viewMore, pm::show, () -> {
                        EmulationDialogFragment f = new EmulationDialogFragment();
                        f.setMenu(pm.getMenu(),  false);
                        f.showNow(ui.getSupportFragmentManager(), FRAGMENT_SAVESTATE_EXTENDED);
                    }, this::dismiss);
                }
            } else {
                viewMore.setVisibility(View.GONE);
            }
            Button viewSave = rootView.findViewById(R.id.bn_save);
            viewSave.setText(dataExists ? R.string.overwrite : R.string.save);

            setSaveListener(ui, currentScreenshot, timemachineOffset, calledFromTimemachine,
                    state, viewSave, rootFolder);

        }
        private static final int CURRENT_ENTRY = -1;
        private void setSaveListener(final EmulationActivity ui,
                                     final Bitmap currentScreenshot,
                                     final int timemachineOffset,
                                     final boolean calledFromTimemachine,
                                     final SaveState state,
                                     final View viewSave,
                                     final File dir) {
            BaseActivity.setViewListeners(viewSave, () -> {
                final File data = state.getDataFile();
                final File timestamp = state.getLabelFile();
                final File screenshot = state.getScreenshotFile();
                final String tag = EmulationDialogFactory.class.getSimpleName();
                boolean ok = false;
                if (dir != null) {
                    if (!(dir.isDirectory() || dir.mkdirs())
                    ) {
                        Log.e(tag, String.format(
                                "Cannon create %s as snapshot path",
                                dir.getAbsolutePath()));
                    }
                } else {
                    Log.e(tag, "rootFolder is null");
                }
                boolean ret;
                Emulation.PackCurrentStateFunctions pcsf = ui.getViewModel().getEmulation()
                        .getPackCurrentStateFunctions();
                if (calledFromTimemachine) {
                    ret = pcsf != null && pcsf.writeDump(timemachineOffset, data.getAbsoluteFile(),
                            ui.getViewModel().getConfiguration());
                } else {
                    ret = pcsf != null && pcsf.writeDump(CURRENT_ENTRY, data.getAbsoluteFile(),
                            ui.getViewModel().getConfiguration());
                }
                if (ret) {
                    if (currentScreenshot != null) {
                        try {
                            currentScreenshot.compress(Bitmap.CompressFormat.PNG,
                                    FULLQUALITY, new FileOutputStream(screenshot));
                        } catch (FileNotFoundException e) {
                            // that's ok
                        }
                    }
                    Date now = new Date();
                    FileWriter fw;
                    try {
                        fw = new FileWriter(timestamp.getAbsolutePath(), false);
                        fw.write(DateFormat.getDateFormat(
                                ui).format(now) + ", "
                                + DateFormat.getTimeFormat(ui)
                                .format(now));
                        fw.close();
                        ok = true;
                    } catch (IOException e) {
                        Log.e(tag, "error saving snapshot time", e);
                    }
                    if (ok) {
                        dismiss();
                    } else {
                        ui.showErrorMessage(ui.getResources().getString(
                                        R.string.IDS_CANNOT_WRITE_SNAPSHOT_S)
                                .replace(" ’%s’", ""));
                    }
                }

            }, this::dismiss);
        }

        //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
        private @Nullable PopupMenu createSaveStatePopup(final @NonNull EmulationActivity ui,
                                               final View viewMore,
                                               final File timestamp,
                                               final File data,
                                               final File screenshot,
                                               final Bitmap bmp,
                                               final EmulationConfiguration conf,
                                               final Runnable runAfterDelete) {
            PopupMenu pm = new PopupMenu(requireContext(), viewMore);
            pm.inflate(R.menu.menu_savestate_extendend);
            Menu m = pm.getMenu();

            final Emulation.PackCurrentStateFunctions pcsf = ui.getViewModel().getEmulation()
                    .getPackCurrentStateFunctions();
            m.findItem(R.id.mi_delete).setVisible(timestamp != null
                    && (timestamp.exists() || data.exists()) && runAfterDelete != null);
            m.findItem(R.id.add_to_main_activity).setVisible(conf.isBareMachine());
            m.findItem(R.id.update_main_activity).setVisible(conf.getLocalPath() != null);
            m.findItem(R.id.start_from_here).setVisible(conf.getLocalPath() != null
                    && conf.isStream()
                    && ui.getViewModel().getEmulation().getInitialSnapshotStorer() != null);
            m.findItem((R.id.share)).setVisible(!conf.isStream()
                    && ui.getViewModel().getEmulation() != null && pcsf != null);
            pm.setOnMenuItemClickListener(menuItem -> {
                if (menuItem.getItemId() == R.id.mi_delete && runAfterDelete != null) {
                    if (timestamp != null) {
                        //noinspection ResultOfMethodCallIgnored
                        timestamp.delete();
                    }
                    //noinspection ResultOfMethodCallIgnored
                    data.delete();
                    if (screenshot.exists()) {
                        //noinspection ResultOfMethodCallIgnored
                        screenshot.delete();
                    }
                    runAfterDelete.run();
                } else if (menuItem.getItemId() == R.id.add_to_main_activity) {
                    Bundle b = new Bundle();
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    if (bmp != null) {
                        bmp.compress(PNG, FULLQUALITY, baos);
                        b.putByteArray("pixeldata", baos.toByteArray());
                    }
                    if (data != null) {
                        b.putString("savestate", data.getAbsolutePath());
                    }
                    DialogFragment f = EmulationDialogFactory.showCreateLinkDialog();
                    f.setArguments(b);
                    f.showNow(ui.getSupportFragmentManager(), EmulationActivity.FRAGMENT_DIALOG);
                } else if (menuItem.getItemId() == R.id.update_main_activity) {
                    ConfigurationFactory.updateBitmap(conf, bmp);
                    ((BaseActivity) requireActivity()).updateWidgets();
                    if (getContext() != null) {
                        Intent i = new Intent(
                                "de.rainerhock.eightbitwonders.update");
                        i.setPackage("de.rainerhock.eightbitwonders");
                        getContext().sendBroadcast(i);
                    }
                    viewMore.post(this::dismiss);
                    //dismiss();
                } else if (menuItem.getItemId() == R.id.export_screenshot) {
                    try {
                        File f = new File(ui.getCacheDir(), "screenshot.png");
                        Matrix matrix = new Matrix();
                        matrix.postScale(2f * ui.getViewModel().getPixelAspectRatio(
                                ui.getViewModel().getCurrentDisplayId()), 2f);
                        Bitmap share = Bitmap.createBitmap(bmp, 0, 0,
                                bmp.getWidth(), bmp.getHeight(), matrix, true);

                        Intent shareIntent = new Intent();
                        share.compress(PNG, FULLQUALITY, new FileOutputStream(f));
                        Uri uri = FileProvider.getUriForFile(ui,
                                "de.rainerhock.eightbitwonders.fileprovider",
                                f);
                        shareIntent.setAction(Intent.ACTION_SEND);
                        shareIntent.putExtra(Intent.EXTRA_STREAM, uri);
                        shareIntent.putExtra(Intent.EXTRA_TITLE, R.string.screenshot);
                        shareIntent.putExtra(Intent.EXTRA_TEXT,
                                Html.fromHtml(getString(R.string.created_with_8bw,
                                        getString(R.string.app_url))));
                        shareIntent.setType("image/png");
                        ClipData cd = ClipData.newRawUri("", uri);
                        shareIntent.setClipData(cd);
                        shareIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
                        Intent chooser = BaseActivity.createChooserIntent(shareIntent,
                                getString(R.string.share));
                        ui.shareContent(chooser);
                        //ui.startActivityForResult(chooser, EmulationActivity.REQUEST_SHARE);

                    } catch (FileNotFoundException e) {
                        // should not happen
                    }
                } else if (menuItem.getItemId() == R.id.share) {
                    boolean tm = requireArguments()
                            .getInt("timemachine_offset", NO_TIMEMACHINE) != NO_TIMEMACHINE;

                    Runnable r = () -> { };
                    if (pcsf != null) {
                        if (data != null) {
                            r = pcsf.getImportContentsRunnable(data,
                                    ui.getViewModel().getConfiguration());
                        } else {
                            if (tm) {
                                r = pcsf.getDumpContentsRunnable(requireArguments()
                                        .getInt("timemachine_offset"),
                                        ui.getViewModel().getConfiguration());
                            } else {
                                r = pcsf.getDumpContentsRunnable(0,
                                        ui.getViewModel().getConfiguration());
                            }
                        }
                    }
                    Intent i = prepareCreatePackageIntent(ui, r, bmp);
                    if (timestamp != null) {
                        i.putExtra("no_timemachine", true);
                    }
                    ui.shareContent(i);
                    dismiss();
                } else if (menuItem.getItemId() == R.id.start_from_here) {
                    ui.getViewModel().getEmulation().getInitialSnapshotStorer().run();
                    dismiss();
                }
                return true;
            });
            for (int i = 0; i < pm.getMenu().size(); i++) {
                if (pm.getMenu().getItem(i).isVisible()) {
                    return pm;
                }
            }
            return null;
        }
    }
    public static final class CreateLinkDialogFragment extends CommonDialogFragment {
        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            if (getDialog() != null && getDialog().getWindow() != null) {
                getDialog().getWindow().setSoftInputMode(
                        WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);
            }
        }

        @Override
        protected View createUi(final EmulationActivity ui,
                                final @NonNull LayoutInflater inflater,
                      final @Nullable ViewGroup parent, final @Nullable Bundle savedInstanceState) {
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_createpackage, null);
            Emulation.TimeMachineFunctions tmf = ui.getViewModel().getEmulation()
                    .getTimeMachineFunctions();
            if (tmf != null && requireArguments()
                    .getString("savestate", null) == null) {
                byte[] buf = requireArguments().getByteArray("pixeldata");
                if (buf == null) {
                    buf = new byte[1];
                }
                Bitmap currentScreenshot = BitmapFactory.decodeByteArray(buf, 0, buf.length);
                initTimemachineSlider(root.findViewById(R.id.choose_moment),
                        currentScreenshot, tmf);
            } else {
                root.findViewById(R.id.choose_moment).setVisibility(View.GONE);
            }
            Button b = root.findViewById(R.id.pb_next);
            Runnable r = () -> {
                root.findViewById(R.id.part1).setVisibility(View.GONE);
                b.setVisibility(View.GONE);
                root.findViewById(R.id.part2).setVisibility(View.VISIBLE);
            };
            if (b != null) {
                if (root.findViewById(R.id.choose_moment).getVisibility() == View.GONE) {
                    r.run();
                } else {
                    b.setOnClickListener(view -> {
                        b.setVisibility(View.GONE);
                        root.findViewById(R.id.part1).setVisibility(View.GONE);
                        root.findViewById(R.id.part2).setVisibility(View.VISIBLE);
                        root.post(() -> root.findViewById(R.id.tv_name).requestFocus());
                        for (int action : Arrays.asList(MotionEvent.ACTION_DOWN,
                                MotionEvent.ACTION_UP)) {
                            MotionEvent me = MotionEvent.obtain(SystemClock.uptimeMillis(),
                                    SystemClock.uptimeMillis(),
                                    action, 0f, 0f, 0);
                            root.findViewById(R.id.tv_name).dispatchTouchEvent(me);
                            me.recycle();
                        }
                    });
                }


            }
            root.findViewById(R.id.bn_ok).setOnClickListener(view -> {
                int time;
                if (tmf != null) {
                    SeekBar sb = root.findViewById(R.id.sb_seekbar);
                    time = sb.getProgress();
                } else {
                    time = 0;
                }
                TextView tv = root.findViewById(R.id.tv_name);
                String path = requireArguments().getString("savestate", null);
                String name = String.valueOf(tv.getText());
                if (path != null) {
                    byte[] buf = requireArguments().getByteArray("pixeldata");
                    Bitmap bmp;
                    if (buf != null) {
                        bmp = BitmapFactory.decodeByteArray(buf, 0, buf.length);
                    } else {
                        bmp = null;
                    }
                    ConfigurationFactory.saveConfiguration(ui, name, bmp, new File(path));
                } else {
                    ConfigurationFactory.saveConfiguration(ui, name, time);
                }
                dismiss();
                String s = getResources().getString(R.string.added_to_start_screen, name);
                Toast.makeText(getContext(), s, Toast.LENGTH_SHORT).show();
            });
            return root;
        }
    }
    abstract static class AnnyRunnableDialog extends CommonDialogFragment {
        Emulation.MenuFeature createMenuFeature(final String name,
                                                final int iconResource,
                                                final Runnable runnable) {
            return new Emulation.MenuFeature() {
                @Override
                public String getName() {
                    return name;
                }

                @Override
                public int getIconResource() {
                    return iconResource;
                }

                @Override
                public Runnable getRunnable() {
                    return runnable;
                }
            };
        }


        abstract List<Emulation.MenuFeature> getFeatures(Emulation e);
        private Drawable createEmptyDrawable(final Drawable sameSize) {
            ShapeDrawable ret = new ShapeDrawable(new OvalShape());
            ret.getPaint().setColor(
                    requireContext().getResources().getColor(android.R.color.transparent));
            //getContext().getResources().getColor(android.R.color.transparent));
            ret.setIntrinsicHeight(sameSize.getIntrinsicHeight());
            ret.setIntrinsicWidth(sameSize.getIntrinsicWidth());
            return ret;
        }
        @Override
        protected View createUi(final EmulationActivity ui,
                                final @NonNull LayoutInflater inflater,
                                final @Nullable ViewGroup parent,
                                final @Nullable Bundle savedInstanceState) {
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_menu_singlerow, parent,
                    false);
            boolean first = true;
            Emulation emulation = null;
            if (ui != null) {
                emulation = ui.getViewModel().getEmulation();
            }
            for (Emulation.MenuFeature mf : getFeatures(emulation)) {
                ViewGroup entryView = (ViewGroup) LayoutInflater.from(getContext()).inflate(
                        R.layout.fragment_popup_dialogitem, root, false);
                Button b = (Button) LayoutInflater.from(getContext()).inflate(first
                        ? R.layout.view_dialog_first_button : R.layout.view_dialog_button,
                        entryView, false);
                first = false;
                Drawable[] drawables = b.getCompoundDrawables();
                if (mf.getIconResource() > 0) {
                    drawables[0] = ResourcesCompat.getDrawable(requireContext().getResources(),
                            mf.getIconResource(), requireContext().getTheme());
                } else {
                    drawables[0] = createEmptyDrawable(
                            Objects.requireNonNull(
                                    ResourcesCompat.getDrawable(requireContext().getResources(),
                                            android.R.drawable.ic_media_play,
                                            requireContext().getTheme())));

                }
                drawables[2] = createEmptyDrawable(
                        Objects.requireNonNull(
                                ResourcesCompat.getDrawable(requireContext().getResources(),
                                        android.R.drawable.ic_media_play,
                                        requireContext().getTheme())));

                b.setCompoundDrawablesRelativeWithIntrinsicBounds(drawables[0],
                        drawables[1],
                        drawables[2],
                        drawables[3]);
                b.setText(mf.getName());
                if (mf.getRunnable() != null) {
                    BaseActivity.setViewListeners(b, () -> {
                        mf.getRunnable().run();
                        dismiss();
                    }, this::dismiss);
                } else {
                    b.setEnabled(false);
                }
                entryView.addView(b);
                ((ViewGroup) root.findViewById(R.id.singlecol)).addView(entryView);
            }

            root.requestLayout();
            return root;
        }

    }
    private static final int VIEW_ID_NONE = View.generateViewId();
    private static final int ERROR_COLOR = android.R.color.holo_red_light;
    private static void updateJoystickport(final EmulationActivity activity,
                                           final Joystick joy,
                                           final int port) {
        joy.setPortnumber(port);
        activity.getCurrentUseropts().setValue(
                Useropts.Scope.CONFIGURATION, "JOYSTICK_" + joy.getId() + "_PORT", port);

    }

    // CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
    private static RadioGroup addJoystickportView(final EmulationActivity activity,
                                                  final LinearLayout rootView,
                                                  final Emulation.JoystickFunctions jf,
                                                  final int port,
                                                  final boolean isRequired,
                                                  final Emulation.InputDeviceType requiredIdt,
                                                  final List<Joystick> availableJoysticks,
                                                  final Map<Integer, Joystick> joystickViewIds) {
        LayoutInflater inflater = (LayoutInflater) activity
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        @SuppressLint("InflateParams")
        LinearLayout ll = (LinearLayout) inflater.inflate(
                R.layout.view_assign_joystick, null);
        TextView tvRequired = ll.findViewById(R.id.tv_required);
        ((TextView) ll.findViewById(R.id.tv_joystickport)).setText(
                jf.getJoystickports().get(port));
        tvRequired.setVisibility(
                isRequired ? View.VISIBLE : View.GONE);
        RadioGroup rg = ll.findViewById(R.id.rg_hardware);
        @SuppressLint("InflateParams")
        RadioButton rbNone = (RadioButton) inflater
                .inflate(R.layout.view_radiobutton_joystick, null);
        int defaultColor = tvRequired.getCurrentTextColor();
        int errorColor = activity.getResources().getColor(ERROR_COLOR);
        rbNone.setText(R.string.IDS_NONE);
        rbNone.setId(VIEW_ID_NONE);
        rbNone.setOnCheckedChangeListener((compoundButton, b) -> {
            if (b) {
                tvRequired.setTextColor(errorColor);
            } else {
                tvRequired.setTextColor(defaultColor);
            }
        });
        rbNone.setOnClickListener(view -> {
            for (Joystick joy: availableJoysticks) {
                joy.readUseropts(activity);
                if (joy.getPortnumber() == port) {
                    updateJoystickport(activity, joy, Joystick.PORT_NOT_CONNECTED);
                }
            }
        });
        rg.addView(rbNone);
        boolean checked = false;
        for (int viewId : joystickViewIds.keySet()) {
            Joystick joy = joystickViewIds.get(viewId);
            if (joy != null && joy.getSupportedDeviceTypes().contains(requiredIdt)) {
                @SuppressLint("InflateParams")
                RadioButton rb = (RadioButton) inflater
                        .inflate(R.layout.view_radiobutton_joystick, null);
                rb.setId(viewId);
                rb.setText(Objects.requireNonNull(joystickViewIds.get(viewId)).toString());
                joy.setCurrentdevicetype(requiredIdt);
                joy.readUseropts(activity);
                if (joy.getPortnumber() == port) {
                    rb.setChecked(true);
                    checked = true;
                }
                rb.setOnClickListener(view -> {
                    Joystick thisJoy = joystickViewIds.get(viewId);
                    if (thisJoy != null) {
                        for (Joystick otherJoy : availableJoysticks) {
                            if (otherJoy.getHardwareDescriptor()
                                    .equals(thisJoy.getHardwareDescriptor())) {
                                updateJoystickport(activity, otherJoy, Joystick.PORT_NOT_CONNECTED);
                            }
                            if (otherJoy.getPortnumber() == port) {
                                updateJoystickport(activity, otherJoy, Joystick.PORT_NOT_CONNECTED);
                            }
                        }
                        updateJoystickport(activity, thisJoy, port);
                    }
                });

                rg.addView(rb);
            }
        }
        if (!checked) {
            rbNone.setChecked(true);
        }
        if (rg.getCheckedRadioButtonId() == VIEW_ID_NONE && isRequired) {
            tvRequired.setTextColor(activity.getResources().getColor(ERROR_COLOR));
        }

        rg.requestLayout();
        ll.requestLayout();
        rootView.addView(ll);
        return rg;

    }
    public static final class RequiredJoysticksFragment extends CommonDialogFragment {
        @Override
        protected View createUi(final @NonNull EmulationActivity activity,
                                final @NonNull LayoutInflater inflater,
                                final @Nullable ViewGroup parent,
                                final @Nullable Bundle savedInstanceState) {
            Emulation emulation = Emulationslist.getCurrentEmulation();
            List<Joystick> availableJoysticks = activity.getAvailableJoysticks();
            @SuppressWarnings("unchecked")
            HashMap<Integer, EmulationConfiguration.PreferredJoystickType> joysticktypes
                    = (HashMap<Integer, EmulationConfiguration.PreferredJoystickType>)
                    requireArguments().getSerializable("joysticktypes");

            if (joysticktypes == null) {
                joysticktypes = new HashMap<>();
            }
            @SuppressWarnings("unchecked")
            List<Integer> required = (List<Integer>)
                    requireArguments().getSerializable(REQUIREDJOYSTICKS);
            if (required == null) {
                required = new LinkedList<>();
            }
            @SuppressWarnings("unchecked")
            Map<Integer, Emulation.InputDeviceType> inputdevicestypes =
                    (Map<Integer, Emulation.InputDeviceType>) requireArguments()
                            .getSerializable(INPUTDEVICETYPES);
            if (inputdevicestypes == null) {
                inputdevicestypes = new LinkedHashMap<>();
            }
            Map<Integer, Joystick> joystickViewIds = new HashMap<>();

            for (Joystick joy: availableJoysticks) {
                joy.readUseropts(activity);
                joystickViewIds.put(View.generateViewId(), joy);
            }
            final List<RadioGroup> allGroups = new ArrayList<>();
            @SuppressLint("InflateParams")
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_menu_singlerow, null);
            LinearLayout llJoysticks = root.findViewById(R.id.singlecol);
            for (int port : joysticktypes.keySet()) {
                Emulation.InputDeviceType requiredIdt;
                if (inputdevicestypes.containsKey(port)) {
                    requiredIdt = inputdevicestypes.get(port);
                } else {
                    requiredIdt = Emulation.InputDeviceType.DSTICK;
                }
                allGroups.add(addJoystickportView(
                        activity, llJoysticks, emulation.getJoystickFunctions(),
                        port, required.contains(port), requiredIdt,
                        availableJoysticks, joystickViewIds));
            }
            for (RadioGroup rg:allGroups) {
                rg.setOnCheckedChangeListener((radioGroup, i) -> {
                    if (i != VIEW_ID_NONE) {
                        for (RadioGroup otherGroup : allGroups) {
                            if (otherGroup != radioGroup) {
                                Joystick otherJoy = joystickViewIds.get(
                                        otherGroup.getCheckedRadioButtonId());
                                Joystick thisJoy = joystickViewIds.get(i);
                                if (otherJoy != null && thisJoy != null) {
                                    if (otherJoy.getHardwareDescriptor()
                                            .equals(thisJoy.getHardwareDescriptor())) {
                                        ((RadioButton) otherGroup.findViewById(VIEW_ID_NONE))
                                                .setChecked(true);
                                        updateJoystickport(activity, otherJoy,
                                                Joystick.PORT_NOT_CONNECTED);
                                    }
                                }
                            }
                        }
                    }
                });
            }
            @SuppressLint("InflateParams")
            Button btnContinue = (Button) inflater.inflate(R.layout.view_dialog_button, null);
            btnContinue.setText(R.string.apply);
            BaseActivity.setViewListeners(btnContinue, () -> {
                activity.setDynamicUiElements();
                dismiss();
            }, this::dismiss);

            llJoysticks.addView(btnContinue);
            TextView tv = root.findViewById(R.id.tv_title);
            tv.setVisibility(View.VISIBLE);
            tv.setText(R.string.required_hardware);
            return root;
        }
    }
    private static final String INPUTDEVICETYPES = "INPUTDEVICETYPES";
    private static final String REQUIREDJOYSTICKS = "REQUIREDJOYSTICKS";
    static DialogFragment showRequiredJoysticksFragment(
            final Map<Integer, EmulationConfiguration.PreferredJoystickType> joysticktypes,
            final Map<Integer, Emulation.InputDeviceType> inputdevicetypes,
            final List<Integer> required) {
        DialogFragment ret = new RequiredJoysticksFragment();
        Bundle b = new Bundle();
        HashMap<Integer, EmulationConfiguration.PreferredJoystickType> jt
                = new HashMap<>(joysticktypes);
        b.putSerializable("joysticktypes", jt);
        LinkedList<Integer> r = new LinkedList<>(required);
        b.putSerializable(REQUIREDJOYSTICKS, r);
        b.putSerializable(INPUTDEVICETYPES, new LinkedHashMap<>(inputdevicetypes));
        ret.setArguments(b);
        return ret;
    }
    public static final class TapeFunctionsDialogFragment extends AnnyRunnableDialog {

        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {

            Emulation.TapeDeviceFunctions tdf = e.getTapeDeviceFunctions();
            List<Emulation.MenuFeature> functions = new LinkedList<>();
            if (tdf != null) {
                functions.add(createMenuFeature(getString(R.string.press_play),
                        R.drawable.ic_media_play, tdf.getPlayCommand()));
                functions.add(createMenuFeature(getString(R.string.IDS_MI_DATASETTE_STOP),
                        R.drawable.ic_media_stop, tdf.getStopCommand()));
                functions.add(createMenuFeature(getString(R.string.IDS_MI_DATASETTE_FORWARD),
                        R.drawable.ic_media_ff, tdf.getForwardCommand()));
                functions.add(createMenuFeature(getString(R.string.IDS_MI_DATASETTE_REWIND),
                        R.drawable.ic_media_rew, tdf.getRewindCommand()));
                functions.add(createMenuFeature(getString(R.string.press_record_and_play),
                        R.drawable.ic_media_record, tdf.getRecordCommand()));
            }
            return functions;
        }
    }
    public static final class ResetFunctionsDialogFragment extends AnnyRunnableDialog {
        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {
            return e.getResetFunctions().getFunctions();
        }
    }
    public static final class SwitchMonitorDialog extends AnnyRunnableDialog {
        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {
            if (e.getDualMonitorFunctions() != null) {
                List<Emulation.MenuFeature> ret = new LinkedList<>();
                EmulationActivity ea = ((EmulationActivity) getActivity());
                for (int i : e.getDualMonitorFunctions().getDisplays().keySet()) {
                    if (ea != null && ea.getViewModel() != null
                            && ea.getViewModel().getCurrentDisplayId() != i) {
                        ret.add(new Emulation.MenuFeature() {
                            @Override
                            public String getName() {
                                return e.getDualMonitorFunctions().getDisplays().get(i);
                            }

                            @Override
                            public int getIconResource() {
                                return View.NO_ID;
                            }

                            @Override
                            public Runnable getRunnable() {
                                return () -> ea.getViewModel().setCurrentDisplayId(i);
                            }
                        });
                    }
                }
                return ret;
            }
            return null;
        }
    }
    public static final class DetachFunctionsDialogFragment extends AnnyRunnableDialog {
        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {
            return e.getFileFunctions().getDetachFunctions().getFunctions();
        }
    }
    public static final class FliplistDialogFragment extends CommonDialogFragment {
        protected View createUi(final @NonNull EmulationActivity ui,
                                final @NonNull LayoutInflater inflater,
                                final @Nullable ViewGroup parent,
                                final @Nullable Bundle savedInstanceState) {
            Set<Uri> fl = ui.getViewModel().getConfiguration().getFliplist();
            if (fl.isEmpty()) {
                fl = ui.getViewModel().getEmulation().getFliplistFunctions(null).getFilesInList();
            }
            Emulation.FliplistFunctions ff = ui.getViewModel().getEmulation()
                    .getFliplistFunctions(fl);
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_menu_singlerow, null);
            ViewGroup contents = root.findViewById(R.id.singlecol);
            boolean first = true;
            for (int drive : ff.getTargetDevices().keySet()) {
                @SuppressLint("InflateParams")
                TextView tv = (TextView) inflater.inflate(R.layout.view_dialog_sectiontitle, null);
                tv.setText(ff.getTargetDevices().get(drive));
                contents.addView(tv);
                for (Uri f : fl) {
                    if (ff.canBePart(f)) {
                        @SuppressLint("InflateParams")
                        Button b = (Button) inflater.inflate(first
                                ? R.layout.view_dialog_first_button
                                : R.layout.view_dialog_button, null);
                        b.setText(f.getLastPathSegment());
                        b.setTag(drive);
                        if (ff.getAttachedImage(drive) != null) {
                            if (Objects.equals(f.getPath(), ff.getAttachedImage(drive).getPath())) {
                                b.setEnabled(false);
                                b.setFocusable(false);
                                first = false;
                            }
                        }
                        if (b.isEnabled()) {
                            BaseActivity.setViewListeners(b, () -> {
                                if (ff.attach(drive, f)) {
                                    dismiss();
                                }
                            }, this::dismiss);
                        }
                        contents.addView(b);
                    }
                }
            }
            return root;
        }
    }
    static DialogFragment showTimemachineDialog() {
        return new TimeMachineDialogFragment();

    }
    static final int NO_TIMEMACHINE = -1;
    static EmulationDialogFragment showSaveStatesDialog(final EmulationActivity ui,
                                               final int timemachineOffset,
                                               final Bitmap bmp) {
        EmulationDialogFragment f = new SavestateDialogFragment();
        Bundle b = new Bundle();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        if (bmp != null) {
            bmp.compress(PNG, FULLQUALITY, baos);
            b.putByteArray("pixeldata", baos.toByteArray());
            b.putInt("bitmapwidth", bmp.getWidth());
        }

        b.putSerializable("configuration", ui.getViewModel().getConfiguration());
        b.putString("rootpath", ui.getViewModel().getSnapshotPath());
        b.putInt("timemachine_offset", timemachineOffset);
        f.setArguments(b);
        return f;

    }
    static DialogFragment showSoftkeyDialog() {
        return new SoftkeysDialogFragment();
    }
    static DialogFragment showFliplistDialog() {
        return new FliplistDialogFragment();
    }
    static DialogFragment showCreateLinkDialog() {
        return new CreateLinkDialogFragment();
    }
    static DialogFragment showTapeFunctionsDialog() {
        return new TapeFunctionsDialogFragment();
    }
    static DialogFragment showResetFunctionsDialog() {
        return new ResetFunctionsDialogFragment();
    }
    static DialogFragment showSwitchMonitorFunctionsDialog() {
        return new SwitchMonitorDialog();
    }
    static DialogFragment showDetachFunctionsDialog() {
        return new DetachFunctionsDialogFragment();

    }
    public static final class QuitDialogFragment  extends AnnyRunnableDialog {
        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            ret.add(new Emulation.MenuFeature() {
                @Override
                public String getName() {
                    return requireContext().getResources().getString(R.string.global_action_logout);
                }

                @Override
                public int getIconResource() {
                    return 0;
                }

                @Override
                public Runnable getRunnable() {
                    return () -> {
                        EmulationActivity a = (EmulationActivity) requireActivity();
                        e.terminate(() -> a.runOnUiThread(a::onEmulatorFinished));
                    };
                }
            });
            return ret;
        }
    }
    public static final class PauseDialogFragment  extends AnnyRunnableDialog {
        @Override
        protected View createUi(final EmulationActivity ui,
                                final @NonNull LayoutInflater inflater,
                                final @Nullable ViewGroup parent,
                                final @Nullable Bundle savedInstanceState) {
            View ret = super.createUi(ui, inflater, parent, savedInstanceState);
            TextView tv = ret.findViewById(R.id.tv_title);
            if (requireArguments().getBoolean("extended", true)) {
                tv.setVisibility(View.GONE);
            } else {
                tv.setVisibility(View.VISIBLE);
                tv.setText(getResources().getString(R.string.pause));
            }
            return ret;
        }
        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation e) {
            List<Emulation.MenuFeature> ret = new LinkedList<>();
            if (requireArguments().getBoolean("extended", true)
                    && getActivity() instanceof EmulationActivity) {
                EmulationActivity ea = (EmulationActivity) getActivity();
                ret.add(createMenuFeature(requireContext().getResources().getString(R.string.menu),
                        0,
                        () -> {
                            ea.getViewModel().setNotPausedByUser();
                            dismiss();
                            ea.getSupportFragmentManager().beginTransaction().remove(this)
                                    .commitNow();
                            ea.showMainMenu();
                        }));
                EmulationViewModel vm = ea.getViewModel();
                if (ea.isTv() && !(vm.getSoftkeyIds().isEmpty() && vm.getUiSoftkeys().isEmpty())) {
                    ret.add(createMenuFeature(requireContext().getResources()
                                    .getString(R.string.softkeys),
                            0,
                            () -> {
                                ea.getViewModel().setNotPausedByUser();
                                dismiss();
                                ea.getSupportFragmentManager().beginTransaction()
                                        .remove(this).commitNow();
                                ea.showSoftkeyDialog();
                            }));

                }
            }
            ret.add(createMenuFeature(requireContext().getResources().getString(R.string.cont), 0,
                    () -> {
                        EmulationActivity ea = (EmulationActivity) getActivity();
                        if (ea != null && ea.getViewModel() != null) {
                            ea.getViewModel().setNotPausedByUser();
                            new Thread(() -> ea.runOnUiThread(ea::clearPauseReasons)).start();
                        }
                    }));
            return ret;
        }
    }
    static EmulationDialogFragment showQuitDialog() {
        return new QuitDialogFragment();
    }
    static EmulationDialogFragment showPauseDialog(final boolean extended) {
        Bundle b = new Bundle();
        b.putBoolean("extended", extended);
        EmulationDialogFragment f = new PauseDialogFragment();
        f.setArguments(b);
        f.setTitle(R.string.pause);
        return f;
    }
    public static final class BePreparedDialogFragment extends CommonDialogFragment {
        @Override
        protected View createUi(final EmulationActivity ui,
                                final @NonNull LayoutInflater inflater,
                                final @Nullable ViewGroup parent,
                                final @Nullable Bundle savedInstanceState) {
            @SuppressLint("InflateParams")
            ViewGroup root = (ViewGroup) inflater.inflate(
                    R.layout.dialog_be_prepared, null, false);

            return root;
        }
        @Override
        public void onDetach() {
            if (mThread != null) {
                mThread.interrupt();
            }
            mRunAfterDismiss.run();
            mThread = null;
            super.onDetach();
        }

        private Thread mThread = null;
        private void startCounting() {
            final int count = requireArguments().getInt("tick_count", 0);
            int delay = requireArguments().getInt("tick_duration", 0);
            if (count > 0 && delay > 0) {
                ProgressBar b = requireView().findViewById(R.id.pb_be_prepared_progress);
                b.setMax(count);
                b.setProgress(count);

                mThread = new Thread(() ->  {
                    int c = count;
                    while (c > 0) {
                        try {
                            Thread.sleep(delay);
                        } catch (InterruptedException e) {
                            return;
                        }
                        c--;
                        final int c2 = c;
                        requireActivity().runOnUiThread(() -> b.setProgress(c2));

                    }
                    requireActivity().runOnUiThread(new Thread(this::dismiss));

                });

                mThread.start();
            }
        }

        private Runnable mRunAfterDismiss = () -> { };
        public void setAfterDismissRunnable(final @NonNull Runnable runAfterDismiss) {
            mRunAfterDismiss = runAfterDismiss;
        }
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup parent,
                                 final @Nullable Bundle savedInstanceState) {
            View ret = super.onCreateView(inflater, parent, savedInstanceState);
            new Handler(Looper.getMainLooper()).post(this::startCounting);
            return ret;
        }

    }
    static DialogFragment createBePreparedDialog(final Runnable runAfterDismiss) {
        Bundle b = new Bundle();
        b.putInt("tick_count", EmulationActivity.PAUSE_TICK_COUNT);
        b.putInt("tick_duration", EmulationActivity.PAUSE_TICK_DURATION);
        BePreparedDialogFragment dlg = new BePreparedDialogFragment();
        dlg.setArguments(b);
        dlg.setAfterDismissRunnable(runAfterDismiss);
        return dlg;
    }
    private static final int UNINITIALIZED = -1;
    public static final class TimeMachineDialogFragment extends CommonDialogFragment {
        private int mProgress = UNINITIALIZED;
        private int mNewestMoment  = UNINITIALIZED;


        @Override
        protected View createUi(@NonNull final EmulationActivity  ui,
                             @NonNull final LayoutInflater inflater,
                             @Nullable final ViewGroup parent,
                             @Nullable final Bundle savedInstanceState) {
            Emulation.TimeMachineFunctions rewindfunctions = ui.getViewModel()
                    .getEmulation().getTimeMachineFunctions();
            ViewGroup root = (ViewGroup) inflater.inflate(R.layout.dialog_timemachine, null, false);
            SeekBar sb = root.findViewById(R.id.sb_seekbar);
            if (mNewestMoment < 0) {
                mNewestMoment = rewindfunctions.getNewestMoment();
            }
            sb.setMax(rewindfunctions.getOldestMoment() - mNewestMoment);
            Bitmap bmp = rewindfunctions.getScreenPreview(mNewestMoment);
            View viewApply = root.findViewById(R.id.bn_apply);
            ImageView viewBitmap = root.findViewById(R.id.iv_bitmap);
            viewBitmap.setImageBitmap(bmp);
            BaseActivity.setViewListeners(root.findViewById(R.id.bn_save), () -> {
                EmulationDialogFactory.showSaveStatesDialog(ui,
                        sb.getProgress() + mNewestMoment,
                        rewindfunctions.getScreenPreview(sb.getProgress() + mNewestMoment))
                        .showNow(ui.getSupportFragmentManager(), EmulationActivity.FRAGMENT_DIALOG);
                dismiss();
            }, this::dismiss);
            sb.setProgress(mProgress > 0  ? mProgress + mNewestMoment : mNewestMoment);
            TextView tv = root.findViewById(R.id.tv_time_offset);
            tv.setText(elapsedTimeAsString(inflater.getContext(), sb.getProgress()));

            Runnable r = () -> {
                Bitmap b = rewindfunctions.getScreenPreview(sb.getProgress() + mNewestMoment);
                if (b != null) {
                    ui.setBitmap(ui.getViewModel().getCurrentDisplayId(), b);
                }
                rewindfunctions.travelTo(sb.getProgress() + mNewestMoment);
                delayAfterDismiss();
                dismiss();
            };
            BaseActivity.setViewListeners(viewBitmap, r, this::dismiss);
            BaseActivity.setViewListeners(viewApply, r, this::dismiss);

            sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
                @Override
                public void onProgressChanged(final SeekBar seekBar, final int i, final boolean b) {
                    if (b) {
                        viewBitmap.setImageBitmap(
                                rewindfunctions.getScreenPreview(i + mNewestMoment));
                        tv.setText(elapsedTimeAsString(
                                inflater.getContext(), i + mNewestMoment));
                        mProgress = i;
                    }

                }

                @Override
                public void onStartTrackingTouch(final SeekBar seekBar) {
                }

                @Override
                public void onStopTrackingTouch(final SeekBar seekBar) {
                }
            });

            return root;
        }


    }
    public static class SoftkeysDialogFragment extends CommonDialogFragment {
        /**
         * Needed by android, do not use.
         */
        // CHECKSTYLE DISABLE RedundantModifier FOR 1 LINES
        public SoftkeysDialogFragment() {
            super();
        }
        @Override
        public void onViewCreated(@NonNull final View view,
                                  @Nullable final Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            Window window = requireDialog().getWindow();
            if (window != null) {
                WindowManager.LayoutParams wlp = window.getAttributes();
                wlp.gravity = Gravity.BOTTOM;
                wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
                window.setAttributes(wlp);
            }

        }

        @Override
        protected View createUi(@NonNull final EmulationActivity  ui,
                                @NonNull final LayoutInflater inflater,
                                @Nullable final ViewGroup parent,
                                @Nullable final Bundle savedInstanceState) {
            SoftkeysLinearLayout root = (SoftkeysLinearLayout)
                    inflater.inflate(R.layout.view_tv_softkeys, parent, false);
            if (getActivity() instanceof EmulationActivity) {
                ((EmulationActivity) getActivity()).addSoftkeyViews(root,
                        this::continueEmulation, this::dismiss);
            }
            root.post(root::requestFocus);
            root.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                //CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                @Override
                public void onLayoutChange(final View view, final int i, final int i1,
                                           final int i2, final int i3, final int i4,
                                           final int i5, final int i6, final int i7) {
                    root.post(root::requestFocus);
                    root.removeOnLayoutChangeListener(this);
                }
            });

            return root;
        }
    }

    private EmulationDialogFactory() { }

    private static final  float MILLIS_PER_S = 1_000f;
    private static String elapsedTimeAsString(final Context context, final int elapsedMilliSecs) {
        //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
        String s = String.format(context.getResources().getConfiguration().locale,
                "%.2f", elapsedMilliSecs / MILLIS_PER_S);
        return context.getResources().getString(R.string.elapsed, s);

    }
    interface SaveState {
        @NonNull  File getLabelFile();
        @NonNull  File getDataFile();
        @Nullable File getScreenshotFile();
    }
    private static List<String> getSubdirectories(final Context context) {
        LayoutInflater inflater
                = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        @SuppressLint("InflateParams") ViewGroup root = (ViewGroup)
                inflater.inflate(R.layout.dialog_savestates, null);
        int count = getSingleSnapshotViews(root).size();
        List<String> ret = new LinkedList<>();
        for (int i = 0; i < count; i++) {
            ret.add(String.valueOf(i));
        }
        return ret;

    }
    static void initTimemachineSlider(final @NonNull  ViewGroup root,
                                      final @NonNull Bitmap currentBitmap,
                                      final @NonNull Emulation.TimeMachineFunctions tmf) {
        SeekBar sb = root.findViewById(R.id.sb_seekbar);
        int newestSnapshot = tmf.getNewestMoment();
        sb.setMax(tmf.getOldestMoment() - newestSnapshot);
        Bitmap bmp = tmf.getScreenPreview(newestSnapshot);
        ImageView iv = root.findViewById(R.id.iv_bitmap);
        iv.setImageBitmap(bmp);
        TextView tv = root.findViewById(R.id.tv_time_offset);
        tv.setText(elapsedTimeAsString(root.getContext(), newestSnapshot));
        sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int i, final boolean b) {
                if (b) {
                    if (i > 0) {
                        iv.setImageBitmap(tmf.getScreenPreview(i + newestSnapshot));
                        tv.setText(elapsedTimeAsString(root.getContext(), i + newestSnapshot));
                    } else {
                        iv.setImageBitmap(currentBitmap);
                        tv.setText(R.string.now);
                    }

                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        sb.setProgress(0);

    }
    private static List<SaveState> getSaveStates(
            final Context context, final File snapshotsRoot) {
        List<SaveState> savestates = new LinkedList<>();
        final String logTag = EmulationDialogFactory.class.getSimpleName() + ".showSaveStateDialog";
        if (!snapshotsRoot.exists()) {
            if (!snapshotsRoot.mkdirs()) {
                Log.e(logTag, String.format("Cannon create %s as snapshot path",
                        snapshotsRoot.getAbsolutePath()));
            }
        }
        for (String subdir : getSubdirectories(context)) {
            File path = new File(snapshotsRoot, subdir);
            savestates.add(new SaveState() {

                @NonNull
                @Override
                public File getLabelFile() {
                    return new File(path, "timestamp");
                }

                @NonNull
                @Override
                public File getDataFile() {
                    return new File(path, "data");
                }

                @NonNull
                @Override
                public File getScreenshotFile() {
                    return new File(path, "screenshot");
                }
            });
        }
        return savestates;
    }
    static void recursivelyFindChildren(final View view, final List<LinearLayout> results) {

        if ("single_snapshot".equals(view.getTag())) {
            results.add((LinearLayout) view);
        } else {
            if (view instanceof ViewGroup) {
                ViewGroup viewGroup = (ViewGroup) view;
                for (int i = 0; i < viewGroup.getChildCount(); i++) {
                    recursivelyFindChildren(viewGroup.getChildAt(i), results);
                }
            }
        }
    }
    private static List<LinearLayout> getSingleSnapshotViews(final ViewGroup root) {
        List<LinearLayout> snapshotViews = new LinkedList<>();
        recursivelyFindChildren(root, snapshotViews);
        return snapshotViews;
    }
    public static class SpinnerDialogFragment extends AnnyRunnableDialog {
        public View onCreateView(@NonNull final LayoutInflater inflater,
                                 @Nullable final ViewGroup parent,
                                 @Nullable final Bundle savedInstanceState) {
            return createUi(null,
                        inflater, parent, savedInstanceState);
        }

        @Override
        List<Emulation.MenuFeature> getFeatures(final Emulation emulation) {
            List<Emulation.MenuFeature> functions = new LinkedList<>();
            try {
                @SuppressWarnings("unchecked")
                List<String> arguments = (List<String>) requireArguments()
                        .getSerializable("values");
                if (arguments == null) {
                    arguments = new LinkedList<>();
                }
                int pos = 0;
                for (String s : arguments) {
                    final int p = pos;
                    functions.add(new Emulation.MenuFeature() {
                        @Override
                        public String getName() {
                            return s;
                        }

                        @Override
                        public int getIconResource() {
                            return 0;
                        }

                        @Override
                        public Runnable getRunnable() {
                            return () -> {
                                FragmentActivity a = requireActivity();
                                int id = requireArguments().getInt("tempViewId");
                                View v = a.findViewById(id);
                                if (v == null) {
                                    for (Fragment f
                                            : a.getSupportFragmentManager().getFragments()) {
                                        if (f != null && f.getView() != null) {
                                            v = f.getView().findViewById(id);
                                            if (v != null) {
                                                break;
                                            }

                                        }
                                    }
                                }
                                if (v != null) {
                                    if (v instanceof Spinner) {
                                        Spinner sp = (Spinner) v;
                                        sp.setSelection(p);
                                    }
                                    int oldId = requireArguments().getInt("oldViewId", View.NO_ID);
                                    if (oldId != View.NO_ID) {
                                        v.setId(oldId);
                                    }
                                }
                            };
                        }
                    });
                    pos++;
                }

            } catch (ClassCastException e) {
                // make compiler smile
            }
            return functions;
        }
    }
    static DialogFragment showSpinnerDialog(final MappedSpinner view, final List<String> values) {
        Bundle b = new Bundle();
        DialogFragment ret = new SpinnerDialogFragment();
        b.putSerializable("values", (Serializable) values);
        b.putInt("viewId", view.getId());
        if (view.getId() != View.NO_ID) {
            b.putInt("oldViewId", view.getId());
        }
        b.putInt("tempViewId", View.generateViewId());

        view.setId(b.getInt("tempViewId"));
        ret.setArguments(b);
        return ret;
    }

}
