//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.View;
import androidx.annotation.NonNull;

import java.util.HashSet;

class DpadJoystick extends SimpleSettingsJoystick implements View.OnKeyListener {
    private final InputDevice mDevice;
    private final boolean mHasSecondaryButton;
    DpadJoystick(final InputDevice device) {
        mDevice = device;
        boolean hasSecondary = false;
        if (mDevice == null && BaseActivity.isInUnitTest()) {
            hasSecondary = true;
        } else {
            for (int key : SECONDARY_KEYS) {
                if (mDevice.hasKeys(key)[0]) {
                    hasSecondary = true;
                    break;
                }
            }
        }
        mHasSecondaryButton = hasSecondary;
    }
    private final HashSet<Integer> mPressedButtons = new HashSet<>();
    @Override
    public boolean onKey(final View view, final int i, final KeyEvent keyEvent) {
        if (keyEvent.getDeviceId() != getHardwareId()) {
            return false;
        }
        boolean pressed = keyEvent.getAction() == KeyEvent.ACTION_DOWN;
        switch (keyEvent.getKeyCode()) {
            case KeyEvent.KEYCODE_DPAD_UP:
                setYValue(pressed ? -FULL_AMPLITUDE : 0);
                break;
            case KeyEvent.KEYCODE_DPAD_DOWN:
                setYValue(pressed ? FULL_AMPLITUDE : 0);
                break;
            case KeyEvent.KEYCODE_DPAD_LEFT:
                setXValue(pressed ? -FULL_AMPLITUDE : 0);
                break;
            case KeyEvent.KEYCODE_DPAD_RIGHT:
                setXValue(pressed ? FULL_AMPLITUDE : 0);
                break;
            case KeyEvent.KEYCODE_BUTTON_SELECT:
            case KeyEvent.KEYCODE_BUTTON_START:
                return false;
            default:
                if (KeyEvent.isGamepadButton(keyEvent.getKeyCode())) {
                    if (SECONDARY_KEYS.contains(keyEvent.getKeyCode())
                            && getSecondaryButtonUsage() != SecondaryUsage.FIRE) {
                        setSecondaryButtonPressed(pressed);
                    } else {
                        if (pressed) {
                            mPressedButtons.add(keyEvent.getKeyCode());
                        } else {
                            mPressedButtons.remove(keyEvent.getKeyCode());
                        }
                        setRealButtonsPressed(mPressedButtons);
                    }
                } else {
                    return false;
                }
        }
        return notifyListener();
    }

    @Override
    String getId() {
        return mDevice.getDescriptor();
    }

    @Override
    String getHardwareDescriptor() {
        return mDevice.getDescriptor();
    }

    @Override
    int getHardwareId() {
        return mDevice.getId();
    }

    @Override
    boolean canLockDiagonals() {
        return false;
    }

    @Override
    boolean hasSecondaryButton() {
        return mHasSecondaryButton;
    }

    @Override
    boolean deviceHasButtonWithKeycode(final int keycode) {
        return mDevice.hasKeys(keycode)[0];
    }

    @Override
    protected int getRessourceId() {
        return 0;
    }

    @NonNull
    @Override
    public String toString() {
        return mDevice.getName();
    }

}
