//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import static android.view.ViewGroup.LayoutParams.WRAP_CONTENT;

import static de.rainerhock.eightbitwonders.DownloaderFactory.DOWNLOADS_FOLDER;

import android.Manifest;
import android.annotation.SuppressLint;
import android.appwidget.AppWidgetManager;
import android.content.ActivityNotFoundException;
import android.content.ClipData;
import android.content.ComponentName;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.net.Uri;
import android.net.http.SslError;
import android.os.Build;
import android.os.Bundle;
import android.text.Editable;
import android.text.Html;
import android.text.TextWatcher;
import android.util.Base64;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;

import android.view.WindowManager;
import android.webkit.CookieManager;
import android.webkit.JavascriptInterface;
import android.webkit.SslErrorHandler;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContract;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.core.content.FileProvider;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.net.HttpURLConnection;

import java.net.MalformedURLException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.UUID;

import javax.net.ssl.SSLHandshakeException;

import de.rainerhock.eightbitwonders.vice.AndroidCompatibilityUtils;

/**
 * This class implements common utility functions.
 */
abstract class BaseActivity extends AppCompatActivity {
    static final int CHUNK_SIZE = 4096;
    static final int HTTP_STATUS_SSL_ERROR = 999;
    protected static final Map<String, Boolean> SUBFOLDERS = new HashMap<String, Boolean>() {{
       put("/packages/user", true);
       put("/packages/imported", false);
        put("/packages/streams", false);
    }};
    protected static final Map<String, Boolean> IS_IMPORTED = new HashMap<String, Boolean>() {{
        put("/packages/user", false);
        put("/packages/imported", true);
        put("/packages/streams", false);
    }};
    protected static final int DEFAULT_WAIT_TIME_IN_JUNIT_TEST = 2000;
    private static final String TAG = BaseActivity.class.getSimpleName();
    private static boolean mUnittestNetworkEnabled = true;
    @SuppressLint("WrongConstant")
    private final Map<Boolean, ActivityResultLauncher<Boolean>> mOpenDocumentLauncher
            = new LinkedHashMap<Boolean, ActivityResultLauncher<Boolean>>() {{
        for (Boolean b: Arrays.asList(true, false)) {
            put(b, registerForActivityResult(new FileSelectionContract(),
                    result -> {
                        if (result != null && result.getData() != null) {
                            onFileOpened(result.getData());
                        } else {
                            onFileOpenCancelled(b);
                        }
                        handleCommonLaunchresult(result);
                    }));


        }
    }};
    protected void handleCommonLaunchresult(final Intent resultData) {

    }
    private Useropts mUseropts = null;
    private final ActivityResultLauncher<Intent> mSharePackageContract = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(),
            result -> {
                if (result.getResultCode() == RESULT_OK) {
                    setResult(RESULT_OK);
                    finish();
                }
            });
    private boolean mRemovePreparingDialogs = false;

    static KeyEvent remapKeyEvent(final KeyEvent event) {
        if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(event.getKeyCode())) {
            return new KeyEvent(event.getDownTime(), event.getEventTime(),
                    event.getAction(),
                    KeyEvent.KEYCODE_DPAD_CENTER,
                    event.getRepeatCount(), event.getMetaState(), event.getDeviceId(),
                    event.getFlags(), event.getSource());
        } else if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(event.getKeyCode())) {
            return new KeyEvent(event.getDownTime(), event.getEventTime(),
                    event.getAction(),
                    KeyEvent.KEYCODE_BACK,
                    event.getRepeatCount(), event.getMetaState(), event.getDeviceId(),
                    event.getFlags(), event.getSource());
        } else {
            return event;
        }
    }


    /**
     * Set a TextView's typeface to Commodore-8-Bit style.
     * @param tv TextView to set typeface
     */
    final void setCBMFont(final TextView tv) {
        Typeface font = Typeface.createFromAsset(tv.getContext().getAssets(), "fonts/cbm.ttf");
        tv.setTypeface(font);
        //tv.setTextScaleX(0.8f);
        tv.setText(tv.getText().toString().toUpperCase(Locale.ROOT));
    }

    static void setViewListeners(final View view,
                                 final Runnable onNormalSelect,
                                 final Runnable onDpadSelect,
                                 final Runnable onDismiss) {
        view.setOnClickListener(v -> onNormalSelect.run());

        view.setOnKeyListener((v, keycode, keyEvent) -> {
            if (keyEvent.getAction() == KeyEvent.ACTION_UP
                    && EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)) {
                onDpadSelect.run();
                return true;
            }
            if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(keycode)) {
                if (keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    onDismiss.run();
                }
                return true;
            }
            return false;
        });
    }
    static void setViewListeners(final View view,
                                 final Runnable onSelect, final Runnable onDismiss) {
        setViewListeners(view, onSelect, onSelect, onDismiss);

    }
    private static String[] readAssetlist(final AssetManager am, final String folder) {
        try {
            InputStream is = am.open(folder + "/__files__");
            BufferedReader br = new BufferedReader(new InputStreamReader(is));
            List<String> l = new LinkedList<>();
            String line;
            while ((line = br.readLine()) != null) {
                l.add(line);
            }
            return l.toArray(new String[0]);
        } catch (IOException e) {
            Log.e(BaseActivity.class.getSimpleName(),
                    "Cannot open " + folder + " from assets.", e);
            // make compiler smile
        }
        try {
            return am.list(folder);
        } catch (IOException e) {
            return null;
        }

    }
    /**
     * Similar to View.getAssets(folder).list(), but checks for a file "__files__"
     * containing the list.
     * This may be a decent amount faster on slower device.
     * @param folder asset-Folder to be searched.
     *               See {@link android.content.res.AssetManager#list(String)} for furhter details.
     * @param v the view, for which the related assets will be used.
     * @return filenames
     * See {@link android.content.res.AssetManager#list(String)} for furhter details.
     */

    public static String[] getAssetsFromFolder(final View v, final String folder) {
        return readAssetlist(v.getContext().getAssets(), folder);
    }
    /**
     * Similar to getAssets(folder).list(), but checks for a file "__files__" containing the list.
     * This may be a decent amount faster on slower device.
     *
     * @param am AssetManager to use. Is ususally {@link #getAssets()},
     *           other values on instrumented test.
     * @param folder asset-Folder to be searched.
     *               See {@link AssetManager#list(String)} for furhter details.
     * @return filenames
     * See {@link android.content.res.AssetManager#list(String)} for furhter details.
     */
    public final String[] getAssetsFromFolder(final AssetManager am, final String folder) {
        return readAssetlist(am, folder);

    }
    protected final void waitIfInJunitTest() {

        try {
            Class<?> clz = Objects.requireNonNull(getClass().getClassLoader())
                    .loadClass("org.junit.Test");
            if (clz != null) {
                new Thread(() -> {
                //final long end = System.currentTimeMillis();
                    try {
                        Thread.sleep(DEFAULT_WAIT_TIME_IN_JUNIT_TEST);
                    } catch (InterruptedException e) {
                        // ok
                    }
                }).start();
            }
        } catch (ClassNotFoundException e) {
            // nothing;
        }
    }

    protected final void cancelRunningEmulation(final String oldName, final String newName,
                                          final Runnable runOnOk, final Runnable runOnCancel) {
        Emulation currentEmulation = Emulationslist.getCurrentEmulation();
        if (currentEmulation != null) {
            AlertDialogBuilder builder = new AlertDialogBuilder(this);
            final String titleName = oldName != null ? oldName : getResources().getString(
                    R.string.another_emulation);
            final String messageName = oldName != null ? oldName : getResources().getString(
                    R.string.other_emulation);
            builder.setIcon(android.R.drawable.ic_dialog_alert)
                    .setTitle(getResources().getString(R.string.currently_running, titleName))
                    .setMessage(getResources().getString(R.string.replace_running,
                            messageName, newName))
                    .setPositiveButton(android.R.string.yes, (dialogInterface, i)
                            -> currentEmulation.terminate(() -> runOnUiThread(() -> {
                        dialogInterface.dismiss();
                        runOnOk.run();
                    })))
                    .setNegativeButton(android.R.string.no, (dialogInterface, i)
                            -> runOnUiThread(() -> {
                                dialogInterface.dismiss();
                                runOnCancel.run();
                            })).create().show();
        } else {
            runOnOk.run();
        }
    }
    private static int mgrantAccessId = 0;
    private static final  Map<Integer, Runnable> RUN_AFTER_ACCESS_GRANTED = new HashMap<>();
    private static final  Map<Integer, Runnable> RUN_AFTER_ACCESS_DENIED = new HashMap<>();
    protected final void executeWithFilesystemAccess(@Nullable final Runnable runWhenRequesting,
                                               @NonNull final Runnable runIfAlreadyGranted,
                                               @Nullable final Runnable runIfDenied,
                                               @NonNull final Runnable runAfterGranting) {
        if ((ContextCompat.checkSelfPermission(

                this,
                android.Manifest.permission.READ_EXTERNAL_STORAGE)
                == PackageManager.PERMISSION_DENIED
                || ContextCompat.checkSelfPermission(
                this,
                android.Manifest.permission.WRITE_EXTERNAL_STORAGE)
                == PackageManager.PERMISSION_DENIED)
                && (Build.VERSION.SDK_INT < Build.VERSION_CODES.TIRAMISU)) {
            if (runWhenRequesting != null) {
                runWhenRequesting.run();
            }
            synchronized (RUN_AFTER_ACCESS_GRANTED) {
                mgrantAccessId++;
                RUN_AFTER_ACCESS_GRANTED.put(mgrantAccessId, runAfterGranting);
                RUN_AFTER_ACCESS_DENIED.put(mgrantAccessId, runIfDenied);
            }
            ActivityCompat
                    .requestPermissions(this,
                            new String[]{android.Manifest.permission.READ_EXTERNAL_STORAGE,
                                    Manifest.permission.WRITE_EXTERNAL_STORAGE},
                            mgrantAccessId);
        } else {
            runIfAlreadyGranted.run();

        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        mUseropts = null;
    }

    @Override
    public void onRequestPermissionsResult(final int requestCode,
                                           @NonNull final String[] permissions,
                                           @NonNull final int[] grantResults) {
        if (RUN_AFTER_ACCESS_GRANTED.containsKey(requestCode)) {
            Runnable run = RUN_AFTER_ACCESS_GRANTED.get(requestCode);
            boolean granted = false;
            if (run != null) {
                for (int r : grantResults) {
                    if (r != PackageManager.PERMISSION_DENIED) {
                        run.run();
                        granted = true;
                        break;
                    }
                }
            }
            if (!granted && RUN_AFTER_ACCESS_DENIED.containsKey(requestCode)) {
                Toast.makeText(this, R.string.access_denied, Toast.LENGTH_SHORT).show();
                Runnable r = RUN_AFTER_ACCESS_DENIED.get(requestCode);
                if (r != null) {
                    r.run();
                }

            }
            super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }
    }

    protected final void storeZip(final Map<String, ByteArrayOutputStream> contents,
                            final Properties props,
                            final Map<String, ByteArrayOutputStream> savestates)
            throws IOException {

        final String targetpath = getApplicationContext().getFilesDir()
                + "/packages/imported/" + props.getProperty("id") + "/";

        if (new File(targetpath + ".originals").mkdirs()
                || new File(targetpath + ".originals").isDirectory()) {
            for (String filename : contents.keySet()) {
                for (String f : new String[]{targetpath + filename,
                        targetpath + ".originals/" + filename}) {
                    FileOutputStream fos = new FileOutputStream(f);
                    fos.write(Objects.requireNonNull(contents
                            .get(filename)).toByteArray());
                    fos.close();
                }
            }
            File snapshotRoot = getSnapshotPath(
                    props.getProperty("emulation"),
                    props.getProperty("id"));
            for (String filename : savestates.keySet()) {
                ByteArrayOutputStream baos = savestates.get(filename);
                File parent = new File(snapshotRoot, filename)
                        .getParentFile();
                if (baos != null && parent != null) {
                    if (parent.exists() || parent.mkdirs()) {
                        FileOutputStream fos = new FileOutputStream(
                                new File(snapshotRoot, filename));
                        fos.write(baos.toByteArray());
                        fos.close();
                    }
                }
            }

            Intent i = new Intent(
                    "de.rainerhock.eightbitwonders.update");
            i.setPackage("de.rainerhock.eightbitwonders");
            sendBroadcast(i);
            onContentImported(props.getProperty("name"));
        }
    }
    final Useropts getCurrentUseropts() {
        if (Emulationslist.getCurrentEmulation() == null) {
            return new Useropts();
        }
        if (mUseropts == null) {
            mUseropts = new Useropts();
            if (Emulationslist.getCurrentEmulation() != null) {
                mUseropts.setCurrentEmulation(this,
                        Emulationslist.getCurrentEmulation().getEmulatorId(),
                        Emulationslist.getCurrentEmulation().getConfigurationId());
            }
        }
        return mUseropts;
    }

    final void executeEmulationStart(final EmulationConfiguration conf, final Runnable r) {
        if (!conf.getProperty("canBeAutostarted", "true").equals("true")) {
            AlertDialogBuilder builder = new AlertDialogBuilder(this);
            builder.setTitle(R.string.IDS_AUTOSTART)
                    .setMessage(R.string.config_has_no_autostart)
                    .setPositiveButton(android.R.string.ok,
                            (dialogInterface, i) -> {
                                r.run();
                                dialogInterface.dismiss();
                            })
                    .setNegativeButton(android.R.string.cancel,
                            (dialogInterface, i) -> dialogInterface.dismiss());
            runOnUiThread(() -> builder.create().show());

        } else {
            runOnUiThread(r);
        }
    }
    protected static final String DOWNLOAD_DIALOG = "DOWNLOAD_DIALOG";

    protected interface ErrorHandler {
        void handleException(@NonNull DownloaderFactory.AdditionalDownloads.DownloadException e);
    }
    protected void hidePreparingDialogs() {
        FragmentManager fm = getSupportFragmentManager();
        FragmentTransaction fta = fm.beginTransaction();
        Fragment downloadFragment = fm.findFragmentByTag(DOWNLOAD_DIALOG);
        if (downloadFragment != null) {
            fta = fta.remove(downloadFragment);
        }
        fta.commitNow();
    }
    protected static final String STARTED_EMULATION = "STARTED_EMULATION";
    protected void showDownloadDialog() {
        runOnUiThread(() -> {
            FragmentManager fm = getSupportFragmentManager();
            if (fm.findFragmentByTag(DOWNLOAD_DIALOG) == null) {
                try {
                    fm.beginTransaction().add(
                            new EmulationLauncherActivity.DownloadDialogFragment(), DOWNLOAD_DIALOG)
                            .commitNow();
                } catch (IllegalStateException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    } else {
                        Log.v(TAG, "showDownloadDialog failed", e);
                    }
                }
            }
        });
    }
    protected final void downloadFiles(final EmulationConfiguration conf,
                                 @NonNull final Runnable runLaunchSuccess,
                                 @NonNull final ErrorHandler runOnError) {
        try {
            DownloaderFactory.Downloader prepare = Emulationslist.getEmulationPreparer(
                    this, conf);
            DownloaderFactory.Downloader addition = conf.getAdditionalDownloader(this);
            if (prepare != null || addition != null) {
                new Thread(() -> {

                    boolean serverOnline = true;
                    boolean downloadsFinished = true;
                    if (prepare != null) {

                        downloadsFinished = prepare.isContentFullyDownloaded();
                        if (!downloadsFinished) {
                            serverOnline = prepare.checkForServices();
                        }
                    }
                    if (addition != null && downloadsFinished) {
                        downloadsFinished = addition.isContentFullyDownloaded();
                    }
                    if (addition != null && serverOnline && !downloadsFinished) {
                        serverOnline = addition.checkForServices();
                    }

                    Runnable dismissDialog;
                    //ProgressDialog statusdialog;
                    if (!downloadsFinished) {
                        if (serverOnline) {
                            final FragmentManager fm = getSupportFragmentManager();
                            dismissDialog = () -> {
                                mRemovePreparingDialogs = true;
                                Fragment f = fm.findFragmentByTag(DOWNLOAD_DIALOG);
                                if (f != null) {
                                    try {
                                        fm.beginTransaction().remove(f).commitNow();
                                    } catch (IllegalStateException e) {
                                        if (BuildConfig.DEBUG) {
                                            throw new RuntimeException(e);
                                        }
                                    }
                                }
                            };
                            showDownloadDialog();
                        } else {
                            dismissDialog = () -> {
                            };
                        }
                        List<DownloaderFactory.HttpRequest> httpRequests = getHttpRequests();
                        for (DownloaderFactory.Downloader downloader
                                : Arrays.asList(prepare, addition)) {
                            if (downloader != null) {
                                try {
                                    downloader.run(httpRequests);
                                } catch (IOException e) {
                                    Log.v(getClass().getSimpleName(),
                                            "Downloader +" + downloader
                                                    + " failed with Exception",
                                            e);
                                    waitIfInJunitTest();
                                    runOnUiThread(() -> {
                                        dismissDialog.run();
                                        showErrorDialog(
                                                getResources().getString(
                                                        R.string.IDS_CANNOT_LOAD_ROMSET_FILE),
                                                e.getLocalizedMessage());
                                    });
                                    return;

                                } catch (DownloaderFactory.AdditionalDownloads.DownloadException
                                        de) {
                                    runOnUiThread(() -> {
                                        dismissDialog.run();
                                        runOnError.handleException(de);
                                    });
                                    return;
                                }
                            }
                        }
                    } else {
                        dismissDialog = () -> { };
                    }
                    runOnUiThread(() -> {

                        try {
                            runLaunchSuccess.run();
                            dismissDialog.run();
                        } catch (IllegalArgumentException  | IllegalStateException e) {
                            if (BuildConfig.DEBUG) {
                                throw (e);
                            }
                        }
                    });
                }).start();
            } else {
                runLaunchSuccess.run();
            }

        } catch (Exception e) {
            showErrorDialog(
                    getResources().getString(R.string.IDS_CANNOT_LOAD_ROMSET_FILE),
                    e.getLocalizedMessage());
        }
    }

    interface MyTextWatcher {
        void onTextChanged(String newValue);
    }

    static class FileCreateionContract extends ActivityResultContract<Intent, Intent> {

        @NonNull
        @Override
        public Intent createIntent(final @NonNull Context context, final Intent intent) {
            return intent;
        }

        @Override
        public Intent parseResult(final int resultCode, final @Nullable Intent intent) {
            return intent;
        }
    }
    static class FileSelectionContract extends ActivityResultContract<Boolean, Intent> {

        @NonNull
        @Override
        public Intent createIntent(final @NonNull Context context, final Boolean retry) {
            Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
            intent.addCategory(Intent.CATEGORY_OPENABLE);
            if (retry) {
                intent.putExtra("retry", true);
            }
            intent.setType("application/*");
            return BaseActivity.createChooserIntent(intent, null);

        }

        @Override
        public Intent parseResult(final int resultCode, final @Nullable Intent intent) {
            return intent;
        }
    }

    static TextWatcher createWatcher(final MyTextWatcher lambda) {
        return new TextWatcher() {
            @Override
            public void beforeTextChanged(final CharSequence charSequence,
                                          final int i, final int i1, final int i2) {
            }
            @Override
            public void onTextChanged(final CharSequence charSequence,
                                      final int i, final int i1, final int i2) {
                lambda.onTextChanged(charSequence.toString());
            }

            @Override
            public void afterTextChanged(final Editable editable) {
            }
        };
    }

    /**
     * Sets the requested screen orientation if the intent has a data element "screen_orientation".
     * @param savedInstanceState passed to {@link AppCompatActivity#onCreate(Bundle)}
     */
    @Override
    protected void onCreate(final @Nullable Bundle savedInstanceState) {
        int defaultval = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
        int val = getIntent().getIntExtra("screen_orientation", defaultval);
        if (getIntent() != null) {
            if (val != defaultval) {
                try {
                    setRequestedOrientation(val);
                } catch (IllegalStateException e) {
                    // that's ok, may occure on API 26, but doesn't matter.
                }
            }
        }
        super.onCreate(savedInstanceState);
        AndroidCompatibilityUtils.fixInsets(this);
    }

    protected final void shareFile(final File pkg, final String title) {
        Intent shareIntent = new Intent();
        Uri uri = FileProvider.getUriForFile(this, "de.rainerhock.eightbitwonders.fileprovider",
                pkg);
        shareIntent.setAction(Intent.ACTION_SEND);
        shareIntent.putExtra(Intent.EXTRA_STREAM, uri);
        shareIntent.putExtra(Intent.EXTRA_TITLE, title);
        shareIntent.putExtra(Intent.EXTRA_TEXT,
                Html.fromHtml(getString(R.string.use_8bw, getString(R.string.app_url))));
        shareIntent.setType("application/octet-stream");
        ClipData cd = ClipData.newRawUri("", uri);
        shareIntent.setClipData(cd);
        shareIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        mSharePackageContract.launch(BaseActivity.createChooserIntent(
                shareIntent, getString(R.string.share)));


    }

    protected void onFileOpened(final Uri uri) { }
    protected void onFileOpenCancelled(final boolean wasRetry) { }

    /**
     * Start Opening a file.
     * @param isRetry if this is the second attempt
     * @return if the activity to select a file could be started.
     */
    protected boolean openFile(final boolean isRetry) {
        try {
            Objects.requireNonNull(mOpenDocumentLauncher.get(isRetry)).launch(isRetry);
            return true;
        } catch (ActivityNotFoundException e) {
            return false;
        }
    }
    protected void onFileCreated(final Uri uri) {

    }

    /**
     * Show an error message.
     * @param title Message title
     * @param message Message text
     * @param runOnDismiss Runnable to be run after message was confirmed.
     */
    protected void showErrorDialog(final String title, final String message,
                                   final @NonNull Runnable runOnDismiss) {
        AlertDialog.Builder builder = new AlertDialogBuilder(this);
        builder.setIcon(android.R.drawable.stat_notify_error);
        if (title != null) {
            builder.setTitle(title);
        }
        if (message != null) {
            builder.setMessage(message);
        }
        builder.setPositiveButton(android.R.string.ok,
                (dialogInterface, i) -> dialogInterface.dismiss());

        builder.setOnDismissListener(dialogInterface -> {
            runOnDismiss.run();
            dialogInterface.dismiss();
        });
        builder.create().show();

    }
    protected final boolean isTv() {
        if (getPackageManager().hasSystemFeature(PackageManager.FEATURE_TELEVISION)
                || getPackageManager().hasSystemFeature(PackageManager.FEATURE_LEANBACK)) {
            return true;
        }
        return !getPackageManager().hasSystemFeature(PackageManager.FEATURE_TOUCHSCREEN);
    }

    /**
     * Show an error message.
     * @param title Message title
     * @param message Message text
     */
    protected void showErrorDialog(final String title, final String message) {
        showErrorDialog(title, message, () -> { });
    }
    static class AlertDialogBuilder extends AlertDialog.Builder {
        AlertDialogBuilder(@NonNull final Context context) {
            super(context);
            init(context);
        }
        private EmulationActivity mActivity = null;
        private String mUuid = null;
        private void init(@NonNull final Context context) {
            if (context instanceof EmulationActivity) {
                mActivity = (EmulationActivity) context;
                mUuid = UUID.randomUUID().toString();
            } else {
                mActivity = null;
            }
        }

        @Override
        public AlertDialogBuilder setOnDismissListener(
                final DialogInterface.OnDismissListener onDismissListener) {

            DialogInterface.OnDismissListener newListener;

            if (mActivity != null) {
                newListener = dialogInterface -> {
                    Fragment f = mActivity.getSupportFragmentManager().findFragmentByTag(mUuid);
                    if (f != null) {
                        mActivity.getSupportFragmentManager().beginTransaction().remove(f).commit();
                    }
                    onDismissListener.onDismiss(dialogInterface);
                };
            } else {
                newListener = onDismissListener;
            }
            super.setOnDismissListener(newListener);
            return this;
        }

        @Override
        public AlertDialog show() {
            if (mActivity != null) {
                mActivity.getSupportFragmentManager().beginTransaction()
                        .add(new EmulationDialogFragment(), mUuid)
                        .commit();
            }
            AlertDialog ret = super.show();

            ret.setOnShowListener(dialogInterface -> {
                Rect r1 = new Rect();
                if (ret.getWindow() != null) {
                    ret.getWindow().getDecorView().getGlobalVisibleRect(r1);
                }
            });
            if (ret.getWindow() != null) {
                WindowManager.LayoutParams layoutParams = new WindowManager.LayoutParams();
                layoutParams.copyFrom(ret.getWindow().getAttributes());
                layoutParams.width = WRAP_CONTENT;
                layoutParams.height = WRAP_CONTENT;
                ret.getWindow().setAttributes(layoutParams);
            }
            return ret;
        }
    }
    private int mJsDownloadStatuscode = 0;
    private final ByteArrayOutputStream mJsDownloadData = new ByteArrayOutputStream();
    //CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
    @JavascriptInterface
    public final void onJsDownloadFinished(final int resultcode, final String base64body) {
        Log.v("WebViewDownloader",
                String.format("onJsDownloadFinished (%d, %s)", resultcode,
                        base64body.isEmpty() ? "empty string" : "data string"));
        synchronized (mJsDownloadData) {
            mJsDownloadData.reset();
            byte[] data = Base64.decode(base64body, Base64.DEFAULT);
            mJsDownloadData.write(data, 0, data.length);
            mJsDownloadStatuscode = resultcode;
            mJsDownloadData.notifyAll();
        }
    }
    //CHECKSTYLE DISABLE MissingJavadocMethod FOR 1 LINES
    @JavascriptInterface
    public final void log(final String s) {
        Log.v("WebViewDownloader", s);
    }

    abstract static class HttpRequestBase implements DownloaderFactory.HttpRequest {
        private final List<String> mIgnoreHosts = new LinkedList<>();
        protected void ignoreHost(final URL url) {
            mIgnoreHosts.add(url.getHost());
        }
        protected boolean hostCanBeIgnored(final URL url) {
            return mIgnoreHosts.contains(url.getHost());
        }
        @NonNull
        @Override
        public abstract DownloaderFactory.HttpResult execute(URL url) throws IOException;
    }
    static class SSlErrorResult implements DownloaderFactory.HttpResult {
        @Override
        public int getResultcode() {
            return HttpURLConnection.HTTP_NOT_ACCEPTABLE;
        }

        @Override
        public byte[] getBody() {
            return null;
        }
    }

    final @NonNull List<DownloaderFactory.HttpRequest> getHttpRequests() {
        List<DownloaderFactory.HttpRequest> ret = new LinkedList<>();
        ret.add(createApiHttpClient());
        try {
            View v = findViewById(R.id.wv_downloads);
            CookieManager.getInstance();
            if (v instanceof WebView) {
                ret.add(createWebViewHttpClient((WebView) v));
            }
        } catch (Exception e) {
            // ok
        }
        return ret;
    }
    @NonNull
    private HttpRequestBase createWebViewHttpClient(final WebView wv) {
        // StandardCharsets.UTF_8.name() > JDK 7
        //noinspection RedundantThrows
        return new HttpRequestBase() {
            @NonNull
            @Override
            @SuppressLint("SetJavaScriptEnabled")
            public DownloaderFactory.HttpResult execute(final URL url) throws IOException {
                if (hostCanBeIgnored(url)) {
                    return new SSlErrorResult();
                }
                runOnUiThread(() -> {
                    wv.setVisibility(View.VISIBLE);
                    wv.getSettings().setJavaScriptEnabled(true);
                    wv.addJavascriptInterface(BaseActivity.this, "callback");
                    wv.setWebViewClient(new WebViewClient() {
                        @SuppressLint("WebViewClientOnReceivedSslError")
                        @Override
                        public void onReceivedSslError(final WebView view,
                                                       final SslErrorHandler handler,
                                                       final SslError error) {
                            // Since we hash check the results
                            // the man in the middle is not a problem
                            handler.cancel();
                            ignoreHost(url);
                            synchronized (mJsDownloadData) {
                                mJsDownloadStatuscode = HTTP_STATUS_SSL_ERROR;
                                mJsDownloadData.notifyAll();
                            }
                        }

                        public boolean shouldOverrideUrlLoading(final WebView view,
                                                                final WebResourceRequest
                                                                        request) {
                            return false;
                        }

                        @Override
                        public void onPageFinished(final WebView view, final String ignore) {
                            super.onPageFinished(view, url.toString());
                            wv.evaluateJavascript(
                                    String.format("javascript:download('%s');", url), null);
                        }
                    });
                    InputStream is = getResources().openRawResource(R.raw.download);
                    ByteArrayOutputStream result = new ByteArrayOutputStream();
                    byte[] buffer = new byte[CHUNK_SIZE];
                    try {
                        for (int length; (length = is.read(buffer)) >= 0;) {
                            result.write(buffer, 0, length);
                        }
                        is.close();
                    } catch (IOException e) {
                        // ok
                    }
                    try {
                        String baseUrl = new URL(url.getProtocol(), url.getHost(), "")
                                .toString();
                        wv.loadDataWithBaseURL(baseUrl, result.toString(),
                                "text/html", "utf-8", baseUrl);
                    } catch (MalformedURLException e) {
                        throw new RuntimeException(e);
                    }
                });
                try {
                    final int[] statuscode = new int[1];
                    Thread t = webDownloaderThread(url, statuscode);
                    t.join();
                    return new DownloaderFactory.HttpResult() {

                        @Override
                        public int getResultcode() {
                            return mJsDownloadStatuscode;
                        }

                        @Override
                        public byte[] getBody() {
                            if (statuscode[0] >= HttpURLConnection.HTTP_OK
                                    && statuscode[0] < HttpURLConnection.HTTP_MOVED_PERM) {
                                return mJsDownloadData.toByteArray();
                            }
                            return null;
                        }
                    };
                } catch (InterruptedException e) {
                    return new DownloaderFactory.HttpResult() {

                        @Override
                        public int getResultcode() {
                            return HttpURLConnection.HTTP_CLIENT_TIMEOUT;
                        }

                        @Override
                        public byte[] getBody() {
                            return null;
                        }
                    };
                }
            }

            @NonNull
            private Thread webDownloaderThread(final URL url, final int[] statuscode) {
                Thread t = new Thread(() -> {
                    synchronized (mJsDownloadData) {
                        try {
                            Log.v("WebViewDownloader",
                                    "waiting for download " + url);
                            mJsDownloadData.wait();
                            statuscode[0] = mJsDownloadStatuscode;
                            Log.v("WebViewDownloader",
                                    "download finished (" + statuscode[0] + ")");
                        } catch (InterruptedException e) {
                            throw new RuntimeException(e);
                        }
                    }
                });
                t.start();
                return t;
            }
        };
    }
    private static final Map<String, Map<String, byte[]>> GLOBAL_CACHED_DATA = new HashMap<>();
    protected final void cacheStreamingData(final EmulationConfiguration conf,
                                      final URL url, final byte[] data) {
        if (!GLOBAL_CACHED_DATA.containsKey(conf.getId())) {
            GLOBAL_CACHED_DATA.put(conf.getId(), new HashMap<>());
        }
        Objects.requireNonNull(GLOBAL_CACHED_DATA.get(conf.getId())).put(url.toString(), data);
    }
    protected final void removeStreamData(final EmulationConfiguration conf) {
        GLOBAL_CACHED_DATA.remove(conf.getId());
    }
    protected final byte[] getStreamData(final String configurationId, final String url) {
        if (GLOBAL_CACHED_DATA.get(configurationId) != null) {
            return Objects.requireNonNull(GLOBAL_CACHED_DATA.get(configurationId)).get(url);
        }
        return null;
    }
    private static final char[] HEX_ARRAY = "0123456789abcdef".toCharArray();
    private static final int HEX_MASK = 0x0F;

    /**
     * Create sha256 has of given data as hexadecimal String (00-ff * n).
     * @param data to be hashed
     * @return hexadecimal string
     */
    public static String hexSha256Hash(final byte[] data) {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(data);
            byte[] digest = md.digest();
            char[] hexChars = new char[digest.length * 2];
            for (int j = 0; j < digest.length; j++) {
                int v = digest[j] & 0xFF;
                hexChars[j * 2] = HEX_ARRAY[v >>> 4];
                hexChars[j * 2 + 1] = HEX_ARRAY[v & HEX_MASK];
            }
            return new String(hexChars);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Create sha256 has of given data.
     * @param data to be hashed
     * @return hash
     */
    public static byte[] sha256Hash(final byte[] data) {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("SHA-256");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        md.update(data);
        return md.digest();
    }
    @NonNull
    private static HttpRequestBase createApiHttpClient() {
        return new HttpRequestBase() {
            @NonNull
            @Override
            public DownloaderFactory.HttpResult execute(final URL url) throws IOException {
                try {
                    if (hostCanBeIgnored(url)) {
                        return new SSlErrorResult();
                    }
                    HttpURLConnection conn = (HttpURLConnection) url.openConnection();
                    final int res = conn.getResponseCode();
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();

                    byte[] chunk = new byte[CHUNK_SIZE];
                    while (true) {
                        int read = conn.getInputStream().read(chunk);
                        if (read < 0) {
                            break;
                        }
                        baos.write(chunk, 0, read);
                    }
                    final byte[] data = baos.toByteArray();
                    conn.disconnect();
                    return new DownloaderFactory.HttpResult() {
                        @Override
                        public int getResultcode() {
                            return res;
                        }

                        @Override
                        public byte[] getBody() {
                            return data;
                        }
                    };
                } catch (SSLHandshakeException e) {
                    ignoreHost(url);
                    throw (e);
                }
            }
        };
    }
    protected final byte[] getContentData(final EmulationConfiguration conf, final String address) {
        byte[] cached = getStreamData(conf.getId(), address);
        if (cached != null) {
            File f = new File(conf.getFilepath(
                    BaseActivity.hexSha256Hash(cached) + ".ovl"));
            if (f.exists()) {
                try {
                    FileInputStream fis = new FileInputStream(f);
                    byte[] overlay = new byte[fis.available()];
                    if (fis.read(overlay) >= 0) {
                        for (int i = 0; i < Math.min(cached.length, overlay.length); i++) {
                            cached[i] += overlay[i];
                        }
                    }
                    fis.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            return cached;
        }
        return FileUtil.getContentData(this, address);
    }
    protected final void putContentData(final EmulationConfiguration conf,
                                        final String address, final byte[] data) {
        byte[] original = getStreamData(conf.getId(), address);
        if (original != null) {
            if (!Arrays.equals(original, data)) {
                byte[] overlay = Arrays.copyOf(original, original.length);
                for (int i = 0; i < Math.min(original.length, data.length); i++) {
                    overlay[i] -= original[i];
                }
                File f = new File(conf.getFilepath(
                        BaseActivity.hexSha256Hash(original) + ".ovl"));
                if (f.getParentFile() != null && f.getParentFile().mkdirs()) {
                    try {
                        FileOutputStream fos = new  FileOutputStream(f);
                        fos.write(overlay);
                        fos.close();
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        } else {
            FileUtil.putContentData(this, address, data);
        }
    }
    final void removeWidget(final EmulationConfiguration conf) {
        LaunchTileWidget.removeWidget(this, conf);
        updateWidgets();
    }
    final void updateWidgets() {
        Intent updateWidgetsIntent = new Intent(this, LaunchTileWidget.class);
        updateWidgetsIntent.setAction(AppWidgetManager.ACTION_APPWIDGET_UPDATE);
        int[] ids = AppWidgetManager.getInstance(getApplication()).getAppWidgetIds(
                new ComponentName(getApplication(), LaunchTileWidget.class));
        updateWidgetsIntent.putExtra(AppWidgetManager.EXTRA_APPWIDGET_IDS, ids);
        sendBroadcast(updateWidgetsIntent);
    }

    protected final void showReallyDeleteDialog(final @NonNull Runnable clearDownloadOnly,
                                                final @Nullable Runnable clearWithUseropts) {
        AlertDialog.Builder builder = new AlertDialogBuilder(this);
        builder.setIcon(android.R.drawable.ic_dialog_alert).setTitle(R.string.really_delete);
        final boolean[] items = new boolean[
                getResources().getStringArray(R.array.delete_options).length];
        builder.setPositiveButton(android.R.string.ok,
                (dialogInterface, i) -> {
                        if (items[0] && clearWithUseropts != null) {
                            clearWithUseropts.run();
                        } else {
                            clearDownloadOnly.run();
                        }
                        updateUsercontentUI();
                        dialogInterface.dismiss();
                })
                .setNegativeButton(android.R.string.cancel,
                    ((dialogInterface, i) -> dialogInterface.dismiss()))
                .setCancelable(true);
        if (clearWithUseropts != null && clearWithUseropts != clearDownloadOnly) {
            builder.setMultiChoiceItems(R.array.delete_options, items,
                    (dialogInterface, i, b) -> items[i] = b);
        }
        if (clearWithUseropts == clearDownloadOnly) {
            builder.setMessage(R.string.will_delete_everything);
        }
        builder.create().show();


    }

    void updateUsercontentUI() {
    }

    protected final boolean restrictPayments() {
        return !getClass().getName().startsWith(getPackageName());
    }
    protected final File getSnapshotPath(final EmulationConfiguration conf) {
        return getSnapshotPath(conf.getEmulatorId(), conf.getId());
    }
    protected final File getSnapshotPath(final String emulatorId, final String configId) {
        return new File(getFilesDir() + File.separator
                + "snapshots" + File.separator
                + emulatorId + File.separator + configId);

    }

    static class Deleter implements Runnable, Serializable {
        private void deleteFolder(final File file) {
            if (file.listFiles() != null) {
                for (File subFile : Objects.requireNonNull(file.listFiles())) {
                    if (subFile.isDirectory()) {
                        deleteFolder(subFile);
                    } else {
                        if (!subFile.delete()) {
                            Log.v(getClass().getSimpleName(), "Could not delete folder, ignored.");
                        }
                    }
                }
            }
            if (!file.delete()) {
                Log.v(getClass().getSimpleName(), "Could not delete folder, ignored.");
            }
        }

        private final String mPath;
        Deleter(final File folder) {
            mPath = folder.getPath();
        }
        @Override
        public void run() {
            deleteFolder(new File(mPath));
        }
    }
    private static final Map<String, EmulationConfiguration.Orientation> ORIENTATIONS
            = new LinkedHashMap<>();
    protected final EmulationConfiguration.Orientation getEmulationScreenOrientation(
            final EmulationConfiguration conf) {
        return getEmulationScreenOrientation(this, conf);
    }
    static EmulationConfiguration.Orientation getEmulationScreenOrientation(
            final Context ctx, final EmulationConfiguration conf) {
        if (ORIENTATIONS.isEmpty()) {
            ORIENTATIONS.put(ctx.getResources().getString(R.string.value_orientation_portrait),
                    EmulationConfiguration.Orientation.PORTRAIT);
            ORIENTATIONS.put(ctx.getResources().getString(R.string.value_orientation_landscape),
                    EmulationConfiguration.Orientation.LANDSCAPE);
            ORIENTATIONS.put(ctx.getResources().getString(R.string.value_orientation_device),
                    EmulationConfiguration.Orientation.DEFAULT);
        }
        Useropts opts = new Useropts();
        opts.setCurrentEmulation(ctx, conf.getEmulatorId(), conf.getId());
        String key = ctx.getResources().getString(R.string.useropt_default_orientation);
        String opt = opts.getStringValue(key, "");
        if (ORIENTATIONS.containsKey(opt)) {
            return ORIENTATIONS.get(opt);
        } else {
            return conf.getBestOrientation();
        }
    }
    private File getInitialSnapshotRoot(final EmulationConfiguration conf) {
        String confVersion = conf.getProperty("internal_version", "default");
        File rootDir = new File(getCacheDir(), "initial_snapshots");
        File configDir = new File(rootDir,
                conf.getEmulatorId() + "_" + conf.getId() + "_" + confVersion);
        if (configDir.isDirectory() || configDir.mkdirs()) {
            return configDir;
        }
        return null;
    }
    //CHECKSTYLE DISABLE _MissingJavadocMethod FOR 1 LINES
    protected final File getInitialSnapshotPathImpl(final EmulationConfiguration conf,
                                                final int level) {
        File configDir = getInitialSnapshotRoot(conf);
        if (configDir != null) {
            return new File(configDir, level + ".vsf");
        }
        return null;
    }
    protected final void removeInitialSnapshots(final EmulationConfiguration conf) {
        File configDir = getInitialSnapshotRoot(conf);
        if (configDir != null) {
            File[] files = configDir.listFiles();
            if (files != null) {
                for (File f : files) {
                    //noinspection ResultOfMethodCallIgnored
                    f.delete();
                }
            }
            //noinspection ResultOfMethodCallIgnored
            configDir.delete();
        }
    }
    protected final List<EmulationConfiguration> getLocalConfigurations() {
        List<EmulationConfiguration> result = new LinkedList<>();
        for (String subfolder: SUBFOLDERS.keySet()) {
            String parentFolder = getApplicationContext().getFilesDir() + subfolder;
            File userconfigs = new File(parentFolder);

            if (userconfigs.exists() && userconfigs.isDirectory()) {

                for (String path : Objects.requireNonNull(userconfigs.list())) {
                    final File folder = new File(userconfigs, path);
                    if (folder.isDirectory()) {
                        EmulationConfiguration configuration
                                = new ConfigurationFactory().createConfiguration(this,
                                s -> {
                                    try {
                                        //noinspection IOStreamConstructor
                                        return new FileInputStream(new File(folder, s));
                                    } catch (IOException e) {
                                        return null;
                                    }
                                },
                                () -> {
                                    final List<String> ret = new LinkedList<>();
                                    for (String name : Objects.requireNonNull(folder.list())) {
                                        String subpath = folder + File.separator + name;
                                        if (new File(subpath).isFile()) {
                                            ret.add(name);
                                        }
                                    }
                                    return ret;
                                },
                                new Deleter(folder),
                                Boolean.TRUE.equals(SUBFOLDERS.get(subfolder)),
                                Boolean.TRUE.equals(IS_IMPORTED.get(subfolder)),
                                folder);
                        if (configuration != null) {
                            result.add(configuration);
                        }
                    }
                }
            }
        }
        return result;
    }
    protected final List<EmulationConfiguration> getMachineConfigurations() {
        List<EmulationConfiguration> result = new LinkedList<>();
        for (String emulationId : Emulationslist.getEmulationIds()) {
            EmulationConfiguration conf = new ConfigurationFactory().createConfiguration(
                    this, emulationId);
            result.add(conf);
        }
        return result;
    }
    protected final List<EmulationConfiguration> getAssetConfigurations(final AssetManager am,
                                                                  final String assetFolder,
                                                                  final String filesystemFolder) {
        List<EmulationConfiguration> result = new LinkedList<>();
        try {

            final File destination;
            if (filesystemFolder != null) {
                destination = new File(filesystemFolder);
            } else {
                destination = null;
            }

            for (String folder : getAssetsFromFolder(am, assetFolder)) {
                Log.v(getClass().getSimpleName(), "Reading package " + folder);
                String path = assetFolder + "/" + folder;
                EmulationConfiguration configuration =
                        new ConfigurationFactory().createConfiguration(this,
                                filename -> {
                                    try {
                                        return am.open(path + "/" + filename);
                                    } catch (IOException e) {
                                        return null;
                                    }
                                },
                                () -> {
                                    try {
                                        List<String> ret = new LinkedList<>();
                                        InputStreamReader ir = new InputStreamReader(
                                                am.open(path + "/__files__"));
                                        BufferedReader br = new BufferedReader(ir);
                                        String line = "";
                                        while (line != null) {
                                            line = br.readLine();
                                            if (line != null) {
                                                ret.add(line);
                                            }
                                        }
                                        return ret;
                                    } catch (IOException e) {
                                        return null;
                                    }
                                },
                                null,
                                false,
                                false,
                                destination);
                if (configuration != null) {
                    result.add(configuration);
                }
            }
        } catch (Exception e) {
            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            }

        }
        return result;
    }

    protected final void requestFocusForFirstTile(final TileView tv) {
        tv.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
            @Override
            //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
            public void onLayoutChange(final View view, final int i, final int i1,
                                       final int i2, final int i3, final int i4,
                                       final int i5, final int i6, final int i7) {
                tv.post(() -> {
                    tv.requestFocus();
                    tv.getChildAt(0).requestFocus();
                });
                tv.removeOnLayoutChangeListener(this);
            }
        });

    }

    /**
     * To be run when local content was successfully imported.
     * @param name of the content package.
     */
    protected void onContentImported(final String name) {
        Toast.makeText(BaseActivity.this,
                getResources().getString(
                        R.string.import_success,
                        name),
                Toast.LENGTH_SHORT).show();
    }
    static void setUnittestNetworkEnabled(final boolean value) {
        mUnittestNetworkEnabled = value;
    }
    static boolean getUnittestNetworkEnabled() {
        return mUnittestNetworkEnabled;
    }
    static boolean isInUnitTest() {
        try {
            Class<?> clz = Objects.requireNonNull(BaseActivity.class.getClassLoader())
                    .loadClass("org.junit.Test");
            return (clz != null);
        } catch (ClassNotFoundException e) {
            return false;
        }
    }
    static boolean fileExists(final Context context, final String path, final String filename) {
        String fullpath;
        if (new File(path).isAbsolute()) {
            fullpath = path + "/" + filename;
        } else {
            fullpath = context.getCacheDir() + DOWNLOADS_FOLDER + path + "/" + filename;
        }
        if (!new File(fullpath).canRead()) {
            Log.v(TAG, String.format("%s missing", fullpath));
            return false;
        } else {
            if (!new File(fullpath + "__md5__").canRead()) {
                Log.v(TAG, String.format("%s hash missing", fullpath));
                return false;
            }
        }
        return true;
    }

    public static void fakeNetworkError(final Context context, final String path,
                                        final DownloaderFactory.AdditionalDownloads downloads)
            throws DownloaderFactory.AdditionalDownloads.DownloadException {
        if (!mUnittestNetworkEnabled) {
            for (String localfile : downloads.getRequiredFiles()) {
                String targetFolder = context.getCacheDir() + DOWNLOADS_FOLDER + path;
                if (!fileExists(context, targetFolder, localfile)) {

                    throw new DownloaderFactory.AdditionalDownloads.DownloadException(
                            targetFolder + "/" + localfile,
                            new File(path).getName() + "/" + localfile,
                            downloads.getAllHashes(localfile),
                            downloads.getZipURL(),
                            false);
                }
            }
        }
    }
    protected void removePreparingDialogs() {
        FragmentManager fm = getSupportFragmentManager();
        Fragment f = fm.findFragmentByTag(DOWNLOAD_DIALOG);
        if (f != null) {
            try {
                fm.beginTransaction().remove(f).commitNow();
            } catch (IllegalStateException e) {
                if (BuildConfig.DEBUG) {
                    throw new RuntimeException(e);
                }
            }
        }
    }
    @Override
    protected void onPause() {
        if (mRemovePreparingDialogs) {
            mRemovePreparingDialogs = false;
            removePreparingDialogs();
        }
        super.onPause();
    }
    static Intent createChooserIntent(final Intent source, final CharSequence title) {
        Intent ret = Intent.createChooser(source, title);
        if (source.getBooleanExtra("retry", false)) {
            ret.putExtra("retry", true);
        }
        return ret;

    }
}
