#!/usr/bin/env python3

# Build required runtime-linkings between libviceshared.so and lib<machine>.so by parsing headers.
# Run after a vice update when linker errors occur.

import os.path
import re
from fnmatch import fnmatch

ROOTFOLDER = "../app/src/main/jni/"


class Function:

    def __init__(self, return_type, parameters, label, definition_start):
        self.return_type = return_type.strip()
        self.params = parameters.strip()
        self.label = label.strip()
        while "  " in definition_start:
            definition_start = definition_start.replace("  ", " ")
        self.definition_start = definition_start.strip()

    def applyformat(self, template: str) -> str:
        return template.replace("$R", self.return_type).replace("$L", self.label).replace("$P", self.params)

    def __str__(self):
        return self.applyformat("$R $L ($P);")
    def __eq__(self, other):
        if isinstance(other,Function):
            if self.return_type == other.return_type and self.params == other.params and self.label == other.label:
                return True
            otherlabel = other.label
            otherparams = other.params
            while otherlabel.otherparams("*"):
                otherparams=otherparams[0:-1]
                otherlabel = "*"+otherlabel
                if self.return_type == other.return_type and self.params == otherparams and self.label == otherlabel:
                    return True


    @staticmethod
    def create_from_declaration(declaration, required_functions):
        if RE_DECLARATION.match(declaration.strip()):
            head = declaration.split("(", 1)[0]
            parameterstring = declaration.split("(", 1)[1]
            values = head.split()[1:]
            parameters = parameterstring.rsplit(")",1)[0]
            rawparams = parameters.split(",")
            parameters = []
            for count, value in enumerate(rawparams):
                if not " " in value and value != "void":
                    value = "%s p%d" % (value, count)
                parameters.append(value)
            parameterstring = ",".join(parameters)
            if not "..." in parameters:
                label = values[-1]
                return_type = " ".join(values[0:-1])
                while label[0] == "*":
                    label = label[1:]
                    return_type = return_type + "*"
                if return_type and return_type[0] != "(":
                    if not required_functions or label in required_functions:
                        if not "..." in parameterstring:
                            if RE_LABEL.match(label):
                                return Function(return_type, parameterstring, label, head.replace("extern ","",1))


RE_DECLARATION = re.compile("^extern\s*\w*\s*.*\w*\(.*\);$")
RE_DECLARATION_BEGIN=re.compile("^extern\s*\w*\s*.*\w*\(.*")
RE_DECLARATION_END=re.compile(".*\);");
RE_LABEL=re.compile("^\w*$")

MACHINE_SPECIFIC_FOLDERS=["arch", "c64", "c64dtv", "c128", "cbm2", "pet", "plus4"]

#                  "diskimage.h", "vice-event.h", "fileio.h", "gfxoutput.h", "vsyncapi.h", "printer.h", "rsuser.h",

HEADER_BLACKLIST=["vice_sdl.h", "archapi.h", "infocontrib.h","info.h", "_config.h", "rawnet.h", "rawnetarch.h",
                  "usleep.h", "cia.h", "rsuser.h", "tap.h", "console.h", "signals.h", "fullscreen.h", "translate_funcs.h",
                  "hardsid.h", "gcr.h", "mon.h", "sid-resources.h", "resid.h","siddefs-fp.h", "opencbm.h", "archdep_rtc_get_centisecond.h",
                  "javascript_tasklist.h"]
FUNCTION_BLACKLIST=["embedded_check_file", "embedded_palette_load", "video_canvas_init", "mon_register_list_get", "parallel_cpu_set_atn",
                    "sid2_dump", "sid2_peek", "sid2_read", "sid2_store",
                    "sid3_dump", "sid3_peek", "sid3_read", "sid3_store",
                    "sid4_dump", "sid4_peek", "sid4_read", "sid4_store",
                    "sid5_dump", "sid5_peek", "sid5_read", "sid5_store",
                    "sid6_dump", "sid6_peek", "sid6_read", "sid6_store",
                    "sid7_dump", "sid7_peek", "sid7_read", "sid7_store",
                    "sid8_dump", "sid8_peek", "sid8_read", "sid8_store",
                    "vice_thread_shutdown", "fixpoint_mult", "soundclk_mult", "gerror",
                    "lib_AllocVec_pinpoint", "lib_FreeVec_pinpoint", "lib_AllocMem_pinpoint",
                    "lib_FreeMem_pinpoint", "lib_AllocVec", "lib_FreeVec", "lib_AllocMem", "lib_FreeMem",
                    "vice_atexit", "vice_exit", "rawfile_open"]
HEADERDIRS=[ROOTFOLDER, ROOTFOLDER+ "datasette/", ROOTFOLDER+"sid/", ROOTFOLDER+"/arch/shared/"]
IMPLEMENTATION_DIRS=[ROOTFOLDER, ROOTFOLDER+"arch/android/"]+\
                    [ROOTFOLDER + p + "/" for p in os.listdir(ROOTFOLDER)
                        if os.path.isdir(ROOTFOLDER + p) and p not in ["c128", "c64", "cbm2", "plus4", "vic20", "c64dtv", "pet"]]

#SOURCES_BLACKLIST=["../app/src/main/jni/" + s for s in ["c128/c128.c", "c64/c64.c", "cbm2/cbm5x0.c", "cbm2/cbm2.c", "plus4/plus4.c", "vic20/vic20.c", "c64dtv/c64dtv.c", "c64/vsid.c", "pet/pet.c" ]]
SOURCES_BLACKLIST=[ROOTFOLDER + f for f in ["main.c", "maincpu.c", "mainc64cpu.c", "mainviccpu.c"]]
if __name__ == "__main__":
    functions = []
    packing=False
    packed=""
    headers = [h for h in os.listdir(ROOTFOLDER) if fnmatch(h, "*.h") and not h in HEADER_BLACKLIST ]
    allheaders=[]
    labels = FUNCTION_BLACKLIST
    for folder in HEADERDIRS:
        headers = [h for h in os.listdir(folder) if fnmatch(h, "*.h") and not h in HEADER_BLACKLIST]
        for filename in headers:
            print("=== %s ===" % filename)
            with open(folder+filename, "r", encoding="iso-8859-1") as f:
                for l in f.readlines():
                    line = l.strip();
                    if RE_DECLARATION_BEGIN.match(line):
                        packing = True
                        packed = line
                    else:
                        if packing:
                            packed = packed + line;
                    if RE_DECLARATION_END.match(line):
                        if packing:
                            packed = packed.split(");")[0] + ");"
                            f = Function.create_from_declaration(packed, None)
                            packed = ""
                            packing = False
                            if f:
                                if not f.label in labels:
                                    functions.append(f)
                                    labels.append(f.label)
        allheaders=allheaders + headers;
    missing_implementations=[f.definition_start for f in functions]

    for folder in IMPLEMENTATION_DIRS:
        srcs = [s for s in os.listdir(folder) if fnmatch(s, "*.c") ]
        for filename in srcs:
            print("=== %s ===" % filename)

            if folder+filename not in SOURCES_BLACKLIST:
                with open(folder+filename,"r",encoding="iso-8859-1") as f:
                    for line in f.readlines():
                        for i in missing_implementations:
                            if line.startswith(i+"(") or line.startswith("extern "+i+"(") \
                                    or line.startswith(i+" (") or line.startswith("extern "+i+" ("):
                                if i in missing_implementations:
                                    missing_implementations.remove(i)
    functions = [f for f in functions if f.definition_start in missing_implementations]
    functions.append(Function.create_from_declaration("extern uint8_t mem_read(uint16_t addr);",None))
    functions.append(Function.create_from_declaration("extern void mem_store(uint16_t addr, uint8_t value);",None))
    with open(ROOTFOLDER+"arch/android/gen_machine_specific_bindings.h", "w+") as header:
        lines = ["#ifndef EIGHTBITWONDERS_GEN_MACHINE_SPECIFIC_BINDINGS_H",
                 "#define EIGHTBITWONDERS_GEN_MACHINE_SPECIFIC_BINDINGS_H"]
        for include in allheaders:
            lines.append("#include \"%s\"" % os.path.basename(include))
        lines.append("typedef struct {")
        for f in functions:
            lines.append(f.applyformat("\t$R(*$L)($P);"))
        lines.append("} t_machine_specific_bindings;")
        lines.append("#endif")
        header.write("".join(open("gpl-header.txt","r").readlines()))
        header.write("\n".join(lines))
    with open(ROOTFOLDER + "arch/android/gen_machine_specific_bindings.c", "w+") as source:
        lines = []
        lines.extend(["#ifdef HAVE_COMMON_SO",
                 "#include <string.h>",
                 "#include \"machine_specific_bindings.h\"",
                 "#include \"gen_machine_specific_bindings.h\"",
                 "",
                 "static t_machine_specific_bindings bindings = {" + ((len(functions) - 1) * "NULL, ") + " NULL};",
                 "",
                 "void raiseException(const char* missing_function);",
                 "",
                 "void populate_callbacks(t_machine_specific_bindings machine_bindings) {",
                 "\tmemcpy(&bindings, &machine_bindings, sizeof(t_machine_specific_bindings));",
                 "}"
                 ])
        for f in functions:
            paramlabels=[]
            for param in f.params.split(","):
                p = param.split(" ")[-1];
                while p[0] == "*":
                    p = p[1:]
                    if not p:
                        break
                if p.startswith("(*"):
                    p = p[2:].split(")")[0]
                if p.endswith("[]"):
                        p = p[0:-2]
                paramlabels.append(p)
            if paramlabels == ["void", ]:
                paramnames = ""
            else:
                paramnames = " COMMA ".join(paramlabels);

            if f.return_type == "void":
                lines.append("VOID_FUNCTION(%s,%s,%s);" % (f.label, f.params.replace(",", " COMMA "), paramnames))
            else:
                lines.append("VALUE_RETURN_FUNCTION (%s,%s,%s,%s,(%s)0);" % (
                f.return_type, f.label, f.params.replace(",", " COMMA "), paramnames, f.return_type))

        lines.append("#endif")
        source.write("".join(open("gpl-header.txt","r").readlines()))
        source.write("\n".join(lines))
    with open(ROOTFOLDER+"arch/android/machine/gen_machine_populate_specific_bindings.c", "w+") as source:
        lines = []
        lines.extend(["#ifdef HAVE_COMMON_SO", "#include \"machine_specific_bindings.h\"",
                 "#include \"gen_machine_specific_bindings.h\"", "void populate_machine_specific_functions() {"])
        labels = [f.label for f in functions]
        lines.append("\tt_machine_specific_bindings bindings = {%s};\n" % ",\\\n\t\t".join(labels))
        lines.append("\tpopulate_callbacks (bindings);")
        lines.append("}")
        lines.append("#endif")
        source.write("".join(open("gpl-header.txt","r").readlines()))
        source.write("\n".join(lines))
