/*/*
 * vdc-snapshot.c - Snapshot functionality for the MOS 8563 (VDC)
 * emulation.
 *
 * Written by
 *  Ettore Perazzoli <ettore@comm2000.it>
 *  Markus Brenner <markus@brenner.de>
 *  Andreas Boose <viceteam@t-online.de>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <stdio.h>
#include <stdint.h>

#include "log.h"
#include "snapshot.h"
#include "types.h"
#include "vdc-snapshot.h"
#include "vdc.h"
#include "vdctypes.h"

/*

   This is the format of the VDC snapshot module.

   Name               Type   Size   Description

   (nothing yet)

 */


static char snap_module_name[] = "VDC";
#define SNAP_MAJOR 0
#define SNAP_MINOR 0


int vdc_snapshot_write_module(snapshot_t *s)
{
    snapshot_module_t *m;

    m = snapshot_module_create (s, snap_module_name, SNAP_MAJOR, SNAP_MINOR);
    if (m == NULL) {
        return -1;
    }
    if ((SMW_W(m, (uint16_t)vdc.initialized) < 0)     
        || (SMW_W(m, (uint16_t)vdc.screen_height) < 0)
        || (SMW_W(m, (uint16_t)vdc.screen_xpix) < 0)
        || (SMW_W(m, (uint16_t)vdc.screen_ypix) < 0)
        || (SMW_W(m, (uint16_t)vdc.first_displayed_line) < 0)
        || (SMW_W(m, (uint16_t)vdc.last_displayed_line) < 0)
        || (SMW_W(m, (uint16_t)vdc.border_height) < 0)
        || (SMW_W(m, (uint16_t)vdc.border_width) < 0)
        || (SMW_W(m, (uint16_t)vdc.raster_ycounter_max) < 0)
        || (SMW_W(m, (uint16_t)vdc.screen_textlines) < 0)
        || (SMW_W(m, (uint16_t)vdc.hsync_shift) < 0)
        || (SMW_W(m, (uint16_t)vdc.xchars_total) < 0)
        || (SMW_W(m, (uint16_t)vdc.xsync_increment) < 0)
        || (SMW_W(m, (int16_t)vdc.update_reg) < 0)
        || (SMW_W(m, (uint16_t)vdc.screen_text_cols) < 0)
        || (SMW_W(m, (uint16_t)vdc.screen_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.cursor_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.update_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.attribute_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.chargen_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.mem_counter) < 0)
        || (SMW_W(m, (uint16_t)vdc.bitmap_counter) < 0)
        || (SMW_W(m, (uint16_t)vdc.bytes_per_char) < 0)
        || (SMW_W(m, (uint16_t)vdc.charwidth) < 0)
        || (SMW_W(m, (uint16_t)vdc.mem_counter_inc) < 0)
        || (SMW_W(m, (int16_t)vdc.log) < 0)
        || (SMW_W(m, (int16_t)vdc.vdc_address_mask) < 0)
        || (SMW_W(m, (int16_t)vdc.frame_counter) < 0)
        || (SMW_W(m, (int16_t)vdc.attribute_blink) < 0)
        || (SMW_W(m, (int16_t)vdc.crsrpos) < 0)
        || (SMW_W(m, (int16_t)vdc.force_repaint) < 0)
        || (SMW_W(m, (int16_t)vdc.force_resize) < 0)
        || (SMW_W(m, (int16_t)vdc.force_cache_flush) < 0)
        || (SMW_W(m, (int16_t)vdc.update_geometry) < 0)
        || (SMW_W(m, (uint16_t)vdc.xsmooth) < 0)
        || (SMW_W(m, (uint16_t)vdc.revision) < 0)
        || (SMW_BA(m, vdc.ram, 0x10000) < 0)
        || (SMW_DW(m, (uint32_t)vdc.vdc_line_start) < 0)
        || (SMW_W(m, (uint16_t)vdc.old_reg27) < 0)
        || (SMW_W(m, (uint16_t)vdc.row_counter) < 0)
        || (SMW_W(m, (uint16_t)vdc.row_counter_y) < 0)
        || (SMW_W(m, (uint16_t)vdc.attribute_offset) < 0)
        || (SMW_W(m, (uint16_t)vdc.display_enable) < 0)
        || (SMW_W(m, (uint16_t)vdc.prime_draw) < 0)
        || (SMW_W(m, (uint16_t)vdc.draw_active) < 0)
        || (SMW_W(m, (uint16_t)vdc.draw_finished) < 0)
        || (SMW_W(m, (uint16_t)vdc.vsync) < 0)
        || (SMW_W(m, (uint16_t)vdc.vsync_counter) < 0)
        || (SMW_W(m, (uint16_t)vdc.vsync_height) < 0)
        || (SMW_W(m, (uint16_t)vdc.draw_counter) < 0)
        || (SMW_W(m, (uint16_t)vdc.draw_counter_y) < 0)
        || (SMW_W(m, (uint16_t)vdc.revision) < 0)
        || (SMW_W(m, (uint16_t)vdc.old_screen_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.old_attribute_adr) < 0)
        || (SMW_W(m, (uint16_t)vdc.interlaced) < 0)
        || (SMW_W(m, (uint16_t)vdc.canvas_width) < 0)
        || (SMW_W(m, (uint16_t)vdc.canvas_height) < 0)
        || (SMW_W(m, (uint16_t)vdc.canvas_width_old) < 0)
        || (SMW_W(m, (uint16_t)vdc.canvas_height_old) < 0)
        || (SMW_BA(m, vdc.scrnbuf, 0x200) < 0)
        || (SMW_W(m, (uint16_t)vdc.scrnbufdraw) < 0)
        || (SMW_BA(m, vdc.attrbuf, 0x200) < 0)
        || (SMW_W(m, (uint16_t)vdc.attrbufdraw) < 0)
        || (SMW_W(m, (uint16_t)vdc.light_pen.triggered) <0)
        || (SMW_W(m, (uint16_t)vdc.light_pen.x) <0)
        || (SMW_W(m, (uint16_t)vdc.light_pen.y) <0)) {
            return -1;
    }
    for (int i = 0; i < sizeof(vdc.regs)/sizeof(vdc.regs[0]); i++) {
        if (SMW_B(m, vdc.regs[i]) < 0) {
            return -1;
        }
        
    }
    return snapshot_module_close(m);
    

/*
fail:
  if (m != NULL)
      snapshot_module_close(m);
  return -1;
*/
}

/* Helper functions.  */

int vdc_snapshot_read_module(snapshot_t *s)
{
    uint8_t major_version, minor_version;
    snapshot_module_t *m;

    m = snapshot_module_open(s, snap_module_name,
                             &major_version, &minor_version);
    if (m == NULL) {
        return -1;
    }

    if (snapshot_version_is_bigger(major_version, minor_version, SNAP_MAJOR, SNAP_MINOR)) {
        log_error(vdc.log,
                  "Snapshot module version (%d.%d) newer than %d.%d.",
                  major_version, minor_version,
                  SNAP_MAJOR, SNAP_MINOR);
        goto fail;
    }
    if ((SMR_W(m, (uint16_t *) &vdc.initialized) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_height) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_xpix) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_ypix) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.first_displayed_line) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.last_displayed_line) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.border_height) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.border_width) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.raster_ycounter_max) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_textlines) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.hsync_shift) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.xchars_total) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.xsync_increment) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.update_reg) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_text_cols) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.screen_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.cursor_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.update_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.attribute_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.chargen_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.mem_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.bitmap_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.bytes_per_char) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.charwidth) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.mem_counter_inc) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.log) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.vdc_address_mask) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.frame_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.attribute_blink) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.crsrpos) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.force_repaint) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.force_resize) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.force_cache_flush) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.update_geometry) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.xsmooth) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.revision) < 0)
        || (SMR_BA(m, vdc.ram, 0x10000) < 0)
        || (SMR_DW(m, (uint32_t*) &vdc.vdc_line_start) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.old_reg27) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.row_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.row_counter_y) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.attribute_offset) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.display_enable) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.prime_draw) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.draw_active) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.draw_finished) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.vsync) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.vsync_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.vsync_height) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.draw_counter) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.draw_counter_y) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.revision) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.old_screen_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.old_attribute_adr) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.interlaced) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.canvas_width) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.canvas_height) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.canvas_width_old) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.canvas_height_old) < 0)
        || (SMR_BA(m, vdc.scrnbuf, 0x200) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.scrnbufdraw) < 0)
        || (SMR_BA(m, vdc.attrbuf, 0x200) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.attrbufdraw) < 0)
        || (SMR_W(m, (uint16_t *) &vdc.light_pen.triggered) <0)
        || (SMR_W(m, (uint16_t *) &vdc.light_pen.x) <0)
        || (SMR_W(m, (uint16_t *) &vdc.light_pen.y) <0)) {
        goto fail;
    }
    for (int i = 0; i < sizeof(vdc.regs)/sizeof(vdc.regs[0]); i++) {
        if (SMR_B(m, vdc.regs+i) < 0) {
            goto fail;
        }

    }
    return 0;

fail:
    if (m != NULL) {
        snapshot_module_close(m);
    }
    return -1;
}
