#include "javascript_tasklist.h"
#include "javascript.h"
#include <mutex>
#include <list>
extern "C" {
#include "charset.h"
#include "mem.h"
#include "logginghelpers.h"
#include "lib.h"
#include "javascript_callbacks.h"

void keyboard_rawkey_pressed(int row, int column, int shift);
void keyboard_rawkey_released(int row, int column, int shift);

}


//
// Created by rainer on 21.01.24.
//


Listener::Listener(JSContext *ctx, JSValue  func, JSValue  this_obj, int argc, JSValueConst *argv) {
    mCtx = ctx;
    mFunc = JS_DupValue(ctx, func);
    mArgc = argc;
    mThisObj = JS_DupValue(ctx, this_obj);
    if (argv) {
        mArgv = new JSValueConst[argc]();
        for (int i = 0; i < argc; i++) {
            mArgv[i] = JS_DupValue(ctx, argv[i]);
        }
    }
}
Listener::~Listener() {
    for (int i = 0; i < mArgc && mArgv; i++) {
        JS_FreeValue(mCtx, mArgv[i]);
    }
    delete(mArgv);
    mArgc = 0;
    JS_FreeValue(mCtx, mFunc);
    JS_FreeValue(mCtx, mThisObj);

}

uint32_t JavascriptTasklist::getNexId() {
    static uint32_t nextId = 0;
    nextId++;
    return nextId;
}
JavascriptTasklist::JavascriptTasklist() = default;
JavascriptTasklist::~JavascriptTasklist() {
    cleanup();
}
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantConditionsOC"
uint32_t JavascriptTasklist::addTask(JavaScriptTask* t) {
    if (!mInTaskExecution) {
        std::lock_guard<std::mutex> guard(myMutex);
    }
    mTasks.insert(mTasks.end(), t);
    return t->getId();
}

void JavascriptTasklist::callTasks() {
    std::list<JavaScriptTask*>::iterator it;
    mInTaskExecution = true;
    for (it = mTasks.begin(); it != mTasks.end(); ++it) {
        setCurrentListenerId(nullptr, (*it)->getId());
        if ((*it)->isEnabled()) {
            setCurrentListenerId(nullptr,(*it)->getId());
            std::lock_guard<std::mutex> guard(myMutex);
            (*it)->fullfill();
        }
    }
    mCurrentListenerId = NOT_IN_LISTENER;
    mInTaskExecution = false;
}
JavascriptTasklist* JavascriptTasklist::mInstance = nullptr;
JavascriptTasklist* JavascriptTasklist::getInstance() {
    if (JavascriptTasklist::mInstance == nullptr) {
        JavascriptTasklist::mInstance = new JavascriptTasklist();
    }
    return JavascriptTasklist::mInstance;
}

void JavascriptTasklist::cleanup() {
    const std::lock_guard<std::mutex> guard(myMutex);
    std::for_each(mTasks.begin(),mTasks.end(), [](JavaScriptTask* t) {delete(t);});
    mTasks.clear();
}

bool JavascriptTasklist::handleJoysticks(unsigned int port, uint8_t value) {
    bool found=false;

    std::for_each(mTasks.begin(), mTasks.end(),[port, value, &found](JavaScriptTask* t) -> void {
        if (auto *jl = dynamic_cast<JoystickListener *>(t)) {
            if (jl->isEnabled()) {
                JavascriptTasklist::getInstance()->setCurrentListenerId(jl->getContext(), jl->getId());
                if (jl->handle(port, value)) {
                    found = true;
                }
            }
        }
    });

    mCurrentListenerId = NOT_IN_LISTENER;
    return found;

}
bool JavascriptTasklist::handleGamepadAction(bool pressed, int keycode) {
    bool found = false;
    std::for_each(mTasks.begin(), mTasks.end(),[pressed, keycode,&found](JavaScriptTask* t) -> void {
        if (auto *kl = dynamic_cast<GamepadButtonListener *>(t)) {
            JavascriptTasklist::getInstance()->setCurrentListenerId(kl->getContext(), kl->getId());
            if (kl->isEnabled()) {
                if (kl->handle(pressed, keycode)) {
                    found = true;
                }
            }
        }
    });
    return found;
}

bool JavascriptTasklist::handleKeyboardAction(bool pressed, int row, int col) {
    bool found=false;
    std::for_each(mTasks.begin(), mTasks.end(),[pressed, row, col, &found](JavaScriptTask* t) -> void {
        if (auto *kl = dynamic_cast<KeyboardListener *>(t)) {
            JavascriptTasklist::getInstance()->setCurrentListenerId(kl->getContext(), kl->getId());
            if (kl->isEnabled()) {
                if (kl->handle(pressed, row, col)) {
                    found = true;
                }
            }
        }
    });
    mCurrentListenerId = NOT_IN_LISTENER;
    return found;
}

JSValue JavascriptTasklist::enableTask(JSContext *ctx, JSValue taskId) {
    return setTaskState(ctx, taskId, ENABLED);
}

JSValue JavascriptTasklist::disableTask(JSContext *ctx, JSValue taskId) {
    return setTaskState(ctx, taskId, DISABLED);
}

JSValue JavascriptTasklist::removeTask(JSContext *ctx, JSValue taskId) {
    return setTaskState(ctx, taskId, REMOVED);
}

JSValue JavascriptTasklist::setTaskState(JSContext *ctx, JSValue taskId,
                                         JavascriptTasklist::requestedTaskTaste newState) {
    if (!mInTaskExecution) {
        std::lock_guard<std::mutex> guard(myMutex);
    }
    uint32_t id;
    bool ret = false;
    JS_ToUint32(ctx, &id, taskId);
    std::for_each(mTasks.begin(),mTasks.end(), [id,newState,&ret](JavaScriptTask* t) {
        if (t->getId() == id) {
            ret = true;
            switch (newState) {
                case ENABLED:
                    t->setEnabled(true);
                    break;
                case DISABLED:
                    t->setEnabled(false);
                    break;
                case REMOVED:
                    t->setGarbage();
                    break;
            }
        }
    });
    return JS_NewBool(ctx,ret);

}

void JavaScriptTask::setEnabled (bool value) {
    mEnabled = value;
}
bool JavaScriptTask::check() {
    return false;
}
JavaScriptTask::JavaScriptTask() {
    mId = JavascriptTasklist::getNexId();
}

uint32_t JavaScriptTask::getId() const {
    return mId;
}
bool JavaScriptTask::isEnabled() const  {
    return mEnabled;
}

void JavaScriptTask::setGarbage() {
    setEnabled(false);
    mGarbage = true;
}

bool JavaScriptTask::isGarbage() const {
    return mGarbage;
}

JSContext* Listener::getContext() {
    return mCtx;
}

bool Listener::execute() {
    bool ret;
    JSValue f = JS_DupValue(mCtx, mFunc);
    if (isEnabled() &&!isGarbage()) {
        JSValue callRet = JS_Call(mCtx, f, mThisObj, 0, nullptr);
        JS_FreeValue(mCtx, f);
        if (JS_IsException(callRet)) {
            char *exception = js_dump_error(mCtx);
            LOGE("%s", exception);
            lib_free(exception);
            ret = false;
        } else {
            if (JS_IsBool(callRet)) {
                ret = JS_ToBool(mCtx, callRet);
            } else {
                uint32_t intRet = 0;
                if (JS_ToUint32(mCtx, &intRet, callRet)) {
                    ret = false;
                } else {
                    ret = intRet;
                }
            }
        }
        return ret;
    }
    return false;
}

void Listener::fullfill() {
    if (check()) {
        execute();
    }
}
MemoryChangedListener::MemoryChangedListener(JSContext *ctx, uint16_t adress, uint32_t memory_operator, uint8_t value, JSValue func, JSValue thisObj)
        : Listener(ctx, func, thisObj, 0, nullptr){
    mAdress = adress;
    mMemoryOperator = memory_operator;
    mValue = value;
    mStoredValue = mem_read(adress);
}

bool MemoryChangedListener::check()  {
    uint8_t value=mem_read(mAdress);
    if (mStoredValue != value) {
        mStoredValue = value;
        // EQUAL
        if ((mMemoryOperator & EQUAL) && mValue == value) {
            return true;
        }
        // LESS
        if ((mMemoryOperator & LESS) && value < mValue) {
            return true;
        }
        // MORE
        if ((mMemoryOperator & MORE) && value > mValue) {
            return true;
        }
    }
    return false;
}
TextVisibilityListener::TextVisibilityListener(JSContext *ctx, uint8_t *screentext,
                                               uint8_t *screentext_reverse, size_t len,
                                               uint32_t mode, JSValue func, JSValue thisObj)
        : Listener(ctx, func, thisObj, 0, nullptr) {
    mScreentext = screentext;
    mScreentextReverse = screentext_reverse;
    mTextlen = len;
    mTextVisibilityFilter = mode;
}
bool TextVisibilityListener::isTextVisible() {
    uint16_t screenbase = 0;
    uint8_t rows = 0;
    uint8_t cols = 0;
    uint8_t *base = get_ram();
    int bank = 0;
    void *addr = nullptr;
    mem_get_screen_parameter(&screenbase, &rows, &cols, &bank);
    if (screenbase && rows && cols) {
        addr = memmem(base + screenbase, rows * cols,
                      mScreentext,
                      mTextlen);
        if (!addr) {
            addr = memmem(base + screenbase, rows * cols,
                          mScreentextReverse,
                          mTextlen);
        }
    }
    return addr && ((unsigned long long) addr - (unsigned long long) base -
                    (unsigned long long) screenbase < rows * cols);
}
bool TextVisibilityListener::check() {
    int changed = 0;
    int visible = isTextVisible();
    if (visible
        && !mTextWasVisible
        && mTextVisibilityFilter & 1) {
        changed = 1;
    }
    if (!visible
        && mTextWasVisible
        && mTextVisibilityFilter & 2) {
        changed = 1;
    }
    if (mTextWasVisible != visible) {
        mTextWasVisible = visible;
    }
    return changed;
}

JoystickListener::JoystickListener(JSContext *ctx, unsigned int joyport, JSValue func, JSValue thisObj)
        : Listener(ctx, func, thisObj, 0, nullptr) {
    mJoyport = joyport;
    mLastValue = 255;
}
bool JoystickListener::handle(unsigned int port, uint8_t value) {
    JSValue elements[5];
    bool ret;
    if (mCtx && mLastValue != value && port == mJoyport) {
        for (int e = 0; e < sizeof(elements)/sizeof(elements[0]);e++) {
            elements[e] = JS_NewBool(mCtx, value & (1 << e));
        }
        JSValue callRet = JS_Call(mCtx, mFunc, mThisObj, 5, elements);
        if (JS_IsException(callRet)) {
            char *exception = js_dump_error(mCtx);
            LOGE("%s", exception);
            lib_free(exception);
            ret = false;
        } else {
            if (JS_IsBool(callRet)) {
                ret = JS_ToBool(mCtx, callRet) != 0;
            } else {
                uint32_t intRet;
                if (JS_ToUint32(mCtx, &intRet, callRet)) {
                    ret = false;
                } else {
                    ret = intRet;
                }
            }
        }
        for (JSValue element : elements) {
            JS_FreeValue(mCtx, element);
        }
        mLastValue = value;
    } else {
        ret = false;
    }
    return ret;
}



KeyboardListener::KeyboardListener(JSContext *ctx, int row, int col, unsigned int action,
                                   JSValue func, JSValue thisObj)
        : Listener(ctx, func, thisObj, 0, nullptr){
    mRow = row;
    mCol = col;
    mAction = action;
}
bool KeyboardListener::handle(bool pressed, int row, int col) {
    bool actionMatch = ((mAction & ACTION_PRESS) && pressed)
            || ((mAction & ACTION_RELEASE) && !pressed);
    return mRow == row && mCol == col && actionMatch && execute();
}
Usertask::Usertask(JSContext* ctx, JSValue  func, JSValue  this_obj, uint32_t vsyncs2skip)
        : Listener(ctx, func,this_obj, 0, nullptr) {
    mVsyncs2Skip = vsyncs2skip;
}

bool Usertask::check() {
    if (mVsyncs2Skip==0) {
        return true;
    } else {
        mVsyncs2Skip--;
        return false;
    }
}

bool Usertask::execute() {
    auto ret= Listener::execute();

    setGarbage();
    return ret;
}


void JavascriptTasklist::setCurrentListenerId([[maybe_unused]] JSContext* ctx, uint32_t id) {
    mCurrentListenerId = id;
}
JSValue JavascriptTasklist::getCurrentListenerId(JSContext* ctx) const {
    return JS_NewUint32(ctx, mCurrentListenerId);
}

void JavascriptTasklist::gc() {
    std::list<JavaScriptTask *> allTasks=mTasks;
    bool taskRemaining = false;
    std::for_each(mTasks.begin(),mTasks.end(), [&allTasks,&taskRemaining](JavaScriptTask* t) {
        if (t->isGarbage()) {
            allTasks.remove(t);
            delete (t);
        } else {
            taskRemaining = true;
        }
    });
    if (!taskRemaining) {
        js_notify_tasks_deleted();
    }
    mTasks = allTasks;
}


GamepadButtonListener::GamepadButtonListener(JSContext *ctx, unsigned int action,
                                             unsigned int keycode, JSValue func,
                                             JSValue thisObj)
        : Listener(ctx, func, thisObj, 0, nullptr){
    mAction = action;
    mKeycode = keycode;
}

bool GamepadButtonListener::handle(bool pressed, int keycode) {
    bool actionMatch = ((mAction & ACTION_PRESS) && pressed)
                       || ((mAction & ACTION_RELEASE) && !pressed);
    return mKeycode == keycode && actionMatch && execute();
}
extern "C" {
static JSValue
addMemoryChangedListener(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
                         JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "adress",   JS_TAG_INT,    1, 0, 65535},
            {(char *) "operator", JS_TAG_INT,    1, 1, 7},
            {(char *) "value",    JS_TAG_INT,    1, 0, 255},
            {(char *) "function", JS_TAG_OBJECT, 0, 0}
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t adress = 0;
        uint32_t _operator = 0;
        uint32_t value = 0;
        JS_ToUint32(ctx, &adress, argv[0]);
        JS_ToUint32(ctx, &_operator, argv[1]);
        JS_ToUint32(ctx, &value, argv[2]);
        auto *l = new MemoryChangedListener(ctx, adress, _operator, value, argv[3], this_val);
        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(l));

    }
    return ret;
}
static JSValue
addTextVisibilityListener(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
                          JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "text",     JS_TAG_UNDEFINED, 0, 0, 0},
            {(char *) "mode",     JS_TAG_INT,    1, 1, 2},
            {(char *) "function", JS_TAG_OBJECT, 0, 0, 0}
    };
    unsigned char* screentext = nullptr;
    unsigned char* screentext_reverse;
    size_t len = 0;

    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        if (JS_VALUE_GET_TAG(argv[0]) == JS_TAG_STRING) {
            JS_ToCStringLen(ctx, &len, argv[0]);
            const char *asciitext = lib_strdup(JS_ToCString(ctx, argv[0]));
            const char *petscii = (char *) charset_petconv_stralloc((uint8_t *) asciitext,
                                                                    CONVERT_TO_PETSCII);
            screentext = (uint8_t *) lib_malloc(len);
            screentext_reverse = (uint8_t *) lib_malloc(len);
            for (int i = 0; i < len; i++) {
                screentext[i] = charset_petcii_to_screencode(petscii[i], 0);
                screentext_reverse[i] = charset_petcii_to_screencode(petscii[i], 1);
            }
            lib_free((void *) petscii);
        } else if (JS_IsObject(argv[0]) && JS_VALUE_GET_OBJ(argv[0])) {

            uint8_t *buf = JS_GetArrayBuffer(ctx, &len, argv[0]);
            if (buf) {
                screentext = (uint8_t *) lib_malloc(len);
                screentext_reverse = (uint8_t *) lib_malloc(len);
                memcpy(screentext, buf, len);
                memcpy(screentext_reverse, buf, len);
            }
        }
        if (!screentext || !screentext_reverse) {
            return JS_ThrowTypeError(ctx, "%s.%s expects %s as parameter %d",
                                     GLOBAL_OBJECT_NAME, __FUNCTION__ , "string or ArrayBuffer", 1);
        }
        uint32_t mode = 0;
        JS_ToUint32(ctx, &mode, argv[1]);
        auto *l = new TextVisibilityListener(ctx, screentext, screentext_reverse, len, mode,
                                             argv[2], this_val);

        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(l));

    }
    return ret;
}
static JSValue
addGamepadButtonListener(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
        JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "action", JS_TAG_INT, 1, 1, 2},
            {(char *) "button",    JS_TAG_INT, 0, 0, 0},
            {(char *) "function", JS_TAG_OBJECT, 0, 0}
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        LOGV("addGamepadButtonListener no exception");
        uint32_t action = 0;
        uint32_t button = 0;
        JS_ToUint32(ctx, &action, argv[0]);
        JS_ToUint32(ctx, &button, argv[1]);
        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(
                new GamepadButtonListener(ctx, (int) action, (int) button, argv[2], this_val)));
        js_notify_gamepad_task_created();
    } else {
        LOGV("addGamepadButtonListener exception");
    }
    return ret;
}
static JSValue
addKeyboardListener(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
                    JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "action",   JS_TAG_INT,    1, 1,  2},
            {(char *) "row",      JS_TAG_INT,    1, -5, 7},
            {(char *) "col",      JS_TAG_INT,    1, 0,  7},
            {(char *) "function", JS_TAG_OBJECT, 0, 0}
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t row = 0;
        uint32_t col = 0;
        uint32_t action = 0;
        JS_ToUint32(ctx, &action, argv[0]);
        JS_ToUint32(ctx, &row, argv[1]);
        JS_ToUint32(ctx, &col, argv[2]);
        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(
                new KeyboardListener(ctx, (int) row, (int) col, (int) action, argv[3], this_val)));
    }
    return ret;
}
static JSValue
addJoystickListener(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
                    JSValueConst *argv) {
    static int maxJoysticks = -1;
    if (maxJoysticks < 0) {
        maxJoysticks = js_getjoystickcount();
    }
    expected_parameters parameters[] = {
            {(char *) "port",     JS_TAG_INT,    2, maxJoysticks > 0 ? 1 : 0, maxJoysticks},
            {(char *) "function", JS_TAG_OBJECT, 0, 0}
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t port = 0;
        JS_ToUint32(ctx, &port, argv[0]);
        auto *l = new JoystickListener(ctx, port, argv[1], this_val);
        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(l));
    }
    return ret;
}
void js_vsync_actions_check() {
    JavascriptTasklist::getInstance()->callTasks();
    JavascriptTasklist::getInstance()->gc();
}
int js_joystick_actions_check(unsigned int port, int value) {
    if (JavascriptTasklist::getInstance()->handleJoysticks(port, value)) {
        return 1;
    } else {
        return 0;
    }
}
void tasklist_cleanup() {
    JavascriptTasklist::getInstance()->cleanup();
}
static int internalKeyAction = 0;
int js_keyboard_actions_check(int pressed, int row, int col) {
    if (internalKeyAction == 0) {
        if (JavascriptTasklist::getInstance()->handleKeyboardAction(pressed, row, col)) {
            return 1;
        }
    }
    return 0;
}
static int js_gamepad_actions_check(int pressed, int keycode) {
    if (JavascriptTasklist::getInstance()->handleGamepadAction(pressed, keycode)) {
        return 1;
    }
    return 0;
}
__attribute__((used)) int js_gamepad_pressed_check(int keycode) {
    return js_gamepad_actions_check(1, keycode);

}
__attribute__((used)) int js_gamepad_released_check(int keycode) {
    return js_gamepad_actions_check(0, keycode);

}
static JSValue joystickAction(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char*)"port", JS_TAG_INT,1,1,js_getjoystickcount()},
            {(char*)"north",JS_TAG_BOOL,0,0,0},
            {(char*)"south",JS_TAG_BOOL,0,0,0},
            {(char*)"west",JS_TAG_BOOL,0,0,0},
            {(char*)"east",JS_TAG_BOOL,0,0,0},
            {(char*)"fire",JS_TAG_INT,1,0,7},
    };
    JSValue  ret = js_checksyntax(__FUNCTION__ , ctx, argc, argv,  parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        int val = 0;
        uint32_t port = 0;
        JS_ToUint32(ctx, &port, argv[0]);
        for (int i= 0; i < 5; i++) {
            val += JS_ToBool(ctx, argv[i+1]) ? 0 : 2^i;
        }
        arch_joystick_set_value_absolute(port, val);
        ret = JS_UNDEFINED;
    }
    return ret;
}
static JSValue key_action(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc, JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char*)"action", JS_TAG_INT,1,1,2},
            {(char*)"row",JS_TAG_INT,1,-5,7},
            {(char*)"col",JS_TAG_INT,1,0,7},
    };
    JSValue  ret = js_checksyntax(__FUNCTION__ , ctx, argc, argv,  parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t action = 0;
        int32_t row = 0;
        uint32_t col = 0;
        JS_ToUint32(ctx, &action, argv[0]);
        JS_ToInt32(ctx, &row, argv[1]);
        JS_ToUint32(ctx, &col, argv[2]);
#pragma clang diagnostic push
#pragma ide diagnostic ignored "UnusedValue"
        internalKeyAction = 1;
#pragma clang diagnostic pop
        if (action == KeyboardListener::ACTION_PRESS) {
            keyboard_rawkey_pressed((int) row, (int) col, 0);
        }
        if (action == KeyboardListener::ACTION_RELEASE) {
            keyboard_rawkey_released((int) row, (int) col, 0);
        }
        internalKeyAction = 0;
        ret = JS_UNDEFINED;
    }
    return ret;

}

static
JSValue enableListener(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc,
                       JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "id", JS_TAG_INT, 0, 0, 0},

    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t id = 0;
        JS_ToUint32(ctx, &id, argv[0]);
        ret = JavascriptTasklist::getInstance()->enableTask(ctx, argv[0]);
    }
    return ret;
}
static
JSValue disableListener(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc,
                        JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "id", JS_TAG_INT, 0, 0, 0},

    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t id = 0;
        JS_ToUint32(ctx, &id, argv[0]);
        ret = JavascriptTasklist::getInstance()->disableTask(ctx, argv[0]);
    }
    return ret;

}
static
JSValue removeListener(JSContext *ctx, __attribute__((unused)) JSValueConst this_val, int argc,
                       JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char *) "id", JS_TAG_INT, 0, 0, 0},

    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t id = 0;
        JS_ToUint32(ctx, &id, argv[0]);
        ret = JavascriptTasklist::getInstance()->removeTask(ctx, argv[0]);
    }
    return ret;

}
static JSValue _addtask(JSContext *ctx, __attribute__((unused))  JSValueConst this_val, int argc,
                        JSValueConst *argv) {
    expected_parameters parameters[] = {
            {(char*)"function", JS_TAG_OBJECT, 0, 0},
            {(char*)"delay",    JS_TAG_INT,    1, 0, INT32_MAX}
    };
    JSValue ret = js_checksyntax(__FUNCTION__, ctx, argc, argv, parameters, LEN(parameters));
    if (!JS_IsException(ret)) {
        uint32_t skip;
        JS_ToUint32(ctx, &skip, argv[1]);
        ret = JS_NewUint32(ctx, JavascriptTasklist::getInstance()->addTask(
                new Usertask(ctx, argv[0], this_val, skip)));
    }
    return ret;
}
static JSValue getCurrentListenerId(JSContext *ctx, __attribute__((unused)) JSValueConst this_val,
                                    [[maybe_unused]] int argc,
                                    __attribute__((unused)) JSValueConst *argv) {
    return JavascriptTasklist::getInstance()->getCurrentListenerId(ctx);
}
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wc99-designator"
JSCFunctionListEntry triggerfunctions[] = {
        JS_CFUNC_DEF("addTextVisibilityListener", 3, addTextVisibilityListener),
        JS_CFUNC_DEF("addKeyboardListener", 4, addKeyboardListener),
        JS_CFUNC_DEF("addGamepadButtonListener", 3, addGamepadButtonListener),
        JS_CFUNC_DEF("addJoystickListener", 1, addJoystickListener),
        JS_CFUNC_DEF("addMemoryChangedListener", 4, addMemoryChangedListener),
        JS_CFUNC_DEF("enableListener", 1, enableListener),
        JS_CFUNC_DEF("disableListener", 1, disableListener),
        JS_CFUNC_DEF("removeListener", 1, removeListener),
        JS_CFUNC_DEF("_addtask2_",2, _addtask),
        JS_CFUNC_DEF("getCurrentListenerId", 0, getCurrentListenerId),
        JS_CFUNC_DEF("keyAction", 4, key_action),
        JS_CFUNC_DEF("joystickAction", 5, joystickAction),
        JS_PROP_INT32_DEF("NOT_IN_LISTENER", JavascriptTasklist::NOT_IN_LISTENER, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("EQUAL", MemoryChangedListener::EQUAL, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("LESS", MemoryChangedListener::LESS, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("MORE", MemoryChangedListener::MORE, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("TO_VISIBLE", TextVisibilityListener::TO_VISIBLE, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("TO_INVISIBLE", TextVisibilityListener::TO_INVISIBLE, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("KEY_PRESSED", KeyboardListener::ACTION_PRESS, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("KEY_RELEASED", KeyboardListener::ACTION_RELEASE, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("BTN_PRESSED", KeyboardListener::ACTION_PRESS, JS_PROP_CONFIGURABLE),
        JS_PROP_INT32_DEF("BTN_RELEASED", KeyboardListener::ACTION_RELEASE, JS_PROP_CONFIGURABLE)


};

#pragma clang diagnostic pop
int triggerfunctions_size = sizeof(triggerfunctions)/sizeof(triggerfunctions[0]);
}


