/** \file   archdep_stat.c
 * \brief   Simplified stat(2) call
 * \author  Bas Wassink <b.wassink@ziggo.nl>
 *
 * Determine if file exists, its length and if its a directory.
 *
 */

/*
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include <string.h>
#include "vice.h"
#include "archdep_defs.h"

#include "lib.h"
#include "util.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "archdep_stat.h"
#include "jnihelpers.h"


/** \brief  Determine the size of \a path and whether it's a directory
 *
 * \param[in]   path    pathname
 * \param[out]  len     length of file \a pathname
 * \param[out]  isdir   pathname is a directory
 *
 * \return  0 on success, -1 on failure
 */
int archdep_stat(const char *path, size_t *len, unsigned int *isdir)
{
    struct stat statbuf;
    if (strstr(path,"content://")==path || strstr(path, "file://") == path
        || strstr(path, "http://") == path || strstr(path, "https://") == path) {
        JNIEnv *env = getAactiveenv();
        char *uri = strdup(path);
        static jmethodID mthGetContentData = NULL;
        if (!mthGetContentData) {
            mthGetContentData = GetMethodID(GetObjectClass(CurrentActivity()), "getContentData",
                                            "(Ljava/lang/String;)[B");
        }
        if (mthGetContentData) {
            int ret;
            jsize size = 0;
            jstring jpath = NewStringUTF(path);
            jbyteArray array = CallObjectMethod(CurrentActivity(), mthGetContentData, jpath);
            if (array) {
                ret = 0;
                size = (*env)->GetArrayLength(env, array);
            } else {
                ret = -1;
            }
            DeleteLocalRef(jpath);
            *len = size;
            *isdir = 0;
            return ret;
        }
        return -1;
    }
    if (stat(path, &statbuf) < 0) {
        *len = 0;
        *isdir = 0;
        return -1;
    }
    *len = statbuf.st_size;
    *isdir = S_ISDIR(statbuf.st_mode);
    return 0;
}
