//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


#include <stdint.h>
#include "video.h"
#include <stdlib.h>
#include <string.h>
#include "jnihelpers.h"
#include "logginghelpers.h"
#include "video.h"
#include "videoarch.h"
#include "palette.h"
#include "lib.h"
#include <jni.h>
#include <pthread.h>
#include <resources.h>
#include <util.h>
#include <raster/raster.h>
#include <android/bitmap.h>
#include "timemachine.h"


extern timemachine* active_timemachine;
static unsigned int current_canvas_index = 0;

extern int video_init(void)
{
    current_canvas_index = 0;
	return 0;
}
extern void video_shutdown(void)
{
}
extern struct video_canvas_s *video_canvas_create(struct video_canvas_s *canvas,
												  __unused unsigned int *width, __unused unsigned int *height, // NOLINT(readability-non-const-parameter)
												  __unused int mapped)
{
    canvas->index = current_canvas_index;
    current_canvas_index++;
    return canvas;
}
static jobject create_bitmap(struct video_canvas_s* canvas)
{
	static jclass class_bitmap=NULL;
	static jclass class_bitmap_config=NULL;
	static jmethodID mth_on_canvas_size_changed=NULL;
	static jmethodID mth_newbitmap=NULL;
	char* config_as_string;
	if (!class_bitmap)
	{
		class_bitmap=(*canvas->env)->FindClass(canvas->env,"android/graphics/Bitmap");
		mth_newbitmap=(*canvas->env)->GetStaticMethodID(canvas->env,class_bitmap,"createBitmap","(IILandroid/graphics/Bitmap$Config;)Landroid/graphics/Bitmap;");
		class_bitmap_config=(*canvas->env)->FindClass(canvas->env,"android/graphics/Bitmap$Config");
	}
	switch(canvas->depth)
	{
		case 4*8:
			config_as_string="ARGB_8888";
			break;
		case 2*8:
			config_as_string="RGB_565";
			break;
		default:
			config_as_string=NULL;
			break;


	}

	if (config_as_string && class_bitmap && class_bitmap_config && mth_newbitmap
    && canvas->draw_buffer->canvas_physical_width && canvas->draw_buffer->canvas_physical_height) {
		if (!mth_on_canvas_size_changed)
		{
			mth_on_canvas_size_changed=GetMethodID(GetObjectClass(canvas->emulator),"onCanvasSizeChanged","(IIIFF)V");
		}
		float screen_aspect_ratio=canvas->geometry->pixel_aspect_ratio*(float)canvas->draw_buffer->canvas_physical_width/(float)canvas->draw_buffer->canvas_physical_height;
		CallVoidMethod(canvas->emulator,mth_on_canvas_size_changed,
                 canvas->index,
				 canvas->draw_buffer->canvas_physical_width,
				 canvas->draw_buffer->canvas_physical_height,
				 canvas->geometry->pixel_aspect_ratio,screen_aspect_ratio);

		jfieldID fid = (*canvas->env)->GetStaticFieldID(canvas->env, class_bitmap_config,
														   config_as_string,
														   "Landroid/graphics/Bitmap$Config;");
		jobject configpar = (*canvas->env)->GetStaticObjectField(canvas->env, class_bitmap_config,fid);
        if ((jint)canvas->draw_buffer->canvas_physical_width && (jint)canvas->draw_buffer->canvas_physical_height) {

            return (*canvas->env)->CallStaticObjectMethod(canvas->env, class_bitmap, mth_newbitmap,
                                                          (jint) canvas->draw_buffer->canvas_physical_width,
                                                          (jint) canvas->draw_buffer->canvas_physical_height,
                                                          configpar);
        }
	}
	return NULL;

}

static inline void update_bitmap (struct video_canvas_s* canvas)
{
	static jclass cls=NULL;
	static jmethodID mth=NULL;

	if (!cls) {
		cls = (*(canvas->env))->GetObjectClass(canvas->env, canvas->emulator);
	}
	if (!mth) {
		mth = (*(canvas->env))->GetMethodID((canvas->env), cls, "onCanvasContentChanged", "(ILandroid/graphics/Bitmap;)V");
	}
	(*(canvas->env))->CallVoidMethod(canvas->env, canvas->emulator, mth,canvas->index, canvas->bitmapobject);

}

extern void video_arch_canvas_init(struct video_canvas_s *canvas)
{
	LOGV(">video_arch_canvas_init (%p)",canvas);
    canvas->depth= 16;
	canvas->bitmapobject=NULL;
	canvas->env=getAactiveenv();
	canvas->emulator=CurrentActivity();
	canvas->callback.create_bitmap=create_bitmap;
	canvas->callback.update_bitmap=update_bitmap;
	LOGV("<video_arch_canvas_init (%p)",canvas);

}

static unsigned int active_canvas = 0;

__attribute__((used))
void set_active_monitor(int new_monitor) {
    active_canvas = new_monitor;
}

extern void video_canvas_refresh(struct video_canvas_s *canvas,
								unsigned int xs, unsigned int ys,
                                 unsigned int xi, unsigned int yi,
                                 unsigned int w, unsigned int h)
{
	uint32_t          *pixels;
	//LOGV("entering video_canvas_refresh");
    if (AndroidBitmap_lockPixels((canvas->env), canvas->bitmapobject, (void**)&pixels)==0) {

		video_canvas_render(canvas,
					  (void *) pixels,
					  (int) w*canvas->videoconfig->scalex,
					  (int) h*canvas->videoconfig->scaley,
					  (int) xs,
					  (int) ys,
					  (int) xi*canvas->videoconfig->scalex,
					  (int) yi*canvas->videoconfig->scalex,
							(int) canvas->draw_buffer->canvas_physical_width * (int) (canvas->depth) / 8, (int) canvas->depth);
        if (active_timemachine && canvas->index == active_canvas) {
            active_timemachine->set_current_screendata(active_timemachine,
                                                       canvas->draw_buffer->canvas_physical_width,
                                                       canvas->draw_buffer->canvas_physical_height,
                                                       canvas->depth,
                                                       (jbyte*)pixels);
        }
		AndroidBitmap_unlockPixels(canvas->env, canvas->bitmapobject);
		canvas->callback.update_bitmap(canvas);
	}
	//LOGV("leaving video_canvas_refresh");

}
static uint32_t makecol_argb8888(unsigned int r, unsigned int g, unsigned int b)
{
	return (255<<24)|(r ) | (g << 8) | b<<16; // NOLINT(cppcoreguidelines-narrowing-conversions,hicpp-signed-bitwise)
}
static uint32_t makecol_dummy(__attribute__((unused)) unsigned int r,
                              __attribute__((unused)) unsigned int g,
                              __attribute__((unused)) unsigned int b)
{
	static int error_reported=0;
	if (!error_reported) {
		LOGE("canvas depth not supported!");
		error_reported=1;
	}
	return 0;
}
static uint32_t makecol_rgb565(unsigned int r, unsigned int g, unsigned int b)
{
	return ((r & 0b11111000) << 8) | ((g & 0b11111100) << 3) | (b >> 3);
}
extern int video_canvas_set_palette(struct video_canvas_s *canvas,
                                    struct palette_s *palette)
{

	if (palette)
	{
		int b,i;
		uint32_t col;
		uint32_t (*colormaker)(unsigned int r, unsigned int g, unsigned int b);
		switch (canvas -> depth) {
			case 16:
				colormaker = makecol_rgb565;
				break;
			case 32:
				colormaker = makecol_argb8888;
				break;
			default:
				colormaker = makecol_dummy;
				break;
		}
		for (i = 0; i < palette->num_entries; i++) {
			col = colormaker(palette->entries[i].red,
							 palette->entries[i].green,
							 palette->entries[i].blue);
			video_render_setphysicalcolor(canvas->videoconfig, i, col, (int) canvas->depth);
		}
		video_render_setphysicalcolor(canvas->videoconfig, i, col, (int)canvas->depth);
	    if (canvas->depth>8) {
			for (b = 0; b < 256; b++) {
				video_render_setrawrgb(b,
									   colormaker(b, 0, 0),
									   colormaker(0, b, 0),
									   colormaker(0, 0, b));
			}
			video_render_initraw(canvas->videoconfig);
		}
		video_render_initraw(canvas->videoconfig);
	}
	canvas->palette=palette;
	return 0;
}
extern void video_canvas_destroy(__unused struct video_canvas_s *canvas)
{
	(*canvas->env)->DeleteGlobalRef(canvas->env,canvas->bitmapobject);
}
extern void video_canvas_resize(struct video_canvas_s *canvas, __unused char resize_canvas)
{
	if (!(canvas && canvas->draw_buffer && canvas->videoconfig)) {
		return;
	}
	video_canvas_set_palette(canvas,canvas->palette);
	jobject bitmapobject=canvas->callback.create_bitmap(canvas);
	if (canvas->bitmapobject)
	{
		(*canvas->env)->DeleteGlobalRef(canvas->env,canvas->bitmapobject);
	}
	canvas->bitmapobject=(*canvas->env)->NewGlobalRef(canvas->env,bitmapobject);
	(*canvas->env)->DeleteLocalRef(canvas->env,bitmapobject);



}
extern char video_canvas_can_resize(__unused struct video_canvas_s *canvas)
{
	return (char)1;
}
static char* viciiexternalpalette;
static char* crtcexternalpalette;
#pragma clang diagnostic push
#pragma ide diagnostic ignored "ConstantFunctionResult"
static int set_externalpalette(const char* chipname, const char* newval) {
    LOGE("(1) chipname = %s", chipname);


    char* parameter = util_concat(chipname, "PaletteFile", NULL);
    LOGE("(2) chipname = %s", chipname);
    char* externalpalette = util_concat(chipname, "ExternalPalette", NULL);
    int ret;
    if (strcmp(newval, "--internal--") != 0)
    {
        ret = resources_set_string(parameter, newval);
        LOGV("resources_set_string (%s, %s) returned %d", parameter, newval, ret);
        ret = ret == 0 ? resources_set_int(externalpalette,1) : ret;
        LOGV("resources_set_int (%s, %d) returned %d", externalpalette, 1, ret);

    } else {
        ret = resources_set_int(externalpalette,0);
        LOGV("resources_set_int (%s, %d) returned %d", externalpalette, 0, ret);
    };
    lib_free(parameter);
    lib_free(externalpalette);
    return 0;
}
static int set_viciiexternalpalette(const char *val, __unused void *param) {
	if (util_string_set(&viciiexternalpalette, val)) {
		return 0;
	}
    return set_externalpalette("VICII", val);
}
static int set_crtcexternalpalette(const char *val, __unused void *param) {
    if (util_string_set(&crtcexternalpalette, val)) {
        return 0;
    }
    return set_externalpalette("Crtc", val);
}
#pragma clang diagnostic pop

static const resource_string_t resources_string[] = {
		{ "VICIIExternalPaletteFile", "--internal--", RES_EVENT_NO, NULL,
		  &viciiexternalpalette, set_viciiexternalpalette, NULL },
        { "CRTCExternalPaletteFile", "--internal--", RES_EVENT_NO, NULL,
          &crtcexternalpalette, set_crtcexternalpalette, NULL },
		RESOURCE_STRING_LIST_END
};

extern int video_arch_resources_init(void) {
	resources_register_string(resources_string);
	return 0;
}
extern void video_arch_resources_shutdown(void)
{
	if (viciiexternalpalette)
	{
		lib_free(viciiexternalpalette);
	}
    if (crtcexternalpalette)
    {
        lib_free(crtcexternalpalette);
    }

}