//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


#include <sys/time.h>
#include <string.h>
#include <stdio.h>
#include <snapshot.h>
#include <tape/tape-snapshot.h>
#include <tape.h>
#include "types.h"
#include "machine.h"
#include "lib.h"
#include "de_rainerhock_eightbitwonders_vice_ViceEmulation.h"
#include "vsync.h"
#include "messages.h"
#include "translate.h"
#include "translate_javastubs.h"
#include "interrupt.h"
#include "fmemopen.h"
#include "md5.h"
#include "interrupt.h"
#include "jnihelpers.h"
#include "logginghelpers.h"
#include "javascript.h"

typedef struct
{
	void* data;
	size_t size;
	struct timeval tv;

} memsnapshot;
typedef struct {
	char* snapshot;
	jobject runnable;
	JNIEnv* env;
} snapshot_with_useropts;

memsnapshot snapshots[10];

typedef struct {
    char* name;
    int version_major;
    int version_minor;

} version;

version versiontab[] = {
        {"C64", 1, 1},
        {"PET", 0, 0},
        {"VIC20", 2, 0}
};
static version* get_version() {
    const char* name = machine_get_name();
    for (int i = 0; i < sizeof (versiontab) / sizeof (versiontab[0]); i++) {
        if (strcmp(name, versiontab[i].name) == 0) {
            return versiontab+i;
        }
    }
    return NULL;
}
static int get_major_version() {
    version* v = get_version();
    return v ? v->version_major : INT_MIN;
}
static int get_minor_version() {
    version* v = get_version();
    return v ? v->version_minor : INT_MIN;
}


static void adjust_tape_image_dev1_type(const char* filename);
static void save_reduced_snapshot_trap(__unused WORD unused_addr, void *rawdata)
{
	snapshot_with_useropts* data = (snapshot_with_useropts*)rawdata;
	if (machine_write_snapshot(data->snapshot, 0, 0, 0) < 0) {
		postError(translate_text_java(IDS_CANNOT_WRITE_SNAPSHOT_S));
	}
	if (data->env && data->runnable) {
		callRun(data->env, data->runnable);
	}
	if (data->snapshot) {
		lib_free(data->snapshot);
	}
	lib_free(data);
	vsync_suspend_speed_eval();
}

static void adjust_tape_image_dev1_type(const char* filename) {
	snapshot_t *s;
	snapshot_module_t *m;
	uint8_t minor, major;
	int snap_type, dummy;
	s = snapshot_open(filename, &major, &minor, machine_get_name());
	if (s != NULL) {
		if (snapshot_version_is_equal(major, minor, get_major_version(), get_minor_version())) {
			m = snapshot_module_open(s, "TAPE",
									 &major, &minor);
			if (m) {
				SMR_B_INT(m, (int *) &dummy);
				SMR_B_INT(m, (int *) &snap_type);
				tape_image_dev1->type = snap_type;
				snapshot_module_close(m);
			} else {
				tape_image_detach_internal(1);
			}
		} else {
			log_error(LOG_DEFAULT, "Snapshot version (%d.%d) not valid: expecting %d.%d.", major,
					  minor, get_major_version(), get_minor_version());
		}
		snapshot_close(s);
	}
}
int load_snapshot_trap(__unused WORD unused_addr, void* filename)
{
    js_reset();
	adjust_tape_image_dev1_type(filename);
    int ret = machine_read_snapshot(filename, 0);
	if (ret !=0)
	{
		LOGE("%s, machine_read_snapshot(%s)->%d", translate_text_java(IDS_CANNOT_READ_SNAPSHOT_IMG), (char*) filename, ret);
		postError(translate_text_java(IDS_CANNOT_READ_SNAPSHOT_IMG));

	} else {
        LOGV("Successfully read snapshot %s", (char*) filename);
    }
	lib_free(filename);

    return ret;
}

static void load_snapshot_callback_trap(__unused WORD unused_addr, void* rawdata) {
	snapshot_with_useropts* data = (snapshot_with_useropts*)rawdata;
    js_reset();
	load_snapshot_trap(unused_addr, data->snapshot);
	if (data->runnable) {
		callRun(data->env, data->runnable);
	}

	lib_free(data);
}
static void load_and_delete_snapshot_trap(__unused WORD unused_addr, void* filename)
{
    js_reset();
	load_snapshot_trap(unused_addr,filename);
	remove(filename);

}



__attribute__((used)) void store_reduced_snapshot(const char* file, jobject runnable)
{
	if (file)
	{
		JNIEnv* env = getAactiveenv();
		snapshot_with_useropts* data = malloc(sizeof(snapshot_with_useropts));
		data->env = env;
		data->snapshot = lib_strdup(file);
		if (data->runnable) {
			data->runnable = (*env)->NewGlobalRef(env, runnable);
		} else {
			data->runnable = NULL;
		}

		interrupt_maincpu_trigger_trap(save_reduced_snapshot_trap, data);

	}
}
static void void_load_snapshot_trap(__unused WORD unused_addr, void* filename) {
    js_reset();
	load_snapshot_trap(unused_addr,filename);

}
__attribute__((used)) void attach_snapshot(const char* file)
{
	if (file)
	{
		interrupt_maincpu_trigger_trap(void_load_snapshot_trap, lib_strdup(file));
	}
}

__attribute__((used)) void attach_snapshot_and_callback(const char* file, jobject runnable) {
	JNIEnv* env = getAactiveenv();
	snapshot_with_useropts* data = malloc(sizeof(snapshot_with_useropts));
	data->env = env;
	data->snapshot = lib_strdup(file);
	if (data->runnable) {
		data->runnable = (*env)->NewGlobalRef(env, runnable);
	} else {
		data->runnable = NULL;
	}

	interrupt_maincpu_trigger_trap(load_snapshot_callback_trap, data);
}

__attribute__((used)) void attach_and_delete_snapshot(const char* file)
{
	if (file)
	{
		interrupt_maincpu_trigger_trap(load_and_delete_snapshot_trap, lib_strdup(file));
	}
}
__attribute__((used)) int check_snapshot(const char* file) {
    int ret = 0;
    snapshot_t *s;
    uint8_t minor, major;

    s = snapshot_open(file, &major, &minor, machine_get_name());
    if (s == NULL) {
        LOGV("check_snapshot: Cannot open %s", file);
        ret = 1;
    } else {
        if (!snapshot_version_is_equal(major, minor, get_major_version(), get_minor_version())) {
            LOGV("check_snapshot: Snapshot version (%d.%d) from %s not valid: expecting %d.%d.", major, minor, file, get_major_version(), get_minor_version());
            ret = 2;
        } else {
            LOGV("check_snapshot: Snapshot version (%d.%d) from %s valid: expecting %d.%d.", major, minor, file, get_major_version(), get_minor_version());
        }
        snapshot_close(s);
    }
    LOGV("check_snapshot: Returning %d", ret);
    return ret;

}