#include <stdint.h>
/*
 * archdep.c - Miscellaneous system-specific stuff.
 *
 * Written by
 *  Ettore Perazzoli <ettore@comm2000.it>
 *  Andreas Boose <viceteam@t-online.de>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <info.h>
#include "archdep.h"
#include "findpath.h"
#include "ioutil.h"
#include "lib.h"
#include "log.h"
#include "machine.h"
#include "platform.h"
#include "ui.h"
#include "util.h"
#include "fmemopen.h"

#include "jnihelpers.h"
#include "logginghelpers.h"
#include "lib.h"
#include "jnihelpers.h"
#include "logginghelpers.h"
#include "lib.h"
#include "machine_specific_bindings.h"
#include "timemachine.h"
#include "javascript.h"
#include "joystick.h"

extern timemachine* active_timemachine;


/* alternate storage of preferences */
const char *archdep_pref_path = NULL; /* NULL -> use home_path + ".vice" */

int archdep_network_init(void)
{
    return 0;
}
int archdep_init(int *argc, char **argv) // NOLINT(readability-non-const-parameter)
{
    int _console_mode = 0;
    int _video_disabled_mode = 0;
    JNIEnv* env = getAactiveenv();
    jmethodID mth_initvalues= (*env)->GetMethodID(env,
                                                  (*env)->GetObjectClass(env,CurrentActivity()),
                                                  "initvalues", "()V");
    (*env)->CallVoidMethod(env, CurrentActivity(), mth_initvalues);


    int i;
    for (i = 1; i < *argc; i++) {
        if ((!strcmp(argv[i], "-console")) || (!strcmp(argv[i], "--console"))) {
            _console_mode = 1;
            _video_disabled_mode = 1;
        }
    }
    set_console_mode(_console_mode);
    set_video_disabled_mode(_video_disabled_mode);
    log_verbose_init(*argc, argv);
    return 0;
}

const char *archdep_program_name(void)
{
    return lib_msprintf("8-Bit-Wonders, based on VICE");
}
/* Return a malloc'ed backup file name for file `fname'.  */
char *archdep_make_backup_filename(const char *fname)
{
    return util_concat(fname, "~", NULL);
}
static char resource_filename[1024]={0};
void set_initial_resource_filename(const char* value) {
    snprintf(resource_filename,1024,"memmap::%p:%d",value,(int)strlen(value)+1);
}
char *archdep_default_resource_file_name(void)
{
    if (resource_filename[0]) {
        return lib_strdup(resource_filename);
    } else {
        if (archdep_pref_path == NULL) {
            const char *home;

            home = archdep_home_path();
            return util_concat(home, "/.vice/vicerc", NULL);
        } else {
            return util_concat(archdep_pref_path, "/vicerc", NULL);
        }
    }
}

char *archdep_default_fliplist_file_name(void)
{
    if (archdep_pref_path == NULL) {
        const char *home;

        home = archdep_home_path();
        return util_concat(home, "/.vice/fliplist-", machine_get_name(), ".vfl", NULL);
    } else {
        return util_concat(archdep_pref_path, "/fliplist-", machine_get_name(), ".vfl", NULL);
    }
}

char *archdep_default_autostart_disk_image_file_name(void)
{
    if (archdep_pref_path == NULL) {
        const char *home;

        home = archdep_home_path();
        return util_concat(home, "/autostart-", machine_get_name(), ".d64", NULL);
    } else {
        return util_concat(archdep_pref_path, "/autostart-", machine_get_name(), ".d64", NULL);
    }
}
/*
int archdep_num_text_lines(void)
{
    char *s;

    s = getenv("LINES");
    if (s == NULL) {
        log_verbose("LINES not set.");
        return -1;
    }
    return atoi(s);
}

int archdep_num_text_columns(void)
{
    char *s;

    s = getenv("COLUMNS");
    if (s == NULL) {
        log_verbose("COLUMNS not set.");
        return -1;
    }
    return atoi(s);
}
*/

int archdep_path_is_relative(const char *path)
{
    if (path == NULL) {
        return 0;
    }
    return *path != '/'
        && strstr(path,"asset@")!=path
        && strstr(path,"content://") != path
        && strstr(path, "file://") != path
        ;
}

int archdep_spawn(__unused const char *name, __unused char **argv, __unused char **pstdout_redir, __unused const char *stderr_redir)
{
	return -1;
}

/* return malloc'd version of full pathname of orig_name */
int archdep_expand_path(char **return_path, const char *orig_name)
{
    /* Unix version.  */
    if (*orig_name == '/') {
        *return_path = lib_strdup(orig_name);
        return 0;
    } else if (strstr(orig_name, "file:///") == orig_name) {
        *return_path = lib_strdup(orig_name+strlen("file://"));
        return 0;
    } else {
        static char *cwd;

        cwd = ioutil_current_dir();
        *return_path = util_concat(cwd, "/", orig_name, NULL);
        lib_free(cwd);
    }
    return 0;
}


char *archdep_filename_parameter(const char *name)
{
    /* nothing special(?) */
    return lib_strdup(name);
}

/*
    "special" chars in *unix are:

    "'\[]()´

    tested unproblematic (no escaping):

    "'()´

    tested problematic (need escaping):

    \[]
    - if the name of a file _inside_ a .zip file contain \, [ or ], then extracting
      it will fail if they are not escaped.

    several problems on autostart remain, which are not quoting but ascii vs petscii related.
*/
char *archdep_quote_parameter(const char *name)
{
    char *a;

    a = util_subst(name, "\\", "\\\\");
    a = util_subst(a, "[", "\\[");
    a = util_subst(a, "]", "\\]");
    return a;
}
static unsigned int tmp_string_counter = 0;
static char* tempdir=NULL;
char *archdep_tmpnam(void)
{
    return lib_strdup("vice1.tmp");
}

__attribute__((unused)) void set_tempdir(char* value)
{
	LOGD("tempdir set to %s",value?value:"NULL");
    tempdir = strdup(value);

}
FILE *archdep_mkstemp_fd(char **filename, const char *mode)
{
    char *tmp;
    FILE *fd=NULL;
    LOGD("archdep_mkstemp_fd: tempdir=%s",tempdir?tempdir:"NULL");
    if (tempdir)
    {
    	tmp_string_counter++;
    	LOGD("archdep_mkstemp_fd: filename=%s/vice2.tmp", tempdir);
		tmp = lib_msprintf("%s/vice2.tmp", tempdir);

		fd = fopen(tmp, mode);

		if (fd == NULL) {
			lib_free(tmp);
			return NULL;
		}

		*filename = tmp;
    }
    return fd;
}
int archdep_mkdir(const char *pathname, int mode)
{
	return mkdir(pathname, (mode_t)mode);
}


/* set permissions of given file to rw, respecting current umask */
int archdep_fix_permissions(const char *file_name)
{
    mode_t mask = umask(0);
    umask(mask);
    return chmod(file_name, mask ^ 0666); // NOLINT(hicpp-signed-bitwise)
}

#define MEMORY_LEAK_DUMP "/memory-leaks.csv"

void archdep_shutdown(void)
{
    archdep_free_log_buffer();
    if (active_timemachine) {
        active_timemachine->shutdown(active_timemachine);
    }
#ifdef DEBUG
	if (tempdir)
	{
        char* path = malloc(strlen(tempdir)+strlen(MEMORY_LEAK_DUMP)+1);
        if (path) {
            sprintf(path, "%s%s", tempdir, MEMORY_LEAK_DUMP);
            dump_memory_leaks(path);
            free(path);
        }
		free(tempdir);
        tempdir = NULL;
	}
#endif
}

FILE *archdep_open_default_log_file(void)
{
    return NULL;
}
char *archdep_default_sysfile_pathlist(const char *emu_id)
{
	const char* val;
	char* ret=NULL;
	jstring js_emu_id;
	jclass cls = GetObjectClass(CurrentActivity());
	jmethodID mid = GetMethodID(cls, "getDefaultSysfilePathlist", "(Ljava/lang/String;)Ljava/lang/String;");
	jstring js_ret;
	LOGD("cls=%p,mid=%p",cls,mid);
	jboolean isCopy;
	if (mid) {
        js_emu_id = NewStringUTF(emu_id);
        LOGD("currentactivity returns %p", CurrentActivity());
        js_ret = (jstring) CallObjectMethod(CurrentActivity(), mid, js_emu_id);
        LOGD("CallObjectMethod returns %p", js_ret);
        val = GetStringUTFChars(js_ret, &isCopy);
        ret = strdup(val);
        if (!isCopy) {
            ReleaseStringUTFChars(js_ret, val);
        }
    }
	return ret;
}
extern void signals_init(__unused int do_core_dumps)
{
}
extern void signals_pipe_set(void)
{
}
extern void signals_pipe_unset(void)
{
}
bool archdep_is_exiting(void)
{
    return false;
}
int video_arch_cmdline_options_init(void)
{
    return 0;
}
vice_team_t core_team[]={{NULL,NULL,NULL}};
void arch_joystick_set_value_absolute(unsigned int joyport, uint8_t value) {
    if (!js_joystick_actions_check(joyport, value)) {
        joystick_set_value_absolute(joyport, value);
    }
}