//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;

import android.content.res.Resources;
import android.net.Uri;

import androidx.annotation.LayoutRes;
import androidx.annotation.StringRes;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.rainerhock.eightbitwonders.EmulationConfiguration;
import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.R;
import de.rainerhock.eightbitwonders.Useropts;

public final class PetEmulation extends ViceEmulation {
    PetEmulation(final EmulationUi ui, final EmulationConfiguration conf) {
        super(ui, conf);
    }

    @Override
    protected boolean isCartridge(final Uri uri) {
        return false;
    }

    @Override
    int getCartridgeAutodetectFlag() {
        return 0;
    }
    @Override
    protected boolean checkRequiredFiles() {
        return true;
    }
    private static final  Map<String, String> PARAMETER_MAPPING
            = new LinkedHashMap<String, String>() {{
        put("Model", "-model");
    }};
    @Override
    protected String getCmdLineParameterValue(final String key, final String val) {
        if (key.equals("Model")) {
            String[] search = getEmulationActivity().getContext().getResources()
                    .getStringArray(R.array.keys_petmodels);
            String[] data = getEmulationActivity().getContext().getResources()
                    .getStringArray(R.array.petmodels);
            for (int i = 0; i < search.length; i++) {
                if (search[i].equals(val)) {
                    return data[i];
                }
            }
        }
        return super.getCmdLineParameterValue(key, val);
    }
    @Override
    protected ViceMachineSettingsFunctions getViceMachineSettingsFunctions() {

        return new ViceMachineSettingsFunctions("petmodel_set",
                Collections.emptyList(),
                Collections.emptyList(),
                Arrays.asList(R.id.gh_petsettings, R.id.gh_crtc_settings,
                        R.id.petmodel, R.id.pet_system_settings),
                Arrays.asList(R.id.gh_petsettings, R.id.gh_crtc_settings),
                PARAMETER_MAPPING) {
            @Override
            protected String getDeviceSpecificModelDefault() {
                return "6";
            }

            @Override
            protected Map<String, String> getDeviceSpecificDefaultValues() {
                Map<String, String> ret = new LinkedHashMap<>();
                ret.put("Drive8Type", "2031");
                ret.put("CrtcFilter", "0");
                ret.put("Basic1", "1");
                ret.put("Model", "6");
                return ret;
            }
        };
    }

    @Override
    public Map<Integer, String> getJoystickports() {
        HashMap<Integer, String> ret = new LinkedHashMap<>();
        Resources res = getEmulationActivity().getContext().getResources();

        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_1));
        ret.put(ret.size() + 1, res.getString(R.string.IDS_JOYSTICK_IN_USERPORT_PORT_2));
        ret.put(KEYMAPPED_JOYSTICK_1, res.getString(R.string.map_to_keyset_1));
        ret.put(KEYMAPPED_JOYSTICK_2, res.getString(R.string.map_to_keyset_2));
        fixJoystickNames(ret);
        return ret;
    }
    @Override
    protected @LayoutRes int getCompactKeyboardLayoutId(final int model) {
        //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
        if (Arrays.asList(0, 1, 2, 3, 5, 6, 2001, 3008, 3016, 3032, 4016, 4032).contains(model)) {
            return R.layout.fragment_pet_grkeyboard;
        } else {
            return R.layout.fragment_pet_bukeyboard;
        }
    }
    protected @LayoutRes int getExactKeyboardLayoutId(final int model) {
        //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
        if (Arrays.asList(0, 1, 2, 3, 5, 6, 2001, 3008, 3016, 3032, 4016, 4032).contains(model)) {
            return R.layout.fragment_pet_grkeyboard_exact;
        } else {
            return R.layout.fragment_pet_bukeyboard_exact;
        }
    }

    private MenuFeature createSwitchProcessorFeature(final String switchTarget,
                                                     final @StringRes int menutext) {
        return new MenuFeature() {
            @Override
            public String getName() {
                return getEmulationActivity().getContext().getResources().getString(menutext);
            }

            @Override
            public int getIconResource() {
                return MenuFeature.NO_ICON;
            }

            @Override
            public Runnable getRunnable() {
                String cpu = getEmulationActivity().getCurrentUseropts()
                        .getStringValue("CPUswitch", "6502");
                if (!cpu.equals(switchTarget)) {
                    return () -> {
                        getEmulationActivity().getCurrentUseropts().setValue(
                                Useropts.Scope.CONFIGURATION, "CPUswitch", switchTarget);
                        getMachineSettingsFunction().applyUseropts(
                                getEmulationActivity().getCurrentUseropts(), null);
                        PetEmulation.this.nativeVoidFunc("machine_trigger_reset", 0);
                    };
                } else {
                    return null;
                }
            }
        };
    }
    @Override
    public AnyRunnableFunctions getResetFunctions() {
        return () -> {
            List<MenuFeature> ret = super.getResetFunctions().getFunctions();
            if ("11".equals(getEmulationActivity().getCurrentUseropts()
                    .getStringValue("Model", null))) {
                ret.add(createSwitchProcessorFeature("6502", R.string.reset_to_6502));
                ret.add(createSwitchProcessorFeature("6809", R.string.reset_to_6809));
                ret.add(createSwitchProcessorFeature("2",
                        R.string.reset_to_programmable));

            }
            return ret;
        };
    }

    private final Pattern mBusinessKeyboardPosFilePattern =
            Pattern.compile("gtk3_buuk_pos_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);

    private final Pattern mGraphicalKeyboardPosFilePattern =
            Pattern.compile("gtk3_grus_pos_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);
    private final Pattern mBusinessKeyboardSymFilePattern =
            Pattern.compile("gtk3_buuk_sym_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);

    private final Pattern mGraphicalKeyboardSymFilePattern =
            Pattern.compile("gtk3_grus_sym_[a-z]{2}\\.vkm", Pattern.CASE_INSENSITIVE);

    @Override
    protected boolean isMatchingVirtualKeyboard(final int model, final String file) {
        if (getCompactKeyboardLayoutId(model) == R.layout.fragment_pet_grkeyboard) {
            return mGraphicalKeyboardPosFilePattern.matcher(file).matches()
                    ||  mGraphicalKeyboardSymFilePattern.matcher(file).matches();
        } else {
            return mBusinessKeyboardPosFilePattern.matcher(file).matches()
                    || mBusinessKeyboardSymFilePattern.matcher(file).matches();
        }
    }

    @Override
    protected String getVirtualKeyboardName(final String file) {
        Pattern p = Pattern.compile("gtk3_...._..._", Pattern.CASE_INSENSITIVE);
        Matcher m = p.matcher(file);
         @StringRes int res;
        if (mBusinessKeyboardPosFilePattern.matcher(file).matches()
            || mGraphicalKeyboardPosFilePattern.matcher(file).matches()) {
            res = R.string.matching_position;
        } else {
            res = R.string.matching_symbol;
        }
        return getLanguageDisplayname(m.replaceAll("").replace(".vkm", "")) + ", "
                + getEmulationActivity().getContext().getString(res);
    }
    @Override
    protected boolean isMatchingKeyboardMapping(final EmulationUi.SettingSpinnerElement e,
                                                final String activeElementId) {
        if (activeElementId != null && e.getId() != null) {
            String neutralId1 = e.getId()
                    .replace("_grus_", "_?_")
                    .replace("_buuk_", "_?_");
            String neutralId2 = activeElementId
                    .replace("_grus_", "_?_")
                    .replace("_buuk_", "_?_");
            return neutralId1.equals(neutralId2);
        }
        return activeElementId == null && e.getId() == null;
    }
}
